/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef metaprogramming_H
#define metaprogramming_H

#include <mrpt/utils/CSerializable.h>

namespace mrpt
{
	namespace utils
	{
		/** A set of utility objects for metaprogramming with STL algorithms.
		  */
		namespace metaprogramming
		{
			/** An object for deleting pointers (intended for STL algorithms) */
			struct ObjectDelete
			{
				template<typename T>
				void operator()(const T *ptr)
				{
					delete ptr;
				}
			};

			/** An object for clearing an object (invokes its method "clear()") given a pointer or smart-pointer, intended for being used in STL algorithms. */
			struct ObjectClear
			{
				template<typename T>
				void operator()(T ptr)
				{
					ptr->clear();
				}
			};

			/** An object for transforming between types/classes, intended for being used in STL algorithms.
			  *  Example of usage:
			  *   \code
			  *     vector_int      v1(10);  // Input
			  *     vector_double   v2(10);  // Output
			  *     std::transform(v1.begin(),v1.end(), v2.begin(), ObjectConvert<double> );
			  *   \endcode
			  */
			template <typename TARGET_TYPE>
			struct ObjectConvert
			{
				template<typename T>
				TARGET_TYPE operator()(const T &val)
				{
					return TARGET_TYPE(val);
				}
			};

			/** An object for making smart pointers unique (ie, making copies if necessary), intended for being used in STL algorithms. */
			struct ObjectMakeUnique
			{
				void operator()(CSerializablePtr &ptr)
				{
					ptr.make_unique();
				}
			};

			/** An object for making smart pointers unique (ie, making copies if necessary), intended for being used in STL algorithms. */
			struct ObjectPairMakeUnique
			{
				template <typename T>
				void operator()(T &ptr)
				{
					ptr.first.make_unique();
					ptr.second.make_unique();
				}
			};

			/** An object for making smart pointers unique (ie, making copies if necessary), intended for being used in STL algorithms. */
			struct ObjectClearUnique
			{
				void operator()(CSerializablePtr &ptr)
				{
					ptr.clear_unique();
				}
			};

			/** An object for reading objects from a stream, intended for being used in STL algorithms. */
			struct ObjectReadFromStream
			{
			private:
				CStream		*m_stream;
			public:
				ObjectReadFromStream(CStream *stream) : m_stream(stream) {  }

				void operator()(CSerializablePtr &ptr)
				{
					(*m_stream) >> ptr;
				}
			};

		} // end metaprogramming
	} // End of namespace
} // end of namespace
#endif
