/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CFeatureExtraction_H
#define CFeatureExtraction_H

#include <mrpt/utils/CMRPTImage.h>
#include <mrpt/utils/CMRPTImageFloat.h>
#include <mrpt/utils/CTicTac.h>
#include <mrpt/vision/CImageConvolution.h>
#include <mrpt/vision/utils.h>
#include <mrpt/vision/CFeature.h>

namespace mrpt
{
	namespace vision
	{
		//class CKanadeLucasTomasi;
		class MRPTDLLIMPEXP CFeatureExtraction
		{
		//friend class CKanadeLucasTomasi;

		public:
			enum TMethodExtraction
			{
				extMethodKLT = 0,
				extMethodHarris,
				extMethodBCD,
				extMethodSIFT,
				extMethodSURF
			};

			enum TSIFTImplementation
			{
				LoweBinary = 0,
				CSBinary,
				VedaldiBinary,
				Hess
			};

			struct MRPTDLLIMPEXP TOptions
			{
				/** Initalizer
				*/
				TOptions();

				/** Method of the extraction
				*/
				TMethodExtraction method;

				/** Size of the patch to extract, or 0 if no patch is desired (default=21).
				  */
				unsigned int patchSize;

				struct MRPTDLLIMPEXP TKLTOptions
				{
					/** KLT Options
					  */
					int		radius;			// size of the block of pixels used
					float	threshold;		// for rejecting weak local maxima (with min_eig < threshold*max(eig_image))
					float	min_distance;	// minimum distance between features
				} KLTOptions;

				struct MRPTDLLIMPEXP THarrisOptions
				{
					/** Harris Options
					  */
					float	threshold;		// for rejecting weak local maxima (with min_eig < threshold*max(eig_image))
					float	k;				// k factor for the Harris algorithm
					float	sigma;			// standard deviation for the gaussian smoothing function
					int		radius;			// size of the block of pixels used
					float	min_distance;	// minimum distance between features
				} harrisOptions;

				struct MRPTDLLIMPEXP TBCDOptions
				{
					/** BCD Options
					  */
				} BCDOptions;

				struct MRPTDLLIMPEXP TSIFTOptions
				{
					/** SIFT Options
					  */
					TSIFTImplementation implementation;
				} SIFTOptions;

			} options;

			/** Constructor
			*/
			CFeatureExtraction();

			/** Virtual destructor.
			*/
			virtual ~CFeatureExtraction();

			/** Extract features from the image based on the method defined in TOptions.
			* \param img (input) The image from where to extract the images.
			* \param feats (output) A complete list of features (containing a patch for each one of them).
			* \param nDesiredFeatures (op. input) Number of features to be extracted. Default: authomatic.
			* \param ROI (op. input) Region of Interest. Default: All the image.
			*/
			void  detectFeatures(	const CMRPTImage		&img,
									CFeatureList			&feats,
									unsigned int			init_ID = 0,
									unsigned int			nDesiredFeatures = 0,
									const TImageROI			&ROI = TImageROI());

			/** Compute the SIFT descriptor of the provided features into the input image
			* \param in_img (input) The image from where to compute the descriptors.
			* \param in_features (input/output) The list of features whose descriptors are going to be computed.
			*/
			void  computeSiftDescriptors( const CMRPTImage	&in_img,
										  CFeatureList		&in_features);

			/** Extract more features from the image (apart from the provided ones) based on the method defined in TOptions .
			* \param img (input) The image from where to extract the images.
			* \param inList (input) The actual features in the image
			* \param outList (output) The list of new features (containing a patch for each one of them).
			* \param nDesiredFeatures (op. input) Number of features to be extracted. Default: authomatic.
			*/
			void  findMoreFeatures( const CMRPTImage &img,
									const CFeatureList &inList,
									CFeatureList &outList,
									unsigned int nDesiredFeats = 0);

		private:
			/** Extract features from the image based on the Harris method.
			* \param img The image from where to extract the images.
			* \param x Vector containing the feature 'x' coordinates.
			* \param y Vector containing the feature 'y' coordinates.
			* \param nDesiredFeatures Number of features to be extracted. Default: authomatic.
			* \param x_min Minimum x coordinate in the image where the extraction is performed. Default: All the image.
			* \param x_length X Length of the region of interest. Default: All the image.
			* \param y_min Minimum y coordinate in the image where the extraction is performed. Default: All the image.
			* \param y_length X Length of the region of interest. Default: All the image.
			*/
			/** /
			void  extractFeaturesHarris(
				const CMRPTImageFloat	&img,
				vector_float			&x,
				vector_float			&y,
				int						nDesiredFeatures = -1,
				unsigned int			x_min = 0,
				unsigned int			x_length = 0,
				unsigned int			y_min = 0,
				unsigned int			y_length = 0);
			/ **/

			/** Extract features from the image based on the Harris method employing the classical algorithm.
			* \param img The image from where to extract the images.
			* \param x Vector containing the feature 'x' coordinates.
			* \param y Vector containing the feature 'y' coordinates.
			* \param nDesiredFeatures Number of features to be extracted. Default: authomatic.
			* \param x_min Minimum x coordinate in the image where the extraction is performed. Default: All the image.
			* \param y_min Minimum y coordinate in the image where the extraction is performed. Default: All the image.
			*/
			/** /
			void  HarrisCommon(
				const CMRPTImageFloat	&imgR,
				vector_float			&x,
				vector_float			&y,
				int						nDesiredFeatures,
				unsigned int			x_min,
				unsigned int			y_min);
			/ **/

			/** Extract features from the image based on the Harris method and distributing the features over the image.
			* \param img The image from where to extract the images.
			* \param x Vector containing the feature 'x' coordinates.
			* \param y Vector containing the feature 'y' coordinates.
			* \param nDesiredFeatures Number of features to be extracted. Default: authomatic.
			* \param x_min Minimum x coordinate in the image where the extraction is performed. Default: All the image.
			* \param y_min Minimum y coordinate in the image where the extraction is performed. Default: All the image.
			*/
			/** /
			void  HarrisMask(
				const CMRPTImageFloat	&imgR,
				vector_float			&x,
				vector_float			&y,
				int						nDesiredFeatures,
				unsigned int			x_min,
				unsigned int			y_min);
			/ **/
			// ------------------------------------------------------------------------------------
			//											KLT
			// ------------------------------------------------------------------------------------
			/** Select good features using the openCV implementation of the KLT method.
			* \param img (input) The image from where to select extract the images.
			* \param feats (output) A complete list of features (containing a patch for each one of them).
			* \param nDesiredFeatures (op. input) Number of features to be extracted. Default: authomatic.
			* \param omitPixels (op. input) A mask for determining the ROI. (0: do not omit this pixel, 1: omit this pixel)
			*/
			void  selectGoodFeaturesKLT(
				const CMRPTImage	&inImg,
				CFeatureList		&feats,
				unsigned int		init_ID = 0,
				unsigned int		nDesiredFeatures = 0,
				void				*mask_ = NULL);

			/** Extract features from the image based on the KLT method.
			* \param img The image from where to extract the images.
			* \param x Vector containing the feature 'x' coordinates.
			* \param y Vector containing the feature 'y' coordinates.
			* \param nDesiredFeatures Number of features to be extracted. Default: authomatic.
			* \param x_min Minimum x coordinate in the image where the extraction is performed. Default: All the image.
			* \param x_length X Length of the region of interest. Default: All the image.
			* \param y_min Minimum y coordinate in the image where the extraction is performed. Default: All the image.
			* \param y_length X Length of the region of interest. Default: All the image.
			*/
			void  extractFeaturesKLT(
				const CMRPTImage		&img,
				CFeatureList			&feats,
				unsigned int			init_ID = 0,
				unsigned int			nDesiredFeatures = 0,
				unsigned int			x_min = 0,
				unsigned int			x_length = 0,
				unsigned int			y_min = 0,
				unsigned int			y_length = 0);

			/** Extract features from the image based on the KLT method.
			* \param img The image from where to extract the images.
			* \param feats The list of extracted features.
			* \param nDesiredFeatures Number of features to be extracted. Default: authomatic.
			* \param ROI (op. input) Region of Interest. Default: All the image.
			*/
			void  extractFeaturesKLT(
				const CMRPTImage		&img,
				CFeatureList			&feats,
				unsigned int			init_ID = 0,
				unsigned int			nDesiredFeatures = 0,
				const TImageROI			&ROI = TImageROI());

			// ------------------------------------------------------------------------------------
			//											BCD
			// ------------------------------------------------------------------------------------
			/** Extract features from the image based on the BCD method.
			* \param img The image from where to extract the images.
			* \param feats The list of extracted features.
			* \param nDesiredFeatures Number of features to be extracted. Default: authomatic.
			* \param ROI (op. input) Region of Interest. Default: All the image.
			*/
			void  extractFeaturesBCD(
				const CMRPTImage 		&img,
				CFeatureList			&feats,
				unsigned int			init_ID = 0,
				unsigned int			nDesiredFeatures = 0,
				const TImageROI			&ROI = TImageROI());

			// ------------------------------------------------------------------------------------
			//											SIFT
			// ------------------------------------------------------------------------------------
			/** Extract features from the image based on the SIFT method.
			* \param img The image from where to extract the images.
			* \param feats The list of extracted features.
			* \param nDesiredFeatures Number of features to be extracted. Default: authomatic.
			* \param ROI (op. input) Region of Interest. Default: All the image.
			*/
			void  extractFeaturesSIFT(
				const CMRPTImage		&img,
				CFeatureList			&feats,
				unsigned int			init_ID = 0,
				unsigned int			nDesiredFeatures = 0,
				const TImageROI			&ROI = TImageROI());

			// ------------------------------------------------------------------------------------
			//											SURF
			// ------------------------------------------------------------------------------------
			/** Extract features from the image based on the SURF method.
			* \param img The image from where to extract the images.
			* \param feats The list of extracted features.
			* \param nDesiredFeatures Number of features to be extracted. Default: authomatic.
			* \param ROI (op. input) Region of Interest. Default: All the image.
			*/
			void  extractFeaturesSURF(
				const CMRPTImage		&img,
				CFeatureList			&feats,
				unsigned int			init_ID = 0,
				unsigned int			nDesiredFeatures = 0,
				const TImageROI			&ROI = TImageROI());


			// ------------------------------------------------------------------------------------
			//								my_scale_space_extrema
			// ------------------------------------------------------------------------------------
			/** Computes extrema in the scale space.
			* \param dog_pyr Pyramid of images.
			* \param octvs Number of considered octaves.
			* \param intvls Number of intervales in octaves.
			*/
			void* my_scale_space_extrema( 
				CFeatureList &featList, void* dog_pyr, 
				int octvs, int intvls, double contr_thr, int curv_thr, 
				void* storage );

			/** Adjust scale if the image was initially doubled.
			* \param features The sequence of features.
			*/
			void	my_adjust_for_img_dbl( void* features );

			/** Gets the number of times that a point in the image is higher or lower than the surroundings in the image-scale space
			* \param dog_pyr Pyramid of images.
			* \param octvs Number of considered octaves.
			* \param intvls Number of intervales in octaves.
			* \param row The row of the feature in the original image.
			* \param col The column of the feature in the original image.
			* \param nMin [out]: Times that the feature is lower than the surroundings.
			* \param nMax [out]: Times that the feature is higher than the surroundings.
			*/
			void	getTimesExtrema( void* dog_pyr, int octvs, int intvls, float row, float col, unsigned int &nMin, unsigned int &nMax );

			/** Computes the Laplacian value of the feature in the corresponing image in the pyramid.
			* \param dog_pyr Pyramid of images.
			* \param octvs Number of considered octaves.
			* \param intvls Number of intervales in octaves.
			* \param row The row of the feature in the original image.
			* \param col The column of the feature in the original image.
			*/
			double	getLaplacianValue( void* dog_pyr, int octvs, int intvls, float row, float col );

			/** Append a sequence of openCV features into an MRPT feature list.
			* \param features The sequence of features.
			* \param list [in-out] The list of MRPT features.
			* \param init_ID [in] The initial ID for the new features.
			*/
			void	insertCvSeqInCFeatureList( void* features, CFeatureList &list, unsigned int init_ID = 0 );

			/** Converts a sequence of openCV features into an MRPT feature list.
			* \param features The sequence of features.
			* \param list [in-out] The list of MRPT features.
			* \param init_ID [in][optional] The initial ID for the features (default = 0).
			* \param ROI [in][optional] The initial ID for the features (default = empty ROI -> not used).
			*/
			void	convertCvSeqInCFeatureList( void* features, CFeatureList &list, unsigned int init_ID = 0, const TImageROI &ROI = TImageROI() );

			/** Launch an external process (for internal use only).
			* \param command [in] The char array containing the command (with parameters)
			*/
			bool	launchExternalApp( char* command );

		}; // end of class
	} // end of namespace
} // end of namespace
#endif
