/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
// ------------------------------------------------------
//  Refer to the description in the wiki:
//  http://babel.isa.uma.es/mrpt/Kalman_Filters
// ------------------------------------------------------

#include <mrpt/core.h>

using namespace mrpt;
using namespace mrpt::bayes;
using namespace mrpt::gui;
using namespace mrpt::math;
using namespace mrpt::slam;
using namespace mrpt::utils;
using namespace std;

#define BEARING_SENSOR_NOISE_STD 	DEG2RAD(15.0f)
#define RANGE_SENSOR_NOISE_STD 		0.3f
#define DELTA_TIME                  	0.1f

#define VEHICLE_INITIAL_X			4.0f
#define VEHICLE_INITIAL_Y			4.0f
#define VEHICLE_INITIAL_V           1.0f
#define VEHICLE_INITIAL_W           DEG2RAD(20.0f)

#define TRANSITION_MODEL_STD_XY 	0.03f
#define TRANSITION_MODEL_STD_VXY 	0.20f

#define NUM_PARTICLES				2000

// ------------------------------------------------------
//		Implementation of the system models as a EKF
// ------------------------------------------------------
class CRangeBearing : public CKalmanFilterCapable
{
public:
	CRangeBearing( );
	virtual ~CRangeBearing();

	void  doProcess( double DeltaTime, double observationRange, double observationBearing );

	void getState( CVectorTemplate<KFTYPE> 	&xkk, CMatrixTemplateNumeric<KFTYPE> 	&pkk)
	{
		xkk = m_xkk;
		pkk = m_pkk;
	}

 protected:

	float m_obsBearing,m_obsRange;
	float m_deltaTime;

	/** @name Virtual methods for Kalman Filter implementation
		@{
	 */

	/** Must return the action vector u.
	  * \param out_u The action vector which will be passed to OnTransitionModel
	  */
	void OnGetAction( CVectorTemplate<KFTYPE> &out_u );

	/** Implements the transition model \f$ \hat{x}_{k|k-1} = f( \hat{x}_{k-1|k-1}, u_k ) \f$
	  * \param in_u The vector returned by OnGetAction.
	  * \param inout_x At input has \f$ \hat{x}_{k-1|k-1} \f$, at output must have \f$ \hat{x}_{k|k-1} \f$.
	  * \param out_skip Set this to true if for some reason you want to skip the prediction step (to do not modify either the vector or the covariance). Default:false
	  */
	void OnTransitionModel(
		const CVectorTemplate<KFTYPE>& in_u,
		CVectorTemplate<KFTYPE>&       inout_x,
		bool &out_skipPrediction
		 );

	/** Implements the transition Jacobian \f$ \frac{\partial f}{\partial x} \f$
	  * \param out_F Must return the Jacobian.
	  *  The returned matrix must be \f$N \times N\f$ with N being either the size of the whole state vector or get_vehicle_size().
	  */
	void OnTransitionJacobian(
		CMatrixTemplateNumeric<KFTYPE> & out_F
		 );

	/** Implements the transition noise covariance \f$ Q_k \f$
	  * \param out_Q Must return the covariance matrix.
	  *  The returned matrix must be of the same size than the jacobian from OnTransitionJacobian
	  */
	void OnTransitionNoise(
		CMatrixTemplateNumeric<KFTYPE> & out_Q
		 );

	/** This is called between the KF prediction step and the update step to allow the application to employ the prior distribution of the system state in the detection of observations (data association), where applicable.
	  * \param out_z A \f$N \times O\f$ matrix, with O being get_observation_size() and N the number of "observations": how many observed landmarks for a map, or just one if not applicable.
	  * \param out_R2 A vector of size O (O being get_observation_size()) with the *variances* of the sensor noise for each of the observation components. This is constant for all the observations (where N>1), and in the naive Kalman method the value TKF_options::veryLargeR2 is used for unobserved map elements.
	  * \param out_data_association An empty vector or, where applicable, a vector where the i'th element corresponds to the position of the observation in the i'th row of out_z within the system state vector, or -1 if it is a new map element and we want to insert it at the end of this KF iteration.
	  *  This method will be called just once for each complete KF iteration.
	  */
	void OnGetObservations(
		CMatrixTemplateNumeric<KFTYPE> &out_z,
		CVectorTemplate<KFTYPE>        &out_R2,
		vector_int                     &out_data_association
		);

	/** Implements the observation prediction \f$ h_i(x) \f$ and the Jacobians \f$ \frac{\partial h_i}{\partial x} \f$ and (when applicable) \f$ \frac{\partial h_i}{\partial y_i} \f$.
	  * \param in_z This is the same matrix returned by OnGetObservations(), passed here for reference.
	  * \param in_data_association The vector returned by OnGetObservations(), passed here for reference.
	  * \param in_full If set to true, all the Jacobians and predictions must be computed at once. Otherwise, just those for the observation in_obsIdx.
	  * \param in_obsIdx If in_full=false, the row of the observation (in in_z and in_data_association) whose innovation & Jacobians are to be returned now.
	  * \param out_innov The difference between the expected observation and the real one: \f$ \tilde{y} = z - h(x) \f$. It must be a vector of length O*N (in_full=true) or O (in_full=false), with O=get_observation_size() and N=number of rows in in_z.
	  * \param out_Hx One or a vertical stack of \f$ \frac{\partial h_i}{\partial x} \f$.
	  * \param out_Hy An empty matrix, or one or a vertical stack of \f$ \frac{\partial h_i}{\partial y_i} \f$.
	  *
	  *  out_Hx must consist of 1 (in_full=false) or N (in_full=true) vertically stacked \f$O \times V\f$ matrices, and out_Hy must consist of 1 or N \f$O \times F\f$ matrices, with:
	  *  - N: number of rows in in_z (number of observed features, or 1 in general).
	  *  - O: get_observation_size()
	  *  - V: get_vehicle_size()
	  *  - F: get_feature_size()
	  */
	void OnObservationModelAndJacobians(
		const CMatrixTemplateNumeric<KFTYPE> &in_z,
		const vector_int                     &in_data_association,
		const bool                           &in_full,
		const int                            &in_obsIdx,
		CVectorTemplate<KFTYPE>       		 &out_innov,
		CMatrixTemplateNumeric<KFTYPE>       &out_Hx,
		CMatrixTemplateNumeric<KFTYPE>       &out_Hy
		);


	/** Must return the dimension of the "vehicle state": either the full state vector or the "vehicle" part if applicable.
	 */
	size_t get_vehicle_size()  const { return 4; /* x y vx vy*/ }

	// get_feature_size Is not implemented since this is not a "mapping" problem. Default method will return 0.

	/** Must return the dimension of each observation (eg, 2 for pixel coordinates, 3 for 3D coordinates,etc).
	 */
	size_t get_observation_size()  const { return 2; /* range+bearing */ }

	/** @}
	 */
};


// ---------------------------------------------------------------
//		Implementation of the system models as a Particle Filter
// ---------------------------------------------------------------
struct CParticleVehicleData
{
	float x,y, vx,vy; // Vehicle state (position & velocities)
};

class CRangeBearingParticleFilter : public mrpt::bayes::CParticleFilterCapable, public mrpt::bayes::CParticleFilterData<CParticleVehicleData>
{
	// This uses CParticleFilterData to implement some methods required for CParticleFilterCapable:
	IMPLEMENT_PARTICLE_FILTER_CAPABLE(CParticleVehicleData);

public:

	 /** Update the m_particles, predicting the posterior of robot pose and map after a movement command.
	  *  This method has additional configuration parameters in "options".
	  *  Performs the update stage of the RBPF, using the sensed Sensorial Frame:
	  *
	  *   \param action This is a pointer to CActionCollection, containing the pose change the robot has been commanded.
	  *   \param observation This must be a pointer to a CSensoryFrame object, with robot sensed observations.
	  *
	  * \sa options
	  */
	void  prediction_and_update_pfStandardProposal(
		const mrpt::slam::CActionCollection	* action,
		const mrpt::slam::CSensoryFrame		* observation,
		const bayes::CParticleFilter::TParticleFilterOptions &PF_options );


	void initializeParticles(size_t  numParticles);

	/** Computes the average velocity & position
   	  */
	void getMean( float &x, float &y, float &vx, float &vy );

};



// ------------------------------------------------------
//				TestBayesianTracking
// ------------------------------------------------------
void TestBayesianTracking()
{
	mrpt::random::Randomize();

	CDisplayWindowPlots		winEKF("Tracking - Extended Kalman Filter",450,400);
	CDisplayWindowPlots		winPF("Tracking - Particle Filter",450,400);

	winEKF.setPos(10,10);
	winPF.setPos(480,10);

	winEKF.axis(-2,20,-10,10); winEKF.axis_equal();
	winPF.axis(-2,20,-10,10);  winPF.axis_equal();

	// Create EKF
	// ----------------------
	CRangeBearing 	EKF;
	EKF.KF_options.verbose = true;
//	EKF.KF_options.method = kfEKFNaive;
	EKF.KF_options.method = kfEKFAlaDavison;
//	EKF.KF_options.method = kfIKFFull;
//	EKF.KF_options.IKF_iterations = 5;

	// Create PF
	// ----------------------
	CParticleFilter::TParticleFilterOptions	PF_options;
	PF_options.adaptiveSampleSize = false;
	PF_options.PF_algorithm = CParticleFilter::pfStandardProposal;
	PF_options.resamplingMethod = CParticleFilter::prSystematic;

	CRangeBearingParticleFilter  particles;
	particles.initializeParticles(NUM_PARTICLES);
	CParticleFilter	PF;
	PF.m_options = PF_options;


	// Init. simulation:
	// -------------------------
	float x=VEHICLE_INITIAL_X,y=VEHICLE_INITIAL_Y,phi=DEG2RAD(-180),v=VEHICLE_INITIAL_V,w=VEHICLE_INITIAL_W;
	float  t=0;

	while (winEKF.isOpen() && winPF.isOpen() && !mrpt::system::os::kbhit() )
	{
		// Update vehicle:
		x+=v*DELTA_TIME*(cos(phi)-sin(phi));
		y+=v*DELTA_TIME*(sin(phi)+cos(phi));
		phi+=w*DELTA_TIME;

		v+=1.0f*DELTA_TIME*cos(t);
		w-=0.1f*DELTA_TIME*sin(t);


		// Simulate noisy observation:
		float realBearing = atan2( y,x );
		float obsBearing = realBearing  + BEARING_SENSOR_NOISE_STD * mrpt::random::normalizedGaussian();
		printf("Real/Simulated bearing: %.03f / %.03f deg\n", RAD2DEG(realBearing), RAD2DEG(obsBearing) );

		float realRange = sqrt(square(x)+square(y));
		float obsRange = max(0.0, realRange  + RANGE_SENSOR_NOISE_STD * mrpt::random::normalizedGaussian() );
		printf("Real/Simulated range: %.03f / %.03f \n", realRange, obsRange );

		// Process with EKF:
		EKF.doProcess(DELTA_TIME,obsRange, obsBearing);

		// Process with PF:
		CSensoryFrame SF;
		CObservationBearingRangePtr obsRangeBear = CObservationBearingRangePtr( new CObservationBearingRange() );
		obsRangeBear->sensedData.resize(1);
		obsRangeBear->sensedData[0].range = obsRange;
		obsRangeBear->sensedData[0].yaw   = obsBearing;
		SF.insert( obsRangeBear );  // memory freed by SF.
		PF.executeOn(particles, NULL,&SF);  // Process in the PF

		// Show EKF state:
		CVectorTemplate<KFTYPE>			EKF_xkk;
		CMatrixTemplateNumeric<KFTYPE> 	EKF_pkk;

		EKF.getState( EKF_xkk, EKF_pkk );

		printf("Real: x:%.03f  y=%.03f heading=%.03f v=%.03f w=%.03f\n",x,y,phi,v,w);
		cout << "EKF: " << EKF_xkk << endl;

		// Show PF state:
		cout << "Particle filter ESS: " << particles.ESS() << endl;

		// Draw EKF state:
		CMatrixTemplateNumeric<float>   COVXY(2,2);
		COVXY(0,0) = EKF_pkk(0,0);
		COVXY(1,1) = EKF_pkk(1,1);
		COVXY(0,1) = COVXY(1,0) = EKF_pkk(0,1);

		winEKF.plotEllipse( EKF_xkk[0], EKF_xkk[1], COVXY, 3, "b-2", "ellipse_EKF" );

		// Draw the velocity vector:
		vector_float vx(2),vy(2);
		vx[0] = EKF_xkk[0];  vx[1] = vx[0] + EKF_xkk[2] * 1;
		vy[0] = EKF_xkk[1];  vy[1] = vy[0] + EKF_xkk[3] * 1;
		winEKF.plot( vx,vy, "g-4", "velocityEKF" );


		// Draw PF state:
		{
			size_t i,N = particles.m_particles.size();
			vector_float   parts_x(N),parts_y(N);
			for (i=0;i<N;i++)
			{
				parts_x[i] = particles.m_particles[i].d->x;
				parts_y[i] = particles.m_particles[i].d->y;
			}

			winPF.plot( parts_x, parts_y, "b.2", "particles" );

			// Draw PF velocities:
			float avrg_x, avrg_y, avrg_vx,avrg_vy;

			particles.getMean(avrg_x, avrg_y, avrg_vx,avrg_vy);

			vector_float vx(2),vy(2);
			vx[0] = avrg_x;  vx[1] = vx[0] + avrg_vx * 1;
			vy[0] = avrg_y;  vy[1] = vy[0] + avrg_vy * 1;
			winPF.plot( vx,vy, "g-4", "velocityPF" );
		}

		// Draw GT:
		winEKF.plot( vector_float(1,x), vector_float(1,y),"k.8","plot_GT");
		winPF.plot( vector_float(1,x), vector_float(1,y),"k.8","plot_GT");


		// Draw noisy observations:
		vector_float  obs_x(2),obs_y(2);
		obs_x[0] = obs_y[0] = 0;
		obs_x[1] = obsRange * cos( obsBearing );
		obs_y[1] = obsRange * sin( obsBearing );

		winEKF.plot(obs_x,obs_y,"r", "plot_obs_ray");
		winPF.plot(obs_x,obs_y,"r", "plot_obs_ray");


		// Delay:
		mrpt::system::sleep((int)(DELTA_TIME*1000));
		t+=DELTA_TIME;
	}
}


// ------------------------------------------------------
//						MAIN
// ------------------------------------------------------
int main()
{
	try
	{
		TestBayesianTracking();
		return 0;
	} catch (std::exception &e)
	{
		std::cout << "MRPT exception caught: " << e.what() << std::endl;
		return -1;
	}
	catch (...)
	{
		printf("Untyped exception!!");
		return -1;
	}
}




CRangeBearing::CRangeBearing()
{
	//KF_options.method = kfEKFNaive;
	KF_options.method = kfEKFAlaDavison;

	// INIT KF STATE
	m_xkk.resize(4,0);	// State: (x,y,heading,v,w)
	m_xkk[0]= VEHICLE_INITIAL_X;
	m_xkk[1]= VEHICLE_INITIAL_Y;
	m_xkk[2]=-VEHICLE_INITIAL_V;
	m_xkk[3]=0;

	// Initial cov:  Large uncertainty
	m_pkk.setSize(4,4);
	m_pkk.unit();
	m_pkk(0,0)=
	m_pkk(1,1)= square( 5.0f );
	m_pkk(2,2)=
	m_pkk(3,3)= square( 1.0f );
}

CRangeBearing::~CRangeBearing()
{

}


void  CRangeBearing::doProcess( double DeltaTime, double observationRange, double observationBearing )
{
	m_deltaTime = (float)DeltaTime;
	m_obsBearing = (float)observationBearing;
	m_obsRange = (float) observationRange;

	runOneKalmanIteration();
}


/** Must return the action vector u.
  * \param out_u The action vector which will be passed to OnTransitionModel
  */
void CRangeBearing::OnGetAction( CVectorTemplate<KFTYPE> &u )
{
	u.resize(1);
	u[0] = m_deltaTime;
}

/** Implements the transition model \f$ \hat{x}_{k|k-1} = f( \hat{x}_{k-1|k-1}, u_k ) \f$
  * \param in_u The vector returned by OnGetAction.
  * \param inout_x At input has \f$ \hat{x}_{k-1|k-1} \f$, at output must have \f$ \hat{x}_{k|k-1} \f$.
  * \param out_skip Set this to true if for some reason you want to skip the prediction step (to do not modify either the vector or the covariance). Default:false
  */
void CRangeBearing::OnTransitionModel(
	const CVectorTemplate<KFTYPE>& in_u,
	CVectorTemplate<KFTYPE>&       inout_x,
	bool &out_skipPrediction
	 )
{
	// in_u[0] : Delta time
	// in_out_x: [0]:x  [1]:y  [2]:vx  [3]: vy
	inout_x[0] += in_u[0] * inout_x[2];
	inout_x[1] += in_u[0] * inout_x[3];

}

/** Implements the transition Jacobian \f$ \frac{\partial f}{\partial x} \f$
  * \param out_F Must return the Jacobian.
  *  The returned matrix must be \f$N \times N\f$ with N being either the size of the whole state vector or get_vehicle_size().
  */
void CRangeBearing::OnTransitionJacobian(
	CMatrixTemplateNumeric<KFTYPE> & F
	 )
{
	F.unit(4);

	F(0,2) = m_deltaTime;
	F(1,3) = m_deltaTime;
}

/** Implements the transition noise covariance \f$ Q_k \f$
  * \param out_Q Must return the covariance matrix.
  *  The returned matrix must be of the same size than the jacobian from OnTransitionJacobian
  */
void CRangeBearing::OnTransitionNoise(
	CMatrixTemplateNumeric<KFTYPE> & Q
	 )
{
	Q.unit(4);
	Q(0,0) =
	Q(1,1) = square( TRANSITION_MODEL_STD_XY );
	Q(2,2) =
	Q(3,3) = square( TRANSITION_MODEL_STD_VXY );
}

/** This is called between the KF prediction step and the update step to allow the application to employ the prior distribution of the system state in the detection of observations (data association), where applicable.
  * \param out_z A \f$N \times O\f$ matrix, with O being get_observation_size() and N the number of "observations": how many observed landmarks for a map, or just one if not applicable.
  * \param out_R2 A vector of size O (O being get_observation_size()) with the *variances* of the sensor noise for each of the observation components. This is constant for all the observations (where N>1), and in the naive Kalman method the value TKF_options::veryLargeR2 is used for unobserved map elements.
  * \param out_data_association An empty vector or, where applicable, a vector where the i'th element corresponds to the position of the observation in the i'th row of out_z within the system state vector, or -1 if it is a new map element and we want to insert it at the end of this KF iteration.
  *  This method will be called just once for each complete KF iteration.
  */
void CRangeBearing::OnGetObservations(
	CMatrixTemplateNumeric<KFTYPE> &out_z,
	CVectorTemplate<KFTYPE>        &out_R2,
	vector_int                     &out_data_association
	)
{
	out_z.setSize(1,2);
	out_z(0,0) = m_obsBearing;
	out_z(0,1) = m_obsRange;

	out_R2.resize(2);
	out_R2[0] = square( BEARING_SENSOR_NOISE_STD );
	out_R2[1] = square( RANGE_SENSOR_NOISE_STD );

	out_data_association.clear(); // Not used
}

/** Implements the observation prediction \f$ h_i(x) \f$ and the Jacobians \f$ \frac{\partial h_i}{\partial x} \f$ and (when applicable) \f$ \frac{\partial h_i}{\partial y_i} \f$.
  * \param in_z This is the same matrix returned by OnGetObservations(), passed here for reference.
  * \param in_data_association The vector returned by OnGetObservations(), passed here for reference.
  * \param in_full If set to true, all the Jacobians and predictions must be computed at once. Otherwise, just those for the observation in_obsIdx.
  * \param in_obsIdx If in_full=false, the row of the observation (in in_z and in_data_association) whose innovation & Jacobians are to be returned now.
  * \param out_innov The difference between the expected observation and the real one: \f$ \tilde{y} = z - h(x) \f$. It must be a vector of length O*N (in_full=true) or O (in_full=false), with O=get_observation_size() and N=number of rows in in_z.
  * \param out_Hx One or a vertical stack of \f$ \frac{\partial h_i}{\partial x} \f$.
  * \param out_Hy An empty matrix, or one or a vertical stack of \f$ \frac{\partial h_i}{\partial y_i} \f$.
  *
  *  out_Hx must consist of 1 (in_full=false) or N (in_full=true) vertically stacked \f$O \times V\f$ matrices, and out_Hy must consist of 1 or N \f$O \times F\f$ matrices, with:
  *  - N: number of rows in in_z (number of observed features, or 1 in general).
  *  - O: get_observation_size()
  *  - V: get_vehicle_size()
  *  - F: get_feature_size()
  */
void CRangeBearing::OnObservationModelAndJacobians(
	const CMatrixTemplateNumeric<KFTYPE> &in_z,
	const vector_int                     &in_data_association,
	const bool                           &in_full,
	const int                            &in_obsIdx,
	CVectorTemplate<KFTYPE>       		 &ytilde,
	CMatrixTemplateNumeric<KFTYPE>       &out_Hx,
	CMatrixTemplateNumeric<KFTYPE>       &out_Hy
	)
{
	// predicted bearing:
	KFTYPE x = m_xkk[0];
	KFTYPE y = m_xkk[1];

	KFTYPE h_bear = atan2(y,x);
	KFTYPE h_range = sqrt(square(x)+square(y));

	ytilde.resize(2);
	ytilde[0] =  math::wrapToPi( in_z(0,0) - h_bear );
	ytilde[1] =  in_z(0,1) - h_range;

	out_Hx.setSize(2,4);
	out_Hx.zeros();
	out_Hx(0,0) = -y/(square(x)+square(y));
	out_Hx(0,1) = 1/(x*(1+square(y/x)));

	out_Hx(1,0) = x/sqrt(square(x)+square(y));
	out_Hx(1,1) = y/sqrt(square(x)+square(y));

	out_Hy.setSize(0,0); // Not used
}



/** Update the m_particles, predicting the posterior of robot pose and map after a movement command.
*  This method has additional configuration parameters in "options".
*  Performs the update stage of the RBPF, using the sensed Sensorial Frame:
*
*   \param action This is a pointer to CActionCollection, containing the pose change the robot has been commanded.
*   \param observation This must be a pointer to a CSensoryFrame object, with robot sensed observations.
*
* \sa options
*/
void  CRangeBearingParticleFilter::prediction_and_update_pfStandardProposal(
		const mrpt::slam::CActionCollection	* action,
		const mrpt::slam::CSensoryFrame		* observation,
		const bayes::CParticleFilter::TParticleFilterOptions &PF_options )
{
	size_t i,N = m_particles.size();

	// Transition model:
	for (i=0;i<N;i++)
	{
		m_particles[i].d->x += DELTA_TIME*m_particles[i].d->vx + TRANSITION_MODEL_STD_XY * mrpt::random::normalizedGaussian();
		m_particles[i].d->y += DELTA_TIME*m_particles[i].d->vy + TRANSITION_MODEL_STD_XY * mrpt::random::normalizedGaussian();

		m_particles[i].d->vx += TRANSITION_MODEL_STD_VXY * mrpt::random::normalizedGaussian();
		m_particles[i].d->vy += TRANSITION_MODEL_STD_VXY * mrpt::random::normalizedGaussian();
	}

	CObservationBearingRangePtr obs = observation->getObservationByClass<CObservationBearingRange>();
	ASSERT_(obs);
	ASSERT_(obs->sensedData.size()==1);
	float obsRange = obs->sensedData[0].range;
	float obsBearing = obs->sensedData[0].yaw;

	// Update weights
	for (i=0;i<N;i++)
	{
		float predicted_range   = sqrt( square(m_particles[i].d->x)+square(m_particles[i].d->y));
		float predicted_bearing = atan2( m_particles[i].d->y, m_particles[i].d->x );

		m_particles[i].log_w +=
			log( math::normalPDF( predicted_range-obsRange, 0, RANGE_SENSOR_NOISE_STD ) ) +
			log( math::normalPDF( math::wrapToPi( predicted_bearing-obsBearing), 0, BEARING_SENSOR_NOISE_STD ) );
	}

	// Resample is automatically performed by CParticleFilter when required.
}


void  CRangeBearingParticleFilter::initializeParticles(size_t  M)
{
	CParticleList::iterator		it;

	clearParticles();
	m_particles.resize(M);
	for (it=m_particles.begin();it!=m_particles.end();it++)
		it->d = new CParticleVehicleData();

	for (it=m_particles.begin();it!=m_particles.end();it++)
	{
		(*it).d->x  = mrpt::random::RandomUni( VEHICLE_INITIAL_X - 2.0f, VEHICLE_INITIAL_X + 2.0f );
		(*it).d->y  = mrpt::random::RandomUni( VEHICLE_INITIAL_Y - 2.0f, VEHICLE_INITIAL_Y + 2.0f );

		(*it).d->vx = mrpt::random::RandomNormal( -VEHICLE_INITIAL_V, 0.2f );
		(*it).d->vy = mrpt::random::RandomNormal( 0, 0.2f );

		it->log_w	= 0;
	}

}

/** Computes the average velocity
  */
void CRangeBearingParticleFilter::getMean( float &x, float &y, float &vx, float &vy )
{
	CParticleList::iterator		it;

	double sumW=0;
	for (it=m_particles.begin();it!=m_particles.end();it++)
		sumW+=exp( it->log_w );

	ASSERT_(sumW>0)

	x = y = vx = vy = 0;

	for (it=m_particles.begin();it!=m_particles.end();it++)
	{
		double w = exp(it->log_w) / sumW;

		x += (float)w * (*it).d->x;
		y += (float)w * (*it).d->y;
		vx+= (float)w * (*it).d->vx;
		vy+= (float)w * (*it).d->vy;

		it->log_w	= 0;
	}
}

