/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 


#include <mrpt/bayes/CKalmanFilterCapable.h>
#include <mrpt/system/os.h>
#include <mrpt/utils/CTicTac.h>
#include <mrpt/utils/CFileOutputStream.h>

using namespace mrpt::bayes;
using namespace mrpt::utils;
using namespace mrpt::math;
using namespace std;

/*---------------------------------------------------------------
			Constructor
  ---------------------------------------------------------------*/
CKalmanFilterCapable::CKalmanFilterCapable()
{
}

/*---------------------------------------------------------------
			Destructor
  ---------------------------------------------------------------*/
CKalmanFilterCapable::~CKalmanFilterCapable()
{
}


/*---------------------------------------------------------------
			runOneKalmanIteration
  ---------------------------------------------------------------*/
void CKalmanFilterCapable::runOneKalmanIteration()
{
	MRPT_TRY_START

	// Size of vectors, jacobians,etc..:
	const size_t vehicle_size   = get_vehicle_size();
	const size_t feature_size   = get_feature_size();
	const size_t obs_size       = get_observation_size();

	// Start of vehicle & landmarks within the joint state vector/matrix:
	const size_t vehicle_offset = get_vehicle_state_offset();
	const size_t feature_offset = get_feature_state_offset();

	// =============================================================
	//  1. CREATE ACTION MATRIX u FROM ODOMETRY
	// =============================================================
	CVectorTemplate<KFTYPE>  u;

	OnGetAction(u);

	static utils::CTicTac  stopwatch;
	double tim_kalman_prediction=0,tim_kalman_update=0,tim_generate_observations=0, tim_kalman_map_update=0;

	if (KF_options.verbose)
		stopwatch.Tic();

	// =============================================================
	//  2. PREDICTION OF NEW POSE xv_{k+1|k}
	// =============================================================
	size_t i,num_feats_in_map = 0;

	CVectorTemplate<KFTYPE>	xv( vehicle_size ); // Vehicle pose
	for (i=0;i<vehicle_size;i++)
		xv[i]=m_xkk[vehicle_offset+i];


	if (feature_size)
	{
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG)
		ASSERT_( (m_xkk.size() - vehicle_size) % feature_size == 0 );
#endif
		num_feats_in_map = (m_xkk.size() - vehicle_size) / feature_size;
	}

	bool skipPrediction=false; // Wether to skip the prediction step (in SLAM this is desired for the first iteration...)

	// Update mean: xv will have the updated pose until we update it in the filterState later.
	//  This is to maintain a copy of the last robot pose in the state vector, required for the Jacobian computation.
	OnTransitionModel(u, xv, skipPrediction);

	if ( !skipPrediction )
	{
		// =============================================================
		//  3. PREDICTION OF COVARIANCE P_{k+1|k}
		//  Details in the notebook --> technical report!!!
		// =============================================================
		CMatrixTemplateNumeric<KFTYPE> dfv_dxv,Q;

		//First we compute de Jacobian fv_by_xv  (derivative of f_vehicle wrt x_vehicle):
		OnTransitionJacobian(dfv_dxv);

		//Q is the process noise covariance matrix, is associated to the robot movement and is necesary to calculate the prediction P(k+1|k)
		OnTransitionNoise(Q);

		// ====================================
		//  3.1:  Pxx submatrix
		// ====================================
		CMatrixTemplateNumeric<KFTYPE> Pkk_new(vehicle_size,vehicle_size);

		// Perform a the multiplication over a sub-matrix of Pkk:
		dfv_dxv.multiplyByMatrixAndByTranspose(
			m_pkk,
			Pkk_new, 			// Output here
			true, 				// Perform submatrix operation,
			vehicle_offset      // Starting point of the covariance submatrix
		);
		//Pkk_new = dfv_dxv * m_pkk.extractMatrix(0,0,vehicle_size,vehicle_size) * ~dfv_dxv;
		Pkk_new += Q; // Add transition uncertainty

		//cout << "Pkk_new: " << Pkk_new;

		// Replace old covariance:
		m_pkk.insertMatrix(vehicle_offset,vehicle_offset, Pkk_new );

		// ====================================
		//  3.2:  All Pxy_i
		// ====================================
		// Now, update the cov. of landmarks, if any:
		CMatrixTemplateNumeric<KFTYPE> aux;
		for (i=0 ; i<num_feats_in_map ; i++)
		{
			dfv_dxv.multiplySubMatrix(
				m_pkk,       // F_parcial * Pkk * ~F_parcial
				aux,            // Output
				feature_offset+i*feature_size,   // Offset col
				vehicle_offset,                  // Offset row
				feature_size                     // Number of columns desired in output
			);

			m_pkk.insertMatrix         (vehicle_offset,                feature_offset+i*feature_size,  aux );
			m_pkk.insertMatrixTranspose(feature_offset+i*feature_size, vehicle_offset               ,  aux );
		}

		// =============================================================
		//  4. NOW WE CAN OVERWRITE THE NEW STATE VECTOR
		// =============================================================
		for (size_t i=0;i<get_vehicle_size();i++)
			m_xkk[vehicle_offset+i]=xv[i];

		// Normalize, if neccesary.
		OnNormalizeStateVector();

	} // end of "if" update vehicle state?


	if (KF_options.verbose)
	{
		tim_kalman_prediction=stopwatch.Tac();
		stopwatch.Tic();
	}

	// =============================================================
	//  5. PREDICTION OF OBSERVATIONS (h) AND COMPUTE JACOBIANS
	// =============================================================
	// (Inside the switch below)

	// Sensor uncertainty (covariance matrix): R
	//CVectorTemplate<KFTYPE>          R2(obs_size);
	CMatrixTemplateNumeric<KFTYPE>     R;

	// Each row is one observation:
	CMatrixTemplateNumeric<KFTYPE>   Z;
	vector_int    					 data_association;  // -1: New map feature.>=0: Indexes in the state vector

	OnGetObservations( Z, R, data_association );

#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG)
	if (Z.getRowCount()>0)
	{
		ASSERT_( Z.getColCount() == obs_size );

		ASSERT_( R.getColCount() == obs_size );
		ASSERT_( R.getRowCount() == obs_size*Z.getRowCount() );
	}
#endif


	if (KF_options.verbose)
	{
		tim_generate_observations =stopwatch.Tac();
		stopwatch.Tic();
	}

	// =============================================================
	//  6. COMPUTE KALMAN INNOVATION MATRIX "S"
	// =============================================================
	// Inside the switch below...

	// =============================================================
	//  7. UPDATE USING THE KALMAN GAIN
	// =============================================================
	// Update, only if there are observations!
	if ( Z.getRowCount()>0 )
	{
		static CTicTac  timer2;
		timer2.Tic();

		switch (KF_options.method)
		{
			// -----------------------
			//  FULL KF- METHOD
			// -----------------------
		case kfEKFNaive:
		case kfIKFFull:
		{
			// Build the whole Jacobian dh_dx matrix
			// ---------------------------------------------
			//MEASURE PREDICTION
			CMatrixTemplateNumeric<KFTYPE> h,Hx,Hy;
			CVectorTemplate<KFTYPE>		   ytilde;
			//size_t                       nObs = Z.getRowCount();

			// Just one, or several update iterations??
			size_t nKF_iterations = KF_options.method==kfEKFNaive ?  1 : KF_options.IKF_iterations;

			CVectorTemplate<KFTYPE>    xkk_0 = m_xkk;

			// For each IKF iteration (or 1 for EKF)
			for (size_t IKF_iteration=0;IKF_iteration<nKF_iterations;IKF_iteration++)
			{
				// For IKF:
				if (IKF_iteration>0)
				{

				}

				// h:  Lx3
				// Hx: 3Lx6
				// Hy: 3Lx3
				// function_h(h,Hx,Hy);
				OnObservationModelAndJacobians(
					Z,
					data_association,
					true, // FULL JACOBIANS
					0,    // Unused in this case
					ytilde,
					Hx,
					Hy );

#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG)
				if (feature_size>0)
					ASSERT_(ytilde.size()==num_feats_in_map*obs_size)
					else ASSERT_(ytilde.size()==obs_size)

						ASSERT_(Hx.getColCount()==vehicle_size)

						if (feature_size>0)
							ASSERT_(Hx.getRowCount()==num_feats_in_map*obs_size)
							else ASSERT_(Hx.getRowCount()==obs_size)

								if (feature_size>0)
								{
									ASSERT_(Hy.getColCount()==feature_size)
									ASSERT_(Hy.getRowCount()==num_feats_in_map*obs_size)
								}
#endif

				// Build the whole jacobian "dh_dx":
				CMatrixTemplateNumeric<KFTYPE> dh_dx( Hx.getRowCount(), vehicle_size + feature_size * num_feats_in_map );
				dh_dx.zeros();

				// Copy the "dhi_xv" elements:
				dh_dx.insertMatrix( 0, vehicle_offset, Hx );

				// Copy the "dhi_yi" elements:
				for (i=0;i<num_feats_in_map;i++)
				{
					CMatrixTemplateNumeric<KFTYPE> aux(obs_size,feature_size);
					Hy.extractMatrix(i*obs_size,0, aux);
					dh_dx.insertMatrix( obs_size*i,feature_offset+obs_size*i , aux );
				}

				// Compute S: slow, exact way
				// ------------------------------------
				CMatrixTemplateNumeric<KFTYPE>        S;
				// S = H P ~H + R
				m_pkk.force_symmetry();
				dh_dx.multiplyByMatrixAndByTranspose(m_pkk,S);
				//S = dh_dx * m_pkk *  (~dh_dx);

				if ( feature_size>0 )
				{
					for (i=0;i<num_feats_in_map;i++)
					{
						// Has been this landmark observed now??
						// Look for any entry in "data_association" with the value "i":
						bool 					found = false;
						size_t  				index_in_z;
						vector_int::const_iterator it;
						for (it=data_association.begin(),index_in_z=0;it!=data_association.end();it++,index_in_z++)
						{
							if (*it==(int)i)
							{
								found=true;
								break;  // index_in_z -> position in "data_association"
							}
						}

						size_t  obs_idx_off = i*obs_size;

						if (found)
						{
							for (size_t j=0;j<obs_size;j++)
								for (size_t k=0;k<obs_size;k++)
									S(obs_idx_off+j,obs_idx_off+k) += R(j,k);
						}
						else
						{
							for (size_t j=0;j<obs_size;j++)
								S(obs_idx_off+j,obs_idx_off+j) += (KFTYPE) KF_options.veryLargeR2;
						}
					}
				}
				else
				{
					// Not a SLAM-like EKF problem:
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG)
					ASSERT_(R.getColCount() == S.getColCount() );
#endif
					S+=R;
				}

				// Compute the full K matrix:
				// ------------------------------
				CMatrixTemplateNumeric<KFTYPE> K(m_pkk.getRowCount(), S.getColCount() );

				// K = m_pkk * (~dh_dx) * S.inv() );
				K.multiply_ABt(m_pkk, dh_dx);
				CMatrixTemplateNumeric<KFTYPE> S_1( S.getRowCount(), S.getColCount() );
				S.inv_fast(S_1);
				K.multiply( K, S_1 );

				// Use the full K matrix to update the mean:
				if (nKF_iterations==1)
				{
					m_xkk += K * ytilde;
				}
				else
				{
					CMatrixTemplateNumeric<KFTYPE>  HAx( dh_dx * ( xkk_0 - m_xkk ) );
					CVectorTemplate<KFTYPE>  HAx_column;
					HAx.extractCol(0, HAx_column );

					m_xkk = xkk_0 ;
					m_xkk += K * ( ytilde  - HAx_column );
				}

				// Update the covariance just at the end of iterations:
				if (IKF_iteration == (nKF_iterations-1) )
				{
					// Use the full K matrix to update the covariance:
					CMatrixTemplateNumeric<KFTYPE> I(m_pkk.getColCount(),m_pkk.getColCount());
					I.unit();

					//m_pkk = (I - K*dh_dx ) * m_pkk;
					CMatrixTemplateNumeric<KFTYPE>	aux_K_dh_dx;
					aux_K_dh_dx.multiply(K,dh_dx);
					I -= aux_K_dh_dx;
					m_pkk.multiply_result_is_symmetric( I, m_pkk );
				}
			} // end for each IKF iteration


		}
		break;

		// --------------------------------------------------------------------
		// - EKF 'a la' Davison: One observation element at once
		// --------------------------------------------------------------------
		case kfEKFAlaDavison:
		{
			// m_IDs[id] -> index in m_xkk & Pkk
			CMatrixTemplateNumeric<KFTYPE> h,Hx,Hy;

			// Just one, or several update iterations??
			size_t nKF_iterations = 1; //KF_options.method==kfEKFAlaDavison ?  1 : KF_options.IKF_iterations;

			// To avoid wasting time, if we are doing 1 iteration only, do not reserve memory for this matrix:
			CMatrixTemplateNumeric<KFTYPE>		*saved_Pkk=NULL;
			if (nKF_iterations>1)
			{
				// Create a copy of Pkk for later restoring it at the beginning of each iteration:
				saved_Pkk = new CMatrixTemplateNumeric<KFTYPE>( m_pkk );
			}

			CVectorTemplate<KFTYPE>				xkk_0 = m_xkk; // First state vector at the beginning of IKF
			CVectorTemplate<KFTYPE>				xkk_next_iter = m_xkk;  // for IKF only


			// For each IKF iteration (or 1 for EKF)
			for (size_t IKF_iteration=0;IKF_iteration<nKF_iterations;IKF_iteration++)
			{
				// Restore initial covariance matrix.
				// The updated mean is stored in m_xkk and is improved with each estimation,
				//  and so do the Jacobians which use that improving mean.
				if (IKF_iteration>0)
				{
					m_pkk = *saved_Pkk;
					//m_xkk = xkk_saved;

					xkk_next_iter = xkk_0;
				}

				// For each observed landmark/whole system state:
				for (size_t obsIdx=0;obsIdx<Z.getRowCount();obsIdx++)
				{
					// Known & mapped landmark?
					bool   doit;
					size_t idxInTheFilter=0;

					if (data_association.empty())
					{
						doit = true;
					}
					else
					{
						doit = data_association[obsIdx] >= 0;
						if (doit)
							idxInTheFilter = data_association[obsIdx];
					}

					if ( doit )
					{
						// Already mapped: OK
						const size_t idx_off      = feature_offset + idxInTheFilter *feature_size; // The offset in m_xkk & Pkk.
						const size_t R_row_offset = obsIdx*obs_size;  // Offset row in R

						// Compute just the part of the Jacobian that we need using the current updated m_xkk:
						//function_h(h,Hx,Hy, idxInTheFilter );

						CVectorTemplate<KFTYPE>   ytilde; // Diff. observation - prediction
						OnObservationModelAndJacobians(
							Z,
							data_association,
							false, // Only a partial Jacobian
							(int)obsIdx,  // Which row from Z
							ytilde,
							Hx,
							Hy );

#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG)
						ASSERT_(ytilde.size() == obs_size )
						ASSERT_(Hx.getRowCount() == obs_size )
						ASSERT_(Hx.getColCount() == vehicle_size )

						if (feature_size>0)
						{
							ASSERT_(Hy.getRowCount() == obs_size )
							ASSERT_(Hy.getColCount() == feature_size )
						}
#endif

						// For each component of the observation:
						for (size_t j=0;j<obs_size;j++)
						{
							// Compute the scalar S_i for each component j of the observation:
							// Sij = dhij_dxv Pxx dhij_dxv^t + 2 * dhij_dyi Pyix dhij_dxv + dhij_dyi Pyiyi dhij_dyi^t + R
							//          ^^
							//         Hx:(O*LxSv)
							//       \----------------------/ \--------------------------/  \------------------------/ \-/
							//               TERM 1                   TERM 2                        TERM 3              R
							//
							// O: Observation size (3)
							// L: # landmarks
							// Sv: Vehicle state size (6)
							//

#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG)
							{
								// This algorithm is applicable only if the scalar component of the sensor noise are INDEPENDENT:
								for (size_t a=0;a<obs_size;a++)
									for (size_t b=0;b<obs_size;b++)
										if ( a!=b )
											if (R(R_row_offset+a,b)!=0)
												THROW_EXCEPTION("This KF algorithm assumes independent noise components in the observation (matrix R). Select another KF algorithm.")
											}
#endif
							// R:
							KFTYPE Sij = R(R_row_offset+j,j); //R2[j]; //R(j,j);
							size_t k;

							// TERM 1:
							for (k=0;k<vehicle_size;k++)
							{
								KFTYPE accum = 0;
								for (size_t q=0;q<vehicle_size;q++)
									accum += Hx.get_unsafe(j,q) * m_pkk.get_unsafe(vehicle_offset+q,vehicle_offset+k);
								Sij+= Hx.get_unsafe(j,k) * accum;
							}

							// TERM 2:
							KFTYPE term2=0;
							for (k=0;k<vehicle_size;k++)
							{
								KFTYPE accum = 0;
								for (size_t q=0;q<feature_size;q++)
									accum += Hy.get_unsafe(j,q) * m_pkk.get_unsafe(idx_off+q,vehicle_offset+k);
								term2+= Hx.get_unsafe(j,k) * accum;
							}
							Sij += 2 * term2;

							// TERM 3:
							for (k=0;k<feature_size;k++)
							{
								KFTYPE accum = 0;
								for (size_t q=0;q<feature_size;q++)
									accum += Hy.get_unsafe(j,q) * m_pkk.get_unsafe(idx_off+q,idx_off+k);
								Sij+= Hy.get_unsafe(j,k) * accum;
							}

							// Compute the Kalman gain "Kij" for this observation element:
							// -->  K = m_pkk * (~dh_dx) * S.inv() );
							size_t N = m_pkk.getColCount();
							vector<KFTYPE> Kij( N );

							for (k=0;k<N;k++)
							{
								KFTYPE K_tmp = 0;

								// dhi_dxv term
								size_t q;
								for (q=0;q<vehicle_size;q++)
									K_tmp+= m_pkk.get_unsafe(k,vehicle_offset+q) * Hx.get_unsafe(j,q);

								// dhi_dyi term
								for (q=0;q<feature_size;q++)
									K_tmp+= m_pkk.get_unsafe(k,idx_off+q) * Hy.get_unsafe(j,q);

								Kij[k] = K_tmp / Sij;
							} // end for k


							// Update the state vector m_xkk:
							if (nKF_iterations==1)
							{
								// EKF:
								//  x' = x + Kij * ytilde(ij)
								for (k=0;k<N;k++)
									m_xkk[k] += Kij[k] * ytilde[j];
							}
							else
							{
								// IKF:
								KFTYPE  HAx=0;
								// HAx = H*(x0-xi)
								size_t q;
								for (q=0;q<vehicle_size;q++)
									HAx += Hx.get_unsafe(j,q) * (xkk_0[vehicle_offset+q] - m_xkk[vehicle_offset+q]);

								for (q=0;q<feature_size;q++)
									HAx += Hy.get_unsafe(j,q) * (xkk_0[idx_off+q] - m_xkk[idx_off+q]);

								//  x' = x_0 + Kij * ( ytilde(ij) - H*(x0-xi))   -> xi: i=this iteration
								//  xkk_next_iter is initialized to xkk_0 at each IKF iteration.
								for (k=0;k<N;k++)
									//m_xkk[k] = xkk_0[k] + Kij[k] * (ytilde[j] - HAx );
									xkk_next_iter[k] += Kij[k] * (ytilde[j] - HAx );
							}

							// Update the covariance Pkk:
							// P' =  P - Kij * Sij * Kij^t
							//if (IKF_iteration==(nKF_iterations-1))   // Just for the last IKF iteration
							{
								for (k=0;k<N;k++)
								{
									for (size_t q=k;q<N;q++)  // Half matrix
									{
										m_pkk(k,q) -= Sij * Kij[k] * Kij[q];
										// It is symmetric
										m_pkk(q,k) = m_pkk(k,q);
									}

#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG)
									if (m_pkk(k,k)<0)
									{
										m_pkk.saveToTextFile("Pkk_err.txt");
										system::vectorToTextFile(Kij,"Kij.txt");
										ASSERT_(m_pkk(k,k)>0)
									}
#endif
								}
							}

							// Ok, save

						} // end for j
					} // end if mapped
				} // end for each observed LM.

				// Now, save the new iterated mean:
				if (nKF_iterations>1)
				{
					m_xkk = xkk_next_iter;
				}

			} // end for each IKF_iteration


			// Copy of pkk not required anymore:
			if (saved_Pkk) delete saved_Pkk;

		}
		break;

		// --------------------------------------------------------------------
		// - IKF method, processing each observation vectors secuentially:
		// --------------------------------------------------------------------
		case kfIKF:
		{
			// m_IDs[id] -> index in m_xkk & Pkk
			CMatrixTemplateNumeric<KFTYPE> h,Hx,Hy;

			// Just one, or several update iterations??
			size_t nKF_iterations = KF_options.IKF_iterations;

			// To avoid wasting time, if we are doing 1 iteration only, do not reserve memory for this matrix:
			CMatrixTemplateNumeric<KFTYPE>		*saved_Pkk=NULL;
			if (nKF_iterations>1)
			{
				// Create a copy of Pkk for later restoring it at the beginning of each iteration:
				saved_Pkk = new CMatrixTemplateNumeric<KFTYPE>( m_pkk );
			}

			CVectorTemplate<KFTYPE>				xkk_0 = m_xkk; // First state vector at the beginning of IKF
			CVectorTemplate<KFTYPE>				xkk_next_iter = m_xkk;  // for IKF only

			// For each IKF iteration (or 1 for EKF)
			for (size_t IKF_iteration=0;IKF_iteration<nKF_iterations;IKF_iteration++)
			{
				// Restore initial covariance matrix.
				// The updated mean is stored in m_xkk and is improved with each estimation,
				//  and so do the Jacobians which use that improving mean.
				if (IKF_iteration>0)
				{
					m_pkk = *saved_Pkk;
					//m_xkk = xkk_saved;
					xkk_next_iter = xkk_0;
				}

				// For each observed landmark/whole system state:
				for (size_t obsIdx=0;obsIdx<Z.getRowCount();obsIdx++)
				{
					// Known & mapped landmark?
					bool   doit;
					size_t idxInTheFilter=0;

					if (data_association.empty())
					{
						doit = true;
					}
					else
					{
						doit = data_association[obsIdx] >= 0;
						if (doit)
							idxInTheFilter = data_association[obsIdx];
					}

					if ( doit )
					{
						// Already mapped: OK
						const size_t idx_off      = feature_offset + idxInTheFilter *feature_size; // The offset in m_xkk & Pkk.
						const size_t R_row_offset = obsIdx*obs_size;  // Offset row in R

						// Compute just the part of the Jacobian that we need using the current updated m_xkk:
						CVectorTemplate<KFTYPE>   ytilde; // Diff. observation - prediction
						OnObservationModelAndJacobians(
							Z,
							data_association,
							false, 			// Only a partial Jacobian
							(int)obsIdx,  	// Which row from Z
							ytilde,
							Hx,
							Hy );

#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG)
						ASSERT_(ytilde.size() == obs_size )
						ASSERT_(Hx.getRowCount() == obs_size )
						ASSERT_(Hx.getColCount() == vehicle_size )

						if (feature_size>0)
						{
							ASSERT_(Hy.getRowCount() == obs_size )
							ASSERT_(Hy.getColCount() == feature_size )
						}
#endif

						// Compute the OxO matrix S_i for each observation:
						// Si  = TERM1 + TERM2 + TERM2^t + TERM3 + R
						//
						// TERM1: dhi_dxv Pxx dhi_dxv^t
						//          ^^
						//         Hx:(OxV)
						//
						// TERM2: dhi_dyi Pyix dhi_dxv
						//          ^^
						//         Hy:(OxF)
						//
						// TERM3: dhi_dyi Pyiyi dhi_dyi^t
						//
						// i: obsIdx
						// O: Observation size
						// F: Feature size
						// V: Vehicle state size
						//

						// R:
						CMatrixTemplateNumeric<KFTYPE> Si(obs_size,obs_size);
						R.extractMatrix(R_row_offset,0, Si);

						size_t k;
						CMatrixTemplateNumeric<KFTYPE>	term(obs_size,obs_size);

						// TERM1: dhi_dxv Pxx dhi_dxv^t
						Hx.multiplyByMatrixAndByTranspose(
							m_pkk, 		// The cov. matrix
							Si,			// The output
							true,		// Yes, submatrix of m_pkk only
							vehicle_offset, // Offset in m_pkk
							true		// Yes, accum results in output.
						);

						// TERM2: dhi_dyi Pyix dhi_dxv
						//  + its transpose:
						CMatrixTemplateNumeric<KFTYPE>	Pyix( feature_size, vehicle_size );
						m_pkk.extractMatrix(idx_off,vehicle_offset, Pyix);  // Extract cross cov. to Pyix

						term.multiplyABCt( Hy, Pyix, Hx ); //term = Hy * Pyix * ~Hx;
						Si.addAAt( term );  // Si += A + ~A

						// TERM3: dhi_dyi Pyiyi dhi_dyi^t
						Hy.multiplyByMatrixAndByTranspose(
							m_pkk, 		// The cov. matrix
							Si,			// The output
							true,		// Yes, submatrix of m_pkk only
							idx_off,    // Offset in m_pkk
							true		// Yes, accum results in output.
						);

						// Compute the inverse of Si:
						CMatrixTemplateNumeric<KFTYPE> Si_1(obs_size,obs_size);

						// Compute the Kalman gain "Ki" for this i'th observation:
						// -->  Ki = m_pkk * (~dh_dx) * S.inv();
						size_t N = m_pkk.getColCount();

						CMatrixTemplateNumeric<KFTYPE> Ki( N, obs_size );

						for (k=0;k<N;k++)  // for each row of K
						{
							size_t q;

							for (size_t c=0;c<obs_size;c++)  // for each column of K
							{
								KFTYPE	K_tmp = 0;

								// dhi_dxv term
								for (q=0;q<vehicle_size;q++)
									K_tmp+= m_pkk.get_unsafe(k,vehicle_offset+q) * Hx.get_unsafe(c,q);

								// dhi_dyi term
								for (q=0;q<feature_size;q++)
									K_tmp+= m_pkk.get_unsafe(k,idx_off+q) * Hy.get_unsafe(c,q);

								Ki.set_unsafe(k,c, K_tmp);
							} // end for c
						} // end for k

						Ki.multiply(Ki, Si.inv() );  // K = (...) * S^-1


						// Update the state vector m_xkk:
						if (nKF_iterations==1)
						{
							// EKF:
							//  x' = x + Kij * ytilde(ij)
							for (k=0;k<N;k++)
								for (size_t q=0;q<obs_size;q++)
									m_xkk[k] += Ki.get_unsafe(k,q) * ytilde[q];
						}
						else
						{
							// IKF:
							std::vector<KFTYPE> HAx(obs_size);
							size_t o,q;
							// HAx = H*(x0-xi)
							for (o=0;o<obs_size;o++)
							{
								KFTYPE	tmp = 0;
								for (q=0;q<vehicle_size;q++)
									tmp += Hx.get_unsafe(o,q) * (xkk_0[vehicle_offset+q] - m_xkk[vehicle_offset+q]);

								for (q=0;q<feature_size;q++)
									tmp += Hy.get_unsafe(o,q) * (xkk_0[idx_off+q] - m_xkk[idx_off+q]);

								HAx[o] = tmp;
							}

							// Compute only once (ytilde[j] - HAx)
							for (o=0;o<obs_size;o++)
								HAx[o] = ytilde[o] - HAx[o];

							//  x' = x_0 + Kij * ( ytilde(ij) - H*(x0-xi))   -> xi: i=this iteration
							//  xkk_next_iter is initialized to xkk_0 at each IKF iteration.
							for (k=0;k<N;k++)
							{
								KFTYPE	 tmp = xkk_next_iter[k];
								//m_xkk[k] = xkk_0[k] + Kij[k] * (ytilde[j] - HAx );
								for (o=0;o<obs_size;o++)
									tmp += Ki.get_unsafe(k,o) * HAx[o];

								xkk_next_iter[k] = tmp;
							}
						}

						// Update the covariance Pkk:
						// P' =  P - Kij * Sij * Kij^t
						//if (IKF_iteration==(nKF_iterations-1))   // Just for the last IKF iteration
						{
							// m_pkk -= Ki * Si * ~Ki;
							Ki.multiplyByMatrixAndByTransposeNonSymmetric(
								Si,
								m_pkk,   // Output
								true,    // Accumulate
								true);   // Substract instead of sum.

							m_pkk.force_symmetry();

/*							for (k=0;k<N;k++)
							{
								for (size_t q=k;q<N;q++)  // Half matrix
								{
									m_pkk(k,q) -= Sij * Kij[k] * Kij[q];
									// It is symmetric
									m_pkk(q,k) = m_pkk(k,q);
								}

#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG)
								if (m_pkk(k,k)<0)
								{
									m_pkk.saveToTextFile("Pkk_err.txt");
									system::vectorToTextFile(Kij,"Kij.txt");
									ASSERT_(m_pkk(k,k)>0)
								}
#endif
							}
							*/
						}

					} // end if doit

				} // end for each observed LM.

				// Now, save the new iterated mean:
				if (nKF_iterations>1)
				{
#if 0
					cout << "IKF iter: " << IKF_iteration << " -> " << xkk_next_iter << endl;
#endif
					m_xkk = xkk_next_iter;
				}

			} // end for each IKF_iteration

			// Copy of pkk not required anymore:
			if (saved_Pkk) delete saved_Pkk;

			}
			break;

			default:
				THROW_EXCEPTION("Invalid value of options.KF_method");
			} // end switch method

		}

		OnNormalizeStateVector();

		if (KF_options.verbose)
		{
			tim_kalman_update = stopwatch.Tac();
			stopwatch.Tic();
		}


		// =============================================================
		//  8. INTRODUCE NEW LANDMARKS
		// =============================================================
		if (data_association.size())
		{
			for (size_t idxObs=0;idxObs<Z.getRowCount();idxObs++)
			{
				// Is already in the map?
				//if ( m_IDs.find( itObs->landmarkID ) == m_IDs.end() )

				if ( data_association[idxObs] < 0 )  // Not in the map yet!
				{
					// Add it:

					// Extract the submatrix from R:
					CMatrixTemplateNumeric<KFTYPE>	subR(obs_size,obs_size);
					R.extractMatrix(idxObs*obs_size,0, subR);

					// Append to map of IDs <-> position in the state vector:
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG)
					ASSERT_(feature_size>0)
#endif
					size_t  newIndexInMap = (m_xkk.size() - vehicle_size) / feature_size;

#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG)
					ASSERT_( 0 == ((m_xkk.size() - vehicle_size) % feature_size) ) // Sanity test
#endif
					// Inverse sensor model:
					CMatrixTemplateNumeric<KFTYPE>       dyn_dxv, dyn_dhn;
					CVectorTemplate<KFTYPE>				 yn;

					// Compute the inv. sensor model and its Jacobians:
					OnInverseObservationModel(
						Z,
						idxObs,
						newIndexInMap,
						yn,
						dyn_dxv,
						dyn_dhn );

#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG)
					ASSERT_( yn.size() == feature_size )
#endif
					// Append to m_xkk:
					size_t q;
					size_t idx = m_xkk.size();
					m_xkk.resize( m_xkk.size() + feature_size );

					for (q=0;q<feature_size;q++)
						m_xkk[idx+q] = yn[q];

					// --------------------
					// Append to Pkk:
					// --------------------
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG)
					ASSERT_( m_pkk.getColCount()==idx && m_pkk.getRowCount()==idx );
#endif
					m_pkk.setSize( idx+feature_size,idx+feature_size );

					// Fill the Pxyn term:
					// --------------------
					CMatrixTemplateNumeric<KFTYPE> Pxx;
					m_pkk.extractMatrix(0,0,vehicle_size,vehicle_size, Pxx);
					CMatrixTemplateNumeric<KFTYPE> Pxyn;
					Pxyn.multiply( dyn_dxv, Pxx );

					m_pkk.insertMatrix( idx,0, Pxyn );
					m_pkk.insertMatrixTranspose( 0,idx, Pxyn );

					// Fill the Pyiyn terms:
					// --------------------
					size_t nLMs = (idx-vehicle_size)/feature_size; // Number of previous landmarks:
					CMatrixTemplateNumeric<KFTYPE> P_cross;
					for (q=0;q<nLMs;q++)
					{
						CMatrixTemplateNumeric<KFTYPE>  P_x_yq(vehicle_size,feature_size);
						m_pkk.extractMatrix(0,vehicle_size+q*feature_size,P_x_yq) ;
						P_cross.multiply(dyn_dxv, P_x_yq );

						m_pkk.insertMatrix(idx,vehicle_size+q*feature_size, P_cross );
						m_pkk.insertMatrixTranspose(vehicle_size+q*feature_size,idx, P_cross );
					} // end each previous LM(q)

					// Fill the Pynyn term:
					// --------------------
					CMatrixTemplateNumeric<KFTYPE> P_yn_yn;
					dyn_dxv.multiplyByMatrixAndByTranspose(Pxx , P_yn_yn);
					dyn_dhn.multiplyByMatrixAndByTranspose(subR, P_yn_yn, false,0, true); // Accumulate in P_yn_yn
					//P_yn_yn =  (dyn_dxv * Pxx * ~dyn_dxv) + (dyn_dhn * R * ~dyn_dhn);

					m_pkk.insertMatrix(idx,idx, P_yn_yn );

				}
			}
		} // end if data_association!=empty

		if (KF_options.verbose)
		{
			tim_kalman_map_update = stopwatch.Tac();

			printf("[KF::runOneIteration] Prediction: %fms | Gen. obs: %fms | Kalman update: %fms | Map update: %fms\n",
				   1000*tim_kalman_prediction,
				   1000*tim_generate_observations,
				   1000*tim_kalman_update,
				   1000*tim_kalman_map_update );
		}

		// Post iteration user code:
		OnPostIteration();


		MRPT_TRY_END
	}


	/*---------------------------------------------------------------
				OnInverseObservationModel
	  ---------------------------------------------------------------*/
	void  CKalmanFilterCapable::OnInverseObservationModel(
		const CMatrixTemplateNumeric<KFTYPE> &in_z,
		const size_t                         &in_obsIdx,
		const size_t                         &in_idxNewFeat,
		CVectorTemplate<KFTYPE>            &out_yn,
		CMatrixTemplateNumeric<KFTYPE>     &out_dyn_dxv,
		CMatrixTemplateNumeric<KFTYPE>     &out_dyn_dhn )
	{
		MRPT_TRY_START
		THROW_EXCEPTION("Inverse sensor model required but not implemented in derived class.")
		MRPT_TRY_END
	}

	/*---------------------------------------------------------------
				OnNormalizeStateVector
	  ---------------------------------------------------------------*/
	void CKalmanFilterCapable::OnNormalizeStateVector()
	{
		// Do nothing.
	}

	/*---------------------------------------------------------------
				OnPostIteration
	  ---------------------------------------------------------------*/
	void CKalmanFilterCapable::OnPostIteration()
	{
		// Do nothing.
	}


	/*---------------------------------------------------------------
						dumpToTextStream
	  ---------------------------------------------------------------*/
	void  CKalmanFilterCapable::TKF_options::dumpToTextStream(CStream	&out)
	{
		out.printf("\n----------- [CKalmanFilterCapable::TKF_options] ------------ \n\n");
		out.printf("method                                  = ");
		switch (method)
		{
		case kfEKFNaive:
			out.printf("Naive EKF\n");
			break;
		case kfEKFAlaDavison:
			out.printf("EKF a la Davison\n");
			break;
		case kfIKFFull:
			out.printf("Full IKF\n");
			break;
		case kfIKF:
			out.printf("IKF observation-by-observation\n");
			break;
		default:
			out.printf("NON VALID VALUE!\n");
			break;
		}

		out.printf("veryLargeR2                             = %f\n", veryLargeR2);
		out.printf("verbose                                 = %c\n", verbose ? 'Y':'N');
		out.printf("IKF_iterations                          = %i\n", IKF_iterations);

		out.printf("\n");
	}

	/*---------------------------------------------------------------
						loadFromConfigFile
	  ---------------------------------------------------------------*/
	void  CKalmanFilterCapable::TKF_options::loadFromConfigFile(
		const mrpt::utils::CConfigFileBase  &iniFile,
		const std::string &section)
	{
		MRPT_LOAD_CONFIG_VAR_CAST_NO_DEFAULT(method,int, TKFMethod, iniFile, section  );
		MRPT_LOAD_CONFIG_VAR( veryLargeR2, double   , iniFile, section  );
		MRPT_LOAD_CONFIG_VAR( verbose, bool    , iniFile, section  );
		MRPT_LOAD_CONFIG_VAR( IKF_iterations, int , iniFile, section  );
	}

	/*---------------------------------------------------------------
				TKF_options
	  ---------------------------------------------------------------*/
	CKalmanFilterCapable::TKF_options::TKF_options()
	{
		method      = kfEKFAlaDavison;
		veryLargeR2 = (KFTYPE) 1e16;
		verbose     = false;
		IKF_iterations = 5;
	}


	/*---------------------------------------------------------------
					OnGetObservations
	  ---------------------------------------------------------------*/
	void CKalmanFilterCapable::OnGetObservations(
		CMatrixTemplateNumeric<KFTYPE> &out_z,
		CMatrixTemplateNumeric<KFTYPE> &out_R,
		vector_int                     &out_data_association
	)
	{
		MRPT_TRY_START
		CVectorTemplate<KFTYPE>		temp_R2;

		// If we are here, the user of this class should have implemented the vector version of OnGetObservations:
		OnGetObservations( out_z, temp_R2, out_data_association );

		size_t	nObs = out_z.getRowCount();
		if (nObs>0)
		{
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG)
			ASSERT_( out_z.getColCount() == temp_R2.size() );
#endif
			size_t  O = temp_R2.size();  // size of each observation

			out_R.setSize(nObs*O, O );

			for (size_t obs=0;obs<nObs;obs++)
			{
				size_t  row_off = O*obs;
				for (size_t i=0;i<O;i++)
					for (size_t j=0;j<O;j++)
						out_R.set_unsafe( row_off+i,j, i==j ? temp_R2[i] : 0 );
			}
		}

		MRPT_TRY_END
	}
