/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/poses/CPointPDFSOG.h>
#include <mrpt/poses/CPosePDFGaussian.h>
#include <mrpt/poses/CPosePDF.h>
#include <mrpt/poses/CPose3D.h>
#include <mrpt/opengl.h>
#include <mrpt/bayes/CParticleFilterCapable.h>
#include <mrpt/random.h>

#include <mrpt/math/utils.h>
#include <mrpt/math/distributions.h>

using namespace mrpt::poses;
using namespace mrpt::math;
using namespace mrpt::utils;
using namespace mrpt::bayes;
using namespace std;

IMPLEMENTS_SERIALIZABLE( CPointPDFSOG, CPosePDF, mrpt::poses )


/*---------------------------------------------------------------
	Constructor
  ---------------------------------------------------------------*/
CPointPDFSOG::CPointPDFSOG( size_t nModes ) :
	m_modes(nModes)
{

}

/*---------------------------------------------------------------
						getEstimatedPoint
  Returns an estimate of the pose, (the mean, or mathematical expectation of the PDF)
 ---------------------------------------------------------------*/
CPoint3D  CPointPDFSOG::getEstimatedPoint() const
{
	size_t		N = m_modes.size();

	if (N)
	{
		double		X=0,Y=0,Z=0;

		CListGaussianModes::const_iterator	it;
		double		w,sumW = 0;

		for (it=m_modes.begin();it!=m_modes.end();it++)
		{
			sumW += w = exp(it->log_w);
			X += it->val.mean.x * w;
			Y += it->val.mean.y * w;
			Z += it->val.mean.z * w;
		}
		if (sumW>0)
		{
			X /= sumW;
			Y /= sumW;
			Z /= sumW;
		}

		return CPoint3D( X,Y,Z );
	}
	else
	{
		return CPoint3D(0,0,0);
	}
}

/*---------------------------------------------------------------
						getEstimatedCovariance
 ---------------------------------------------------------------*/
CMatrixD  CPointPDFSOG::getEstimatedCovariance() const
{
	size_t		N = m_modes.size();

	if (N)
	{
		// 1) Get the mean:
		double		w,sumW = 0;
		CMatrixD		estCov(3,3);
		CMatrixD		estMean( getEstimatedPoint() );

		estCov.zeros();

		CListGaussianModes::const_iterator	it;

		for (it=m_modes.begin();it!=m_modes.end();it++)
		{
			sumW += w = exp(it->log_w);
			CMatrixD		estMean_i(it->val.mean);
			estCov += w * ( it->val.cov + ((estMean_i-estMean)*(~(estMean_i-estMean))) );
		}

		if (sumW==0)
				return CMatrixD(3,3);		// Zeros!
		else	return estCov * (1.0/sumW);
	}
	else
	{
		return CMatrixD(3,3);
	}
}

/*---------------------------------------------------------------
						writeToStream
  ---------------------------------------------------------------*/
void  CPointPDFSOG::writeToStream(CStream &out,int *version) const
{
	if (version)
		*version = 1;
	else
	{
		uint32_t	N = m_modes.size();
		CListGaussianModes::const_iterator		it;

		out << N;

		for (it=m_modes.begin();it!=m_modes.end();it++)
		{
			out << it->log_w;
			out << it->val.mean;
			out << it->val.cov(0,0) << it->val.cov(1,1) << it->val.cov(2,2);
			out << it->val.cov(0,1) << it->val.cov(0,2) << it->val.cov(1,2);
		}
	}
}

/*---------------------------------------------------------------
						readFromStream
  ---------------------------------------------------------------*/
void  CPointPDFSOG::readFromStream(CStream &in,int version)
{
	switch(version)
	{
	case 0:
	case 1:
		{
			uint32_t								N;
			CListGaussianModes::iterator		it;
			float									x;

			in >> N;

			m_modes.resize(N);

			for (it=m_modes.begin();it!=m_modes.end();it++)
			{
				in >> it->log_w;

				// In version 0, weights were linear!!
				if (version==0) it->log_w = log(max(1e-300,it->log_w));

				in >> it->val.mean;

				it->val.cov.setSize(3,3);

				in >> x; it->val.cov(0,0) = x;
				in >> x; it->val.cov(1,1) = x;
				in >> x; it->val.cov(2,2) = x;

				in >> x; it->val.cov(1,0) = x; it->val.cov(0,1) = x;
				in >> x; it->val.cov(2,0) = x; it->val.cov(0,2) = x;
				in >> x; it->val.cov(1,2) = x; it->val.cov(2,1) = x;
			}

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}



/*---------------------------------------------------------------
						operator =
  ---------------------------------------------------------------*/
void  CPointPDFSOG::copyFrom(const CPointPDF &o)
{
	MRPT_TRY_START;

	if (this == &o) return;		// It may be used sometimes

	if (o.GetRuntimeClass()==CLASS_ID(CPointPDFSOG))
	{
		m_modes = static_cast<const CPointPDFSOG*>(&o)->m_modes;
	}
	else
	{
		// Approximate as a mono-modal gaussian pdf:
		// TODO: More elaborate!!
		m_modes.resize(1);
		m_modes[0].log_w = 0;
		m_modes[0].val.mean = o.getEstimatedPoint();
		m_modes[0].val.cov = o.getEstimatedCovariance();
	}

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						saveToTextFile
  ---------------------------------------------------------------*/
void  CPointPDFSOG::saveToTextFile(const std::string &file) const
{
	FILE	*f=os::fopen(file.c_str(),"wt");
	if (!f) return;


	for (CListGaussianModes::const_iterator it=m_modes.begin();it!=m_modes.end();it++)
		os::fprintf(f,"%e %e %e %e %e %e %e %e %e %e\n",
			exp(it->log_w),
			it->val.mean.x, it->val.mean.y, it->val.mean.z,
			it->val.cov(0,0),it->val.cov(1,1),it->val.cov(2,2),
			it->val.cov(0,1),it->val.cov(0,2),it->val.cov(1,2) );
	os::fclose(f);
}

/*---------------------------------------------------------------
						changeCoordinatesReference
 ---------------------------------------------------------------*/
void  CPointPDFSOG::changeCoordinatesReference(const CPose3D &newReferenceBase )
{
	CMatrixD		M = newReferenceBase.getHomogeneousMatrix();
	M.setSize(3,3);	// Clip the 4x4 matrix

	// The variance in phi is unmodified:
	M(0,2) = 0; M(1,2) = 0;
	M(2,0) = 0; M(2,1) = 0;
	M(2,2) = 1;

	for (CListGaussianModes::iterator it=m_modes.begin();it!=m_modes.end();it++)
		it->val.changeCoordinatesReference( newReferenceBase );

	assureSymmetry();
}

/*---------------------------------------------------------------
					drawSingleSample
 ---------------------------------------------------------------*/
void CPointPDFSOG::drawSingleSample(CPoint3D  &outSample) const
{
	MRPT_TRY_START

	ASSERT_(m_modes.size()>0);


	// 1st: Select a mode with a probability proportional to its weight:
	vector_double				logWeights( m_modes.size() );
	vector<size_t>				outIdxs;
	vector_double::iterator 	itW;
	CListGaussianModes::const_iterator it;
	for (it=m_modes.begin(),itW=logWeights.begin();it!=m_modes.end();it++,itW++)
		*itW = it->log_w;

	CParticleFilterCapable::computeResampling(
		CParticleFilter::prMultinomial, // Resampling algorithm
		logWeights,                     // input: log weights
		outIdxs                         // output: indexes
		);

	// we need just one: take the first (arbitrary)
	size_t   selectedIdx = outIdxs[0];
	ASSERT_(selectedIdx<m_modes.size());
	const CPointPDFGaussian* selMode = & m_modes[selectedIdx].val;


	// 2nd: Draw a position from the selected Gaussian:
	vector_double vec;
	mrpt::random::randomNormalMultiDimensional(
		selMode->cov,
		vec);

	ASSERT_(vec.size()==3);
	outSample.x = selMode->mean.x + vec[0];
	outSample.y = selMode->mean.y + vec[1];
	outSample.z = selMode->mean.z + vec[2];

	MRPT_TRY_END
}

/*---------------------------------------------------------------
					bayesianFusion
 ---------------------------------------------------------------*/
void  CPointPDFSOG::bayesianFusion( CPointPDF &p1_, CPointPDF &p2_,const double &minMahalanobisDistToDrop )
{
	MRPT_TRY_START;

	// p1: CPointPDFSOG, p2: CPosePDFGaussian:

	ASSERT_( p1_.GetRuntimeClass() == CLASS_ID(CPointPDFSOG) );
	ASSERT_( p2_.GetRuntimeClass() == CLASS_ID(CPointPDFSOG) );

	CPointPDFSOG		*p1 = static_cast<CPointPDFSOG*>( &p1_ );
	CPointPDFSOG		*p2 = static_cast<CPointPDFSOG*>( &p2_ );

	// Compute the new kernel means, covariances, and weights after multiplying to the Gaussian "p2":
	CPointPDFGaussian	auxGaussianProduct,auxSOG_Kernel_i;
	TGaussianMode		newKernel;

	float			minMahalanobisDistToDrop2 = square(minMahalanobisDistToDrop);


	this->m_modes.clear();
	bool is2D = false; // to detect & avoid errors in 3x3 matrix inversions of range=2.

	for (CListGaussianModes::iterator it1 =p1->m_modes.begin();it1!=p1->m_modes.end();it1++)
	{
		// Is a 2D covariance??
		if (it1->val.cov.get_unsafe(2,2)==0)
		{
			is2D = true;
			it1->val.cov.set_unsafe(2,2,1);
		}

		ASSERT_( it1->val.cov(0,0)!=0 && it1->val.cov(0,0)!=0 )

		CMatrixD				covInv( ! it1->val.cov );
		CMatrixD				eta(3,1);
		eta.set_unsafe(0,0, it1->val.mean.x);
		eta.set_unsafe(1,0, it1->val.mean.y);
		eta.set_unsafe(2,0, it1->val.mean.z);
		eta = covInv * eta;

		// Normal distribution canonical form constant:
		// See: http://www-static.cc.gatech.edu/~wujx/paper/Gaussian.pdf
		double a = -0.5*( 3*log(M_2PI) - log( covInv.det() ) +
		            eta.multiplyByMatrixAndByTransposeScalar(it1->val.cov) ); // (~eta * it1->val.cov * eta)(0,0) );

		for (CListGaussianModes::iterator it2 =p2->m_modes.begin();it2!=p2->m_modes.end();it2++)
		{
			auxSOG_Kernel_i = it2->val;
			if (auxSOG_Kernel_i.cov.get_unsafe(2,2)==0) { auxSOG_Kernel_i.cov.set_unsafe(2,2,1); is2D=true; }
			ASSERT_(auxSOG_Kernel_i.cov(0,0)>0 && auxSOG_Kernel_i.cov(1,1)>0 )


			// Should we drop this product term??
			bool reallyComputeThisOne = true;
			if (minMahalanobisDistToDrop>0)
			{
				// Approximate (fast) mahalanobis distance (square):
				float mahaDist2;

				float stdX2 = max(auxSOG_Kernel_i.cov.get_unsafe(0,0) , it1->val.cov.get_unsafe(0,0));
				mahaDist2 = square( auxSOG_Kernel_i.mean.x - it1->val.mean.x )/stdX2;

				float stdY2 = max(auxSOG_Kernel_i.cov.get_unsafe(1,1), it1->val.cov.get_unsafe(1,1));
				mahaDist2 += square( auxSOG_Kernel_i.mean.y - it1->val.mean.y )/stdY2;

				if (!is2D)
				{
					float stdZ2 = max( auxSOG_Kernel_i.cov.get_unsafe(2,2), it1->val.cov.get_unsafe(2,2) );
					mahaDist2 += square( auxSOG_Kernel_i.mean.z - it1->val.mean.z )/stdZ2;
				}

				reallyComputeThisOne = mahaDist2 < minMahalanobisDistToDrop2;
			}

			if (reallyComputeThisOne)
			{
				auxGaussianProduct.bayesianFusion( auxSOG_Kernel_i, it1->val );

				// ----------------------------------------------------------------------
				// The new weight is given by:
				//
				//   w'_i = w_i * exp( a + a_i - a' )
				//
				//      a = -1/2 ( dimensionality * log(2pi) - log(det(Cov^-1)) + (Cov^-1 * mu)^t * Cov^-1 * (Cov^-1 * mu) )
				//
				// ----------------------------------------------------------------------
				newKernel.val = auxGaussianProduct; // Copy mean & cov

				CMatrixD		covInv_i( auxSOG_Kernel_i.cov.inv() );
				CMatrixD		eta_i(3,1);
				eta_i(0,0) = auxSOG_Kernel_i.mean.x;
				eta_i(1,0) = auxSOG_Kernel_i.mean.y;
				eta_i(2,0) = auxSOG_Kernel_i.mean.z;
				eta_i = covInv_i * eta_i;

				CMatrixD		new_covInv_i( newKernel.val.cov.inv() );
				CMatrixD		new_eta_i(3,1);
				new_eta_i(0,0) = newKernel.val.mean.x;
				new_eta_i(1,0) = newKernel.val.mean.y;
				new_eta_i(2,0) = newKernel.val.mean.z;
				new_eta_i = new_covInv_i * new_eta_i;

				double		a_i	    = -0.5*( 3*log(M_2PI) - log( new_covInv_i.det() ) + (~eta_i * auxSOG_Kernel_i.cov * eta_i)(0,0) );
				double		new_a_i = -0.5*( 3*log(M_2PI) - log( new_covInv_i.det() ) + (~new_eta_i * newKernel.val.cov * new_eta_i)(0,0) );

				newKernel.log_w	   = it1->log_w + it2->log_w + a + a_i - new_a_i ;

				// Fix 2D case:
				if (is2D) newKernel.val.cov(2,2)=0;

				// Add to the results (in "this") the new kernel:
				this->m_modes.push_back( newKernel );
			} // end if reallyComputeThisOne
		} // end for it2

		// Restore:
		if (is2D)
		{
			it1->val.cov(2,2)=0;
		}


	} // end for it1

	normalizeWeights();

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						assureSymmetry
 ---------------------------------------------------------------*/
void  CPointPDFSOG::assureSymmetry()
{
	MRPT_TRY_START;
	// Differences, when they exist, appear in the ~15'th significant
	//  digit, so... just take one of them arbitrarily!
	for (CListGaussianModes::iterator it=m_modes.begin();it!=m_modes.end();it++)
	{
		it->val.cov(0,1) = it->val.cov(1,0);
		it->val.cov(0,2) = it->val.cov(2,0);
		it->val.cov(1,2) = it->val.cov(2,1);
	}

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						normalizeWeights
 ---------------------------------------------------------------*/
void  CPointPDFSOG::normalizeWeights()
{
	MRPT_TRY_START;

	if (!m_modes.size()) return;

	CListGaussianModes::iterator		it;
	double									maxW = m_modes[0].log_w;
	for (it=m_modes.begin();it!=m_modes.end();it++)
		maxW = max(maxW,it->log_w);

	for (it=m_modes.begin();it!=m_modes.end();it++)
		it->log_w -= maxW;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						getAs3DObject
 ---------------------------------------------------------------*/
void  CPointPDFSOG::getAs3DObject( mrpt::opengl::CSetOfObjectsPtr	&outObj ) const
{
	// For each gaussian node
	CListGaussianModes::const_iterator it;
	for ( it = m_modes.begin(); it!= m_modes.end();it++)
	{
		opengl::CEllipsoidPtr obj = opengl::CEllipsoid::Create();

		obj->setPose( it->val.mean);

		CMatrixDouble C =it->val.cov;
		if (C(2,2)==0) C.setSize(2,2);
		obj->setCovMatrix(C);

		obj->setQuantiles(3);
		obj->enableDrawSolid3D(false);
		obj->setColor(1,0,0, 0.5);

		outObj->insert( obj );
	} // end for each gaussian node
}

/*---------------------------------------------------------------
						ESS
 ---------------------------------------------------------------*/
double CPointPDFSOG::ESS() const
{
	MRPT_TRY_START;
	CListGaussianModes::const_iterator	it;
	double	cum = 0;

	/* Sum of weights: */
	double sumLinearWeights = 0;
	for (it=m_modes.begin();it!=m_modes.end();it++) sumLinearWeights += exp(it->log_w);

	/* Compute ESS: */
	for (it=m_modes.begin();it!=m_modes.end();it++)
		cum+= square( exp(it->log_w) / sumLinearWeights );

	if (cum==0)
			return 0;
	else	return 1.0/(m_modes.size()*cum);
	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						evaluatePDFInArea
 ---------------------------------------------------------------*/
void  CPointPDFSOG::evaluatePDFInArea(
	float		x_min,
	float		x_max,
	float		y_min,
	float		y_max,
	float		resolutionXY,
	float		z,
	CMatrixD	&outMatrix,
	bool		sumOverAllZs )
{
	MRPT_TRY_START;

	ASSERT_(x_max>x_min);
	ASSERT_(y_max>y_min);
	ASSERT_(resolutionXY>0);

	size_t		Nx = (size_t)ceil((x_max-x_min)/resolutionXY);
	size_t		Ny = (size_t)ceil((y_max-y_min)/resolutionXY);
	size_t		i,j;
	float		x,y;

	outMatrix.setSize(Ny,Nx);

	for (i=0;i<Ny;i++)
	{
		y = y_min + i*resolutionXY;
		for (j=0;j<Nx;j++)
		{
			x = x_min + j*resolutionXY;
			outMatrix(i,j) = evaluatePDF(CPoint3D(x,y,z),sumOverAllZs);
		}
	}


	MRPT_TRY_END;
}


/*---------------------------------------------------------------
						evaluatePDF
 ---------------------------------------------------------------*/
double  CPointPDFSOG::evaluatePDF(
	const	CPoint3D &x,
	bool	sumOverAllZs ) const
{
	if (!sumOverAllZs)
	{
		// Normal evaluation:
		CMatrixD	X(3,1), MU(3,1);
		double	ret = 0;

		X(0,0) = x.x;
		X(1,0) = x.y;
		X(2,0) = x.z;

		for (CListGaussianModes::const_iterator it=m_modes.begin();it!=m_modes.end();it++)
		{
			MU(0,0) = it->val.mean.x;
			MU(1,0) = it->val.mean.y;
			MU(2,0) = it->val.mean.z;

			ret+= exp(it->log_w) * math::normalPDF( X, MU, it->val.cov );
		}

		return ret;
	}
	else
	{
		// Only X,Y:
		CMatrixD	X(2,1), MU(2,1),COV(2,2);
		double	ret = 0;

		X(0,0) = x.x;
		X(1,0) = x.y;

		for (CListGaussianModes::const_iterator it=m_modes.begin();it!=m_modes.end();it++)
		{
			MU(0,0) = it->val.mean.x;
			MU(1,0) = it->val.mean.y;

			COV(0,0) = it->val.cov(0,0);
			COV(1,1) = it->val.cov(1,1);
			COV(0,1) = COV(1,0) = it->val.cov(0,1);

			ret+= exp(it->log_w) * math::normalPDF( X, MU, COV );
		}

		return ret;
	}
}
