/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/poses/CPose2D.h>
#include <mrpt/poses/CPoint2D.h>
#include <mrpt/poses/CPose3D.h>
#include <mrpt/poses/CPoint3D.h>
#include <mrpt/math/utils.h>

using namespace mrpt;
using namespace mrpt::poses;
using namespace mrpt::utils;

IMPLEMENTS_SERIALIZABLE(CPose2D, CPose,mrpt::poses)

/*---------------------------------------------------------------
	Constructors
  ---------------------------------------------------------------*/
CPose2D::CPose2D(const double& x,const double& y,const double& _phi) : aux_HM(0,0), phi(_phi)
{
	this->x		= x;
	this->y		= y;
	this->z		= 0;
	//this->phi	= phi;

	normalizePhi();
	is3D = false;
}

CPose2D::CPose2D(const CPoint2D &p) : aux_HM(0,0), phi(0)
{
	x = p.x;
	y = p.y;
	this->z		= 0;
	//phi = 0;
	is3D = false;
}

CPose2D::CPose2D(const CPose3D &p) : aux_HM(0,0), phi(p.yaw)
{
	x = p.x;
	y = p.y;
	//phi = p.yaw;

	z = 0;

	is3D = false;
}

/*---------------------------------------------------------------
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void  CPose2D::writeToStream(CStream &out,int *version) const
{
	if (version)
		*version = 1;
	else
	{
		// The coordinates:
		out << x << y << phi;
	}
}

/*---------------------------------------------------------------
	Implements the reading from a CStream capability of
		CSerializable objects
  ---------------------------------------------------------------*/
void  CPose2D::readFromStream(CStream &in,int version)
{
	switch(version)
	{
	case 0:
		{
			// The coordinates:
			float x0,y0,phi0;
			in >> x0 >> y0 >> phi0;
			x = x0;
			y = y0;
			phi = phi0;
		} break;
	case 1:
		{
			// The coordinates:
			in >> x >> y >> phi;
		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}

/**  Textual output stream function.
 */
std::ostream& mrpt::poses::operator << (std::ostream& o, const CPose2D& p)
{
	o << format("(%.03f,%.03f,%.02fdeg)",p.x,p.y,RAD2DEG(p.phi));
	return o;
}

/*---------------------------------------------------------------
The operator a="this"+D is the pose compounding operator.
 ---------------------------------------------------------------*/
CPose2D  CPose2D::operator + (const CPose2D& D)const
{
	CPose2D		ret;

	double ccos = cos(phi);
	double ssin = sin(phi);

	ret.x	= x + D.x * ccos - D.y * ssin;
	ret.y	= y + D.x * ssin + D.y * ccos;
	ret.phi = phi + D.phi;

	ret.normalizePhi();

	return ret;
}

/*---------------------------------------------------------------
The operator a="this"+D is the pose compounding operator.
 ---------------------------------------------------------------*/
CPose3D  CPose2D::operator + (const CPose3D& D) const
{
	return CPose3D(*this) + D;
}

/*---------------------------------------------------------------
The operator u'="this"+u is the pose/point compounding operator.
 ---------------------------------------------------------------*/
CPoint2D CPose2D::operator + (const CPoint2D& u)const
{
	CPoint2D		ret;

	double ccos = cos(phi);
	double ssin = sin(phi);

	ret.x	= x + u.x * ccos - u.y * ssin;
	ret.y	= y + u.x * ssin + u.y * ccos;

	return ret;
}

/*---------------------------------------------------------------
The operator u'="this"+u is the pose/point compounding operator.
 ---------------------------------------------------------------*/
CPoint3D CPose2D::operator + (const CPoint3D& u)const
{
	CPoint3D		ret;

	double ccos = cos(phi);
	double ssin = sin(phi);

	ret.x	= x + u.x * ccos - u.y * ssin;
	ret.y	= y + u.x * ssin + u.y * ccos;
	ret.z   = u.z;

	return ret;
}

/*---------------------------------------------------------------
The operator D="this"-b is the pose inverse compounding operator.
   The resulting pose "D" is the diference between this pose and "b"
 ---------------------------------------------------------------*/
CPose2D  CPose2D::operator - (const CPose2D& b)const
{
	CPose2D		ret;

	double ccos = cos(b.phi);
	double ssin = sin(b.phi);

	ret.x	=  (x - b.x) * ccos + (y - b.y) * ssin;
	ret.y	= -(x - b.x) * ssin + (y - b.y) * ccos;
	ret.phi = phi - b.phi;

	ret.normalizePhi();

	return ret;
}



/*---------------------------------------------------------------
 Scalar sum of components: This is diferent from poses
   composition, which is implemented as "+" operators in "CPose" derived classes.
 ---------------------------------------------------------------*/
void CPose2D::AddComponents(CPose2D &p)
{
	x+=p.x;
	y+=p.y;
//	z+=p.z;		// May be undefined, this is a 2D class!!!

	phi+=p.phi;
}


/*---------------------------------------------------------------
 Scalar multiplication.
 ---------------------------------------------------------------*/
void CPose2D::operator *= (const double& s)
{
	x*=s;
	y*=s;
	phi*=s;
}


/*---------------------------------------------------------------
	Returns the corresponding 4x4 homogeneous
	  transformation matrix for the point(translation),
	  or pose (translation+orientation).
---------------------------------------------------------------*/
const math::CMatrixDouble&  CPose2D::getHomogeneousMatrix() const
{
	aux_HM.unit(4);

	aux_HM.set_unsafe(0,3, x );
	aux_HM.set_unsafe(1,3, y );

	const double ccos = cos(phi);
	const double csin = sin(phi);

	aux_HM.set_unsafe(0,0, ccos);  aux_HM.set_unsafe(0,1, -csin);
	aux_HM.set_unsafe(1,0, csin);  aux_HM.set_unsafe(1,1, ccos);

	return aux_HM;
}

/*---------------------------------------------------------------
	Returns the corresponding 4x4 homogeneous
	  transformation matrix for the point(translation),
	  or pose (translation+orientation).
---------------------------------------------------------------*/
void  CPose2D::getHomogeneousMatrix(const math::CMatrixDouble *& m) const
{
	getHomogeneousMatrix();
	m = &aux_HM;
}

/** Forces "phi" to be in the range [-pi,pi];
*/
void  CPose2D::normalizePhi()
{
	phi = math::wrapToPi(phi);
}
