/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.



#include <mrpt/slam/CHeightGridMap2D.h>
#include <mrpt/slam/CObservationGasSensors.h>
#include <mrpt/slam/CObservation2DRangeScan.h>
#include <mrpt/slam/CSimplePointsMap.h>
#include <mrpt/system/os.h>
#include <mrpt/math/utils.h>
#include <mrpt/utils/CTicTac.h>
#include <mrpt/vision/utils.h>
#include <mrpt/opengl/CMesh.h>

using namespace mrpt::slam;
using namespace mrpt::poses;
using namespace mrpt::math;
using namespace mrpt::utils;
using namespace mrpt::system;
using namespace std;

IMPLEMENTS_SERIALIZABLE(CHeightGridMap2D, CMetricMap,mrpt::slam)

/*---------------------------------------------------------------
						Constructor
  ---------------------------------------------------------------*/
CHeightGridMap2D::CHeightGridMap2D(
	TMapRepresentation	mapType,
	float		x_min,
	float		x_max,
	float		y_min,
	float		y_max,
	float		resolution ) :
		CDynamicGrid<THeightGridmapCell>( x_min,x_max,y_min,y_max,resolution ),
		insertionOptions(),
		m_mapType(mapType)
{
}

/*---------------------------------------------------------------
						clear
  ---------------------------------------------------------------*/
void  CHeightGridMap2D::clear()
{
	fill( THeightGridmapCell() );
}

/*---------------------------------------------------------------
						isEmpty
  ---------------------------------------------------------------*/
bool  CHeightGridMap2D::isEmpty() const
{
	return false;
}

/*---------------------------------------------------------------
						insertObservation
  ---------------------------------------------------------------*/
bool  CHeightGridMap2D::insertObservation(
	const CObservation	*obs,
	const CPose3D			*robotPose )
{
	MRPT_TRY_START;

	CPose2D		robotPose2D;
	CPose3D		robotPose3D;

	if (robotPose)
	{
		robotPose2D = CPose2D(*robotPose);
		robotPose3D = (*robotPose);
	}
	else
	{
		// Default values are (0,0,0)
	}

	if ( CLASS_ID( CObservation2DRangeScan )==obs->GetRuntimeClass())
	{
		/********************************************************************
					OBSERVATION TYPE: CObservation2DRangeScan
		********************************************************************/
		const CObservation2DRangeScan	*o = static_cast<const CObservation2DRangeScan*>( obs );

		// Create points map, if not created yet:
		CPointsMap::TInsertionOptions	opts;
		opts.minDistBetweenLaserPoints = insertionOptions.minDistBetweenPointsWhenInserting;
		const CPointsMap	*thePoints = o->buildAuxPointsMap( &opts );

		// And rotate to the robot pose:
		CSimplePointsMap	thePointsMoved;
		thePointsMoved.changeCoordinatesReference(*thePoints, robotPose3D );

		size_t i, N = thePointsMoved.size();
		float	x,y,z;
		float	rel_x,rel_y,rel_z;

		// First compute the bounding box:
/*		float	min_x,max_x, min_y,max_y, min_z, max_z;
		thePointsMoved.boundingBox( min_x,max_x,min_y,max_y, min_z,max_z );

		// Resize grid if necessary:
		const THeightGridmapCell	dummyCell;
		resize( min_x,max_x,min_y, max_y, dummyCell, 3 );
*/

		// Insert z's using the selected method:
		const bool	doWindow = m_mapType==mrSlidingWindow;
		const mrpt::system::TTimeStamp	tim = o->timestamp;

		for (i=0;i<N;i++)
		{
			thePointsMoved.getPoint(i, x,y,z);
			thePoints->getPoint(i,rel_x,rel_y,rel_z);

			THeightGridmapCell *cell = cellByPos(x,y);
			if(!cell) continue; // Out of the map: Ignore if we've not resized before.

			if (!insertionOptions.filterByHeight || (rel_z>=insertionOptions.z_min && rel_z<=insertionOptions.z_max ) )
			{
				cell->u += z;
				cell->v += z*z;
				if (!cell->w)
				{
					cell->h = z;	// First observation
					cell->w = 1;
				}
				else
				{
					float W = cell->w++;	// W = N-1
					cell->h = (cell->h*W + z)/cell->w;
					if (W > 0)
						cell->var = 1/(W) * (cell->v - pow(cell->u,2)/cell->w);
				}

				if (doWindow)
				{
					cell->history_Zs.insert( pair<TTimeStamp,float>(tim, z));
				}

			} // end if really inserted
		} // end for i

		return true;	// Done!

	} // end if "CObservationGasSensors"

	return false;

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
						computeObservationLikelihood
  ---------------------------------------------------------------*/
double	 CHeightGridMap2D::computeObservationLikelihood(
	const CObservation		*obs,
	const CPose3D			&takenFrom )
{
	MRPT_UNUSED_PARAM(obs);MRPT_UNUSED_PARAM(takenFrom);

    THROW_EXCEPTION("Not implemented yet!");
}

/*---------------------------------------------------------------
  Implements the writing to a CStream capability of CSerializable objects
 ---------------------------------------------------------------*/
void  CHeightGridMap2D::writeToStream(CStream &out, int *version) const
{
	if (version)
		*version = 0;
	else
	{
		uint32_t	n;

		// Save the dimensions of the grid:
		out << m_x_min << m_x_max << m_y_min << m_y_max;
		out << m_resolution;
		out << static_cast<uint32_t>(m_size_x) << static_cast<uint32_t>(m_size_y);

		// To assure compatibility: The size of each cell:
		n = static_cast<uint32_t>(sizeof( THeightGridmapCell ));
		out << n;

		// Save the map contents:
		n = static_cast<uint32_t>(m_map.size());
		out << n;
		for (vector<THeightGridmapCell>::const_iterator it=m_map.begin();it!=m_map.end();it++)
		{
			out << it->h << it->w;
			it->history_Zs.write(out);
		}

		// Save the insertion options:
		out << uint8_t(m_mapType);

		out << insertionOptions.filterByHeight
			<< insertionOptions.z_min
			<< insertionOptions.z_max;
	}
}

/*---------------------------------------------------------------
  Implements the reading from a CStream capability of CSerializable objects
 ---------------------------------------------------------------*/
void  CHeightGridMap2D::readFromStream(CStream &in, int version)
{
	switch(version)
	{
	case 0:
		{
			uint32_t	n,i,j;

			// Load the dimensions of the grid:
			in >> m_x_min >> m_x_max >> m_y_min >> m_y_max;
			in >> m_resolution;
			in >> i >> j;
			m_size_x = i;
			m_size_y = j;

			// To assure compatibility: The size of each cell:
			in >> n;
			ASSERT_( n == static_cast<uint32_t>( sizeof( THeightGridmapCell ) ));

			// Save the map contents:
			in >> n;
			m_map.resize(n);
			for (vector<THeightGridmapCell>::iterator it=m_map.begin();it!=m_map.end();it++)
			{
				in >> it->h >> it->w;
				it->history_Zs.read( in );
			}

			// Version 1: Insertion options:
			uint8_t	ty;
			in  >> ty;
			m_mapType = TMapRepresentation(ty);

			in  >> insertionOptions.filterByHeight
				>> insertionOptions.z_min
				>> insertionOptions.z_max;

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};

}

/*---------------------------------------------------------------
					TInsertionOptions
 ---------------------------------------------------------------*/
CHeightGridMap2D::TInsertionOptions::TInsertionOptions() :
	filterByHeight				( false ),
	z_min						( -0.5  ),
	z_max						(  0.5  ),
	minDistBetweenPointsWhenInserting ( 0 )
{
}

/*---------------------------------------------------------------
					dumpToTextStream
  ---------------------------------------------------------------*/
void  CHeightGridMap2D::TInsertionOptions::dumpToTextStream(CStream	&out)
{
	out.printf("\n----------- [CHeightGridMap2D::TInsertionOptions] ------------ \n\n");
	out.printf("filterByHeight                          = %c\n", filterByHeight ? 'y':'n');
	out.printf("z_min                                   = %f\n", z_min);
	out.printf("z_max                                   = %f\n", z_max);
	out.printf("minDistBetweenPointsWhenInserting       = %f\n", minDistBetweenPointsWhenInserting);
	out.printf("\n");
}

/*---------------------------------------------------------------
					loadFromConfigFile
  ---------------------------------------------------------------*/
void  CHeightGridMap2D::TInsertionOptions::loadFromConfigFile(
	const mrpt::utils::CConfigFileBase  &iniFile,
	const std::string &section)
{
	MRPT_LOAD_CONFIG_VAR( filterByHeight,	bool, iniFile, section )
	MRPT_LOAD_CONFIG_VAR( z_min,			float, iniFile, section )
	MRPT_LOAD_CONFIG_VAR( z_max,			float, iniFile, section )
	MRPT_LOAD_CONFIG_VAR( minDistBetweenPointsWhenInserting, float, iniFile, section )
}

/*---------------------------------------------------------------
					saveMetricMapRepresentationToFile
  ---------------------------------------------------------------*/
void  CHeightGridMap2D::saveMetricMapRepresentationToFile(
	const std::string	&filNamePrefix ) const
{
	std::string		fil;

	// Text matrix:
	saveToTextFile( filNamePrefix + std::string("_mean.txt") );
}

/*---------------------------------------------------------------
						getAs3DObject
---------------------------------------------------------------*/
void  CHeightGridMap2D::getAs3DObject( mrpt::opengl::CSetOfObjectsPtr	&outObj ) const
{
	if (m_disableSaveAs3DObject)
		return;

	opengl::CMeshPtr	mesh = opengl::CMesh::Create();

	mesh->setGridLimits(m_x_min,m_x_max, m_y_min, m_y_max);

	mesh->setColor(0.4, 0.4, 0.4 );

	mesh->enableWireFrame(true);
	mesh->enableColorFromZ(true, mrpt::vision::cmJET);

	CMatrixFloat Z,mask;
	mesh->getZ(Z);
	mesh->getMask(mask);

	Z.setSize( m_size_x, m_size_y );
	mask.setSize( m_size_x, m_size_y );

	for (size_t x=0;x<m_size_x;x++)
	{
		for (size_t y=0;y<m_size_y;y++)
		{
			const THeightGridmapCell *c = cellByIndex(x,y);
			ASSERT_(c);
			Z.set_unsafe(x,y, c->h );
			mask.set_unsafe(x,y, c->w ? 1:0 );
		}
	}
	mesh->setZ(Z);
	mesh->setMask(mask);

	outObj->insert( mesh );
}

/*---------------------------------------------------------------
  Computes the ratio in [0,1] of correspondences between "this" and the "otherMap" map, whose 6D pose relative to "this" is "otherMapPose"
 *   In the case of a multi-metric map, this returns the average between the maps. This method always return 0 for grid maps.
 * \param  otherMap					  [IN] The other map to compute the matching with.
 * \param  otherMapPose				  [IN] The 6D pose of the other map as seen from "this".
 * \param  minDistForCorr			  [IN] The minimum distance between 2 non-probabilistic map elements for counting them as a correspondence.
 * \param  minMahaDistForCorr		  [IN] The minimum Mahalanobis distance between 2 probabilistic map elements for counting them as a correspondence.
 *
 * \return The matching ratio [0,1]
 * \sa computeMatchingWith2D
  --------------------------------------------------------------- */
float  CHeightGridMap2D::compute3DMatchingRatio(
		const CMetricMap						*otherMap,
		const CPose3D							&otherMapPose,
		float									minDistForCorr,
		float									minMahaDistForCorr
		) const
{
	MRPT_UNUSED_PARAM(otherMap);
	MRPT_UNUSED_PARAM(otherMapPose);
	MRPT_UNUSED_PARAM(minDistForCorr);
	MRPT_UNUSED_PARAM(minMahaDistForCorr);

	return 0;
}


/*---------------------------------------------------------------
					auxParticleFilterCleanUp
 ---------------------------------------------------------------*/
void  CHeightGridMap2D::auxParticleFilterCleanUp()
{
}

/*---------------------------------------------------------------
					auxParticleFilterCleanUp
 ---------------------------------------------------------------*/
size_t CHeightGridMap2D::removeObservationsByTimestamp( const mrpt::system::TTimeStamp	&tim )
{
	MRPT_TRY_START
	ASSERT_( m_mapType == mrSlidingWindow );
	size_t	N = 0;

	for (vector<THeightGridmapCell>::iterator it=m_map.begin();it!=m_map.end();it++)
	{
		multimap<mrpt::system::TTimeStamp,float>			 &H = it->history_Zs;
		multimap<mrpt::system::TTimeStamp,float>::iterator p;
		while ( (p=H.find(tim))!=H.end())
		{
			H.erase(p);
			N++;
		}
	}
	return N;

	MRPT_TRY_END
}
