/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef  CSERIALIZABLE_H
#define  CSERIALIZABLE_H

#include <mrpt/utils/utils_defs.h>
#include <mrpt/utils/CStream.h>
#include <mrpt/utils/safe_pointers.h>

/*---------------------------------------------------------------
	Class
  ---------------------------------------------------------------*/
namespace mrpt
{
	/** Classes for serialization, sockets, ini-file manipulation, streams, list of properties-values, timewatch, extensions to STL.
	  */
	namespace utils
	{
		class MRPTDLLIMPEXP CSerializable;
		/** A smart pointer to a CRenderizable object */
		typedef stlplus::smart_ptr_clone<CSerializable> CSerializablePtr;

		/** A structure that holds runtime class type information. Use CLASS_ID(<class_name>) to get a reference to the class_name's TRuntimeClassId descriptor.
		  */
		struct MRPTDLLIMPEXP TRuntimeClassId
		{
			const char*			className;
			int				objectSize;
			/** Create an object of the related class, or NULL if it is virtual.
			  */
			CSerializable*			(*ptrCreateObject)();
			/** Gets the base class runtime id.
			  */
			const TRuntimeClassId*		(*getBaseClass)();

			// Operations
			CSerializable*		createObject() const;
			bool				derivedFrom(const TRuntimeClassId* pBaseClass) const;
			bool				derivedFrom(const char* pBaseClass_name) const;

			// Implementation
			void					writeTo(mrpt::utils::CStream& out) const;
			static const TRuntimeClassId* loadFrom(mrpt::utils::CStream& in);
		};

		/** A wrapper class for a "TRuntimeClassId *", well-defined with respect to copy operators and constructors.
		  */
		typedef safe_ptr<TRuntimeClassId> TRuntimeClassIdPtr;

		/** Register a class into the MRPT internal list of "CSerializable" descendents.
		  *  Used internally in the macros DEFINE_SERIALIZABLE, etc...
		  * \sa getAllRegisteredClasses
		  */
		void MRPTDLLIMPEXP registerClass(const mrpt::utils::TRuntimeClassId* pNewClass);

		/** Returns a list with all the classes registered in the system through mrpt::utils::registerClass.
		  * \sa registerClass, findRegisteredClass
		  */
		std::vector<const mrpt::utils::TRuntimeClassId*> MRPTDLLIMPEXP getAllRegisteredClasses();

		/** Return info about a given class by its name, or NULL if the class is not registered
		  * \sa registerClass, getAllRegisteredClasses
		  */
		const TRuntimeClassId MRPTDLLIMPEXP * findRegisteredClass(const std::string &className);


		/** Access to runtime class ID for a defined class name.
		  */
		#define CLASS_ID(class_name) static_cast<const mrpt::utils::TRuntimeClassId*>(&class_name::class##class_name)

		/** Access to runtime class ID for a defined class name.
		  */
		#define CLASS_ID_NAMESPACE(class_name,namespaceName) static_cast<const mrpt::utils::TRuntimeClassId*>(&namespaceName::class_name::class##class_name)

		/** Access to runtime class ID for a defined template class name.
		  */
		#define CLASS_ID_TEMPLATE(class_name,T) static_cast<const mrpt::utils::TRuntimeClassId*>(& template <class T> class_name<T>::class##class_name)

		/** Evaluates to true if the given pointer to an object (derived from mrpt::utils::CSerializable) is of the given class.
		  */
		#define IS_CLASS( ptrObj, class_name )  ((ptrObj)->GetRuntimeClass()==CLASS_ID(class_name))

		/** Evaluates to true if the given pointer to an object (derived from mrpt::utils::CSerializable) is an instance of the given class or any of its derived classes.
		  */
		#define IS_DERIVED( ptrObj, class_name )  ((ptrObj)->GetRuntimeClass()->derivedFrom(CLASS_ID(class_name)))

		/** Auxiliary structure used for CSerializable runtime class ID support.
		  */
		struct MRPTDLLIMPEXP CLASSINIT
		{
			CLASSINIT(const mrpt::utils::TRuntimeClassId* pNewClass)
			{
				registerClass(pNewClass);
			}
		};


		/** The virtual base class which provides a unified interface for all persistent objects in MRPT.
		 *   For each class named <code>CMyClass</code>, a new type named <code>CMyClassPtr</code> will be created as a smart pointer suitable for
		 *    keeping referencing count smart pointers to objects of that class. By default the base class of all these smart pointers is CSerializablePtr.
		 *	 Refer to the tutorial about <a href="http://babel.isa.uma.es/mrpt/index.php/Serialization" >serialization</a> in the wiki.
		 * \sa CStream
		 */
		class MRPTDLLIMPEXP CSerializable
		{
		protected:
			static  mrpt::utils::TRuntimeClassId* _GetBaseClass();
		public:
			static const mrpt::utils::TRuntimeClassId classCSerializable;
			friend class mrpt::utils::CStream;

			/** Returns information about the class of an object in runtime.
			  */
			virtual const mrpt::utils::TRuntimeClassId* GetRuntimeClass() const;

			/** Returns a copy of the object, independtly of its class.
			  */
			virtual CSerializable *duplicate() const = 0;

			/** Cloning interface for smart pointers */
			CSerializable *clone() const { return duplicate(); }

			virtual ~CSerializable();

		protected:
			 /** Introduces a pure virtual method responsible for writing to a CStream.
			  *  This can not be used directly be users, instead use "stream << object;"
			  *   for writing it to a stream.
			  * \param out The output binary stream where object must be dumped.
			  * \param getVersion If NULL, the object must be dumped. If not, only the
			  *		version of the object dump must be returned in this pointer. This enables
			  *     the versioning of objects dumping and backward compatibility with previously
			  *     stored data.
			  *	\exception std::exception On any error, see CStream::WriteBuffer
			  * \sa CStream
			  */
			virtual void  writeToStream(mrpt::utils::CStream &out, int *getVersion) const = 0;

			 /** Introduces a pure virtual method responsible for loading from a CStream
			  *  This can not be used directly be users, instead use "stream >> object;"
			  *   for reading it from a stream or "stream >> object_ptr;" if the class is
			  *   unknown apriori.
			  * \param in The input binary stream where the object data must read from.
			  * \param version The version of the object stored in the stream: use this version
			  *                number in your code to know how to read the incoming data.
			  *	\exception std::exception On any error, see CStream::ReadBuffer
			  * \sa CStream
			  */
			virtual void  readFromStream(mrpt::utils::CStream &in, int version) = 0;
		}; // End of class def.


		/** @name Non-streaming serialization functions
		@{ */

		/** Used to pass MRPT objects into a CORBA-like object (strings). See doc about "Integration with BABEL".
		 * \param o The object to be serialized.
		 * \return The string containing the binay version of object.
		 * \sa StringToObject, <a href="http://babel.isa.uma.es/mrpt/index.php/Integration_with_BABEL">Integration with BABEL</a>
		 */
		std::string MRPTDLLIMPEXP ObjectToString(const CSerializable *o);

		/** Used to pass CORBA-like objects (strings) into a MRPT object.
		 * \param str An string generated with ObjectToString
		 * \param obj A currently empty pointer, where a pointer to the newly created object will be stored.
		 * \exception None On any internal exception, this function returns NULL.
		 * \sa ObjectToString, <a href="http://babel.isa.uma.es/mrpt/index.php/Integration_with_BABEL">Integration with BABEL</a>
		 */
		void MRPTDLLIMPEXP StringToObject(const std::string &str, CSerializablePtr &obj);

		/** Converts (serializes) an MRPT object into an array of bytes.
		 * \param o The object to be serialized.
		 * \param out_vector The vector which at return will contain the data. Size will be set automatically.
		 * \sa OctetVectorToObject, ObjectToString
		 */
		void MRPTDLLIMPEXP ObjectToOctetVector(const CSerializable *o, vector_byte & out_vector);

		/** Converts back (de-serializes) a sequence of binary data into a MRPT object, without prior information about the object's class.
		 * \param in_data The serialized input data representing the object.
		 * \param obj The newly created object will be stored in this smart pointer.
		 * \exception None On any internal exception, this function returns a NULL pointer.
		 * \sa ObjectToOctetVector, StringToObject
		 */
		void MRPTDLLIMPEXP OctetVectorToObject(const vector_byte & in_data, CSerializablePtr &obj);

		/** Converts (serializes) an MRPT object into an array of bytes within a std::string, without codifying to avoid NULL characters.
		 *  This is therefore more efficient than ObjectToString
		 * \param o The object to be serialized.
		 * \param out_vector The string which at return will contain the data. Size will be set automatically.
		 * \sa RawStringToObject, ObjectToOctetVector
		 */
		void MRPTDLLIMPEXP ObjectToRawString(const CSerializable *o, std::string & out_str);

		/** Converts back (de-serializes) a sequence of binary data within a std::string into a MRPT object, without prior information about the object's class.
		 * \param in_data The serialized input data representing the object.
		 * \param obj The newly created object will be stored in this smart pointer.
		 * \exception None On any internal exception, this function returns a NULL pointer.
		 * \sa ObjectToRawString
		 */
		void MRPTDLLIMPEXP RawStringToObject(const std::string & in_str, CSerializablePtr &obj);

		/** @} */



		/** @name Conversion of type to string at compile time
		  * IMPORTANT: See also the implementation of Serialization for STL containers in mrpt/utils/stl_extensions.h
		@{ */

		/** A template to obtain the type of its argument as a string at compile time.
		  *  It works with all classes derived from  CSerializable, plus many specializations for the plain data types (bool, double, uint8_t, etc...)
		  *   For example:
		  *  \code
		  *     cout << TTypeName<double>::get() << endl;                          // "double"
		  *   	cout << TTypeName<CPose2D>::get() << endl;                         // "CPose2D"
		  *   	cout << TTypeName<mrpt::slam::COccupancyGridMap2D>::get() << endl; // "COccupancyGridMap2D"
		  *  \endcode
		  */
		template<typename T>
		struct TTypeName
		{
			static std::string get() {
				return std::string( T::classinfo->className );
			}
		};

		#define MRPT_DECLARE_TTYPENAME(_TYPE) \
			template<> struct TTypeName <_TYPE > { \
				static std::string get() { return std::string(#_TYPE); } };

		#define MRPT_DECLARE_TTYPENAME_PTR(_TYPE) \
			template<> struct TTypeName <_TYPE##Ptr> { \
			static std::string get() { return TTypeName<_TYPE>::get(); }	};

		MRPT_DECLARE_TTYPENAME(bool)
		MRPT_DECLARE_TTYPENAME(double)
		MRPT_DECLARE_TTYPENAME(float)
		MRPT_DECLARE_TTYPENAME(uint64_t)
		MRPT_DECLARE_TTYPENAME(int64_t)
		MRPT_DECLARE_TTYPENAME(uint32_t)
		MRPT_DECLARE_TTYPENAME(int32_t)
		MRPT_DECLARE_TTYPENAME(uint16_t)
		MRPT_DECLARE_TTYPENAME(int16_t)
		MRPT_DECLARE_TTYPENAME(uint8_t)
		MRPT_DECLARE_TTYPENAME(int8_t)

		MRPT_DECLARE_TTYPENAME(std::string)

		#define MRPT_DECLARE_TTYPENAME_CONTAINER(_CONTAINER) \
			template< typename V > \
			struct TTypeName <_CONTAINER<V> > { \
				static std::string get() { \
					return std::string( #_CONTAINER )+std::string("<")+std::string( TTypeName<V>::get() ) + std::string(">"); \
				} \
			};

		MRPT_DECLARE_TTYPENAME_CONTAINER( std::vector )
		MRPT_DECLARE_TTYPENAME_CONTAINER( std::deque )
		MRPT_DECLARE_TTYPENAME_CONTAINER( std::list )
		MRPT_DECLARE_TTYPENAME_CONTAINER( std::set )

		// These ones act as a "translation" between vector_XXX types and their base classes:
		#define MRPT_DECLARE_TTYPENAME_MAP_FOR_VECTOR(_CONT) \
			template<> struct TTypeName <_CONT> : TTypeName<std::vector<_CONT::value_type> > { };

		MRPT_DECLARE_TTYPENAME_MAP_FOR_VECTOR(vector_signed_byte )
		MRPT_DECLARE_TTYPENAME_MAP_FOR_VECTOR(vector_signed_word)
		MRPT_DECLARE_TTYPENAME_MAP_FOR_VECTOR(vector_int)
		MRPT_DECLARE_TTYPENAME_MAP_FOR_VECTOR(vector_long)
		MRPT_DECLARE_TTYPENAME_MAP_FOR_VECTOR(vector_float)
		MRPT_DECLARE_TTYPENAME_MAP_FOR_VECTOR(vector_double)
		MRPT_DECLARE_TTYPENAME_MAP_FOR_VECTOR(vector_byte)
		MRPT_DECLARE_TTYPENAME_MAP_FOR_VECTOR(vector_word)
		MRPT_DECLARE_TTYPENAME_MAP_FOR_VECTOR(vector_uint)

        #if MRPT_WORD_SIZE!=32  // If it's 32 bit, size_t <=> uint32_t
		MRPT_DECLARE_TTYPENAME_MAP_FOR_VECTOR(vector_size_t)
        #endif


		#define MRPT_DECLARE_TTYPENAME_CONTAINER_ASSOC(_CONTAINER) \
			template< typename K, typename V > \
			struct TTypeName <_CONTAINER<K,V> > { \
				static std::string get() { \
					return std::string( #_CONTAINER )+std::string("<")+std::string( TTypeName<K>::get() )+ std::string(",")+std::string( TTypeName<V>::get() )+std::string(">"); \
				} \
			};

		MRPT_DECLARE_TTYPENAME_CONTAINER_ASSOC( std::map )
		MRPT_DECLARE_TTYPENAME_CONTAINER_ASSOC( std::multimap )


		template< typename T1, typename T2 >
		struct TTypeName <std::pair<T1,T2> >	{
			static std::string get() {
				return std::string("std::pair<")+std::string( TTypeName<T1>::get() )+ std::string(",")+std::string( TTypeName<T2>::get() )+std::string(">");
			}
		};

		/** @} */


		/** This declaration must be inserted in all CSerializable classes definition, within the class declaration.
		  */
		#define DEFINE_SERIALIZABLE(class_name) \
		protected: \
			static  const mrpt::utils::TRuntimeClassId* _GetBaseClass(); \
			static mrpt::utils::CLASSINIT _init_##class_name;\
			void  writeToStream(mrpt::utils::CStream &out, int *getVersion) const;\
			void  readFromStream(mrpt::utils::CStream &in, int version);\
		public: \
			typedef class_name##Ptr SmartPtr; \
			static  mrpt::utils::TRuntimeClassId  class##class_name; \
			static  const mrpt::utils::TRuntimeClassId *classinfo; \
			virtual const mrpt::utils::TRuntimeClassId* GetRuntimeClass() const; \
			static  CSerializable* CreateObject(); \
			static class_name##Ptr Create(); \
			virtual CSerializable *duplicate() const;

		/**  This declaration must be inserted in all CSerializable classes definition, before the class declaration.
		  */
		#define DEFINE_SERIALIZABLE_PRE_CUSTOM_LINKAGE(class_name,_LINKAGE_) \
			class class_name; \
			struct _LINKAGE_ class_name##Ptr : public mrpt::utils::CSerializablePtr \
			{ \
				class_name##Ptr() : mrpt::utils::CSerializablePtr(static_cast<mrpt::utils::CSerializable*>(NULL)) { } \
				explicit class_name##Ptr(class_name* p) : mrpt::utils::CSerializablePtr( reinterpret_cast<mrpt::utils::CSerializable*>(p) ) { } \
				explicit class_name##Ptr(const mrpt::utils::CSerializablePtr & p) : mrpt::utils::CSerializablePtr(p)  \
				{  ASSERTMSG_( p->GetRuntimeClass()->derivedFrom(#class_name),::mrpt::format("Wrong typecasting of smart pointers: %s -> %s",p->GetRuntimeClass()->className, #class_name) )  } \
				void setFromPointerDoNotFreeAtDtor(const class_name* p) { \
					this->set(const_cast<mrpt::utils::CSerializable*>(reinterpret_cast<const mrpt::utils::CSerializable*>(p))); m_holder->increment(); \
				} \
				class_name * pointer() { return reinterpret_cast<class_name*>(mrpt::utils::CSerializablePtr::pointer()); } \
				const class_name * pointer() const { return reinterpret_cast<const class_name*>(mrpt::utils::CSerializablePtr::pointer()); } \
				class_name* operator ->(void) { return reinterpret_cast<class_name*>( mrpt::utils::CSerializablePtr::operator ->() ); } \
				const class_name* operator ->(void) const { return reinterpret_cast<const class_name*>( mrpt::utils::CSerializablePtr::operator ->() ); } \
				class_name& operator *(void) { return *reinterpret_cast<class_name*>( mrpt::utils::CSerializablePtr::operator ->() ); } \
				const class_name& operator *(void) const { return *reinterpret_cast<const class_name*>( mrpt::utils::CSerializablePtr::operator ->() ); } \
			};\
			_LINKAGE_ ::mrpt::utils::CStream& operator>>(mrpt::utils::CStream& in, class_name##Ptr &pObj);


		/**  This declaration must be inserted in all CSerializable classes definition, before the class declaration.
		  */
		#define DEFINE_SERIALIZABLE_PRE(class_name) \
			DEFINE_SERIALIZABLE_PRE_CUSTOM_LINKAGE(class_name, MRPTDLLIMPEXP )

		/**  This declaration must be inserted in all CSerializable classes definition, before the class declaration.
		  */
		#define DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE_LINKAGE(class_name, base_name, _LINKAGE_ ) \
			class class_name; \
			struct _LINKAGE_ class_name##Ptr : public base_name##Ptr \
			{ \
				class_name##Ptr() : base_name##Ptr(static_cast<base_name*>(NULL)) { } \
				explicit class_name##Ptr(class_name* p) : base_name##Ptr( reinterpret_cast<base_name*>(p) ) { } \
				explicit class_name##Ptr(const base_name##Ptr & p) : base_name##Ptr(p) { ASSERTMSG_( p->GetRuntimeClass()->derivedFrom(#class_name),::mrpt::format("Wrong typecasting of smart pointers: %s -> %s",p->GetRuntimeClass()->className, #class_name) )  } \
				explicit class_name##Ptr(const mrpt::utils::CSerializablePtr & p) : base_name##Ptr(p) { ASSERTMSG_( p->GetRuntimeClass()->derivedFrom(#class_name),::mrpt::format("Wrong typecasting of smart pointers: %s -> %s",p->GetRuntimeClass()->className, #class_name) )  } \
				class_name * pointer() { return reinterpret_cast<class_name*>(base_name##Ptr::pointer()); } \
				const class_name * pointer() const { return reinterpret_cast<const class_name*>(base_name##Ptr::pointer()); } \
				class_name* operator ->(void) { return reinterpret_cast<class_name*>( base_name##Ptr::operator ->() ); } \
				const class_name* operator ->(void) const { return reinterpret_cast<const class_name*>( base_name##Ptr::operator ->() ); } \
				class_name& operator *(void) { return *reinterpret_cast<class_name*>( base_name##Ptr::operator ->() ); } \
				const class_name& operator *(void) const { return *reinterpret_cast<const class_name*>( base_name##Ptr::operator ->() ); } \
			}; \
			_LINKAGE_ ::mrpt::utils::CStream& operator>>(mrpt::utils::CStream& in, class_name##Ptr &pObj);


		/**  This declaration must be inserted in all CSerializable classes definition, before the class declaration.
		  */
		#define DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE(class_name, base_name) \
			DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE_LINKAGE(class_name, base_name, MRPTDLLIMPEXP )

		/** This must be inserted in all CSerializable classes implementation files:
		  */
		#define IMPLEMENTS_SERIALIZABLE(class_name, base,NameSpace) \
			mrpt::utils::CSerializable* NameSpace::class_name::CreateObject() \
				{ return static_cast<mrpt::utils::CSerializable*>( new NameSpace::class_name ); } \
			NameSpace::class_name##Ptr NameSpace::class_name::Create() \
				{ return NameSpace::class_name##Ptr( new NameSpace::class_name ); } \
			const mrpt::utils::TRuntimeClassId* NameSpace::class_name::_GetBaseClass() \
				{ return CLASS_ID(base); } \
			mrpt::utils::TRuntimeClassId NameSpace::class_name::class##class_name = { \
				#class_name, sizeof(class NameSpace::class_name), NameSpace::class_name::CreateObject, \
					&class_name::_GetBaseClass }; \
			const mrpt::utils::TRuntimeClassId *NameSpace::class_name::classinfo = & NameSpace::class_name::class##class_name; \
			const mrpt::utils::TRuntimeClassId* NameSpace::class_name::GetRuntimeClass() const \
			{ return CLASS_ID_NAMESPACE(class_name,NameSpace); } \
			mrpt::utils::CLASSINIT NameSpace::class_name::_init_##class_name(CLASS_ID(base)); \
			mrpt::utils::CStream& NameSpace::operator>>(mrpt::utils::CStream& in, NameSpace::class_name##Ptr &pObj) \
			{ pObj = NameSpace::class_name##Ptr( in.ReadObject() ); return in; } \
			mrpt::utils::CSerializable * NameSpace::class_name::duplicate() const \
			{ return static_cast<mrpt::utils::CSerializable*>( new NameSpace::class_name(*this) ); }


		/** This declaration must be inserted in virtual CSerializable classes definition:
		  */
		#define DEFINE_VIRTUAL_SERIALIZABLE(class_name) \
		protected: \
			static const mrpt::utils::TRuntimeClassId* _GetBaseClass(); \
		public: \
			static const mrpt::utils::TRuntimeClassId class##class_name; \
			virtual const mrpt::utils::TRuntimeClassId* GetRuntimeClass() const; \
			friend class mrpt::utils::CStream; \

		/** This must be inserted as implementation of some required members for
		  *  virtual CSerializable classes:
		  */
		#define IMPLEMENTS_VIRTUAL_SERIALIZABLE(class_name, base_class_name,NameSpace) \
			const mrpt::utils::TRuntimeClassId* class_name::_GetBaseClass() \
				{ return CLASS_ID(base_class_name); } \
			const mrpt::utils::TRuntimeClassId class_name::class##class_name = { \
				#class_name, sizeof(class class_name), NULL, \
					&class_name::_GetBaseClass }; \
			const mrpt::utils::TRuntimeClassId* class_name::GetRuntimeClass() const \
				{ return CLASS_ID(class_name); } \
			mrpt::utils::CStream& NameSpace::operator>>(mrpt::utils::CStream& in, class_name##Ptr &pObj) \
			{ pObj = class_name##Ptr( in.ReadObject() ); return in; }


	} // End of namespace
} // End of namespace

// I want these operators to reside in std so STL algorithms can always find them.
namespace std
{
	/**  This operator enables comparing two smart pointers with "==" to test whether they point to the same object.
	  */
	template <typename T,typename C>
	bool operator == ( const stlplus::smart_ptr_base<T,C>&a,const stlplus::smart_ptr_base<T,C>&b) {
		return a.aliases(b);
	}
	/**  This operator enables comparing two smart pointers with "!=" to test whether they don't point to the same object.
	  */
	template <typename T,typename C>
	bool operator != ( const stlplus::smart_ptr_base<T,C>&a,const stlplus::smart_ptr_base<T,C>&b) {
		return !a.aliases(b);
	}
}

#endif
