/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef  CBaseGUIWindow_H
#define  CBaseGUIWindow_H

#include <mrpt/synch.h>
#include <mrpt/utils/CSerializable.h>
#include <mrpt/utils/safe_pointers.h>
#include <mrpt/gui/keycodes.h>

namespace mrpt
{
	namespace gui
	{
		using namespace mrpt::utils;

		class CWindowDialog;
		class CWindowDialogPlots;
		class C3DWindowDialog;

		/** The base class for GUI window classes.
		 */
		class MRPTDLLIMPEXP CBaseGUIWindow
		{
			friend class CWindowDialog;
			friend class C3DWindowDialog;
			friend class CWindowDialogPlots;

		protected:
			synch::CSemaphore 	m_semThreadReady;	//!< This semaphore will be signaled when the wx window is built and ready.
			synch::CSemaphore 	m_semWindowDestroyed; //!< This semaphore will be signaled when the wx window is destroyed.
			std::string			m_caption;	//!< The caption of the window
			void_ptr_noncopy	m_hwnd;	//!< The window handle

			/* Auxiliary */
			volatile bool             m_keyPushed;
			volatile int              m_keyPushedCode;
			volatile mrptKeyModifier  m_keyPushedModifier;

		public:
			void * getWxObject() { return m_hwnd.get(); } //!< Read-only access to the wxDialog object.
			void notifyChildWindowDestruction();	//!< Called by wx main thread to set m_hwnd to NULL.
			void notifySemThreadReady();	//!< Called by wx main thread to signal the semaphore that the wx window is built and ready.

		public:
			CBaseGUIWindow(const std::string &initial_caption = std::string() );
			virtual ~CBaseGUIWindow();

			/** Returns false if the user has already closed the window.
			  */
			bool isOpen();

			/** Resizes the window, stretching the image to fit into the display area.
			 */
			virtual void  resize( unsigned int width, unsigned int height ) = 0;

			/** Changes the position of the window on the screen.
			 */
			virtual void  setPos( int x, int y ) = 0;

			/** Changes the window title text.
			  */
			virtual void setWindowTitle( const std::string &str )=0;

			/** Waits for any key to be pushed on the image or the console, and returns the key code.
			  *  This method remove key strokes previous to its call, so it will always wait. To get
			  *   the latest pushed key, see
			  *
			  * \param ignoreControlKeys If set to false, any push of shift, cmd, control, etc... will make this method to return.
			  * \param out_pushModifier If set to !=NULL, the modifiers of the key stroke will be saved here.
			  * \return The virtual key code, as defined in mrptKeyCode (a replication of wxWidgets key codes).
			  *
			  * \sa getPushedKey, Key codes in the enum mrptKeyCode
			  */
			int  waitForKey(bool ignoreControlKeys = true, mrptKeyModifier *out_pushModifier=NULL);

			/** Returns true if a key has been pushed, without blocking waiting for a new key being pushed.
			  * \sa waitForKey, clearKeyHitFlag
			  */
			bool  keyHit() const { return m_keyPushed; }

			/** Assure that "keyHit" will return false until the next pushed key.
			  * \sa keyHit, waitForKey
			  */
			void  clearKeyHitFlag() { m_keyPushed = false; }

			/** Returns the latest pushed key, or 0 if there is no new key stroke.
			  * \param out_pushModifier If set to !=NULL, the modifiers of the key stroke will be saved here.
			  * \return The virtual key code, as defined in mrptKeyCode (a replication of wxWidgets key codes).
			  *
			  * \sa keyHit, waitForKey
			  */
			int getPushedKey(mrptKeyModifier *out_pushModifier=NULL);


		}; // End of class def.

	} // End of namespace

} // End of namespace

#endif
