/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#ifndef CCameraSensor_H
#define CCameraSensor_H

#include <mrpt/poses/CPose3D.h>
#include <mrpt/slam/CObservation.h>
#include <mrpt/utils/CDebugOutputCapable.h>
#include <mrpt/utils/CConfigFileBase.h>
#include <mrpt/hwdrivers/CGenericSensor.h>
#include <mrpt/hwdrivers/CFFMPEG_InputStream.h>
#include <mrpt/vision/CImageGrabber_OpenCV.h>
#include <mrpt/vision/CImageGrabber_dc1394.h>
#include <mrpt/vision/CStereoGrabber_Bumblebee.h>
#include <mrpt/utils/CFileGZInputStream.h>

#include <mrpt/gui/CDisplayWindow.h>

namespace mrpt
{
	namespace hwdrivers
	{
		/** The central class for camera grabbers in MRPT, implementing the "generic sensor" interface.
		  *   This class provides the user with a uniform interface to a variety of other classes which manage only one specific camera "driver" (opencv, ffmpeg, bumblebee,...)
		  *
		  *   Following the "generic sensor" interface, all the parameters must be passed int the form of a configuration file, which may be also formed on the fly (without being a real config file) as in this example:
		  *
		  *  \code
		  *   CCameraSensor myCam;
		  *   string str = "[CONFIG]\n grabber_type=opencv \n";
		  *   CConfigFileMemory	cfg(str);
		  *   myCam.loadConfig(cfg,"CONFIG");
		  *   myCam.initialize();
		  *   CObservationPtr obs = myCam.getNextFrame();
		  *  \endcode
		  *
		  *  Images can be retrieves through the normal "doProcess()" interface, or the specific method "getNextFrame()".
		  *
		  *  These is the list of all the accepted parameters:
		  *
		  *  \code
		  *  PARAMETERS IN THE ".INI"-LIKE CONFIGURATION STRINGS:
		  * -------------------------------------------------------
		  *   [supplied_section_name]
		  *    // Select one of the grabber implementations:
		  *    grabber_type       = opencv | dc1394 | bumblebee | ffmpeg | rawlog
		  *    preview_decimation = 0     // N<=0 (or not present): No preview; N>0, display 1 out of N captured frames.
		  *    preview_reduction  = 0     // 0 or 1 (or not present): The preview shows the actual image. For 2,3,..., reduces the size of the image by that factor, only for the preview window.
		  *    capture_grayscale  = 0     // 1:capture in grayscale, whenever the driver allows it. Default=0
		  *
		  *    // Options for grabber_type= opencv
		  *    cv_camera_index  = 0       // [opencv] Number of camera to open
		  *    cv_camera_type   = CAMERA_CV_AUTODETECT
		  *    cv_frame_width   = 640     // [opencv] Capture width (not present or set to 0 for default)
		  *    cv_frame_height  = 480     // [opencv] Capture height (not present or set to 0 for default)
		  *    cv_fps           = 15      // [opencv] IEEE1394 cams only: Capture FPS (not present or 0 for default)
		  *    cv_gain          = 0       // [opencv] Camera gain, if available (nor present or set to 0 for default).
		  *
		  *    // Options for grabber_type= dc1394
		  *    dc1394_camera_guid   = 0 | 0x11223344    // 0 (or not present): the first camera; A hexadecimal number: The GUID of the camera to open
		  *    dc1394_camera_unit   = 0     			// 0 (or not present): the first camera; 0,1,2,...: The unit number (within the given GUID) of the camera to open (Stereo cameras: 0 or 1)
		  *    dc1394_frame_width	= 640
		  *    dc1394_frame_height	= 480
		  *    dc1394_framerate		= 15					// eg: 7.5, 15, 30, 60, etc... For posibilities see mrpt::vision::TCaptureOptions_dc1394
		  *    dc1394_mode7         = -1                    // -1: Ignore, i>=0, set to MODE7_i
		  *    dc1394_color_coding	= COLOR_CODING_YUV422	// For posibilities see mrpt::vision::TCaptureOptions_dc1394
		  *    dc1394_shutter		= -1	// A value, or -1 (or not present) for not to change this parameter in the camera
		  *    dc1394_gain			= -1	// A value, or -1 (or not present) for not to change this parameter in the camera
		  *    dc1394_gamma			= -1	// A value, or -1 (or not present) for not to change this parameter in the camera
		  *    dc1394_brightness	= -1	// A value, or -1 (or not present) for not to change this parameter in the camera
		  *    dc1394_exposure		= -1	// A value, or -1 (or not present) for not to change this parameter in the camera
		  *    dc1394_sharpness		= -1	// A value, or -1 (or not present) for not to change this parameter in the camera
		  *    dc1394_white_balance	= -1	// A value, or -1 (or not present) for not to change this parameter in the camera
		  *
		  *    // Options for grabber_type= bumblebee
		  *    bumblebee_camera_index  = 0       // [bumblebee] Number of camera within the firewire bus to open (typically = 0)
		  *    bumblebee_frame_width   = 640     // [bumblebee] Capture width (not present or set to 0 for default)
		  *    bumblebee_frame_height  = 480     // [bumblebee] Capture height (not present or set to 0 for default)
		  *    bumblebee_fps           = 15      // [bumblebee] Capture FPS (not present or 0 for default)
		  *    bumblebee_mono          = 0|1     // [bumblebee] OPTIONAL: If this parameter is present, monocular (0:left, 1:right) images will be grabbed instead of stereo pairs.
		  *    bumblebee_get_rectified = 0|1     // [bumblebee] Determines if the camera should grab rectified or raw images (1 is the default)
		  *
		  *    // Options for grabber_type= ffmpeg
		  *    ffmpeg_url             = rtsp://127.0.0.1      // [ffmpeg] The video file or IP camera to open
		  *
		  *    // Options for grabber_type= rawlog
		  *    rawlog_file            = mylog.rawlog          // [rawlog] This can be used to simulate the capture of images already grabbed in the past in the form of a MRPT rawlog.
		  *    rawlog_camera_sensor_label  = CAMERA1          // [rawlog] If this field is not present, all images found in the rawlog will be retrieved. Otherwise, only those observations with a matching sensor label.
		  *
		  *    // For externaly stored images, the format of image files (default=jpg)
		  *    //external_images_format  = jpg
		  *
		  *    // For externaly stored images: whether to spawn independent threads to save the image files.
		  *    //external_images_own_thread  = 1   // 0 or 1
		  *
		  *    // If external_images_own_thread=1, this changes the number of threads to launch
		  *    //  to save image files. The default is determined from mrpt::system::getNumberOfProcessors()
		  *    //  and should be OK unless you want to save processor time for other things.
		  *    //external_images_own_thread_count = 2    // >=1
		  *
		  *    // (Only when external_images_format=jpg): Optional parameter to set the JPEG compression quality:
		  *    //external_images_jpeg_quality = 95    // [1-100]. Default: 95
		  *
		  *    // Pose of the sensor on the robot:
		  *    pose_x=0		; (meters)
		  *    pose_y=0
		  *    pose_z=0
		  *    pose_yaw=0	; (Angles in degrees)
		  *    pose_pitch=0
		  *    pose_roll=0
		  *
		  *  \endcode
		  *
		  *  - "grabber_type" is the class to use internally for image capturing. Choices are "opencv" (Windows & Linux) or "dc1394" (Linux only for now, requires libdc1394-2).
		  *  - For the meaning of cv_camera_type and other parameters, refer to mrpt::vision::CImageGrabber_OpenCV
		  *  - For the parameters of dc1394 parameters, refer to generic IEEE1394 documentation, and to mrpt::vision::TCaptureOptions_dc1394.
		  *
		  *  Images can be saved in the "external storage" mode. See setPathForExternalImages and setExternalImageFormat. These methods
		  *   are called automatically from rawlog-grabber.
		  *
		  *  \note The execution rate (in rawlog-grabber) should be greater than the required capture FPS.
		  *  \note In Linux you may need to execute "chmod 666 /dev/video1394/ * " and "chmod 666 /dev/raw1394" for allowing any user R/W access to firewire cameras.
		  *  \sa mrpt::vision::CImageGrabber_OpenCV, mrpt::vision::CImageGrabber_dc1394, CGenericSensor, prepareVideoSourceFromUserSelection
		  */
		class HWDLLIMPEXP CCameraSensor : public utils::CDebugOutputCapable, public CGenericSensor
		{
			DEFINE_GENERIC_SENSOR(CCameraSensor)

		public:
			/** Constructor
			  *  The camera is not open until "initialize" is called.
			  */
			CCameraSensor();

			/** Destructor
			  */
			virtual ~CCameraSensor();

			/** This method should be called periodically (at least at 1Hz to capture ALL the real-time data)
			*  It is thread safe, i.e. you can call this from one thread, then to other methods from other threads.
			*/
			void  doProcess();

			/** Retrieves the next frame from the video source, raising an exception on any error.
			  * \note The observations can be of the classes CObservationImage or CObservationStereoImages
			  */
			mrpt::slam::CObservationPtr getNextFrame();

			/** Tries to open the camera, after setting all the parameters with a call to loadConfig.
			  *  \exception This method must throw an exception with a descriptive message if some critical error is found.
			  */
			virtual void initialize();

			/** Close the camera (if open).
			  *   This method is called automatically on destruction.
			  */
			void close();

			/**  Set the path where to save off-rawlog images: empty (default) means save images embedded in the rawlog.
			  * \exception std::exception If the directory cannot be created
			  */
			void setPathForExternalImages( const std::string &directory );

			/**  Set the extension ("jpg","gif","png",...) that determines the format of images saved externally
			  *   The default is "jpg".
			  * \sa setPathForExternalImages, setExternalImageJPEGQuality
			  */
			void setExternalImageFormat( const std::string &ext ) {
				m_external_images_format = ext;
			}

			/** The quality of JPEG compression, when external images is enabled and the format is "jpg". \sa setExternalImageFormat */
			void setExternalImageJPEGQuality(const unsigned int quality) {
				m_external_images_jpeg_quality = quality;
			}
			unsigned int getExternalImageJPEGQuality()const  {
				return m_external_images_jpeg_quality;
			}

			/** This must be called before initialize() */
			void enableLaunchOwnThreadForSavingImages(bool enable=true) { m_external_images_own_thread = enable; };

		protected:
			poses::CPose3D		m_sensorPose;
			std::string			m_sensorLabel;

			std::string								m_grabber_type; //!< Can be "opencv",...
			bool									m_capture_grayscale;
			int										m_cv_camera_index;
			std::string								m_cv_camera_type;
			mrpt::vision::TCaptureCVOptions			m_cv_options;

			uint64_t								m_dc1394_camera_guid;
			int										m_dc1394_camera_unit;
			mrpt::vision::TCaptureOptions_dc1394	m_dc1394_options;
			int										m_preview_decimation;
			int										m_preview_reduction;

			int										m_bumblebee_camera_index;
			mrpt::vision::TCaptureOptions_bumblebee	m_bumblebee_options;
			int										m_bumblebee_monocam; // 0:Left, 1: Right, <0,>1 -> Stereo

			std::string								m_ffmpeg_url;

			std::string								m_rawlog_file;
			std::string								m_rawlog_camera_sensor_label;
			std::string								m_rawlog_detected_images_dir;

			std::string			m_path_for_external_images; //!< The path where to save off-rawlog images: empty means save images embedded in the rawlog.
			std::string			m_external_images_format; //!< The extension ("jpg","gif","png",...) that determines the format of images saved externally \sa setPathForExternalImages
			unsigned int		m_external_images_jpeg_quality; //!< For JPEG images, the quality (default=95%).

			bool				m_external_images_own_thread; //!< Whether to launch independent thread
			unsigned int		m_external_image_saver_count; //!< Number of working threads. Default:1, set to 2 in quad cores.
			std::vector<mrpt::system::TThreadHandle>  m_threadImagesSaver;

			bool 	m_threadImagesSaverShouldEnd;

			/** Loads specific configuration for the device from a given source of configuration parameters, for example, an ".ini" file, loading from the section "[iniSection]" (see utils::CConfigFileBase and derived classes)
			  *  See hwdrivers::CCameraSensor for the possible parameters
			  */
			void  loadConfig_sensorSpecific(
				const mrpt::utils::CConfigFileBase &configSource,
				const std::string	  &iniSection );

		private:
			/**  The OpenCV capture object. */
			mrpt::vision::CImageGrabber_OpenCV 		*m_cap_cv;

			/**  The dc1394 capture object. */
			mrpt::vision::CImageGrabber_dc1394 		*m_cap_dc1394;

			/**  The bumblebee capture object. */
			mrpt::vision::CStereoGrabber_Bumblebee *m_cap_bumblebee;

			/** The FFMPEG capture object */
			CFFMPEG_InputStream						*m_cap_ffmpeg;

			/** The input file for rawlogs */
			mrpt::utils::CFileGZInputStream			*m_cap_rawlog;

			int							 m_preview_counter;
			mrpt::gui::CDisplayWindowPtr m_preview_win1,m_preview_win2; //!< Normally we'll use only one window, but for stereo images we'll use two of them.

			mrpt::synch::CCriticalSection	m_csToSaveList;		//!< The critical section for m_toSaveList
			std::vector<TListObservations>	m_toSaveList;		//!< The queues of objects to be returned by getObservations, one for each working thread.

			void thread_save_images(unsigned int my_working_thread_index); //!< Thread to save images to files.

		}; // end class

		typedef stlplus::smart_ptr<CCameraSensor>    CCameraSensorPtr; //!< A smart pointer to a CCameraSensor

		/** Used only from MRPT apps: Use with caution since "panel" MUST be a "mrpt::gui::CPanelCameraSelection *"
		  */
		CCameraSensorPtr HWDLLIMPEXP prepareVideoSourceFromPanel(void *panel);

		/** Parse the user options in the wxWidgets "panel" and write the configuration into the given section of the given configuration file.
		  * Use with caution since "panel" MUST be a "mrpt::gui::CPanelCameraSelection *"
		  * \sa prepareVideoSourceFromUserSelection, prepareVideoSourceFromPanel, readConfigIntoVideoSourcePanel
		  */
		void HWDLLIMPEXP writeConfigFromVideoSourcePanel(
			void *panel,
			const std::string &in_cfgfile_section_name,
			mrpt::utils::CConfigFileBase *out_cfgfile
			);

		/** Parse the given section of the given configuration file and set accordingly the controls of the wxWidgets "panel".
		  * Use with caution since "panel" MUST be a "mrpt::gui::CPanelCameraSelection *"
		  * \sa prepareVideoSourceFromUserSelection, prepareVideoSourceFromPanel, writeConfigFromVideoSourcePanel
		  */
		void HWDLLIMPEXP readConfigIntoVideoSourcePanel(
			void *panel,
			const std::string &in_cfgfile_section_name,
			const mrpt::utils::CConfigFileBase *in_cfgfile
			);

		/** Show to the user a list of possible camera drivers and creates and open the selected camera.
		  */
		CCameraSensorPtr HWDLLIMPEXP prepareVideoSourceFromUserSelection();


	} // end namespace
} // end namespace

#endif
