/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CMATRIXD_H
#define CMATRIXD_H

#include <mrpt/utils/CSerializable.h>
#include <mrpt/math/CMatrixTemplateNumeric.h>
#include <mrpt/math/CVectorTemplate.h>
#include <mrpt/utils/utils_defs.h>


namespace mrpt
{
	namespace poses
	{
		class CPose2D;
		class CPose3D;
		class CPoint2D;
		class CPoint3D;
	}

	namespace math
	{
		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE_PRE( CMatrixD )

		/**  This class is a "CSerializable" wrapper for "CMatrixTemplateNumeric<double>".
		 */
		class MRPTDLLIMPEXP CMatrixD : public mrpt::utils::CSerializable, public CMatrixTemplateNumeric<double>
		{
			// This must be added to any CSerializable derived class:
			DEFINE_SERIALIZABLE( CMatrixD )
		public:
		public:
			/** Constructor */
			CMatrixD() : CMatrixTemplateNumeric<double>(1,1)
			{ }

			/** Constructor */
			CMatrixD(size_t row, size_t col) : CMatrixTemplateNumeric<double>(row,col)
			{ }

			/** Copy constructor */
			CMatrixD( const CMatrixTemplateNumeric<double> &m ) : CMatrixTemplateNumeric<double>(m)
			{ }

			/** Copy constructor
			  */
			CMatrixD( const CMatrixFloat &m ) : CMatrixTemplateNumeric<double>(0,0)
			{
				*this = m;
			}

			/** Constructor from a mrpt::poses::CPose2D, which generates a 3x1 matrix \f$ [x y \phi]^T \f$
			   */
			explicit CMatrixD( const mrpt::poses::CPose2D &p) : CMatrixDouble(p) {}

			/** Constructor from a mrpt::poses::CPose6D, which generates a 6x1 matrix \f$ [x y z yaw pitch roll]^T \f$
			   */
			explicit CMatrixD( const mrpt::poses::CPose3D &p) : CMatrixDouble(p) {}

			/** Constructor from a mrpt::poses::CPoint2D, which generates a 2x1 matrix \f$ [x y]^T \f$
			   */
			explicit CMatrixD( const mrpt::poses::CPoint2D &p) : CMatrixDouble(p) {}

			/** Constructor from a mrpt::poses::CPoint3D, which generates a 3x1 matrix \f$ [x y z]^T \f$
			   */
			explicit CMatrixD( const mrpt::poses::CPoint3D &p) : CMatrixDouble(p) {}



			/** Assignment operator
			*/
			CMatrixD & operator = (const CMatrixTemplateNumeric<double>& m)
			{
				CMatrixTemplateNumeric<double>::operator =(m);
				return *this;
			}

			/** Assignment operator
			*/
			CMatrixD & operator = (const CMatrixFloat& m)
			{
				setSize(m.getRowCount(),m.getColCount());

				for (size_t i=0;i<m.getRowCount();i++)
					for (size_t j=0;j<m.getColCount();j++)
						(*this)(i,j) = static_cast<double>( m(i,j) );
				return *this;
			}

		}; // end of class definition



	} // End of namespace
} // End of namespace

#endif
