/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CVisualOdometryStereo_H
#define CVisualOdometryStereo_H

#include <mrpt/slam/CObservationStereoImages.h>
#include <mrpt/poses/CPose3DPDFGaussian.h>
#include <mrpt/poses/CPose3DPDFParticles.h>
#include <mrpt/poses/CPose3D.h>
#include <mrpt/poses/CPoint2DPDFGaussian.h>
#include <mrpt/slam/CMetricMap.h>
#include <mrpt/slam/CObservationVisualLandmarks.h>
#include <mrpt/scan_matching.h>

#include <mrpt/vision/CFeatureExtraction.h>
#include <mrpt/vision/utils.h>

#include <mrpt/math/CMatrix.h>
#include <mrpt/utils/CImage.h>
#include <mrpt/utils/CConfigFile.h>
#include <mrpt/gui/CDisplayWindow.h>

#include <mrpt/stereoslam/link_pragmas.h>

#define	VISUALODOMETRY_USES_GUI 1

namespace mrpt
{
	namespace stereoslam
	{
		using namespace mrpt;
		using namespace mrpt::vision;

		struct STSLAMDLLIMPEXP TOdometryInfo
		{
			CObservationStereoImagesPtr				m_obs;						//!< The inner observation of the odometer
			CMatchedFeatureList						m_mfList;					//!< The inner list of matched features
			CFeatureList							m_leftFeats, m_rightFeats;	//!< The inner lists of features
			slam::CObservationVisualLandmarks		m_Prev_cloud;				//!< The inner cloud of 3D points

			TOdometryInfo( );
		};

		/** A structure containing options for the visual odometry
		  */
		struct STSLAMDLLIMPEXP TOdometryOptions : public mrpt::utils::CLoadableOptions
		{
			enum TCov_Computation_Method
			{
				TCov_Linear = -1,			//!< Linear propagation of the uncertainty
				TCov_UT,					//!< Using the Unscented Transform (UT)
				TCov_Particles				//!< Using particles (not implemented yet)
			};

			// General
			TCov_Computation_Method		covMethod;					//!< Specifies the method for estimating the covariance of the change in pose
			unsigned int				minNumFeatures;				//!< Minimun number of features within the odometer (to be tracked)
			double						minDispX;					//!< Minimun allowed disparity in the X-coordinate within the image
			double						minDispY;					//!< Minimun allowed disparity in the Y-coordinate within the image
			double						minSetDist;					//!< Minimun allowed disparity in the Y-coordinate within the image
			unsigned int				debug_nsamples;				//!< Number of samples to employ when using process_dbg (for estimating the real covariance)
			bool						VERBOSE;					//!< Display on-screen process information
			bool						PLOT_IMAGES;				//!< Select whether of not to plot both the images and tracked image features which are being processed
			bool						SAVE_FILES;					//!< Save a set of intermediate and final results in files
			bool						USE_DISPERSION;				//!< Use a measure of the dispersion of the image features
			bool						DEBUG_ESTIMATE_COVARIANCE;	//!< Employ the process_dbg method instead of the process (for debugging)
			bool						USE_RANSAC;					//!< Use RANSAC when estimating the change in pose
			bool						TIME_INFO;					//!< Save the timing information in a pair of files: ...
			string						OUT_DIR;					//!< Output directory for storing files and time info

			TOdometryOptions( );

			/** See utils::CLoadableOptions
			  */
			void  loadFromConfigFile(
				const mrpt::utils::CConfigFileBase	&source,
				const std::string		&section);

			/** See utils::CLoadableOptions
			  */
			void  dumpToTextStream(mrpt::utils::CStream		&out) const;

		}; // end struct TOdometryOptions

		/** Visual odometry for stereo cameras.
		  *  The method is described in the paper:
		  *   - F.-A. Moreno, J.-L. Blanco, and J. Gonzalez, "An Efficient Closed-form Solution to Probabilistic 6D Visual Odometry for a Stereo Camera," in Proc. Advanced Concepts for Intelligent Vision Systems, 2007, pp. 932-942.
		  *
 		  *  \note This class is undergoing changes and it's in an experimental status.
 		  */
		class STSLAMDLLIMPEXP CVisualOdometryStereo
		{

		public:

			/** Constructor
			  */
			CVisualOdometryStereo();

			/** Destructor
			  */
			virtual ~CVisualOdometryStereo();

			/** Resets the state of the Visual Odometry system.
			  */
			void  reset();

			/** Load the system options from an INI file.
			  */
			void loadOptions( const std::string &configFile );

			/** Performs one step of the visual odometry process for estimating the real output covariance
			  * (for internal use only)
			  */
			void process_dbg( 
				const slam::CObservationStereoImages &inObs1, 
				const slam::CObservationStereoImages &inObs2, 
				std::vector<CPose3D> &outEst );

			/** Performs one step of the Visual Odometry system. It takes the input stereo observation and estimates the
			  *	camera movement between the previous observation and this one.
			  * If this is the first observation inserted into the visual odometry system, this functions return FALSE.
			  * Otherwise it returns TRUE and yields a CPose3DPDFGaussian object containing the change in the 6D camera pose.
			  */
			bool  process( 
				const mrpt::slam::CObservationStereoImagesPtr &inObs, 
				mrpt::poses::CPose3DPDFGaussian &outEst );

			/** Gets a CONST version of the info stored within the odometer 
			  */
			const TOdometryInfo &getInfo();

			TStereoSystemParams		stereoParams;		//!< Parameters to perform projection from stereo features to 3D landmarks
			TMatchingOptions		matchingOptions;	//!< Parameters to perform stereo matching
			TOdometryOptions		odometryOptions;	//!< Parameters to compute visual odometry

			unsigned int							m_nIter;					//!< Number of iterations performed by the visual odometry system

		protected:

			TOdometryInfo							m_odInfo;					//!< The structure containing all the relevant info of the odometer (can be retrieved through the 'getInfo()' method)

			bool									m_firstObs;					//!< Is this the first observation to introduce into the system?
			CFeatureExtraction						m_fExtract;					//!< The feature extractor of the system

			//CMatchedFeatureList					m_mfList;					//!< A stored set of matched features to perform tracking and projection
			//
			//slam::CObservationVisualLandmarks		m_Prev_cloud;				//!< The set of previous 3D points (to perform the visual odometry estimation)
			//CObservationStereoImagesPtr			m_stereoObs;
			//utils::CImage							m_imgL, m_imgR;				//!< The pair of previous images (to perform keypoints tracking)

			//CFeatureList							m_leftFeats, m_rightFeats;	//!< The sets of keypoints in the images (to perform tracking)

			// DEBUG
			std::vector<CPose3D>					m_dbg_outEstimations;		//!< An internal vector containing the output estimations for dbg
			unsigned int							m_dbg_sampleCount;			//!< An internal counter for the samples in debugging
			CFeatureList							m_dbg_orLF, m_dbg_orRF;		//!< The sets of original keypoints in the images (for debugging)

#if VISUALODOMETRY_USES_GUI
			gui::CDisplayWindow						*m_wind1,*m_wind2;			//!< A pair of display windows to show the images (if enabled)
#endif

			/** Compute the covariance of the change in pose using the method defined in the options
			  * (for internal use only)
			  */
			void  computeRTCovariance(
				const mrpt::poses::CPose3D &meanPose,
				const mrpt::slam::CMetricMap::TMatchingPairList &list,
				const mrpt::slam::CObservationVisualLandmarks &Pre_Cloud,
				const mrpt::slam::CObservationVisualLandmarks &Cur_Cloud,
				math::CMatrixD &RTCov );

			/** Fills in the KLT lists from the matched SIFTS
			  * (for internal use only)
			  */
			void  storeMatchedFeatureList(
				const CMatchedFeatureList &mfList );			// The list of matched features

			/** Adds the new set of matched points into the internal lists of points
			  * (for internal use only)
			  */
			void  addMatches(
				const CMatchedFeatureList &mfList );

			/** Construct a CMatchedFeatureList from the internal lists of features
			  * (for internal use only)
			  */
			void getMatchedList(
				CMatchedFeatureList &mfList );

			/** Get a CFeatureList with positions sampled from an input CFeatureList
			  * (for internal use only)
			  */
			static void dbgGetSampledList(
				const CFeatureList &inList, CFeatureList &outList, const double sigma );

			/** Checks two sets of tracked features and builds an output matched feature list.
			  * (for internal use only)
			  */
			void checkTrackedFeatures( 
				CFeatureList &leftList, 
				CFeatureList &rightList, 
				CMatchedFeatureList &mfList );

		}; // end of class CVisualOdometryStereo

	} // end of namespace
} // end of namespace

#endif
