/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/hwdrivers/CRovio.h>
#include <mrpt/hwdrivers/CJoystick.h>
#include <mrpt/utils/utils_defs.h>
#include <mrpt/core.h>
#include <math.h>
#include <cv.h>
#include <fstream>

#define PI M_PI

using namespace mrpt::hwdrivers;
using namespace mrpt::utils;
using namespace std;
using namespace mrpt::gui;


/*--------------------------------
|            VARIABLES            |
---------------------------------*/
enum matching_sort
	{
		FM_match=0,
		complete_match
	};

struct data_st
{
	CRovio Robot;
	bool showvideo, features_taken, matched, matching_done, fm_calculated;
	CFeatureList	featsHarris, featsHarris2, featsSURF, featsSURF2, featsSIFT_Hess, featsSIFT_Hess2;
	CMatchedFeatureList Harris_matched, SIFT_matched, SURF_matched;
	CMatrixDouble33 FM_Harris, FM_SIFT, FM_SURF;
	matching_sort matching_type;
};

CDisplayWindow   win("Video"),wind1,wind2,wind3,wind4,wind5,wind6; 	//open a window to show the video

/*--------------------------------
|            FUNCTIONS            |
---------------------------------*/

// Int to binary and print.
void binary(int number) {
    int remainder;

    if(number <= 1) {
        cout << number;
        return;
    }
    remainder = number%2;
    binary(number >> 1);
    cout << remainder;
}


//Print the encoders readed from the CRovio Class
void print_Encoders(int *arr)
{
	string field_name[12]={"Packet length","Not Used","Left Wheel:Dir Rotation","Left Wheel:Ticks","Right Wheel:Dir Rotation","Right Wheel:Ticks","Rear Wheel:Dir Rotation","Rear Wheel:Ticks","Not used","Head Position","Batery","Config Status"};

	//Code
		cout << "\n-----------------ENCODERS------------------\n";
		for(int i=0;i<=11;i++)
		{
			cout << field_name[i] <<" -> ";

			if(i==10) cout <<hex << arr[i] <<dec;
			else cout << arr[i];

			if (i==11)
			{
				cout <<" = "; binary(arr[i]);	//print binary value
			}
			cout<<endl;
		}
		cout << "\n-------------------------------------------\n";
}


//Show the Position given by the NorthStar system
void showPosition(data_st &Pdata)
{
	mrpt::math::TPose2D pose_out; //Pose sent to the console
	pose_out=Pdata.Robot.getPosition();
	cout<<"\nX: " <<pose_out.x <<endl;
	cout<<"Y: " <<pose_out.y <<endl;
	cout<<"Theta: " <<pose_out.phi <<"\n\n";
}

void goThere(data_st &Pdata, string& response, string& errormsg)
{
	mrpt::math::TPose2D pose;
	int x_t, y_t; float theta_t;	//Target Position
	float  x_d, y_d, theta_d, distance, angle;	//Distance to the Target
	bool target=false;
	do{

		cout<<"\nEnter the target (x, y & theta)\n";
		cin >> x_t >>y_t >>theta_t;
		pose=Pdata.Robot.getPosition();
		x_d=x_t-pose.x;
		y_d=y_t-pose.y;
		distance=sqrt(x_d*x_d+y_d*y_d);
		while(distance>500)
		{
			theta_d=atan2(x_d,(-y_d));
			angle=theta_d-pose.phi;
			if(angle>PI)
				angle-=2*PI;
			if(angle<(-PI))
				angle+=2*PI;
			while(angle>0.6 || angle<(-0.6))
			{
				if(angle>0.6)
					Pdata.Robot.Moving(18, 5, 10, response, errormsg);
				if(angle<(-0.6))
					Pdata.Robot.Moving(18, 6, 10, response, errormsg);
				Pdata.Robot.getPosition();
				angle=theta_d-pose.phi;
				if(angle>PI)
					angle-=2*PI;
				if(angle<(-PI))
					angle+=2*PI;
			}
			Pdata.Robot.Moving(18, 1, 5, response, errormsg);
			Pdata.Robot.getPosition();
			x_d=x_t-pose.x;
			y_d=y_t-pose.y;
			distance=sqrt(x_d*x_d+y_d*y_d);
		}
		angle=theta_t-pose.phi;
		if(angle>PI)
			angle-=2*PI;
		if(angle<(-PI))
			angle+=2*PI;
		while(angle>0.6 || angle<(-0.6))
		{
			if(angle>0.6)
				Pdata.Robot.Moving(18, 6, 10, response, errormsg);
			if(angle<(-0.6))
				Pdata.Robot.Moving(18, 5, 10, response, errormsg);
			Pdata.Robot.getPosition();
			angle=theta_t-pose.phi;
			if(angle>PI)
				angle-=2*PI;
			if(angle<(-PI))
				angle+=2*PI;
		}
		if(distance<500 && theta_d<0.6 && theta_d>(-0.6))
			target=true;
	}while(!target);
}



//Shows the video in the "win" window.
 void videowindow(data_st* data)
{
	CObservationImagePtr lastimg;

	while(data->showvideo)
	{
		data->Robot.getLastImage2(lastimg);
		if(lastimg)	//Avoid NULL image
			win.showImage(lastimg->image);
		mrpt::system::sleep(50);
	}
}


/*--------------------------------
|       JOYSTICK CONTROLS         |
---------------------------------*/
void JoystickControl(data_st *data)
{
	CJoystick joy;
	float x,y,z;
	vector <bool> buttons;
	string response, errormsg;
	int action, direction;
	//CTicTac tictac;
	while(joy.getJoystickPosition(0,x,y,z,buttons))
	{
		//printf("Joystick readings: %.03f, %.03f, %.03f  (", x, y, z);
		int speed=(int)(10-(x*x+y*y)*9);
		if(x<-0.2 || x>0.2 || y <-0.2 || y>0.2)
		{
			if(y<-0.2)
			{
				if(x<0.2 && x>-0.2){//Move Forward
					action=18; direction=1;}
				if(x<-0.2){//Move Forward & Left
					action=18; direction=7;}
				if(x>0.2){//Move Forward & Right
					action=18; direction=8;}
			}

			if(y>0.2)
			{
				if(x<0.2 && x>-0.2){ //Move Backward
					action=18; direction=2;}
				if(x>0.2){ //Move Backward & Right
					action=18; direction=10;}
				if(x<-0.2){ //Move Backward & Left
					action=18; direction=9;}
			}
			if(y>-0.2 && y<0.2)
			{
				if(x<-0.2){ //Move Left
					action=18; direction=3;}
				if(x>0.2){ //Move Right
					action=18; direction=4;}
			}
			data->Robot.Moving(action, direction, speed, response, errormsg);
		}

		if((buttons[3] && buttons[2])||(buttons[0] && buttons[1])||(y<-0.5 && buttons[2])||(buttons[0] && buttons[2]))
		{
			if(buttons[3] && buttons[2]) //Take head up
				data->Robot.Moving(18, 11, 5, response, errormsg);
			if(buttons[0] && buttons[1]) //Go home
				data->Robot.Moving(12, 12, 5, response, errormsg);
			if(y<-0.5 && buttons[2]) //Go home & dock
				data->Robot.Moving(13, 12, 5, response, errormsg);
			if(buttons[0] && buttons[2]) //Get Report
			{
				data->Robot.Moving(1, 12, 5, response, errormsg);
				cout<<"Response:\n"<<response;
			}
		}
		else
		{
			if(buttons[0]) //Turn to the left
				data->Robot.Moving(18, 5, 5, response, errormsg);
			if(buttons[1]) //Turn to the right
				data->Robot.Moving(18, 6, 5, response, errormsg);
			if(buttons[2]) //Take head to the middle
				data->Robot.Moving(18, 13, 5, response, errormsg);
			if(buttons[3]) //Take head down
				data->Robot.Moving(18, 12, 5, response, errormsg);
		}

		fflush(stdout);
		mrpt::system::sleep(20);
	}
	cout<<"No Joystick connected...\n\n\n"<<endl;
}


/*--------------------------------
|       EXTRACT FEATURES          |
---------------------------------*/
void TestExtractFeatures2(data_st *data)
{
	//CDisplayWindow		wind1,wind2;
	CFeatureExtraction	fExt;

	CImage pict1, pict2;
	CTicTac	tictac;

	if(!data->matched)
	{
////		pict1.loadFromFile("img1.jpg", -1);
		data->Robot.captureImageRect(pict1);
		pict1.saveToFile("img1.jpg");
//		if (!pict)
//		{
//			cerr << "Cannot load image" << endl;
//			return;
//		}
		cout << "1ST IMAGE LOADED" << endl;

		//TImageROI ROI( 0, 0, img.getWidth()/4, img.getHeight()/2 );

		cout << "Extracting Harris features 1... [f_harris1.txt]\n";
		tictac.Tic();
		fExt.options.featsType = featHarris;
		fExt.options.harrisOptions.tile_image = false;
		fExt.detectFeatures( pict1, data->featsHarris);
		cout << "Detected " << data->featsHarris.size() << " features in " << endl;
		cout << format("  %.03fms",tictac.Tac()*1000) << endl;
		data->featsHarris.saveToTextFile("f_harris1.txt");
		wind1.setWindowTitle("Harris detected features");
		wind1.showImageAndPoints( pict1, data->featsHarris );

		cout << "Extracting SIFT features 1... [f_sift_hess1.txt]\n";
		tictac.Tic();
		fExt.options.featsType = featSIFT;
		fExt.options.SIFTOptions.implementation = CFeatureExtraction::Hess;
		fExt.detectFeatures( pict1, data->featsSIFT_Hess );
		cout << "Detected " << data->featsSIFT_Hess.size() << " features in " << endl;
		cout << format("  %.03fms",tictac.Tac()*1000) << endl;
		data->featsSIFT_Hess.saveToTextFile("f_sift_hess1.txt");
		wind3.setWindowTitle("SIFT Hess detected features");
		wind3.showImageAndPoints( pict1, data->featsSIFT_Hess );

		cout << "Extracting SURF features 1... [f_surf1.txt]\n";
		tictac.Tic();
		fExt.options.featsType = featSURF;
		fExt.detectFeatures(pict1, data->featsSURF );
		cout << "Detected " << data->featsSURF.size() << " features in " << endl;
		cout << format("  %.03fms",tictac.Tac()*1000) << endl;
		data->featsSURF.saveToTextFile("f_surf1.txt");
		wind5.setWindowTitle("SURF detected features");
		wind5.showImageAndPoints( pict1, data->featsSURF );

		data->matched=true;
		cout<<"\nFirst image taken & analyzed\n\n";
	}
	else
	{
////		pict2.loadFromFile("img2.jpg", -1);

		data->Robot.captureImageRect(pict2);
		pict2.saveToFile("img2.jpg");
//		if (!pict2)
//		{
//			cerr << "Cannot load image" << endl;
//			return;
//		}
		cout << "2ND IMAGE LOADED" << endl;

		cout << "Extracting Harris features 2... [f_harris2.txt]\n";
		tictac.Tic();
		fExt.options.featsType = featHarris;
		fExt.options.harrisOptions.tile_image = false;
		fExt.detectFeatures( pict2, data->featsHarris2);
		cout << "Detected " << data->featsHarris2.size() << " features in " << endl;
		cout << format("  %.03fms",tictac.Tac()*1000) << endl;
		data->featsHarris.saveToTextFile("f_harris2.txt");
		wind2.setWindowTitle("Harris detected features2");
		wind2.showImageAndPoints( pict2, data->featsHarris2 );

		cout << "Extracting SIFT features 2... [f_sift_hess2.txt]\n";
		tictac.Tic();
		fExt.options.featsType = featSIFT;
		fExt.options.SIFTOptions.implementation = CFeatureExtraction::Hess;
		fExt.detectFeatures( pict2, data->featsSIFT_Hess2 );
		cout << "Detected " << data->featsSIFT_Hess2.size() << " features in " << endl;
		cout << format("  %.03fms",tictac.Tac()*1000) << endl;
		data->featsSIFT_Hess.saveToTextFile("f_sift_hess2.txt");
		wind4.setWindowTitle("SIFT Hess detected features2");
		wind4.showImageAndPoints( pict2, data->featsSIFT_Hess2 );

		cout << "Extracting SURF features 2... [f_surf2.txt]\n";
		tictac.Tic();
		fExt.options.featsType = featSURF;
		fExt.detectFeatures(pict2, data->featsSURF2 );
		cout << "Detected " << data->featsSURF2.size() << " features in " << endl;
		cout << format("  %.03fms",tictac.Tac()*1000) << endl;
		data->featsSURF.saveToTextFile("f_surf2.txt");
		wind6.setWindowTitle("SURF detected features 2");
		wind6.showImageAndPoints( pict2, data->featsSURF2 );

		data->matched=false;
		cout<<"\nSecond image taken & analyzed\n";
	}
	data->features_taken=true;
}

/*--------------------------------
|			MATCHING		      |
---------------------------------*/
void matching(data_st *data)
{
	CTicTac	tictac;
	TMatchingOptions opt;
	opt.useXRestriction=false;
	if(data->matching_type==complete_match)
	{
		opt.useEpipolarRestriction=true;
		opt.epipolar_TH=2;
	}
	else
		opt.useEpipolarRestriction=false;
	cout<<"\nCalculating matches...\n";

//HARRIS

	if(data->matching_type==complete_match)
	{
		//data->FM_Harris.loadFromTextFile("FM_Harris.txt");
		opt.F=data->FM_Harris;
		//opt.minCC_TH=1.0;
		//opt.rCC_TH=0.020;
	}
	opt.matching_method=TMatchingOptions::mmCorrelation;
	tictac.Tic();
	mrpt::vision::utils::matchFeatures2(data->featsHarris, data->featsHarris2, data->Harris_matched, opt);
	cout << "Detected " << data->Harris_matched.size() << " Harris matches in " << endl;
	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
	data->Harris_matched.saveToTextFile("Harris_matches.txt");

//SIFT
	if(data->matching_type==complete_match)
	{
		opt.F=data->FM_SIFT;
		//opt.maxEDSD_TH=0.12;
		//opt.EDSD_RATIO=0.5;
	}
	opt.matching_method=TMatchingOptions::mmDescriptorSIFT;
	tictac.Tic();
	mrpt::vision::utils::matchFeatures2(data->featsSIFT_Hess, data->featsSIFT_Hess2, data->SIFT_matched, opt);
	cout << "Detected " << data->SIFT_matched.size() << " SIFT matches in " << endl;
	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
	data->SIFT_matched.saveToTextFile("SIFT_matches.txt");

//SURF
	if(data->matching_type==complete_match)
	{
		opt.F=data->FM_SURF;
		//opt.maxEDD_TH=70;
		//opt.EDD_RATIO=0.5;
	}
	opt.matching_method=TMatchingOptions::mmDescriptorSURF;
	tictac.Tic();
	mrpt::vision::utils::matchFeatures2(data->featsSURF, data->featsSURF2, data->SURF_matched, opt);
	cout << "Detected " << data->SURF_matched.size() << " SURF matches in " << endl;
	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
	data->SURF_matched.saveToTextFile("SURF_matches.txt");

	data->matching_done=true;
	cout<<"\nMatching finished\n";
}


//GETTING FUNDAMENTAL MATRIX
void getFMat(data_st *data)
{
	CvMat* points1;
	CvMat* points2;
	CvMat* status;
	CvMat* fundamental_matrix;
	ofstream mat;

// HARRIS
	if(data->Harris_matched.size()<8)
		cout<<"WARNING->Less than 8 elements...\n";
	else
	{
		int point_count = data->Harris_matched.size();
		points1 = cvCreateMat(1,point_count,CV_32FC2);
		points2 = cvCreateMat(1,point_count,CV_32FC2);
		status = cvCreateMat(1,point_count,CV_8UC1);

		CMatchedFeatureList::iterator it;
		unsigned int i, j;

		/* Fill the points here ... */
		for(i=0, it=data->Harris_matched.begin(); it!=data->Harris_matched.end(); it++,i++)
		{
			points1->data.fl[i*2] = it->first->x;
			points1->data.fl[i*2+1] = it->first->y;
			points2->data.fl[i*2] = it->second->x;
			points2->data.fl[i*2+1] = it->second->y;
		}

		fundamental_matrix = cvCreateMat(3,3,CV_32FC1);

		int fm_count = cvFindFundamentalMat( points1,points2,fundamental_matrix,
			CV_FM_RANSAC, 1, 0.99, status );

		if(fm_count!=0)
		{
			cout<<"\n\nFUNDAMENTAL MATRIX HARRIS:\n\n";
			for(i=0; i<data->Harris_matched.size();i++)
				cout<< (status->data.ptr[i] == 0 ? "0" : "1");
			cout<<"\n";
			mat.open("FM_Harris.txt");
			if(!mat.is_open())
				cout<<"Fundamental Matrix file not opened";
			for(i=0; i<3; i++)
			{
				for(j=0; j<3; j++)
				{
					data->FM_Harris(i,j)=cvGetReal2D(fundamental_matrix,i,j);
					cout<<data->FM_Harris(i,j)<<"\t";
					mat<<data->FM_Harris(i,j)<<"\t";
				}
				cout<<"\n";
				mat<<"\n";
			}
			mat.close();
		}
		else
			cout<<"\nNo Fundamental Matrix found\n";
	}

// SIFT
	if(data->SIFT_matched.size()<8)
		cout<<"WARNING->Less than 8 elements...\n";
	else
	{
		int point_count = data->SIFT_matched.size();
		points1 = cvCreateMat(1,point_count,CV_32FC2);
		points2 = cvCreateMat(1,point_count,CV_32FC2);
		status = cvCreateMat(1,point_count,CV_8UC1);

		CMatchedFeatureList::iterator it;
		unsigned int i, j;

		// Fill the points here ...
		for(i=0, it=data->SIFT_matched.begin(); it!=data->SIFT_matched.end(); it++,i++)
		{
			points1->data.fl[i*2] = it->first->x;
			points1->data.fl[i*2+1] = it->first->y;
			points2->data.fl[i*2] = it->second->x;
			points2->data.fl[i*2+1] = it->second->y;
		}

		fundamental_matrix = cvCreateMat(3,3,CV_32FC1);
		int fm_count = cvFindFundamentalMat( points1,points2,fundamental_matrix,
											 CV_FM_RANSAC, 1, 0.99, status );

		if(fm_count!=0)
		{
			cout<<"\n\nFUNDAMENTAL MATRIX - SIFT:\n\n";
			for(i=0; i<data->SIFT_matched.size();i++)
				cout<<status->data.ptr[i];
			cout<<"\n";
			mat.open("FM_SIFT.txt");
			if(!mat.is_open())
				cout<<"Fundamental Matrix file not opened";
			for(i=0; i<3; i++)
			{
				for(j=0; j<3; j++)
				{
					data->FM_SIFT(i,j)=cvGetReal2D(fundamental_matrix,i,j);
					cout<<data->FM_SIFT(i,j)<<"\t";
					mat<<data->FM_SIFT(i,j)<<"\t";
				}
				cout<<"\n";
				mat<<"\n";
			}
			mat.close();
		}
		else
			cout<<"\nNo Fundamental Matrix found\n";
	}

// SURF
	if(data->SURF_matched.size()<8)
		cout<<"WARNING->Less than 8 elements...\n";
	else
	{
		int point_count = data->SURF_matched.size();
		points1 = cvCreateMat(1,point_count,CV_32FC2);
		points2 = cvCreateMat(1,point_count,CV_32FC2);
		status = cvCreateMat(1,point_count,CV_8UC1);

		CMatchedFeatureList::iterator it;
		unsigned int i, j;

		// Fill the points here ...
		for(i=0, it=data->SURF_matched.begin(); it!=data->SURF_matched.end(); it++,i++)
		{
			points1->data.fl[i*2] = it->first->x;
			points1->data.fl[i*2+1] = it->first->y;
			points2->data.fl[i*2] = it->second->x;
			points2->data.fl[i*2+1] = it->second->y;
		}

		fundamental_matrix = cvCreateMat(3,3,CV_32FC1);
		int fm_count = cvFindFundamentalMat( points1,points2,fundamental_matrix,
											 CV_FM_RANSAC, 1, 0.99, status );

		if(fm_count!=0)
		{
			cout<<"\n\nFUNDAMENTAL MATRIX SURF:\n\n";
			for(i=0; i<data->SURF_matched.size();i++)
				cout<<status->data.ptr[i];
			cout<<"\n";
			mat.open("FM_SURF.txt");
			if(!mat.is_open())
				cout<<"Fundamental Matrix file not opened";
			for(i=0; i<3; i++)
			{
				for(j=0; j<3; j++)
				{
					data->FM_SURF(i,j)=cvGetReal2D(fundamental_matrix,i,j);
					cout<<data->FM_SURF(i,j)<<"\t";
					mat<<data->FM_SURF(i,j)<<"\t";
				}
				cout<<"\n";
				mat<<"\n";
			}
			mat.close();
		}
		else
			cout<<"\nNo Fundamental Matrix found\n";
	}
	cout<<"\nFundamental Matrix calculation finished\n";
	data->fm_calculated=true;
}

void getFMat_from_txt(data_st *data)
{
	CvMat* points1;
	CvMat* points2;
	CvMat* status;
	CvMat* fundamental_matrix;
	ofstream mat;

	CMatrix A;
	int point_count;
	int i,j;

	// HARRIS
	A.loadFromTextFile("Harris_matches.txt");
	point_count = A.getRowCount();
	if(point_count<8)
		cout<<"WARNING->Less than 8 elements...\n";
	else
	{
		points1 = cvCreateMat(1,point_count,CV_32FC2);
		points2 = cvCreateMat(1,point_count,CV_32FC2);
		status = cvCreateMat(1,point_count,CV_8UC1);
		/* Fill the points here ... */
		for(i=0; i<point_count; i++)
		{
			points1->data.fl[i*2] = A.get_unsafe(i,1);
			points1->data.fl[i*2+1] = A.get_unsafe(i,2);
			points2->data.fl[i*2] = A.get_unsafe(i,4);
			points2->data.fl[i*2+1] = A.get_unsafe(i,5);
		}

		fundamental_matrix = cvCreateMat(3,3,CV_32FC1);

		int fm_count = cvFindFundamentalMat( points1,points2,fundamental_matrix,
			CV_FM_RANSAC, 2, 0.8, status );

		if(fm_count!=0)
		{
			cout<<"\n\nFUNDAMENTAL MATRIX HARRIS:\n\n";
			for(i=0; i<point_count;i++)
				cout<< (status->data.ptr[i]==0 ? "A" : "B");
			cout<<"\n";
			mat.open("FM_Harris.txt");
			if(!mat.is_open())
				cout<<"Fundamental Matrix file not opened";
			for(i=0; i<3; i++)
			{
				for(j=0; j<3; j++)
				{
					data->FM_Harris(i,j)=cvGetReal2D(fundamental_matrix,i,j);
					cout<<data->FM_Harris(i,j)<<"\t";
					mat<<data->FM_Harris(i,j)<<"\t";
				}
				cout<<"\n";
				mat<<"\n";
			}
		}
		else
			cout<<"\nNo Fundamental Matrix found\n";
	}

// SIFT
	A.loadFromTextFile("SIFT_matches.txt");
	point_count = A.getRowCount();

	if(point_count<8)
		cout<<"WARNING->Less than 8 elements...\n";
	else
	{
		points1 = cvCreateMat(1,point_count,CV_32FC2);
		points2 = cvCreateMat(1,point_count,CV_32FC2);
		status = cvCreateMat(1,point_count,CV_8UC1);

		/* Fill the points here ... */
		for(i=0; i<point_count; i++)
		{
			points1->data.fl[i*2] = A.get_unsafe(i,1);
			points1->data.fl[i*2+1] = A.get_unsafe(i,2);
			points2->data.fl[i*2] = A.get_unsafe(i,4);
			points2->data.fl[i*2+1] = A.get_unsafe(i,5);
		}

		fundamental_matrix = cvCreateMat(3,3,CV_32FC1);
		int fm_count = cvFindFundamentalMat( points1,points2,fundamental_matrix,
											 CV_FM_RANSAC, 2, 0.8,  status );

		if(fm_count!=0)
		{
			cout<<"\n\nFUNDAMENTAL MATRIX SIFT:\n\n";
			for(i=0; i<point_count;i++)
				cout<<status->data.ptr[i];
			cout<<"\n";
			mat.open("FM_SIFT.txt");
			for(i=0; i<3; i++)
			{
				for(j=0; j<3; j++)
				{
					data->FM_SIFT(i,j)=cvGetReal2D(fundamental_matrix,i,j);
					cout<<data->FM_SIFT(i,j)<<"\t";
					mat<<data->FM_SIFT(i,j)<<"\t";
				}
				cout<<"\n";
				mat<<"\n";
			}
		}
		else
			cout<<"\nNo Fundamental Matrix found\n";
	}

// SURF
	A.loadFromTextFile("SURF_matches.txt");
	point_count = A.getRowCount();

	if(point_count<8)
		cout<<"WARNING->Less than 8 elements...\n";
	else
	{

		points1 = cvCreateMat(1,point_count,CV_32FC2);
		points2 = cvCreateMat(1,point_count,CV_32FC2);
		status = cvCreateMat(1,point_count,CV_8UC1);

		/* Fill the points here ... */
		for(i=0; i<point_count; i++)
		{
			points1->data.fl[i*2] = A.get_unsafe(i,1);
			points1->data.fl[i*2+1] = A.get_unsafe(i,2);
			points2->data.fl[i*2] = A.get_unsafe(i,4);
			points2->data.fl[i*2+1] = A.get_unsafe(i,5);
		}

		fundamental_matrix = cvCreateMat(3,3,CV_32FC1);
		int fm_count = cvFindFundamentalMat( points1,points2,fundamental_matrix,
											 CV_FM_RANSAC, 2, 0.8,  status );



		if(fm_count!=0)
		{
			cout<<"\n\nFUNDAMENTAL MATRIX SURF:\n\n";
			for(i=0; i<point_count;i++)
				cout<<status->data.ptr[i];
			cout<<"\n";
			mat.open("FM_SURF.txt");
			for(i=0; i<3; i++)
			{
				for(j=0; j<3; j++)
				{
					data->FM_SURF(i,j)=cvGetReal2D(fundamental_matrix,i,j);
					cout<<data->FM_SURF(i,j)<<"\t";
					mat<<data->FM_SURF(i,j)<<"\t";
				}
				cout<<"\n";
				mat<<"\n";
			}
		}
		else
			cout<<"\nNo Fundamental Matrix found\n";
	}
	cout<<"\nFundamental Matrix calculation finished\n";
	data->fm_calculated=true;
}
/*--------------------------------
|            MAIN                 |
---------------------------------*/
int main()
{
	//VAR
	char decision;	//to read from keyboard
	data_st data;
	bool output=false;
	data.showvideo=false;
	string response, errormsg;
	mrpt::system::TThreadHandle screen_hd, featuring_hd, matching_hd, fm_hd;
	int *a_enc;
	int act=0; //Action
	int dir=0; //Direction
	int speed=3; //Speed

	data.matched=false;			//this var checks if there are extracted features
	data.features_taken=false, data.matching_done=false, data.fm_calculated=false;
	//string line;
	//fstream f_harris, f_harris_b;

	CImage Picture;

	string MenuJoystick="---------------------------Joystick Controls---------------------------\n\n|1->Turn to the left		2->Turn to the right			|\n|3->Head to the middle		4->Head down		3&4->Head up	|\n|1&2->Go Home			Up&3->Go Home and Dock	1&3->Get Report	|\n-----------------------------------------------------------------------\n";
	string MenuKeyboard="---------------------------Keyboard Controls---------------------------\n|MOVEMENT			CAMERA			 PATHS		|\n|'w'->up			'c'->Middle		'r'->Start	|\n|'s'->down			'z'->Up			 't'->Stop/Save	|\n|'a'->left			'x'->Down		 'y'->Show	|\n|'d'->up						 'u'->Delete	|\n|'q'->Move left 					 'i'->Run Ford	|\n|'e'->move Right					 'o'->Run Back	|\n|'g'->Move to target position						|\n|									|\n|EXTRA									|\n|'-'->-Speed			'+'->+Speed		'v'->Video	|\n|'h'->Gohome			'j'->menuJY		'r'->menuKB	|\n|'l'->ShowPosition		'n'->ShowEnc		'k'->Get Report	|\n|'f'->Ext Features		'b'->Match Feat		'*'->FMatrix	|\n|				'.'->EXIT				|\n-----------------------------------------------------------------------\n\n";

/*---------------------------Keyboard Controls---------------------------\n
|MOVEMENT			CAMERA			 PATHS		|\n
|'w'->up			'c'->Middle		'r'->Start	|\n
|'s'->down			'z'->Up			 't'->Stop/Save	|\n
|'a'->left			'x'->Down		 'y'->Show	|\n
|'d'->up						 'u'->Delete	|\n
|'q'->Move left 					 'i'->Run Ford	|\n
|'e'->move Right					 'o'->Run Back	|\n
|'g'->Move to target position						|\n
|									|\n
|EXTRA									|\n
|'-'->-Speed			'+'->+Speed		'v'->Video	|\n
|'h'->Gohome			'j'->menuJY		'r'->menuKB	|\n
|'l'->ShowPosition		'n'->ShowEnc		'k'->Get Report	|\n
|'f'->Ext Features		'b'->Match Feat		'*'->FMatrix	|\n
|				'.'->EXIT				|\n
-----------------------------------------------------------------------\n\n*/

/*---------------------------Joystick Controls---------------------------\n\n
|1->Turn to the left		2->Turn to the right			|\n
|3->Head to the middle		4->Head down				3&4->Head up	|\n
|1&2->Go Home				Up&3->Go Home and Dock		1&3->Get Report	|\n
-----------------------------------------------------------------------\n*/

	//Code START here
	try
	{
		//initialization of Rovio Values (Ip, user and password)
		data.Robot.Initialize(errormsg,"150.214.109.134","admin","admin");

		if(errormsg.empty())
		{
			cout<<"What do you want from the robot?\n\n";
			cout<<MenuKeyboard<<endl;
			//cout<<MenuJoystick<<endl;

			//Calling Joystick in other thread
			//createThread(JoystickControl,&data);

			do{
				decision=mrpt::system::os::getch();
				switch(decision)
				{
				//Movement control
				case 'w':
					act=18;dir=1;break;
				case 's':
					act=18;dir=2;break;
				case 'q':
					act=18;dir=3;break;
				case 'e':
					act=18;dir=4;break;
				case 'a':
					act=18;dir=5;break;
				case 'd':
					act=18;dir=6;break;
				//camera head movement
				case 'z':
					act=18;dir=11;break;
				case 'x':
					act=18;dir=12;break;
				case 'c':
					act=18;dir=13;break;
				//Paths
	 			case 'r':	//start
					act=2;
					cout << "Recording Path" << endl; break;
				case 't':	//stop and save path
					act=4;
					cout << "Path Saved" << endl; break;
				case 'y':	//GetPath
					act=6;
					cout << "Getting saved Paths" << endl;
					output=true; break;
				case 'u':	//delete path
					act=5;
					cout << "Path deleted" << endl; break;
				case 'i':	//fordward
					act=7;
					cout << "Running path Fordward mode" << endl; break;
				case 'o':	//backward
					act=8;
					cout << "Running path Backward mode" << endl; break;
				//Speed control
				case '-':
					act=0;
					if (speed<=7)
						speed+=2;
					cout << "Speed: "<< speed <<endl;
					break;
				case '+':
					act=0;
					if(speed>=3)
						speed-=2;
					cout << "Speed: "<< speed <<endl;
					break;
				//Camera On/Off
				case 'v':
					act=0;
					if(!data.showvideo)	//open the window
					{
						cout<<"VIDEO ON"<<endl;
						data.showvideo=!data.showvideo;
						screen_hd=createThread(videowindow, &data);
					}
					else //close the window
					{
						cout<<"VIDEO OFF"<<endl;
						data.showvideo=!data.showvideo; // this kill the thread
						joinThread(screen_hd); //Wait till thread finish
						win.~CDisplayWindow();
					}
					break;
				case 'h':
					cout << "GOING HOME" << endl;
					act=13;break;
				case 'j':
					act=0;
					cout<<MenuJoystick<<endl;break;
				case 'm':
					act=0;
					cout<<MenuKeyboard<<endl;break;
				case 'k':
					act=1; output=true;
					cout <<"--------------------------REPORT------------------------" << endl;
					break;
				case 'l': //Print the position
					act=0; output=false;
					showPosition(data);
					break;
				case 'g': //Go to Target Position
					goThere(data, response, errormsg);
					break;
				case 'n': //Print the encoder's values.
					act=0; output=false;
					a_enc=data.Robot.getEncoders();
					print_Encoders(a_enc);
					break;
				case 'f': //Take Features
					act=0; output=false;
					data.features_taken=false;	//this var checks if the function TestExtractFeatures has finished
					featuring_hd=createThread(TestExtractFeatures2, &data);
					break;
				case 'b': //Match Features
					act=0; output=false;
					data.matching_done=false;	//this var checks if the function TestExtractFeatures has finished
					data.matching_type=FM_match;
					matching_hd=createThread(matching, &data);
					break;
				case '1': //Change the matching to complete matching for the next iteration
					act=0; output=false;
					data.matching_done=false;	//this var checks if the function TestExtractFeatures has finished
					data.matching_type=complete_match;
					matching_hd=createThread(matching, &data);
					break;
				case '*': //Calculate Fundamental Matrix
					act=0; output=false;
					data.fm_calculated=false;	//this var checks if the function TestExtractFeatures has finished
					fm_hd=createThread(getFMat, &data);
					break;
				case '/': //Calculate Fundamental Matrix
					act=0; output=false;
					data.fm_calculated=false;	//this var checks if the function TestExtractFeatures has finished
					fm_hd=createThread(getFMat_from_txt, &data);
					break;
				case 'p': //Take Picture and show it in a window
					act=0; output=false;
					data.Robot.captureImageRect(Picture);
					win.showImage(Picture);
					break;
				default:
					act=0;	//No action is executed
				}//end switch

//EXECUTION
				if (act!=0)
				{
					data.Robot.Moving(act,dir,speed, response, errormsg);
					//a_enc=data.Robot.getEncoders();
					//print_Encoders(a_enc);
				}

				if(output)
				{
					cout << "RESPONSE->" << response <<endl;
					output=false;
				}
				if (!errormsg.empty()){
					cout <<"---------------------------------------------------" << endl;
					cout << "ERROR->" <<errormsg <<endl;
				}

				if(data.features_taken)	//End features thread
					joinThread(featuring_hd);
				if(data.matching_done)	//End matching thread
					joinThread(matching_hd);
				if(data.fm_calculated)	//End getFM thread
					joinThread(fm_hd);

				mrpt::system::sleep(10);
			}while(decision!='.');
		}//end if
		else
			mrpt::system::pause();
	}//end try
	catch(std::exception &e)
	{
		cerr << e.what() << endl;
		return -1;
	}
	return 0;
}
