/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/opengl/COpenGLViewport.h>
#include <mrpt/opengl/COpenGLScene.h>
#include <mrpt/opengl/CSetOfObjects.h>
#include <mrpt/opengl/CTexturedPlane.h>
#include <mrpt/utils/CStringList.h>
#include <mrpt/math/CVectorTemplate.h>

#include "opengl_internals.h"

using namespace mrpt;
using namespace mrpt::opengl;
using namespace mrpt::utils;
using namespace mrpt::math;
using namespace std;

#include <mrpt/utils/metaprogramming.h>
using namespace mrpt::utils::metaprogramming;

IMPLEMENTS_SERIALIZABLE( COpenGLViewport, CSerializable, mrpt::opengl )


/*--------------------------------------------------------------

			IMPLEMENTATION OF COpenGLViewport

  ---------------------------------------------------------------*/

/*--------------------------------------------------------------
					Constructor
  ---------------------------------------------------------------*/
COpenGLViewport::COpenGLViewport( COpenGLScene *parent, const string &name  ) :
	m_camera(),
	m_parent( parent ),
	m_isCloned(false),
	m_isClonedCamera(false),
	m_clonedViewport(),
	m_name(name),
	m_isTransparent(false),
	m_borderWidth(0),
	m_view_x(0),
	m_view_y(0),
	m_view_width(1),
	m_view_height(1),
	m_clip_min(0.1),
	m_clip_max(10000),
	m_objects()
{

}


/*--------------------------------------------------------------
					Destructor
  ---------------------------------------------------------------*/
COpenGLViewport::~COpenGLViewport()
{
	clear();
}

/*--------------------------------------------------------------
					setCloneView
  ---------------------------------------------------------------*/
void COpenGLViewport::setCloneView( const string &clonedViewport )
{
	clear();
	m_isCloned = true;
	m_clonedViewport = clonedViewport;
}

/*--------------------------------------------------------------
					setViewportPosition
  ---------------------------------------------------------------*/
void COpenGLViewport::setViewportPosition(
	const double &x,
	const double &y,
	const double &width,
	const double &height )
{
	MRPT_START
	ASSERT_( m_view_x>=0 && m_view_x<=1 );
	ASSERT_( m_view_y>=0 && m_view_y<=1 );
	ASSERT_( m_view_width>=0 && m_view_width<=1 );
	ASSERT_( m_view_height>=0 && m_view_height<=1 );

	m_view_x = x;
	m_view_y = y;
	m_view_width = width;
	m_view_height = height;

	MRPT_END
}

/*--------------------------------------------------------------
					getViewportPosition
  ---------------------------------------------------------------*/
void COpenGLViewport::getViewportPosition(
	double &x,
	double &y,
	double &width,
	double &height )
{
	x = m_view_x;
	y = m_view_y;
	width = m_view_width;
	height = m_view_height;
}

/*--------------------------------------------------------------
					clear
  ---------------------------------------------------------------*/
void COpenGLViewport::clear()
{
	m_objects.clear();
}

/*--------------------------------------------------------------
					insert
  ---------------------------------------------------------------*/
void COpenGLViewport::insert( const CRenderizablePtr &newObject )
{
	m_objects.push_back(newObject);
}


/*---------------------------------------------------------------
						render
 ---------------------------------------------------------------*/
void  COpenGLViewport::render( const int &render_width, const int &render_height  ) const
{
#if MRPT_HAS_OPENGL_GLUT
	CListOpenGLObjects::const_iterator		it=m_objects.end();
	try
	{
		// Change viewport:
		// -------------------------------------------
        glViewport(
			GLint( render_width * m_view_x ),
			GLint( render_height * m_view_y ),
			GLsizei( render_width * m_view_width ),
			GLsizei( render_height * m_view_height ) );

		// Clear depth&/color buffers:
		// -------------------------------------------
		GLsizei  width  = GLsizei( render_width * m_view_width );
		GLsizei  height = GLsizei( render_height * m_view_height );

        glScissor(
			GLint( render_width * m_view_x ),
			GLint( render_height * m_view_y ),
			width, height );

        glEnable(GL_SCISSOR_TEST);
        if ( !m_isTransparent )
        {   // Clear color & depth buffers:
            // glClearColor(clearColorR,clearColorG,clearColorB,1);  // Use default colors, or those set by invoking program.
            glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT |  GL_ACCUM_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
        }
        else
        {   // Clear depth buffer only:
            glClear(GL_DEPTH_BUFFER_BIT);
        }
        glDisable(GL_SCISSOR_TEST);

		// Set camera:
		// -------------------------------------------
        glMatrixMode(GL_PROJECTION);
        glLoadIdentity();

		const CListOpenGLObjects	*objectsToRender;
		COpenGLViewport		*viewForGetCamera;

		if (m_isCloned)
		{	// Clone: render someone's else objects.
			ASSERT_(m_parent.get()!=NULL);

			COpenGLViewportPtr view = m_parent->getViewport( m_clonedViewport );
			if (!view)
				THROW_EXCEPTION_CUSTOM_MSG1("Cloned viewport '%s' not found in parent COpenGLScene",m_clonedViewport.c_str());

			objectsToRender = &view->m_objects;
			viewForGetCamera = m_isClonedCamera ? view.pointer() : const_cast<COpenGLViewport*>(this);
		}
		else
		{	// Normal case: render our own objects:
			objectsToRender = &m_objects;
			viewForGetCamera = const_cast<COpenGLViewport*>(this);
		}

		// Get camera:
		// 1st: if there is a CCamera in the scene:
		CRenderizablePtr cam_ptr = viewForGetCamera->getByClass<CCamera>();

		CCamera *myCamera=NULL;
		if (cam_ptr.present())
		{
			myCamera = getAs<CCamera>(cam_ptr);
		}

		// 2nd: the internal camera of all viewports:
		if (!myCamera)
			myCamera = &viewForGetCamera->m_camera;

        ASSERT_(m_camera.m_distanceZoom>0);


        float	eye_x = myCamera->m_pointingX +  max(0.01f,myCamera->m_distanceZoom) * cos(DEG2RAD(myCamera->m_azimuthDeg))*cos(DEG2RAD(myCamera->m_elevationDeg));
        float	eye_y = myCamera->m_pointingY +  max(0.01f,myCamera->m_distanceZoom) * sin(DEG2RAD(myCamera->m_azimuthDeg))*cos(DEG2RAD(myCamera->m_elevationDeg));
        float	eye_z = myCamera->m_pointingZ +  max(0.01f,myCamera->m_distanceZoom) * sin(DEG2RAD(myCamera->m_elevationDeg));

        float   up_x,up_y,up_z;

        if (fabs(fabs(myCamera->m_elevationDeg)-90)>1e-6)
        {
            up_x=0;
            up_y=0;
            up_z=1;
        }
        else
        {
            float sgn = myCamera->m_elevationDeg>0 ? 1:-1;
            up_x = -cos(DEG2RAD(myCamera->m_azimuthDeg))*sgn;
            up_y = -sin(DEG2RAD(myCamera->m_azimuthDeg))*sgn;
            up_z = 0;
        }

        if (myCamera->m_projectiveModel)
        {
            gluPerspective( myCamera->m_projectiveFOVdeg, width/(1.0*height),m_clip_min,m_clip_max);
            gluLookAt(
                eye_x,
                eye_y,
                eye_z,
                myCamera->m_pointingX,
                myCamera->m_pointingY,
                myCamera->m_pointingZ,
                up_x,up_y,up_z);
			CRenderizable::checkOpenGLError();
        }
        else
        {
			double ratio = width/double(height);
			double Ax = myCamera->m_distanceZoom*0.5;
			double Ay = myCamera->m_distanceZoom*0.5;

			if (ratio>1)
				Ax *= ratio;
			else
			{
				if (ratio!=0) Ay /=ratio;
			}

            glOrtho( -Ax,Ax,-Ay,Ay,-0.5*m_clip_max,0.5*m_clip_max);
            gluLookAt(
                eye_x,
                eye_y,
                eye_z,
                myCamera->m_pointingX,
                myCamera->m_pointingY,
                myCamera->m_pointingZ,
                up_x,up_y,up_z);
            CRenderizable::checkOpenGLError();
        }

		// Render objects:
		// -------------------------------------------
        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();

        glEnable(GL_DEPTH_TEST);
		glDepthFunc(GL_LESS); // GL_LEQUAL

		for (it=objectsToRender->begin();it!=objectsToRender->end();it++)
		{
			if (!it->present()) continue;

			// 3D coordinates transformation:
			glMatrixMode(GL_MODELVIEW);
			glPushMatrix();

			glPushAttrib(GL_ALL_ATTRIB_BITS);
			CRenderizable::checkOpenGLError();

			// This is the right order so that the transformation results in the standard matrix.
			// The order seems to be wrong, but it's not.
			glTranslated((*it)->m_x, (*it)->m_y, (*it)->m_z);

			//TODO: cambiarlo si resulta estar mal.
			glRotatef((*it)->m_yaw, 0.0, 0.0, 1.0);
			glRotatef((*it)->m_pitch, 0.0, 1.0, 0.0);
			glRotatef((*it)->m_roll, 1.0, 0.0, 0.0);
			/*glRotatef((*it)->m_roll, 1.0, 0.0, 0.0);
			glRotatef((*it)->m_pitch, 0.0, 1.0, 0.0);
			glRotatef((*it)->m_yaw, 0.0, 0.0, 1.0);*/
			// Do scaling after the other transformations!
			glScalef((*it)->m_scale_x,(*it)->m_scale_y,(*it)->m_scale_z);

			// Set color:
			glColor4f( (*it)->m_color_R,(*it)->m_color_G,(*it)->m_color_B,(*it)->m_color_A);

			(*it)->render();

			if ((*it)->m_show_name)
			{
				glDisable(GL_DEPTH_TEST);
				glRasterPos3f(0.0f,0.0f,0.0f);
				glColor3f(1.0f,1.0f,1.0f);

				GLfloat		raster_pos[4];
				glGetFloatv( GL_CURRENT_RASTER_POSITION, raster_pos);
				float eye_distance= raster_pos[3];

				void *font=NULL;
				if (eye_distance<2)
						font = GLUT_BITMAP_TIMES_ROMAN_24;
				else if(eye_distance<200)
					font = GLUT_BITMAP_TIMES_ROMAN_10;

				if (font)
					CRenderizable::renderTextBitmap( (*it)->m_name.c_str(), font);

				glEnable(GL_DEPTH_TEST);
			}

			glPopAttrib();
			CRenderizable::checkOpenGLError();

			glPopMatrix();
			CRenderizable::checkOpenGLError();

		} // end foreach object


		// Finally, draw the border:
		// --------------------------------
		if (m_borderWidth>0)
		{
		    glLineWidth( 2*m_borderWidth );
		    glColor4f(0,0,0,1);
            glDisable(GL_DEPTH_TEST);

            glMatrixMode(GL_MODELVIEW);
            glLoadIdentity();
            glMatrixMode(GL_PROJECTION);
            glLoadIdentity();

            glBegin( GL_LINE_LOOP );
                glVertex2f(-1,-1);
                glVertex2f(-1, 1);
                glVertex2f( 1, 1);
                glVertex2f( 1,-1);
            glEnd();

            glEnable(GL_DEPTH_TEST);
		}

	}
	catch(exception &e)
	{
		string		msg;
		if (it!=m_objects.end())
				msg = format("Exception while rendering a class '%s'\n%s", (*it)->GetRuntimeClass()->className, e.what() );
		else	msg = format("Exception while rendering:\n%s", e.what() );

		THROW_EXCEPTION(msg);
	}
	catch(...)
	{
		THROW_EXCEPTION("Runtime error!");
	}
#else
	THROW_EXCEPTION("The MRPT has been compiled with MRPT_HAS_OPENGL_GLUT=0! OpenGL functions are not implemented");
#endif
}


/*---------------------------------------------------------------
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void  COpenGLViewport::writeToStream(CStream &out,int *version) const
{
	if (version)
		*version = 0;
	else
	{
		// Save data:
		out << m_camera
			<< m_isCloned << m_isClonedCamera << m_clonedViewport
			<< m_name
			<< m_isTransparent
			<< m_borderWidth
			<< m_view_x << m_view_y << m_view_width << m_view_height;

		// Save objects:
		uint32_t	n;
		n = (uint32_t)m_objects.size();
		out << n;
		for (CListOpenGLObjects::const_iterator	it=m_objects.begin();it!=m_objects.end();++it)
			out << **it;
	}
}

/*---------------------------------------------------------------
	Implements the reading from a CStream capability of
		CSerializable objects
  ---------------------------------------------------------------*/
void  COpenGLViewport::readFromStream(CStream &in,int version)
{
	switch(version)
	{
	case 0:
		{
			// Load data:
			in  >> m_camera
				>> m_isCloned >> m_isClonedCamera >> m_clonedViewport
				>> m_name
				>> m_isTransparent
				>> m_borderWidth
				>> m_view_x >> m_view_y >> m_view_width >> m_view_height;

			// Load objects:
			uint32_t	n;
			in >> n;
			clear();
			m_objects.resize(n);

			for_each(m_objects.begin(), m_objects.end(), ObjectReadFromStream(&in) );

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}

/*---------------------------------------------------------------
							getByName
  ---------------------------------------------------------------*/
CRenderizablePtr COpenGLViewport::getByName( const string &str )
{
	for (CListOpenGLObjects::iterator	it=m_objects.begin();it!=m_objects.end();++it)
	{
		if ((*it)->m_name == str)
			return *it;
		else if ( (*it)->GetRuntimeClass() == CLASS_ID_NAMESPACE(CSetOfObjects,opengl))
		{
			CRenderizablePtr ret = getAs<CSetOfObjects>(*it)->getByName(str);
			if (ret.present())
				return ret;
		}
	}
	return CRenderizablePtr();
}

/*---------------------------------------------------------------
					initializeAllTextures
  ---------------------------------------------------------------*/
void  COpenGLViewport::initializeAllTextures()
{
#if MRPT_HAS_OPENGL_GLUT
	for (CListOpenGLObjects::iterator it=m_objects.begin();it!=m_objects.end();++it)
	{
		if ( (*it)->GetRuntimeClass() == CLASS_ID_NAMESPACE(CTexturedPlane,opengl))
		{
			getAs<CTexturedPlane>(*it)->loadTextureInOpenGL();
		}
		else
		if ( (*it)->GetRuntimeClass() == CLASS_ID_NAMESPACE(CSetOfObjects,opengl))
		{
			getAs<CSetOfObjects>(*it)->initializeAllTextures();
		}
	}
#endif
}

/*--------------------------------------------------------------
					dumpListOfObjects
  ---------------------------------------------------------------*/
void COpenGLViewport::dumpListOfObjects( utils::CStringList  &lst )
{
	for (CListOpenGLObjects::iterator	it=m_objects.begin();it!=m_objects.end();++it)
	{
		// Single obj:
		string  s( (*it)->GetRuntimeClass()->className );
		if ((*it)->m_name.size())
			s+= string(" (") +(*it)->m_name + string(")");
		lst.add( s );

		if ((*it)->GetRuntimeClass() == CLASS_ID_NAMESPACE(CSetOfObjects,mrpt::opengl))
		{
			utils::CStringList  auxLst;

			getAs<CSetOfObjects>(*it)->dumpListOfObjects(auxLst);

			for (size_t i=0;i<auxLst.size();i++)
				lst.add( string(" ")+auxLst(i) );
		}
	}
}

/*--------------------------------------------------------------
					removeObject
  ---------------------------------------------------------------*/
void COpenGLViewport::removeObject( const CRenderizablePtr &obj )
{
	for (CListOpenGLObjects::iterator	it=m_objects.begin();it!=m_objects.end();++it)
		if (it->pointer() == obj.pointer())
		{
			m_objects.erase(it);
			return;
		}
		else if ( (*it)->GetRuntimeClass()==CLASS_ID_NAMESPACE(CSetOfObjects,opengl) )
			getAs<CSetOfObjects>(*it)->removeObject(obj);
}

/*--------------------------------------------------------------
					setViewportClipDistances
  ---------------------------------------------------------------*/
void COpenGLViewport::setViewportClipDistances(const double clip_min, const double clip_max)
{
	ASSERT_(clip_max>clip_min);

	m_clip_min = clip_min;
	m_clip_max = clip_max;
}

/*--------------------------------------------------------------
					getViewportClipDistances
  ---------------------------------------------------------------*/
void COpenGLViewport::getViewportClipDistances(double &clip_min, double &clip_max) const
{
	clip_min = m_clip_min;
	clip_max = m_clip_max;
}
