/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <math.h>
#include <string.h>
#include "MRI.h"
#include "vmedian.h"

#define	NLUT 4096 /* 2^12 */

struct ContrastData {
	struct link *next;
	int	width;
	int	height;
	double  darkness, contrast, shadows, saturation;
};

static void
ContrastStart (void *private, int width, int height, int freedata)
{
	struct ContrastData *wd = private;

	wd->width = width;
	wd->height = height;
	(*wd->next->start) (wd->next->private, width, height, freedata);
}

static void
ContrastRow (void *private, void *data)
{
	struct ContrastData *wd = private;
	struct MRI_ScanLine *fp = data;
	int x;
	double *R = fp->R;
	double *G = fp->G;
	double *B = fp->B;

	for (x = 0; x < wd->width; x++) {
		double tr = wd->contrast * (R[x]-wd->darkness);
		double tg, tb, signscale;

		if (wd->saturation < 1.0e-5) {
		    tg = tb = 0.0;
		}
		else if (fabs(G[x]) > fabs(B[x])) {
		    signscale = G[x] >= 0.0 ? 128.0 : -128.0;
		    if (fabs(G[x]) > 128.0) G[x] = signscale;
		    tg = pow (fabs(G[x])/128.0, 1.0/wd->saturation) * signscale;
		    tb = fabs(G[x]) > 1.0e-5 ? B[x]/G[x] * tg : 0.0;
		}
		else {
		    signscale = B[x] >= 0.0 ? 128.0 : -128.0;
		    if (fabs(B[x]) > 128.0) B[x] = signscale;
		    tb = pow (fabs(B[x])/128.0, 1.0/wd->saturation) * signscale;
		    tg = fabs(B[x]) > 1.0e-5 ? G[x]/B[x] * tb : 0.0;
		}
#if 1
		if (tg > 127.0) tg = 127.0; if (tg < -128.0) tg = -128.0;
		if (tb > 127.0) tb = 127.0; if (tb < -128.0) tb = -128.0;
#endif

		if (tr > 1.0) {
			tr = 1.0;
			tg /= 2.0;
			tb /= 2.0;
		}
		if (tr < 0.0) {
			tr = 0.0;
			tg = tb = 0.0;
		}
		tr = pow (tr, wd->shadows) * 100.0;
		R[x] = tr;

		G[x] = tg;
		B[x] = tb;
	}
	(*wd->next->row) (wd->next->private, fp);
}

static void
ContrastClose (void *private)
{
	struct ContrastData *wd = private;
	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}

struct link *
MRI_GenContrastAdjuster (double darkness, double lumascale, double contrast, double shadows, double saturation, 
		     struct link *next)
{
	struct ContrastData *wd = malloc (sizeof (struct ContrastData));
	struct link *ep = malloc (sizeof (*ep));

	if (wd == (struct ContrastData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = ContrastStart;
	ep->row = ContrastRow;
	ep->close = ContrastClose;
	ep->private = wd;
	wd->next = next;
	wd->darkness = darkness * 10.0;
	wd->contrast = lumascale * (1.0+contrast/10.0) / (100.0 - wd->darkness);
	wd->shadows = 1.0 - shadows / 10.0;
	wd->saturation = 1.0 + saturation/5.0;
#if 0
	fprintf (stderr, "Contrast: darkness=%g contrast=%g shadows=%g saturation=%g\n", wd->darkness, wd->contrast, wd->shadows, wd->saturation);
#endif
	return ep;
}
