#include <stdlib.h>
#include <sys/types.h>
#include <dirent.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <unistd.h>

#include "MRI.h"
#include "options.h"

MRI_balance target;
char	*bestWB = NULL;
double  bestDiff;
char *progname;

void
Usage()
{
	fprintf (stderr, "Usage: %s [options...] < mrwfile > white_balance_name\n", progname);
	fprintf (stderr, "Command line options:\n");
	PrintOptions (stderr);
	exit (1);
}

void
CompareWB (const char *dirname, const char *wbFileName)
{
	FILE	*f;
	char	buffer[512];
	int	rb, gb, bb;
	double	diff;

	sprintf (buffer, "%s/%s", dirname, wbFileName);
	if ((f = fopen (buffer, "r")) == NULL)
		return;
	if (fgets (buffer, sizeof(buffer), f) == NULL)
		return;
	fclose(f);
	sscanf (buffer, "%d:%d:%d", &rb, &gb, &bb);
	diff = (target.rgain - rb) * (target.rgain - rb);
	diff += (target.bgain - gb) * (target.bgain - gb);
	diff += (target.bgain - bb) * (target.bgain - bb);
#define DEBUG
#ifdef DEBUG
	fprintf (stderr, "%s: Info: %s: diff == %g\n", progname, wbFileName, sqrt(diff));
#endif
	if (bestWB == NULL || diff < bestDiff) {
		if (bestWB != NULL)
			free (bestWB);
		bestWB = strdup (wbFileName);
		bestDiff = diff;
	}
}

void
SearchDir (const char *dirname, const char *ext)
{
	DIR *dir;
	struct dirent *de;
	int extlen = strlen (ext);

	if ((dir = opendir (dirname)) == NULL) {
		return;
	}
	while ((de = readdir (dir)) != NULL) {
		int len = strlen (de->d_name);
		if (len >= extlen &&
		    strcmp (&de->d_name[len-extlen], ext) == 0)
			CompareWB (dirname, de->d_name);
	}
	closedir (dir);
}

int
main(int argc, char *argv[])
{
	MRI *mri;
	char buffer[512];
	char *dir;
	char *errmsg;

	progname = argv[0];

	InitOptions();
	sprintf (buffer, "%s/mrwguesswb.rc", DATAPREFIX);
	ProcessOptionFile (buffer, "mrwguesswb");
	if (getenv ("HOME")) {
		sprintf (buffer, "%s/.mrwtoppm/mrwguesswb.rc", getenv("HOME"));
		ProcessOptionFile (buffer, "mrwguesswb");
	}
	if (!ProcessCmdLineOptions (argc-1, argv+1))
		Usage();

	mri = MRW_Loader (stdin, &errmsg);
	if (mri == (MRI *)0) {
		fprintf (stderr, "%s: cannot read image from standard input (%s).\n", progname, errmsg);
		Usage ();
	}

	if (MRI_GetFlash (mri)) {
		printf ("flash\n");
		exit (0);
	}
	if (!MRI_GetBalance (&target, mri, "avgestimate")) {
		fprintf (stderr, "Error: unable to compute avg gray world estimate\n");
		exit (0);
	}
	dir = strtok (colorMapPath, ":");
	while (dir) {
#ifdef DEBUG
		fprintf (stderr, "%s: Info: searching %s\n", progname, dir);
#endif
		SearchDir (dir, ".cmap");
		dir = strtok (NULL, ":");
	}
	if (bestWB == NULL) {
		fprintf (stderr, "%s: Error: No white balance files found.\n", progname);
		return 1;
	}
	else {
#ifdef DEBUG
		fprintf (stderr, "%s: Info: Best wb: %s, diff = %g\n", progname, bestWB, sqrt(bestDiff));
#endif
		bestWB[strlen(bestWB)-strlen(".cmap")] = '\0';
		fprintf (stdout, "%s\n", bestWB);
		return 0;
	}
}
