/*
 * msmtp.c
 *
 * This file is part of msmtp, an SMTP client.
 *
 * Copyright (C) 2000, 2003, 2004
 * Martin Lambers <marlam@users.sourceforge.net>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   msmtp is released under the GPL with the additional exemption that
 *   compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
extern int errno;
#include <time.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
extern char *optarg;
extern int optind;
#endif
#ifdef HAVE_SYSEXITS_H
#include <sysexits.h>
#else
#define EX_OK		0
#define EX_USAGE	64
#define EX_DATAERR	65
#define EX_NOINPUT	66
#define EX_NOHOST	68
#define EX_UNAVAILABLE	69
#define EX_SOFTWARE	70
#define EX_OSERR	71
#define EX_IOERR	74
#define EX_TEMPFAIL	75
#define EX_PROTOCOL	76
#define EX_NOPERM	77
#define EX_CONFIG	78
#endif

#include "list.h"
#include "merror.h"
#include "paths.h"
#include "conffile.h"
#include "net.h"
#include "smtp.h"
#ifdef HAVE_SSL
#include "tls.h"
#endif /* HAVE_SSL */

/* Default configuration file. */
#ifdef WINDOWS
#define CONFFILE "msmtprc.txt"
#elif defined (DJGPP)
#define CONFFILE "_msmtprc"
#else /* UNIX */
#define CONFFILE ".msmtprc"
#endif

/* The name of this program */
char *prgname;


/*
 * Translate error codes from net.h, tls.h or smtp.h 
 * to error codes from sysexits.h
 */

int exitcode_net(int net_error_code)
{
    switch (net_error_code)
    {
	case NET_EHOSTNOTFOUND:
	    return EX_NOHOST;

	case NET_ESOCKET:
	    return EX_OSERR;

	case NET_ECONNECT:
	    return EX_TEMPFAIL;	    

	case NET_EIO:
	    return EX_IOERR;
	    
	case NET_ELIBFAILED:
	default:
	    return EX_SOFTWARE;
    }
}

#ifdef HAVE_SSL
int exitcode_tls(int tls_error_code)
{
    switch (tls_error_code)
    {
	case TLS_EIO:
	    return EX_IOERR;

	case TLS_EFILE:
	    return EX_NOINPUT;

	case TLS_EHANDSHAKE:
	    return EX_PROTOCOL;

	case TLS_ECERT:
	    /* did not find anything better... */
	    return EX_UNAVAILABLE;

	case TLS_ELIBFAILED:
	case TLS_ESEED:
	default:
	    return EX_SOFTWARE;
    }
}
#endif /* HAVE_SSL */

int exitcode_smtp(int smtp_error_code)
{
    switch (smtp_error_code)
    {
	case SMTP_EIO:
	    return EX_IOERR;

	case SMTP_EPROTO:
	    return EX_PROTOCOL;

	case SMTP_ENOMEM:
	    return EX_OSERR;

	case SMTP_EINVAL:
	    return EX_DATAERR;
	    
	case SMTP_EAUTHFAIL:
	    return EX_NOPERM;

	case SMTP_EINSECURE:
	case SMTP_EUNAVAIL:
	    return EX_UNAVAILABLE;

	case SMTP_ELIBFAILED:
	default:
	    return EX_SOFTWARE;
    }
}

    
/*
 * msmtp_endsession()
 *
 * Quit an SMTP session and close the connection.
 * QUIT is only sent when the flag 'quit' is set.
 */

void msmtp_endsession(smtp_server_t *srv, int quit)
{
    if (quit)
    {
	(void)smtp_quit(srv);
    }
    smtp_close(srv);
}


/*
 * msmtp_serverinfo()
 *
 * Prints information about the SMTP server specified in the account 'acc'.
 * 'errstr' must be MERROR_BUFSIZE characters long.
 * If an error occured, 'errstr' may contain a descriptive message (or an empty
 * string) and 'msg' may contain the offending message from the SMTP server (or
 * NULL).
 */

int msmtp_serverinfo(account_t *acc, int debug, char *errstr, list_t **msg)
{
    smtp_server_t srv;
    char *server_greeting;
    merror_t e;
    
    errstr[0] = '\0';
    *msg = NULL;
    
    /* create a new smtp_server_t */
    srv = smtp_new(debug ? stdout : NULL);

    /* prepare tls */
#ifdef HAVE_SSL
    if (acc->tls)
    {
	if (!merror_ok(e = smtp_tls_init(&srv, acc->tls_key_file, 
			acc->tls_cert_file, acc->tls_trust_file)))
	{
	    strcpy(errstr, e.errstr);
	    return exitcode_tls(e.number);
	}
    }
#endif /* HAVE_SSL */

    /* connect */
    if (!merror_ok(e = smtp_connect(&srv, acc->host, acc->port)))
    {
    	strcpy(errstr, e.errstr);
	return exitcode_net(e.number);
    }

    /* start tls for ssmtp servers */
#ifdef HAVE_SSL
    if (acc->tls && acc->tls_nostarttls)
    {
	if (!merror_ok(e = smtp_tls(&srv, acc->host, acc->tls_nocertcheck)))
	{
	    strcpy(errstr, e.errstr);
	    msmtp_endsession(&srv, 0);
	    return exitcode_tls(e.number);
	}
    }
#endif /* HAVE_SSL */

    /* get greeting */
    if (!merror_ok(e = smtp_get_greeting(&srv, msg, &server_greeting)))
    {
    	strcpy(errstr, e.errstr);
	msmtp_endsession(&srv, 0);
	return exitcode_smtp(e.number);
    }
		    
    /* initialize session */
    if (!merror_ok(e = smtp_init(&srv, acc->domain, msg)))
    {
    	strcpy(errstr, e.errstr);
	msmtp_endsession(&srv, 1);
	free(server_greeting);
	return exitcode_smtp(e.number);
    }
    
    /* start tls for starttls servers */
#ifdef HAVE_SSL
    if (acc->tls && !acc->tls_nostarttls)
    {
	if (!(srv.cap.flags & SMTP_CAP_STARTTLS))
	{
	    strcpy(errstr, "the SMTP server does not support TLS via the STARTTLS command");
	    msmtp_endsession(&srv, 1);
	    free(server_greeting);
	    return EX_UNAVAILABLE;
	}
	if (!merror_ok(e = smtp_tls_starttls(&srv, msg)))
	{
	    strcpy(errstr, e.errstr);
	    msmtp_endsession(&srv, 1);
	    free(server_greeting);
	    return exitcode_smtp(e.number);
	}
	if (!merror_ok(e = smtp_tls(&srv, acc->host, acc->tls_nocertcheck)))
	{
	    strcpy(errstr, e.errstr);
	    msmtp_endsession(&srv, 0);
	    free(server_greeting);
	    return exitcode_tls(e.number);
	}
	/* initialize again */
	if (!merror_ok(e = smtp_init(&srv, acc->domain, msg)))
	{
	    strcpy(errstr, e.errstr);
	    msmtp_endsession(&srv, 1);
	    free(server_greeting);
	    return exitcode_smtp(e.number);
	}
    }
#endif /* HAVE_SSL */

    /* end session */
    msmtp_endsession(&srv, 1);

    /* print results */
    printf("SMTP server at %s", acc->host);
#ifdef HAVE_SSL
    if ((acc->tls_nostarttls && acc->port != 465) || (!acc->tls_nostarttls && acc->port != 25))
#else /* not HAVE_SSL */
    if (acc->port != 25)
#endif /* not HAVE_SSL */
    {
	printf(", port %d", acc->port);
    }
    printf(": %s\n", server_greeting);
    free(server_greeting);
#ifdef HAVE_SSL
    if (srv.cap.flags == 0 && !(acc->tls && !acc->tls_nostarttls))
#else /* not HAVE_SSL */
    if (srv.cap.flags == 0)
#endif /* not HAVE_SSL */
    {
	printf("No special capabilities.\n");
    }
    else
    {
	if (srv.cap.flags & SMTP_CAP_SIZE)
	{
    	    printf("SIZE %ld - Maximum message size is %ld bytes",
    		    srv.cap.size, srv.cap.size);
    	    if (srv.cap.size > 1024 * 1024)
    	    {
    		printf(" = %.2f MB", (float)srv.cap.size / (1024.0 * 1024.0));
    	    }
    	    else if (srv.cap.size > 1024)
    	    {
    		printf(" = %.2f KB", (float)srv.cap.size / 1024.0);
    	    }
    	    printf("\n");
       	}
	if (srv.cap.flags & SMTP_CAP_PIPELINING)
	{
    	    printf("PIPELINING - Support for command grouping for faster transmission\n");
       	}
	if (srv.cap.flags & SMTP_CAP_DSN)
	{
    	    printf("DSN - Delivery Status Notifications\n");
       	}
#ifdef HAVE_SSL
	if ((acc->tls && !acc->tls_nostarttls) || (srv.cap.flags & SMTP_CAP_STARTTLS))
#else /* not HAVE_SSL */
        if (srv.cap.flags & SMTP_CAP_STARTTLS)
#endif /* not HAVE_SSL */
	{
    	    printf("STARTTLS - Support for TLS encryption via the STARTTLS command\n");
       	}
	if (srv.cap.flags & SMTP_CAP_AUTH)
	{
    	    printf("AUTH - User authentication methods: ");
    	    if (srv.cap.flags & SMTP_CAP_AUTH_PLAIN)
    	    {
    		printf("PLAIN ");
    	    }
    	    if (srv.cap.flags & SMTP_CAP_AUTH_CRAM_MD5)
    	    {
    		printf("CRAM-MD5 ");
    	    }
    	    if (srv.cap.flags & SMTP_CAP_AUTH_DIGEST_MD5)
    	    {
    		printf("DIGEST-MD5 ");
    	    }
    	    if (srv.cap.flags & SMTP_CAP_AUTH_LOGIN)
    	    {
    		printf("LOGIN ");
    	    }
    	    if (srv.cap.flags & SMTP_CAP_AUTH_NTLM)
    	    {
    		printf("NTLM");
    	    }
    	    printf("\n");
       	}
#ifdef HAVE_SSL
	if ((srv.cap.flags & SMTP_CAP_STARTTLS) && !acc->tls)
#else /* not HAVE_SSL */
	if (srv.cap.flags & SMTP_CAP_STARTTLS)
#endif /* not HAVE_SSL */
	{
    	    printf("This server might advertise more or other capabilities when TLS is active.\n");
       	}
    }

    return EX_OK;
}

 
/*
 * msmtp_sendmail()
 *
 * Sends a mail. Returns a value from sysexits.h.
 * 'errstr' must be MERROR_BUFSIZE characters long.
 * If an error occured, 'errstr' may contain a descriptive message (or an empty
 * string) and 'msg' may contain the offending message from the SMTP server (or
 * NULL).
 * In case of success, 'mailsize' contains the number of bytes of the mail 
 * transferred to the SMTP server. In case of failure, its contents are
 * undefined.
 */

int msmtp_sendmail(account_t *acc, char *envelope_from, int rcptc, char *rcptv[], 
	FILE *f, int debug, long *mailsize, char *errstr, list_t **msg)
{
    smtp_server_t srv;
    merror_t e;
    
    errstr[0] = '\0';
    *msg = NULL;
    
    /* create a new smtp_server_t */
    srv = smtp_new(debug ? stdout : NULL);

    /* prepare tls */
#ifdef HAVE_SSL
    if (acc->tls)
    {
	if (!merror_ok(e = smtp_tls_init(&srv, acc->tls_key_file, 
			acc->tls_cert_file, acc->tls_trust_file)))
	{
	    strcpy(errstr, e.errstr);
	    return exitcode_tls(e.number);
	}
    }
#endif /* HAVE_SSL */

    /* connect */
    if (!merror_ok(e = smtp_connect(&srv, acc->host, acc->port)))
    {
    	strcpy(errstr, e.errstr);
	return exitcode_net(e.number);
    }

    /* start tls for ssmtp servers */
#ifdef HAVE_SSL
    if (acc->tls && acc->tls_nostarttls)
    {
	if (!merror_ok(e = smtp_tls(&srv, acc->host, acc->tls_nocertcheck)))
	{
	    strcpy(errstr, e.errstr);
	    msmtp_endsession(&srv, 0);
	    return exitcode_tls(e.number);
	}
    }
#endif /* HAVE_SSL */

    /* get greeting */
    if (!merror_ok(e = smtp_get_greeting(&srv, msg, NULL)))
    {
    	strcpy(errstr, e.errstr);
	msmtp_endsession(&srv, 0);
	return exitcode_smtp(e.number);
    }
		    
    /* initialize session */
    if (!merror_ok(e = smtp_init(&srv, acc->domain, msg)))
    {
	strcpy(errstr, e.errstr);
	msmtp_endsession(&srv, 1);
	return exitcode_smtp(e.number);
    }

    /* start tls for starttls servers */
#ifdef HAVE_SSL
    if (acc->tls && !acc->tls_nostarttls)
    {
	if (!(srv.cap.flags & SMTP_CAP_STARTTLS))
	{
	    strcpy(errstr, "the SMTP server does not support TLS "
		    "via the STARTTLS command");
	    msmtp_endsession(&srv, 1);
	    return EX_UNAVAILABLE;
	}
	if (!merror_ok(e = smtp_tls_starttls(&srv, msg)))
	{
	    strcpy(errstr, e.errstr);
	    msmtp_endsession(&srv, 1);
	    return exitcode_smtp(e.number);
	}
	if (!merror_ok(e = smtp_tls(&srv, acc->host, acc->tls_nocertcheck)))
	{
	    strcpy(errstr, e.errstr);
	    msmtp_endsession(&srv, 0);
	    return exitcode_tls(e.number);
	}
	/* initialize again */
	if (!merror_ok(e = smtp_init(&srv, acc->domain, msg)))
	{
	    strcpy(errstr, e.errstr);
	    msmtp_endsession(&srv, 1);
	    return exitcode_smtp(e.number);
	}
    }
#endif /* HAVE_SSL */

    /* authenticate */
    if (acc->username)
    {
	if (!(srv.cap.flags & SMTP_CAP_AUTH))
	{
	    strcpy(errstr, "the SMTP server does not support authentication");
	    msmtp_endsession(&srv, 1);
	    return EX_UNAVAILABLE;
	}
	if (!merror_ok(e = smtp_auth(&srv, acc->host, acc->username, 
			acc->password, acc->auth_mech, msg)))
	{
	    strcpy(errstr, e.errstr);
	    msmtp_endsession(&srv, 1);
	    return exitcode_smtp(e.number);
	}
    }
    
    /* send mail */
    if (acc->dsn_return || acc->dsn_notify)
    {
	if (!(srv.cap.flags & SMTP_CAP_DSN))
	{
	    strcpy(errstr, "the SMTP server does not support DSN");
	    msmtp_endsession(&srv, 1);
	    return EX_UNAVAILABLE;
	}
    }
    if (!merror_ok(e = smtp_send_mail(&srv, 
		    envelope_from ? envelope_from : acc->from, rcptc, rcptv, 
		    acc->dsn_notify, acc->dsn_return, acc->keepbcc, f, mailsize, msg)))
    {
	strcpy(errstr, e.errstr);
	msmtp_endsession(&srv, 1);
	return exitcode_smtp(e.number);
    }

    /* end session */
    msmtp_endsession(&srv, 1);

    return EX_OK;
}


/*
 * print_error()
 *
 * Print an error message
 */

void print_error(char *format, ...)
{
    va_list args;
    fprintf(stderr, "%s: ", prgname);
    va_start(args, format);
    vfprintf(stderr, format, args);
    va_end(args);
    fprintf(stderr, "\n");
}


/*
 * msmtp_sanitize_smtpmsg()
 *
 * Replaces all non-printable characters in the SMTP server message with a
 * question mark.
 */

void msmtp_sanitize_smtpmsg(list_t *msg)
{
    int i;
    char *s;
    
    while (!list_is_empty(msg))
    {
	msg = msg->next;
	s = msg->data;
	i = 0;
	while (s[i] != '\0')
	{
	    if (!isprint((unsigned char)s[i]))
	    {
		s[i] = '?';
	    }
	    i++;
	}
    }
}


/*
 * msmtp_log()
 *
 * Append a log entry to 'acc->logfile' with the following information:
 * - date/time 
 * - conffile=%s 
 * - account=%s 
 * - from=%s 
 * - recipients=%s,%s,... 
 * - mailsize=%s (only if exitcode == EX_OK)
 * - smtpstatus=%s (only if exitcode != EX_OK and a smtp msg is available)
 * - smtpmsg='%s' (only if exitcode != EX_OK and a smtp msg is available)
 * - errormsg='%s' (only if exitcode != EX_OK and an error msg is available)
 * - exitcode=%s 
 * 'errorcode' must be one of the codes returned by msmtp_sendmail().
 */

void msmtp_log(char *conffile, account_t *acc, 
	char *envelope_from, int rcptc, char *rcptv[], 
	long mailsize, list_t *errmsg, char *errstr, int errorcode)
{
    FILE *f;
    int i;
    time_t t;
    struct tm *tm;
    size_t s;
    list_t *l;
    char *line;
    int n;
    char *p;
    char *failure_reason;
    /* temporary strings: */
    char time_str[64];
    char mailsize_str[80];	/* sufficient for 256 bit longs */
    char *errorcode_str;
    char smtpstatus_str[4];
    char *smtperrmsg_str = NULL;
    
    
    /*
     * gather information for the log line
     */
    
    line = NULL;
    /* time */
    if ((t = time(NULL)) < 0)
    {
	failure_reason = strerror(errno);
	goto log_failure;
    }
    if (!(tm = localtime(&t)))
    {
	failure_reason = "cannot convert UTC time to local time";
	goto log_failure;
    }
    if (strftime(time_str, 64, "%b %d %H:%M:%S", tm) > 63)
    {
	failure_reason = "BUG: time_str is too short";
	goto log_failure;
    }
    /* mailsize */
    if (errorcode == EX_OK)
    {
	if (snprintf(mailsize_str, 80, "%ld", mailsize) >= 80)
	{
	    failure_reason = "BUG: mailsize_str is too short";
	    goto log_failure;
	}
    }	
    /* exitcode */
    switch (errorcode)
    {
	/* These are the exitcodes returned by msmtp_sendmail() */
	case EX_OK:
	    errorcode_str = "EX_OK";
	    break;

	case EX_DATAERR:
	    errorcode_str = "EX_DATAERR";
	    break;

	case EX_NOINPUT:
	    errorcode_str = "EX_NOINPUT";
	    break;

	case EX_NOHOST:
	    errorcode_str = "EX_NOHOST";
	    break;

	case EX_UNAVAILABLE:
	    errorcode_str = "EX_UNAVAILABLE";
	    break;

	case EX_SOFTWARE:
	    errorcode_str = "EX_SOFTWARE";
	    break;

	case EX_OSERR:
	    errorcode_str = "EX_OSERR";
	    break;

	case EX_IOERR:
	    errorcode_str = "EX_IOERR";
	    break;

	case EX_TEMPFAIL:
	    errorcode_str = "EX_TEMPFAIL";
	    break;
	    
	case EX_PROTOCOL:
	    errorcode_str = "EX_PROTOCOL";
	    break;

	case EX_NOPERM:
	    errorcode_str = "EX_NOPERM";
	    break;

	default:
	    errorcode_str = "BUG:UNKNOWN";
	    break;
    }
    /* smtp status and smtp error message */
    if (errorcode != EX_OK && errmsg)
    {
	(void)snprintf(smtpstatus_str, 4, "%d", smtp_msg_status(errmsg));
	msmtp_sanitize_smtpmsg(errmsg);
	l = errmsg;
	s = 0;
	while (!list_is_empty(l))
	{
	    l = l->next;
	    s += strlen(l->data) + 2;
	}
	s += 1;
	if (!(smtperrmsg_str = malloc(s * sizeof(char))))
	{
	    failure_reason = strerror(errno);
	    goto log_failure;
	}
	smtperrmsg_str[0] = '\'';
	i = 1;
	l = errmsg;
	while (!list_is_empty(l))
	{
	    l = l->next;
	    p = l->data;
	    while (*p != '\0')
	    {
		smtperrmsg_str[i] = (*p == '\'') ? '?' : *p;
		p++;
		i++;
	    }
	    smtperrmsg_str[i++] = '\\';
	    smtperrmsg_str[i++] = 'n';
	}
	i -= 2;
	smtperrmsg_str[i++] = '\'';
	smtperrmsg_str[i++] = '\0';
    }
    
    /*
     * calculate the length of the log line
     */
    
    s = 0;
    /* time: "%b %d %H:%M:%S " */
    s += strlen(time_str) + 1;
    /* "conffile=%s " */
    s += 9 + strlen(conffile) + 1;
    /* "account=%s " */
    s += 8 + strlen(acc->id) + 1;
    /* "from=%s " */
    s += 5 + strlen(envelope_from ? envelope_from : acc->from) + 1;
    /* "recipients=%s,%s,... " */
    s += 11;
    for (i = 0; i < rcptc; i++)
    {
	s += strlen(rcptv[i]) + 1;
    }
    /* "mailsize=%s " */
    if (errorcode == EX_OK)
    {
	s += 9 + strlen(mailsize_str) + 1;
    }
    /* "smtpstatus=%s smtpmsg=%s " */
    if (errorcode != EX_OK && errmsg)
    {
	s += 11 + strlen(smtpstatus_str) + 1 + 8 + strlen(smtperrmsg_str) + 1;
    }
    /* "errormsg='%s' */
    if (errorcode != EX_OK && errstr[0] != '\0')
    {
	s += 10 + strlen(errstr) + 2;
    }
    /* "exitcode=%s" */
    s += 9 + strlen(errorcode_str);
    /* '\0' */
    s++;
    
    if (!(line = malloc(s * sizeof(char))))
    {
	failure_reason = strerror(errno);
	goto log_failure;
    }
    
    /*
     * build the log line
     */
    
    /* line should be long enough */
    n = snprintf(line, s, "%s conffile=%s account=%s from=%s recipients=", 
	    time_str, conffile, acc->id, envelope_from ? envelope_from : acc->from);
    if ((size_t)n >= s)
    {
	goto line_too_short;
    }
    s -= (size_t)n;
    p = line + n;
    for (i = 0; i < rcptc; i++)
    {
	n = snprintf(p, s, "%s,", rcptv[i]);
	if ((size_t)n >= s)
	{
	    goto line_too_short;
	}
	s -= (size_t)n;
	p += n;
    }
    /* delete the last ',' */
    *(p - 1) = ' ';
    if (errorcode == EX_OK)
    {
	n = snprintf(p, s, "mailsize=%s ", mailsize_str);
	if ((size_t)n >= s)
	{
	    goto line_too_short;
	}
	s -= (size_t)n;
	p += n;
    }
    if (errorcode != EX_OK && errmsg)
    {
	n = snprintf(p, s, "smtpstatus=%s smtpmsg=%s ", smtpstatus_str, smtperrmsg_str);
	if ((size_t)n >= s)
	{
	    goto line_too_short;
	}
	s -= (size_t)n;
	p += n;
    }
    if (errorcode != EX_OK && errstr[0] != '\0')
    {
	n = snprintf(p, s, "errormsg='%s' ", errstr);
	if ((size_t)n >= s)
	{
	    goto line_too_short;
	}
	s -= (size_t)n;
	p += n;
    }
    n = snprintf(p, s, "exitcode=%s", errorcode_str);
    if ((size_t)n >= s)
    {
	goto line_too_short;
    }
    
    /*
     * write log line to file
     */
    
    if (strcmp(acc->logfile, "-") == 0)
    {
	f = stdout;
    }
    else
    {
	if (!(f = fopen(acc->logfile, "a")))
	{
	    failure_reason = strerror(errno);
	    goto log_failure;
	}
    }
    if ((fputs(line, f) == EOF) || (fputc('\n', f) == EOF))
    {
	failure_reason = "output error";
	goto log_failure;
    }
    if (f != stdout && fclose(f) != 0)
    {
	failure_reason = strerror(errno);
	goto log_failure;
    }

    return;
    
    /*
     * error exit targets
     */
    
line_too_short:
    failure_reason = "BUG: log line too short";
    free(line);
    line = NULL;	

log_failure:
    free(smtperrmsg_str);
    print_error("cannot log to %s: %s", acc->logfile, failure_reason);
    if (line && (line[0] != '\0'))
    {
	print_error("log line was: %s", line);
    }
    free(line);
}


/*
 * The main function.
 * It returns values from sysexits.h (like sendmail does).
 */

int main(int argc, char *argv[])
{
    /* the configuration */
    int print_version;
    int print_help;
    int print_conf;
    int debug;
    int sendmail;
    int serverinfo;
    char *envelope_from;
    char *conffile;
    char *account_id;
    list_t *account_list;
    account_t *account;
    /* the recipients of the mail */
    int rcptc;
    char **rcptv;
    /* error handling */
    int error_code;
    merror_t e;
    char errstr[MERROR_BUFSIZE];
    list_t *errmsg;
    list_t *lp;
    /* misc */
    int c;
    long mailsize;
#ifdef HAVE_GETOPT_LONG
    int option_index;
    struct option options[] =
    {
	{ "version",    no_argument, 0, 'v' },
	{ "help",       no_argument, 0, 'h' },
	{ "pretend",    no_argument, 0, 'p' },
	{ "debug",      no_argument, 0, 'd' },
	{ "serverinfo", no_argument, 0, 'S' },
	{ "from",       required_argument, 0, 'f' },
	{ "file",       required_argument, 0, 'F' },
	{ "account",    required_argument, 0, 'a' },
	{ 0, 0, 0, 0 }
    };
#endif /* HAVE_GETOPT_LONG */
    
    
    /* process the command line */
    prgname = get_prgname(argv[0]);
    error_code = 0;
    print_version = 0;
    print_help = 0;
    print_conf = 0;
    debug = 0;
    sendmail = 1;
    serverinfo = 0;
    envelope_from = NULL;
    conffile = NULL;
    account_id = NULL;
    for (;;)
    {
#ifdef HAVE_GETOPT_LONG
	c = getopt_long(argc, argv, "vhpdSf:F:a:i", options, &option_index);
#else
	c = getopt(argc, argv, "vhpdSf:F:a:i");
#endif /* HAVE_GETOPT_LONG */
	if (c == -1)
	{
	    break;
	}
	switch(c)
	{
	    case 'v':
		print_version = 1;
		sendmail = 0;
		serverinfo = 0;
		break;

	    case 'h':
		print_help = 1;
		sendmail = 0;
		serverinfo = 0;
		break;

	    case 'p':
		print_conf = 1;
		sendmail = 0;
		serverinfo = 0;
		break;
		
	    case 'd':
		print_conf = 1;
		debug = 1;
		break;
		
	    case 'S':
		serverinfo = 1;
		sendmail = 0;
		break;

	    case 'f':
		envelope_from = optarg;
		break;

	    case 'F':
		conffile = optarg;
		break;

	    case 'a':
		account_id = optarg;
		break;

	    case 'i':
		/* ignore this flag; it exists for mail(1) compatibility */
		break;
		
	    default:
		error_code = 1;
		break;
	}
	if (error_code)
	{
	    break;
	}
    }
    if (error_code)
    {
	return EX_USAGE;
    }

    if (print_version)
    {
	printf("%s version %s\nTLS/SSL library: %s\n"
		"Authentication library: %s\n"
		"Supported authentication methods: ",
		PACKAGE_NAME, VERSION,
#ifdef HAVE_GNUTLS
		"GnuTLS"
#elif defined (HAVE_OPENSSL)
		"OpenSSL"
#else
		"none"
#endif
		,
#ifdef USE_GSASL
		"GNU SASL"
#else
		"built-in"
#endif /* USE_GSASL */
	      );
	if (smtp_authmech_is_supported("DIGEST-MD5"))
	{
	    printf("digest-md5 ");
	}
	if (smtp_authmech_is_supported("CRAM-MD5"))
	{
	    printf("cram-md5 ");
	}
	if (smtp_authmech_is_supported("PLAIN"))
	{
	    printf("plain ");
	}
	if (smtp_authmech_is_supported("NTLM"))
	{
	    printf("ntlm ");
	}
	if (smtp_authmech_is_supported("LOGIN"))
	{
	    printf("login");
	}
	printf("\n\nCopyright (C) 2004 Martin Lambers and others.\n"
		"This is free software; see the source for copying conditions.  There is NO\n"
		"warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n");
    }
    if (print_help)
    {
	printf("Usage: %s [option...] [--] recipient...\n"
		"Reads a mail from standard input and transmits it to an SMTP server.\n"
		"Options are:\n", prgname);
#ifdef HAVE_GETOPT_LONG
	printf("-v, --version        print version and exit\n"
		"-h, --help           print help and exit\n"
		"-p, --pretend        print configuration info and exit\n"
		"-d, --debug          print debugging information\n"
		"-S, --serverinfo     print information about the SMTP server\n"
		"-f, --from=address   set envelope from address\n"
		"-F, --file=filename  set configuration file\n"
		"-a, --account=id     use other account than default\n");
#else
	printf("-v           print version and exit\n"
		"-h           print help and exit\n"
		"-p           print configuration info and exit\n"
		"-d           print debugging information\n"
		"-S           print information about the SMTP server\n"
		"-f address   set envelope from address\n"
		"-F filename  set configuration file\n"
		"-a id        use other account than default\n");
#endif /* HAVE_GETOPT_LONG */
	printf("Report bugs to <%s>.\n", PACKAGE_BUGREPORT);
    }
    
    if (!sendmail && !serverinfo && !print_conf)
    {
	return EX_OK;
    }

    /* get the list of recipients */
    rcptc = argc - optind;
    rcptv = &(argv[optind]);
    if (rcptc == 0 && sendmail)
    {
	print_error("no recipients given");
	return EX_USAGE;
    }
    if (rcptc != 0 && serverinfo)
    {
	print_error("too many arguments");
	return EX_USAGE;
    }

    /* read the configuration file */
    if (!conffile)
    {
	if (!merror_ok(e = get_default_conffile(CONFFILE, &conffile)))
	{
	    print_error("cannot get default configuration file name: %s", e.errstr);
	    return EX_CONFIG;
	}
    }
    if (!merror_ok(e = get_conf(conffile, &account_list)))
    {
	print_error("%s: %s", conffile, e.errstr);
	if (e.number == CONFFILE_ENOMEM)
	{
	    return EX_OSERR;
	}
	else if (e.number == CONFFILE_EIO)
	{
	    return EX_IOERR;
	}
	else
	{
	    return EX_CONFIG;
	}
    }

    /* get the account data */
    if (!account_id)
    {
	account_id = "default";
    }
    if (!(account = find_account(account_list, account_id)))
    {
	print_error("no account %s found in configuration file %s",
		account_id, conffile);
	return EX_CONFIG;
    }
    if (sendmail && !account->from && !envelope_from)
    {
	print_error("no envelope from address for account %s from file %s", 
		account_id, conffile);
	return EX_USAGE;
    }

    if (print_conf)
    {
    	printf("using account %s from %s:\n"
		"host            = %s\n"
		"port            = %d\n"
		"domain          = %s\n"
		"from            = %s\n",
		account->id, conffile,
		account->host,
		account->port,
		account->domain,
		envelope_from ? envelope_from : account->from);
	printf("auth            = ");
	if (!account->auth_mech)
	{
	    printf("none\n");
	}
	else if (account->auth_mech[0] == '\0')
	{
	    printf("choose\n");
	}
	else
	{
	    printf("%s\n", account->auth_mech);
	}
	printf("user            = %s\n"
		"password        = %s\n"
		"tls             = %s\n"
		"tls_trust_file  = %s\n"
		"tls_key_file    = %s\n"
		"tls_cert_file   = %s\n"
		"tls_nostarttls  = %s\n"
		"tls_nocertcheck = %s\n"
		"dsn_notify      = %s\n"
		"dsn_return      = %s\n"
		"keepbcc         = %s\n"
		"logfile         = %s\n",
		account->username ? account->username : "",
		account->password ? "*" : "",
		account->tls ? "true" : "false", 
		account->tls_trust_file ? account->tls_trust_file : "",
		account->tls_key_file ? account->tls_key_file : "",
		account->tls_cert_file ? account->tls_cert_file : "",
		account->tls_nostarttls ? "true" : "false",
		account->tls_nocertcheck ? "true" : "false",
		account->dsn_notify ? account->dsn_notify : "",
		account->dsn_return ? account->dsn_return : "",
		account->keepbcc ? "true" : "false",
		account->logfile ? account->logfile : "");
    }
    if (!sendmail && !serverinfo)
    {
	return EX_OK;
    }
    
    /* initialize libraries */
    if (!merror_ok(e = net_lib_init()))
    {
	print_error("cannot initialize network library: %s", e.errstr);
	list_xfree(account_list, account_free);
	return EX_SOFTWARE;	
    }
    if (account->auth_mech && (strcmp(account->auth_mech, "") != 0) 
	    && !smtp_authmech_is_supported(account->auth_mech))
    {
	print_error("support for authentication method %s is not compiled in", 
		account->auth_mech);
	list_xfree(account_list, account_free);
	return EX_UNAVAILABLE;
    }
    if (account->tls)
    {
#ifdef HAVE_SSL
	if (!merror_ok(e = tls_lib_init()))
	{
	    print_error("cannot initialize TLS library: %s", e.errstr);
	    list_xfree(account_list, account_free);
	    return EX_SOFTWARE;
	}
#else /* not HAVE_SSL */
	print_error("support for TLS is not compiled in");
	list_xfree(account_list, account_free);
	return EX_UNAVAILABLE;
#endif /* not HAVE_SSL */
    }

    /* do the work */
    if (serverinfo)
    {
	if ((error_code = msmtp_serverinfo(account, debug, 
			errstr, &errmsg)) != EX_OK)
	{
	    if (errstr[0] != '\0')
	    {
		print_error("%s", errstr);
	    }
	    if (errmsg)
	    {
		msmtp_sanitize_smtpmsg(errmsg);
		lp = errmsg;
		while (!list_is_empty(lp))
		{
		    lp = lp->next;
		    print_error("SMTP server message: %s", lp->data);
		}
	    }		    
	}
    }
    else /* if (sendmail) */
    {
	if ((error_code = msmtp_sendmail(account, envelope_from, rcptc, rcptv, 
			stdin, debug, &mailsize, errstr, &errmsg)) != EX_OK)
	{
	    if (errstr[0] != '\0')
	    {
		print_error("%s", errstr);
	    }
	    if (errmsg)
	    {
		msmtp_sanitize_smtpmsg(errmsg);
		lp = errmsg;
		while (!list_is_empty(lp))
		{
		    lp = lp->next;
		    print_error("SMTP server message: %s", lp->data);
		}
	    }		    
	    print_error("could not send mail (account %s from %s)",
	    	    account_id, conffile);
	}
	if (account->logfile)
	{
	    msmtp_log(conffile, account, envelope_from, rcptc, rcptv, 
		    mailsize, errmsg, errstr, error_code);
	}
    }

    /* clean up */
#ifdef HAVE_SSL
    if (account->tls)
    {
	tls_lib_deinit();
    }
#endif /* HAVE_SSL */
    net_lib_deinit();
    list_xfree(account_list, account_free);

    return error_code;
}
