//streams.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010
 *
 *  This file is part of RoarD,
 *  a sound server daemon for using the RoarAudio protocol.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  or (at your option) any later version as published by
 *  the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "muroard.h"

int stream_init(void) {
 memset(g_stream, 0, sizeof(g_stream)); // unused = 0
 return 0;
}

int stream_free(void) {
 int i;
 int ret = 0;

 for (i = 0; i < MUROAR_MAX_STREAMS; i++) {
  if ( g_stream[i].state != STREAM_STATE_UNUSED ) {
   if ( stream_delete(i) == -1 )
    ret = -1;
  }
 }

 return ret;
}

int stream_new(int client, int dir, struct muroar_audio_info * info) {
#if defined(MUROAR_FEATURE_UPSCALE_INPUT) || defined(MUROAR_FEATURE_DOWNSCALE_OUTPUT)
 int need_scale = 0;
#define _NEED_SCALE need_scale
#endif
#if defined(MUROAR_FEATURE_UPMIX_INPUT) || defined(MUROAR_FEATURE_DOWNMIX_OUTPUT)
 int need_mix   = 0;
#define _NEED_MIX need_mix
#endif
 int i;

#ifdef _NEED_SCALE
 if ( info->bits == 8 ) {
  need_scale = 1;
  switch (info->codec) {
/*
#define MUROAR_CODEC_PCM_S_LE     0x01
#define MUROAR_CODEC_PCM_S_BE     0x02
#define MUROAR_CODEC_PCM_S_PDP    0x03
#define MUROAR_CODEC_PCM_U_LE     0x05
#define MUROAR_CODEC_PCM_U_BE     0x06
#define MUROAR_CODEC_PCM_U_PDP    0x07
*/
   case MUROAR_CODEC_PCM_S_LE:
   case MUROAR_CODEC_PCM_S_BE:
   case MUROAR_CODEC_PCM_S_PDP:
     info->codec = MUROAR_CODEC_PCM_S;
    break;
#ifdef MUROAR_FEATURE_UNSIGNED_8BIT
   case MUROAR_CODEC_PCM_U_LE:
   case MUROAR_CODEC_PCM_U_BE:
   case MUROAR_CODEC_PCM_U_PDP:
     info->codec = MUROAR_CODEC_PCM_U;
    break;
#endif
  }
 } else if ( info->bits != 16 ) {
  return -1;
 }
#else
 if ( info->bits != 16 )
  return -1;
#endif

#ifdef _NEED_MIX
 if ( info->channels == 1 ) {
  need_mix = 1;
 } else if ( info->channels != g_sa_channels ) {
  return -1;
 }
#else
 if ( info->channels != g_sa_channels )
  return -1;
#endif

#ifdef MUROAR_FEATURE_BYTE_SWAP
 if ( info->bits > 16 && info->codec != MUROAR_CODEC_PCM )
  return -1;

 switch (info->codec) {
  case MUROAR_CODEC_PCM_S_BE:
  case MUROAR_CODEC_PCM_S_LE:
  case MUROAR_CODEC_PCM_S_PDP:
#ifdef MUROAR_FEATURE_UNSIGNED_8BIT
  case MUROAR_CODEC_PCM_U_LE:
  case MUROAR_CODEC_PCM_U_BE:
  case MUROAR_CODEC_PCM_U_PDP:
#endif
   break;
  default:
    return -1;
   break;
 }
#else
 if ( info->codec != MUROAR_CODEC_PCM )
  return -1;
#endif

 if ( client_get_stream(client) != -1 )
  return -1;

 for (i = 0; i < MUROAR_MAX_STREAMS; i++) {
  if ( g_stream[i].state == STREAM_STATE_UNUSED ) {
   g_stream[i].iobuf = NULL;

   switch (dir) {
    case MUROAR_PLAY_WAVE:
      g_stream[i].datadir = STREAM_DATADIR_IN;
      if ( (g_stream[i].iobuf = muroard_malloc(g_abuffer_size)) == NULL ) {
       return -1;
      }
      memset(g_stream[i].iobuf, 0, g_abuffer_size); // clean memory to avoid noise
     break;
#ifdef MUROAR_FEATURE_MONITOR
    case MUROAR_MONITOR_WAVE:
       g_stream[i].datadir = STREAM_DATADIR_OUT;
      // no neet to set up anything
     break;
#endif
    default:
      return -1;
     break;
   }

#ifdef _NEED_SCALE
   if ( _NEED_SCALE ) {
    switch (g_stream[i].datadir) {
#ifdef MUROAR_FEATURE_UPSCALE_INPUT
     case STREAM_DATADIR_IN: break;
#endif
#ifdef MUROAR_FEATURE_DOWNSCALE_OUTPUT
     case STREAM_DATADIR_OUT: break;
#endif
     default:
       if ( g_stream[i].iobuf != NULL )
        muroard_free(g_stream[i].iobuf);
       return -1;
    }
   }
#endif

#ifdef _NEED_MIX
   if ( _NEED_MIX ) {
    switch (g_stream[i].datadir) {
#ifdef MUROAR_FEATURE_UPMIX_INPUT
     case STREAM_DATADIR_IN: break;
#endif
#ifdef MUROAR_FEATURE_DOWNMIX_OUTPUT
     case STREAM_DATADIR_OUT: break;
#endif
     default:
       if ( g_stream[i].iobuf != NULL )
        muroard_free(g_stream[i].iobuf);
       return -1;
    }
   }
#endif

   g_stream[i].sock   = -1;
   g_stream[i].client = client;
   g_stream[i].dir    = dir;

   memcpy(&(g_stream[i].info), info, sizeof(g_stream[i].info));

   g_stream[i].state = STREAM_STATE_NEW;
   client_set_stream(client, i); // TODO: check errors here
   return i;
  }
 }

 return -1;
}

int stream_delete(int id) {
 int state;
 int ret   = 0;

 if ( id >= MUROAR_MAX_STREAMS || id < 0 )
  return -1;

 state = g_stream[id].state;

 if ( state == STREAM_STATE_UNUSED  ||
      state == STREAM_STATE_CLOSING )
  return 0;

 g_stream[id].state = STREAM_STATE_CLOSING;

 if ( state == STREAM_STATE_EXECED ) {
  if ( client_delete(g_stream[id].client) == -1 )
   ret = -1;
 } else {
  g_client[g_stream[id].client].stream = -1;
 }

 if ( g_stream[id].sock != -1 ) {
  network_close(g_stream[id].sock);
 }

 if ( g_stream[id].iobuf != NULL )
  muroard_free(g_stream[id].iobuf);

 g_stream[id].state = STREAM_STATE_UNUSED;

 return ret;
}

int stream_exec(int id) {
 int sock;

 if ( (sock = client_exec(g_stream[id].client)) == -1 )
  return -1;

 g_stream[id].sock  = sock;
 g_stream[id].state = STREAM_STATE_EXECED;

 return 0;
}

int stream_read(int id) {
 ssize_t len;
 size_t readsize = g_abuffer_size;

#ifdef MUROAR_FEATURE_UPSCALE_INPUT
 if ( g_stream[id].info.bits == 8 )
  readsize /= 2;
#endif

#ifdef MUROAR_FEATURE_UPMIX_INPUT
 if ( g_stream[id].info.channels == 1 && g_sa_channels != 1 )
  readsize /= g_sa_channels;
#endif

 len = network_read(g_stream[id].sock, g_stream[id].iobuf, readsize);

 if ( len == (ssize_t) readsize ) {
  // no op here.
 } else if ( len == 0 ) {
  return stream_delete(id);
 } else if ( len == -1 ) {
  stream_delete(id);
  return -1;
 } else {
  memset(((void*)g_stream[id].iobuf)+len, 0, g_abuffer_size - len);
 }

#ifdef MUROAR_FEATURE_UPSCALE_INPUT
 if ( g_stream[id].info.bits == 8 ) {
  dsp_upscale(g_stream[id].iobuf, len);

  len *= 2;
 }
#endif

#ifdef MUROAR_FEATURE_BYTE_SWAP
 if ( dsp_need_swap_s(g_stream[id].info.codec) )
  dsp_swap16(g_stream[id].iobuf, len/2);
#endif

#ifdef MUROAR_FEATURE_UPMIX_INPUT
 if ( g_stream[id].info.channels == 1 && g_sa_channels != 1 )
  dsp_upmix(g_stream[id].iobuf, len/2);
#endif

 return 0;
}

#ifdef MUROAR_FEATURE_MONITOR
int stream_write(int id, int16_t * buf) {
 ssize_t len;
 size_t  writesize = g_abuffer_size;
 int     sock;
#if defined(MUROAR_FEATURE_DOWNSCALE_OUTPUT) || defined(MUROAR_FEATURE_DOWNMIX_OUTPUT) || defined(MUROAR_FEATURE_BYTE_SWAP)
 size_t   need_buf = 0;
#define _NEED_BUF need_buf
#endif

 if ( (sock = stream_get_sock(id)) == -1 )
  return -1;

#ifdef MUROAR_FEATURE_DOWNSCALE_OUTPUT
 if ( g_stream[id].info.bits == 8 ) {
  writesize /= 2;
  need_buf   = 1;
 }
#endif

#ifdef MUROAR_FEATURE_DOWNMIX_OUTPUT
 if ( g_stream[id].info.channels == 1 && g_sa_channels != 1 ) {
  writesize /= g_sa_channels;
  need_buf   = 1;
 }
#endif

#ifdef MUROAR_FEATURE_BYTE_SWAP
 if ( dsp_need_swap_s(g_stream[id].info.codec) )
  need_buf   = 1;
#endif

#ifdef _NEED_BUF
 if ( _NEED_BUF ) {
  if ( g_stream[id].iobuf == NULL ) {
   g_stream[id].iobuf = muroard_malloc(g_abuffer_size);
   if ( g_stream[id].iobuf == NULL )
    return -1;
  }

  memcpy(g_stream[id].iobuf, buf, g_abuffer_size);
  buf = g_stream[id].iobuf;

#ifdef MUROAR_FEATURE_DOWNMIX_OUTPUT
  if ( g_stream[id].info.channels == 1 && g_sa_channels != 1 ) {
   dsp_downmix(buf, g_abuffer_size/(2*g_sa_channels));
  }
#endif

#ifdef MUROAR_FEATURE_BYTE_SWAP
 if ( dsp_need_swap_s(g_stream[id].info.codec) )
  dsp_swap16(buf, (g_abuffer_size*g_stream[id].info.channels)/(2*g_sa_channels));
#endif

#ifdef MUROAR_FEATURE_DOWNSCALE_OUTPUT
  if ( g_stream[id].info.bits == 8 ) {
   dsp_downscale(buf, writesize);
  }
#endif
 }
#endif

 len = network_write(sock, buf, writesize);

 if ( len < 1 ) {
  stream_delete(id);
  return -1;
 }

 return 0;
}
#endif

#ifdef MUROAR_FEATURE_CMD_ATTACH
int stream_move_client(int id, int client) {
 if ( g_stream[id].state == STREAM_STATE_UNUSED )
  return -1;

 // if we are execed we need to close the client, too.
 // we use a little trick:
 // we tell the stream it is closing and close the client.
 // after the client has been closed we reset the stream to NEW.
 if ( g_stream[id].state == STREAM_STATE_EXECED ) {
  g_stream[id].state = STREAM_STATE_CLOSING;
  if ( client_delete(g_stream[id].client) == -1 ) {
   g_stream[id].state = STREAM_STATE_NEW;
   stream_delete(id);
   return -1;
  }
  g_stream[id].state = STREAM_STATE_NEW;
 }

 // now we have a free client we need to attach to the client like stream_new() does:
 g_stream[id].client = client;
 client_set_stream(client, id); // TODO: check errors here

 return 0;
}
#endif

//ll
