//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: seqmsg.cpp,v 1.32.2.1 2004/12/28 23:19:55 lunar_shuttle Exp $
//
//  (C) Copyright 2001 Werner Schweer (ws@seh.de)
//=========================================================

#include <stdio.h>

#include "song.h"
#include "midiport.h"
#include "minstrument.h"
#include "app.h"
#include "amixer.h"
#include "tempo.h"
#include "sig.h"
#include "audio.h"
#include "mididev.h"
#include "audiodev.h"
#include "alsamidi.h"
#include "audio.h"
#include "arranger.h"

//---------------------------------------------------------
//   sendMsg
//---------------------------------------------------------

void Audio::sendMsg(AudioMsg* m)
      {
      static int sno = 0;

      if (_running) {
            m->serialNo = sno++;
            //DEBUG:
            msg = m;
            // wait for next audio "process" call to finish operation
            int no = -1;
            int rv = read(fromThreadFdr, &no, sizeof(int));
            if (rv != sizeof(int))
                  perror("Audio: read pipe failed");
            else if (no != (sno-1)) {
                  fprintf(stderr, "audio: bad serial number, read %d expected %d\n",
                     no, sno-1);
                  }
            }
      else {
            // if audio is not running (during initialization)
            // process commands immediatly
            processMsg(m);
            }
      }

//---------------------------------------------------------
//   sendMessage
//    send request from gui to sequencer
//    wait until request is processed
//---------------------------------------------------------

bool Audio::sendMessage(AudioMsg* m, bool doUndo)
      {
      if (doUndo)
           song->startUndo();
      sendMsg(m);
      if (doUndo)
            song->endUndo(0);       // song->endMsgCmd();
      return false;
      }

//---------------------------------------------------------
//   msgRemoveRoute
//---------------------------------------------------------

void Audio::msgRemoveRoute(Route src, Route dst)
      {
      msgRemoveRoute1(src, dst);
      if (src.type == JACK_ROUTE)
            audioDevice->disconnect(src.jackPort, ((AudioInput*)dst.track)->jackPort(dst.channel));
      else if (dst.type == JACK_ROUTE)
            audioDevice->disconnect(((AudioOutput*)src.track)->jackPort(src.channel), dst.jackPort);
      }

//---------------------------------------------------------
//   msgRemoveRoute1
//---------------------------------------------------------

void Audio::msgRemoveRoute1(Route src, Route dst)
      {
      AudioMsg msg;
      msg.id     = AUDIO_ROUTEREMOVE;
      msg.sroute = src;
      msg.droute = dst;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgAddRoute
//---------------------------------------------------------

void Audio::msgAddRoute(Route src, Route dst)
      {
      if (src.type == JACK_ROUTE) {
            if (isRunning())
                  audioDevice->connect(src.jackPort, ((AudioInput*)dst.track)->jackPort(dst.channel));
            }
      else if (dst.type == JACK_ROUTE) {
            if (audio->isRunning())
                  audioDevice->connect(((AudioOutput*)src.track)->jackPort(dst.channel), dst.jackPort);
            }
      msgAddRoute1(src, dst);
      }

//---------------------------------------------------------
//   msgAddRoute1
//---------------------------------------------------------

void Audio::msgAddRoute1(Route src, Route dst)
      {
      AudioMsg msg;
      msg.id = AUDIO_ROUTEADD;
      msg.sroute = src;
      msg.droute = dst;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgAddPlugin
//---------------------------------------------------------

void Audio::msgAddPlugin(AudioTrack* node, int idx, PluginI* plugin)
      {
      AudioMsg msg;
      msg.id     = AUDIO_ADDPLUGIN;
      msg.snode  = node;
      msg.ival   = idx;
      msg.plugin = plugin;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgSetRecord
//---------------------------------------------------------

void Audio::msgSetRecord(AudioTrack* node, bool val)
      {
      AudioMsg msg;
      msg.id     = AUDIO_RECORD;
      msg.snode  = node;
      msg.ival   = int(val);
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgSetVolume
//---------------------------------------------------------

void Audio::msgSetVolume(AudioTrack* src, double val)
      {
      AudioMsg msg;
      msg.id    = AUDIO_VOL;
      msg.snode    = src;
      msg.dval  = val;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgSetPan
//---------------------------------------------------------

void Audio::msgSetPan(AudioTrack* node, double val)
      {
      AudioMsg msg;
      msg.id    = AUDIO_PAN;
      msg.snode = node;
      msg.dval  = val;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgSetPrefader
//---------------------------------------------------------

void Audio::msgSetPrefader(AudioTrack* node, int val)
      {
      AudioMsg msg;
      msg.id    = AUDIO_SET_PREFADER;
      msg.snode = node;
      msg.ival  = val;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgSetChannels
//---------------------------------------------------------

void Audio::msgSetChannels(AudioTrack* node, int n)
      {
      if (n == node->channels())
            return;
      QString name = node->name();
      int mc       = std::max(n, node->channels());

      if (!name.isEmpty()) {
            if (node->type() == Track::AUDIO_INPUT) {
                  AudioInput* ai = (AudioInput*)node;
                  for (int i = 0; i < mc; ++i) {
                        if (i < n && ai->jackPort(i) == 0) {
                              char buffer[128];
                              snprintf(buffer, 128, "%s-%d", name.latin1(), i);
                              ai->setJackPort(i, audioDevice->registerInPort(buffer));
                              }
                        else if ((i >= n) && ai->jackPort(i)) {
                              RouteList* ir = node->inRoutes();
                              for (iRoute ii = ir->begin(); ii != ir->end(); ++ii) {
                                    Route r = *ii;
                                    if ((r.type == JACK_ROUTE) && (r.channel == i)) {
                                          msgRemoveRoute(r, Route(node,i));
                                          break;
                                          }
                                    }
                              audioDevice->unregisterPort(ai->jackPort(i));
                              ai->setJackPort(i, 0);
                              }
                        }
                  }
            else if (node->type() == Track::AUDIO_OUTPUT) {
                  AudioOutput* ao = (AudioOutput*)node;
                  for (int i = 0; i < mc; ++i) {
                        void* jp = ao->jackPort(i);
                        if (i < n && jp == 0) {
                              char buffer[128];
                              snprintf(buffer, 128, "%s-%d", name.latin1(), i);
                              ao->setJackPort(i, audioDevice->registerOutPort(buffer));
                              }
                        else if (i >= n && jp) {
                              RouteList* ir = node->outRoutes();
                              for (iRoute ii = ir->begin(); ii != ir->end(); ++ii) {
                                    Route r = *ii;
                                    if ((r.type == JACK_ROUTE) && (r.channel == i)) {
                                          msgRemoveRoute(Route(node,i), r);
                                          break;
                                          }
                                    }
                              audioDevice->unregisterPort(jp);
                              ao->setJackPort(i, 0);
                              }
                        }
                  }
            }
      AudioMsg msg;
      msg.id    = AUDIO_SET_CHANNELS;
      msg.snode = node;
      msg.ival  = n;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgSetSegSize
//---------------------------------------------------------

void Audio::msgSetSegSize(int bs, int sr)
      {
      AudioMsg msg;
      msg.id = AUDIO_SET_SEG_SIZE;
      msg.ival = bs;
      msg.iival = sr;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgSeek
//---------------------------------------------------------

void Audio::msgSeek(const Pos& pos)
      {
      audioDevice->seekTransport(pos.frame());
      }

//---------------------------------------------------------
//   msgUndo
//---------------------------------------------------------

void Audio::msgUndo()
      {
      AudioMsg msg;
      msg.id = SEQM_UNDO;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgRedo
//---------------------------------------------------------

void Audio::msgRedo()
      {
      AudioMsg msg;
      msg.id = SEQM_REDO;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgPlay
//---------------------------------------------------------

void Audio::msgPlay(bool val)
      {
      if (val)
            audioDevice->startTransport();
      else
            audioDevice->stopTransport();
      }

//---------------------------------------------------------
//   msgShowInstrumentGui
//---------------------------------------------------------

void Audio::msgShowInstrumentGui(MidiInstrument* instr, bool val)
      {
      instr->showGui(val);
      AudioMsg msg;
      msg.id = MIDI_SHOW_INSTR_GUI;
      msg.p1 = instr;
      msg.a  = val;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   msgAddTrack
//---------------------------------------------------------

void Song::msgInsertTrack(Track* track, int idx, bool doUndoFlag)
      {
      AudioMsg msg;
      msg.id = SEQM_ADD_TRACK;
      msg.track = track;
      msg.ival  = idx;
      if (doUndoFlag) {
            song->startUndo();
            undoOp(UndoOp::AddTrack, idx, track);
            }
      audio->sendMsg(&msg);
      if (doUndoFlag)
            endUndo(SC_TRACK_INSERTED);
      }

//---------------------------------------------------------
//   msgRemoveTrack
//---------------------------------------------------------

void Audio::msgRemoveTrack(Track* track, bool doUndoFlag)
      {
      AudioMsg msg;
      msg.id = SEQM_REMOVE_TRACK;
      msg.track = track;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgRemoveTracks
//    remove all selected tracks
//---------------------------------------------------------

void Audio::msgRemoveTracks()
      {
      bool loop;
      do {
            loop = false;
            TrackList* tl = song->tracks();
            for (iTrack t = tl->begin(); t != tl->end(); ++t) {
                  if ((*t)->selected()) {
                        song->removeTrack1(*t);
                        msgRemoveTrack(*t, false);
                        song->removeTrack3(*t);
                        loop = true;
                        break;
                        }
                  }
            } while (loop);
      }

//---------------------------------------------------------
//   msgChangeTrack
//    oldTrack - copy of the original track befor modification
//    newTrack - modified original track
//---------------------------------------------------------

void Audio::msgChangeTrack(Track* oldTrack, Track* newTrack, bool doUndoFlag)
      {
      AudioMsg msg;
      msg.id = SEQM_CHANGE_TRACK;
      msg.p1 = oldTrack;
      msg.p2 = newTrack;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgMoveTrack
//    move track idx1 to slot idx2
//---------------------------------------------------------

void Audio::msgMoveTrack(int idx1, int idx2, bool doUndoFlag)
      {
      int n = song->tracks()->size();
      if (idx1 >= n || idx2 >= n)   // sanity check
            return;
      AudioMsg msg;
      msg.id = SEQM_MOVE_TRACK;
      msg.a = idx1;
      msg.b = idx2;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgAddPart
//---------------------------------------------------------

void Audio::msgAddPart(Part* part, bool doUndoFlag)
      {
      AudioMsg msg;
      msg.id = SEQM_ADD_PART;
      msg.p1 = part;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgRemovePart
//---------------------------------------------------------

void Audio::msgRemovePart(Part* part, bool doUndoFlag)
      {
      AudioMsg msg;
      msg.id = SEQM_REMOVE_PART;
      msg.p1 = part;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgRemoveParts
//    remove selected parts; return true if any part was
//    removed
//---------------------------------------------------------

bool Song::msgRemoveParts()
      {
      bool loop;
      bool partSelected = false;
      do {
            loop = false;
            TrackList* tl = song->tracks();

            for (iTrack it = tl->begin(); it != tl->end(); ++it) {
                  PartList* pl = (*it)->parts();
                  for (iPart ip = pl->begin(); ip != pl->end(); ++ip) {
                        if (ip->second->selected()) {
                              if ((*it)->type() == Track::WAVE) {
                                    audio->msgRemovePart((WavePart*)(ip->second));
                                    }
                              else {
                                    audio->msgRemovePart(ip->second, false);
                                    }
                              loop = true;
                              partSelected = true;
                              break;
                              }
                        }
                  if (loop)
                        break;
                  }
            } while (loop);
      return partSelected;
      }

//---------------------------------------------------------
//   msgChangePart
//---------------------------------------------------------

void Audio::msgChangePart(Part* oldPart, Part* newPart, bool doUndoFlag)
      {
      AudioMsg msg;
      msg.id = SEQM_CHANGE_PART;
      msg.p1 = oldPart;
      msg.p2 = newPart;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgAddEvent
//---------------------------------------------------------

void Audio::msgAddEvent(Event& event, Part* part, bool doUndoFlag)
      {
      AudioMsg msg;
      msg.id = SEQM_ADD_EVENT;
      msg.ev1 = event;
      msg.p2 = part;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgDeleteEvent
//---------------------------------------------------------

void Audio::msgDeleteEvent(Event& event, Part* part, bool doUndoFlag)
      {
      AudioMsg msg;
      msg.id = SEQM_REMOVE_EVENT;
      msg.ev1 = event;
      msg.p2 = part;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgChangeEvent
//---------------------------------------------------------

void Audio::msgChangeEvent(Event& oe, Event& ne, Part* part, bool doUndoFlag)
      {
      AudioMsg msg;
      msg.id = SEQM_CHANGE_EVENT;
      msg.ev1 = oe;
      msg.ev2 = ne;
      msg.p3 = part;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgAddTempo
//---------------------------------------------------------

void Audio::msgAddTempo(int tick, int tempo, bool doUndoFlag)
      {
      AudioMsg msg;
      msg.id = SEQM_ADD_TEMPO;
      msg.a = tick;
      msg.b = tempo;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgSetTempo
//---------------------------------------------------------

void Audio::msgSetTempo(int tick, int tempo, bool doUndoFlag)
      {
      AudioMsg msg;
      msg.id = SEQM_SET_TEMPO;
      msg.a = tick;
      msg.b = tempo;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgSetGlobalTempo
//---------------------------------------------------------

void Audio::msgSetGlobalTempo(int val)
      {
      AudioMsg msg;
      msg.id = SEQM_SET_GLOBAL_TEMPO;
      msg.a = val;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   msgDeleteTempo
//---------------------------------------------------------

void Audio::msgDeleteTempo(int tick, int tempo, bool doUndoFlag)
      {
      AudioMsg msg;
      msg.id = SEQM_REMOVE_TEMPO;
      msg.a = tick;
      msg.b = tempo;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgAddSig
//---------------------------------------------------------

void Audio::msgAddSig(int tick, int z, int n, bool doUndoFlag)
      {
      AudioMsg msg;
      msg.id = SEQM_ADD_SIG;
      msg.a = tick;
      msg.b = z;
      msg.c = n;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgRemoveSig
//! sends remove tempo signature message
//---------------------------------------------------------

void Audio::msgRemoveSig(int tick, int z, int n, bool doUndoFlag)
      {
      AudioMsg msg;
      msg.id = SEQM_REMOVE_SIG;
      msg.a = tick;
      msg.b = z;
      msg.c = n;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgScanAlsaMidiPorts
//---------------------------------------------------------

void Audio::msgScanAlsaMidiPorts()
      {
      AudioMsg msg;
      msg.id = SEQM_SCAN_ALSA_MIDI_PORTS;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   msgResetMidiDevices
//---------------------------------------------------------

void Audio::msgResetMidiDevices()
      {
      AudioMsg msg;
      msg.id = SEQM_RESET_DEVICES;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   msgInitMidiDevices
//---------------------------------------------------------

void Audio::msgInitMidiDevices()
      {
      AudioMsg msg;
      msg.id = SEQM_INIT_DEVICES;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   panic
//---------------------------------------------------------

void Audio::msgPanic()
      {
      AudioMsg msg;
      msg.id = SEQM_PANIC;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   localOff
//---------------------------------------------------------

void Audio::msgLocalOff()
      {
      AudioMsg msg;
      msg.id = SEQM_MIDI_LOCAL_OFF;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   msgSetAux
//---------------------------------------------------------

void Audio::msgSetAux(AudioTrack* track, int idx, double val)
      {
      AudioMsg msg;
      msg.id    = SEQM_SET_AUX;
      msg.snode = track;
      msg.ival  = idx;
      msg.dval  = val;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   msgPlayMidiEvent
//---------------------------------------------------------

void Audio::msgPlayMidiEvent(const MidiPlayEvent* event)
      {
      AudioMsg msg;
      msg.id = SEQM_PLAY_MIDI_EVENT;
      msg.p1 = event;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   msgBounce
//    start bounce operation
//---------------------------------------------------------

void Audio::msgBounce()
      {
      _bounce = true;
      audioDevice->seekTransport(song->lPos().frame());
      }

//---------------------------------------------------------
//   msgIdle
//---------------------------------------------------------

void Audio::msgIdle(bool on)
      {
      AudioMsg msg;
      msg.id = SEQM_IDLE;
      msg.a  = on;
      sendMessage(&msg, false);
      }

