# tageditor.py - a GTK+ dialog for editing tags
#
#   Copyright (C) 2003 Daniel Burrows <dburrows@debian.org>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import gtk
import tags

# Wraps a tag editor window.
class TagEditor:
    def __init__(self, file, toplevel):
        self.file=file
        self.toplevel=toplevel
        self.file.add_listener(self.file_updated)

        label=file.get_label()

        if label == None:
            title="Edit tags"
        else:
            title="Edit tags of %s"%label

        self.widget=gtk.Dialog(title,
                               None,
                               gtk.DIALOG_DESTROY_WITH_PARENT,
                               (gtk.STOCK_OK, 1,
                                gtk.STOCK_CANCEL, 0))

        alltags=tags.known_tags.keys()

        alltags.sort()

        table=gtk.Table(len(alltags), 2, False)
        self.widget.vbox.add(table)

        # tag_entries is a list of tuples (tag, entry, original_value)
        # which is used to test whether the value changed (in which
        # case the tag will be set)
        self.tag_entries=[]

        # build the table
        for row in range(0,len(alltags)):
            tag=tags.known_tags[alltags[row]]
            # FIXME: handle multiple values properly
            label=gtk.Label(tag.title)
            entry=gtk.Entry()
            entry_text=self.__get_entry_text(tag.tag)
            entry.set_text(entry_text)
            self.tag_entries.append((tag.tag,entry,entry_text))

            table.attach(label, 0, 1, row, row+1, 0, 0)
            table.attach(entry, 1, 2, row, row+1, gtk.EXPAND|gtk.FILL, 0)

        self.widget.connect('response', self.response)

    # common code from the constructor and file_updated -- get the
    # text to be placed in an entry for the file.
    def __get_entry_text(self, tag):
        current_value=self.file.get_tag(tag)
        if len(current_value)==0:
            current_value=['']

        return current_value[0]

    def show(self):
        self.widget.show_all()

    def file_updated(self, file, oldcomments):
        new_entries=[]
        for tag,entry,orig_val in self.tag_entries:
            new_text=self.__get_entry_text(tag)

            if entry.get_text()==orig_val:
                entry.set_text(new_text)

            new_entries.append((tag, entry, new_text))

        self.tag_entries=new_entries

    def response(self, widget, response):
        if response==1: # Ok was pressed
            # Validate everything up-front.  (does it make sense to
            # test this when a field is changed?  I find those
            # 'instant popups' annoying..)
            for tag,entry,orig_val in self.tag_entries:
                text=entry.get_text()
                if text <> '' and not tags.known_tags[tag].validate(self.file,
                                                                    text):
                    return

            modified=0
            # could this be more efficient?  eg, setting a bunch of
            # tags all at once.
            for tag,entry,orig_val in self.tag_entries:
                if entry.get_text() <> orig_val:
                    modified=1
                    # FIXME: this relies heavily on the behavior of
                    #        FileList, and the fact that we place
                    #        real files in it, not GUI file proxies.
                    #
                    # (ie: GUI file proxies take strings in set_tag, and
                    #  the underlying files take lists)
                    self.file.set_tag(tag, [entry.get_text()])

            if modified:
                self.toplevel.purge_empty()

        self.widget.destroy()
        self.file.remove_listener(self.file_updated)
