/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 gnudownload.cpp  -  Definition of a file download in the gnutella network

 the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)

    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "structures.h"

#include "asyncsocket.h"
#include "packet.h"
#include "common.h"
#include "conversions.h"
#include "event.h"

#include "gnudirector.h"
#include "gnusearch.h"
#include "gnudownload.h"

#include "asyncfile.h"
#include "messages.h"

#include "property.h"
#include "preferences.h"

#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>

#define MAX_DOWNLOADERS 16

class MDownFileMng;

MEvFileMoved::MEvFileMoved(const CString& sPath, int nFileSize) :
	TSimpleEvent<CString>(ET_MESSAGE, ES_UNIMPORTANT, sPath, MSG_DOWNLOAD_FILE_MOVED, MSGSRC_DOWNLOAD),
	m_nFileSize(nFileSize) {}

CString MEvFileMoved::Format()
{
	return "filename = '" + m_value + "'";
}

class MDownloadFile : public MAsyncFile {
public:
	MDownloadFile(MDownFileMng*);
	virtual void OnSuccessDelayed(DWORD dwSeqReqID, int nReqType, const SAFileState& FileState);
	virtual void OnErrorDelayed(DWORD dwSeqReqID, int nReqType, const SAFileState& FileState);
	virtual ~MDownloadFile();
private:
	MDownFileMng* m_pFileMng;
};

class MRequestMoveCompl : public MCustomAFRequest
{
public:
	MRequestMoveCompl(const CString& pathOld, const CString& pathNew) : m_pathOld(pathOld), m_pathNew(pathNew) {}
	virtual bool Do(MAFRequest* pRequest)
	{
		CString path = m_pathNew;
		// make sure we dont overwrite anything
		for (int n = 1; FileExists(path) && n<1000; n++)
		{
			CString sName = m_pathNew;
			CString sExt;
			int nDotPos = m_pathNew.rfind(".");
			if (nDotPos != -1)
			{
				sName = m_pathNew.substr(0,nDotPos);
				sExt = m_pathNew.substr(nDotPos);
			}
			path = sName + "." + DWrdtoStr(n) + sExt;
		}
		//
		pRequest->file_state.nReturn = rename(m_pathOld.c_str(), path.c_str());
		if(pRequest->file_state.nReturn)
		{
			POST_EVENT( MStringEvent(
				ET_ERROR,
				ES_IMPORTANT,
				CString("filename=`") + m_pathOld + "' error: " + strerror(errno),
				MSG_DOWNLOAD_FILE_MOVE_FAILED,
				MSGSRC_DOWNLOAD
			));
		}
		else
		{
			POST_EVENT( MEvFileMoved(
				path,
				pRequest->file_state.nSize
			));
		}
		return pRequest->file_state.nReturn == 0;
	}
	void Predict(SAFileState& state)
	{
		// this does not really change the file
		state.sPath = m_pathNew;
	}
private:
	CString m_pathOld;
	CString m_pathNew;
};

class MDownloadBuffer : public MAFBufferTrivial
{
public:
	MDownloadBuffer(int nBufferSize = 16384) : MAFBufferTrivial(nBufferSize) {
		m_nContains = 0;
		m_nOffset = 0;
	}
	// TODO: mutex-protect it all
	inline char* data(){return m_pBuffer + m_nOffset;}
	inline char* space(){ if (remains()>0) return m_pBuffer + m_nContains; return NULL; }
	inline int   contains(){return m_nContains;}
	inline int   size(){return m_nSize;}
	inline int   remains(){return m_nSize - m_nContains;}
	inline bool  increment(int nBy){ASSERT(nBy>=0); ASSERT(m_nContains>=0); m_nContains += nBy; ASSERT(m_nContains <= m_nSize); return true;}
	inline bool  isFull(){return m_nContains == m_nSize;}
	inline bool  isEmpty(){return m_nContains==0;}
	inline void  clear(){m_nContains = 0; m_nOffset = 0;}
	inline void  trim(int nTo){ if (m_nContains > nTo) m_nContains = nTo;}
	//
	inline bool  offset(int nBy) {ASSERT(nBy>=0); ASSERT(m_nOffset>=0); m_nOffset += nBy; ASSERT(m_nOffset <= m_nSize); return true;}
	virtual char* buffer(){return m_pBuffer + m_nOffset;}
	virtual int   volume(){return m_nSize - m_nOffset;}
protected:
	int    m_nContains;
	int    m_nOffset;
	//
	~MDownloadBuffer(){}
};

struct SStoreHelper
{
	MDownloader* pD;
	MDownloadBuffer* pB;
	int nRequestedPos;
	int nRequestedSize;
	int nStoredPos;
	int nStoredSize;
};

struct SRequestStatus{
	int  nRequestType;
	// general status
	bool bSuccess;
	int  nReturn;
	int  nErrNo;
	// read/write specific
	//int  nBufSize;
	int  nFilePos;
	int  nFileSize;
	//char * pBuffer;
};
	
/////////////////////////////////////////////////////////////////////////////////////////

enum DownManState{
	DMS_New,
	DMS_Opening,
	DMS_Opened,
	DMS_Closing,
	DMS_Closed,
	DMS_Error
};

/////////////////////////////////////////////////////////////////////////////////////////
// MDownFileMng is inherited from MAsyncSocket only to enable a trick with safe callbacks
class MDownFileMng
{
public:
	MDownFileMng(MGnuDownload* pOwner, MGnuPreferences* pPrefs);
	~MDownFileMng();
	void RequestOpen(const CString& sName, int nFileSize);
	void RequestClose();
	StoreResult StoreDataAt(MDownloader* pDownloader, int nPos, MDownloadBuffer* pBuff, int nSize);
	bool RequestRename(const CString& sNewName);
	bool RequestDelete();
	bool RequestStartPoint(MDownloader* pDownloader);
	void OnDisconnect(MDownloader* pDownloader);
	bool NeedMoreSockets(){return m_pActiveDownloader==NULL;}
	//
	int  GetSizeCompleted(){return m_nPartSize;}
	//
	const CString& GetName(){return m_sName;}
	const CString& GetPath(){return m_sPath;}
	const CString& GetPartPath(){return m_sPartPath;}
	//
	void OnFileEvent(const SRequestStatus& );
protected:
	MGnuDownload*    m_pDownload;
	MGnuPreferences* m_pPrefs;
	CString m_sName;
	CString m_sPath;
	CString m_sPartPath;
	int     m_nFileSize;
	// state
	int     m_nState;
	//
	int     m_nPartSize;
	int     m_nPartSizeQueued; // async write requests may be in the queue
	//
	MMutex m_mutex;
	//
	set< TSmartPtr<MDownloadBuffer> > m_BuffersInUse;
	MAFBuffer* m_pOverlapBuff;
	int    m_nOverlapPos;
	list<MDownloader*> m_listStartRequesters;
	map<int,SStoreHelper> m_mapPos;
	MDownloader* m_pActiveDownloader;
	// one file for now
	MDownloadFile* m_pFile;
	//
	void SetPaths(const CString& newName, int nFileSize);
};

////////////////////////////////////////////////////////////////////////
class MDownloader : public MAsyncSocket
{
public:
	MDownloader(MGnuDownload* pDownl, MGnuDirector* pDirector, const Result& result);
	~MDownloader();
	//
	void OnTimer(int nTimeStamp);
	bool OnIncomingPush(const CString& sFileName, int nIndex, SOCKET hPushSock);
	void ForceDisconnect(int nReason, int nStatus);
	int  GetStatus(){return m_nStatus;}
	const Result& GetResult(){return m_result;}
	const IP& GetHostIP(){return m_result.Host;}
	void UpdateResult(const Result& newerRes);
	DWORD GetRate(){return m_dwRate;}
	void CopyTo(SGnuDownloadItem& item);
	//
	int GetStatusChangeTime(){return m_nLastStatusChange;}
	//
	void Initiate(); // call Connect or request push
	void Enqueue(); // changes status to "TRANSFER_QUEUED"
	bool OnSetStartPoint(int nFilePos); // return true if start point was accepted
	void OnDataWritten(DWORD dwPos, DWORD dwSize, MDownloadBuffer* pB);
protected:
	Result m_result;
	//
	int m_nStatus;
	int m_nDisconnectReason;
	//
	int  m_nConErrors;
	int  m_nPushTimeouts;
	bool m_bServerIsUp;
	bool m_bForceUriRes;
	//
	int m_nFilePos;
	// timers, etc
	int m_nLastStatusChange;
	int m_nLastActive;
	int m_nSecBytesReceived; // bytes received in the last second
	int m_dwRate;
	int m_nSecsUnderLimit;
	// double buffering
	MDownloadBuffer* m_pReceiveBuff;
	MDownloadBuffer* m_pStoreBuff;
	//
	MGnuDownload* m_pDownload;
	MGnuDirector* m_pDirector;
	// virtual overrides
	virtual void OnConnect(int nErrorCode);
	virtual void OnReceive(int nErrorCode);
	virtual void OnClose(int nErrorCode);
	// service functions
	void SendRequest(); // called either from OnConnect or OnIncomingPush
	void SetStatus(int nNewStatus);
	void StoreStoreBuff();
};

/////////////////////////////////////////////////////////////////////////////////////////////////////////////

MDownloadFile::MDownloadFile(MDownFileMng* pMng) :
	MAsyncFile(AFM_READWRITE|AFM_CREATEPATH),
	m_pFileMng(pMng)
{
}

MDownloadFile::~MDownloadFile()
{
}

void MDownloadFile::OnSuccessDelayed(DWORD dwSeqReqID, int nReqType, const SAFileState& FileState)
{
	//TRACE2("MDownloadFile::OnSuccess; request=", m_pRequest->type_compl);
	SRequestStatus rs;
	rs.bSuccess     = true;
	rs.nRequestType = nReqType;
	rs.nReturn      = FileState.nReturn;
	rs.nErrNo       = FileState.nErrNo;
	rs.nFilePos     = FileState.nPos;
	rs.nFileSize    = FileState.nSize;
	//
	//TRACE("OnSuccess: sending structure...");
	m_pFileMng->OnFileEvent(rs);
	//TRACE("OnSuccess: structure sent...");
}

void MDownloadFile::OnErrorDelayed(DWORD dwSeqReqID, int nReqType, const SAFileState& FileState)
{
	//TRACE2("MDownloadFile::OnError; request=", m_pRequest->type_compl);
	SRequestStatus rs;
	rs.bSuccess     = false;
	rs.nRequestType = nReqType;
	rs.nReturn      = FileState.nReturn;
	rs.nErrNo       = FileState.nErrNo;
	rs.nFilePos     = FileState.nPos;
	rs.nFileSize    = FileState.nSize;
	//
	//TRACE("OnError: sending structure...");
	m_pFileMng->OnFileEvent(rs);
	//TRACE("OnError: structure sent...");
}


///////////////////////////////////////////////////////////////////
MDownFileMng::MDownFileMng(MGnuDownload* pOwner, MGnuPreferences* pPrefs)
{
	m_pDownload = pOwner;
	m_pPrefs = pPrefs;
	//
	m_pFile = new MDownloadFile(this);
	ASSERT(m_pFile);
	m_nFileSize = 0;
	m_nPartSize = 0;
	m_nPartSizeQueued = 0;
	m_nOverlapPos = 0;
	m_pActiveDownloader = NULL;
	m_pOverlapBuff = NULL;
	//
	m_nState = DMS_New;
}

MDownFileMng::~MDownFileMng()
{
	if (m_pFile)
		delete m_pFile;
	if (m_pOverlapBuff)
		m_pOverlapBuff->Release();
}

void MDownFileMng::OnFileEvent(const SRequestStatus& rs)
{
	// update status accordingly
	// call apropriate MGnuDownload functions
	// release the buffer -- remove it from the set
	if (!rs.bSuccess)
	{
		m_nState = DMS_Error;
		m_pDownload->OnFileError(rs.nErrNo);
	}
	else
	{
		switch (rs.nRequestType)
		{
			case AFR_OPEN:
				// so file is open -- we are ready to operate
				m_nPartSize = rs.nFileSize;
				m_pDownload->OnFileManReady();
				// check if we possibly completed the file
				if (rs.nFileSize==m_nFileSize)
				{
					// request file move and so on
					MRequestMoveCompl* pMove = new MRequestMoveCompl(m_sPartPath, m_sPath);
					ASSERT(pMove);
					m_pFile->CustomRequest(pMove);
					pMove->Release();
					// call OnFileCompleted
					#warning may be we have to call OnFileCompleted() after the move?
					m_pDownload->OnFileCompleted();
				}
			break;
			case AFR_WRITE|AFR_SEEK:
			case AFR_WRITE:
				{
					map<int,SStoreHelper>::iterator itm = m_mapPos.find(rs.nFilePos-rs.nReturn);
					if (itm==m_mapPos.end())
					{
						TRACE("MDownFileMng::OnFileEvent(): Failed to find request");
						m_pDownload->OnFileError(0);
						itm->second.pB->Release();
						return;
					}
					// this basically changes wherer the whole of the buffer was saved
					// the first case can legaly happen when storing the rests of the buffer
					// the second case will fail for the first store request for the partial file
					if (itm->second.nStoredSize != rs.nReturn)
					{
						TRACE("MDownFileMng::OnFileEvent() detected incomplete save:\n  " <<
						      itm->second.nStoredSize        << " !=  " << rs.nReturn << "\n  " <<
						      "rs.nFilePos       = " << rs.nFilePos << "\n  " <<
						      "m_nPartSizeQueued = " << m_nPartSizeQueued  << "\n  " <<
						      "m_nPartSize       = " << m_nPartSize );
						m_pDownload->OnFileError(0);
						itm->second.pB->Release();
						return;
					}
					//TRACE4("AFR_WRITE: Old size was ", m_nPartSize, ", new is ", rs.nFileSize);
					m_nPartSize = rs.nFileSize;
					// file write invalidates the overlap buffer
					if (m_pOverlapBuff)
					{
						m_pOverlapBuff->Release();
						m_pOverlapBuff = NULL;
						m_nOverlapPos = 0;
					}
					//
					m_pDownload->OnDataWritten(itm->second.pD, itm->second.nRequestedPos, itm->second.nRequestedSize, itm->second.pB);
					itm->second.pB->Release();
					m_mapPos.erase(itm);
					// check if we possibly completed the file
					if (rs.nFileSize>=m_nFileSize)
					{
						if (rs.nFileSize>m_nFileSize)
						{
							TRACE("rs.nFileSize>m_nFileSize! this should never happen!");
						}
						// request file move and so on
						MRequestMoveCompl* pMove = new MRequestMoveCompl(m_sPartPath, m_sPath);
						ASSERT(pMove);
						m_pFile->CustomRequest(pMove);
						pMove->Release();
						// call OnFileCompleted
						#warning may be we have to call OnFileCompleted() after the move?
						m_pDownload->OnFileCompleted();
					}
				}
			break;
			case AFR_READ:
			case AFR_READ|AFR_SEEK:
				// overlap buff is ready
				while (m_listStartRequesters.size())
				{
					m_pDownload->OnSetStartPoint(m_listStartRequesters.front(), m_nOverlapPos);
					m_listStartRequesters.pop_front();
				}
			break;
			case AFR_CLOSE:
				m_pDownload->OnFileManClosed();
			break;
			case AFR_NONE:
				TRACE("MDownFileMng::OnFileEvent(): AFR_NONE - should not happen");
			break;
		}
	}
}

void MDownFileMng::SetPaths(const CString& sName, int nFileSize)
{
	m_pDownload->GetDirector()->BuildDownloadPaths(sName,nFileSize,&m_sName,&m_sPath,&m_sPartPath);
	m_nFileSize = nFileSize;
}

void MDownFileMng::RequestOpen(const CString& sName, int nFileSize)
{
	// assert(IsOpen || IsOpening)
	ASSERT(m_nState == DMS_New || m_nState == DMS_Closed);
	// generate partial file name etc
	SetPaths(sName, nFileSize);
	// open file
	m_nState == DMS_Opening;
	//
	VERIFY(m_pFile->Open(m_sPartPath.c_str())); // non-blocking call should never fail
}

void MDownFileMng::RequestClose()
{
	// assert(IsClosed)
	// Close file
	m_nState == DMS_Closing;
	//
	if (m_pFile->IsOpen())
		VERIFY(m_pFile->Close());
	else
		m_pDownload->OnFileManClosed();
}

bool MDownFileMng::RequestRename(const CString& sNewName)
{
	if (m_pFile && m_sPartPath.length())
	{
		TRACE("Renaming partial file");
		CString sOldPartPath = m_sPartPath;
		SetPaths(sNewName, m_nFileSize);
		MRequestMove* pMove = new MRequestMove(sOldPartPath, m_sPartPath);
		ASSERT(pMove);
		m_pFile->CustomRequest(pMove);
		pMove->Release();
		return true;
	}
	return false;
}

bool MDownFileMng::RequestDelete()
{
	if (m_pFile && m_sPartPath.length())
	{
		MRequestDelete* pDel = new MRequestDelete(m_sPartPath);
		ASSERT(pDel);
		m_pFile->CustomRequest(pDel);
		pDel->Release();
		return true;
	}
	return false;
}

StoreResult MDownFileMng::StoreDataAt(MDownloader* pDownloader, int nPos0, MDownloadBuffer* pBuff, int nSize0)
{
	if (m_pActiveDownloader &&
		m_pActiveDownloader != pDownloader)
		return SR_Stop; // the place is taken!
	// assert(IsOpen)
	// assert(!BufInSet)
	int nPos =  nPos0;
	int nSize = nSize0;
	char * pData = pBuff->data();
	// first check against the overlap buffer
	if (m_nOverlapPos && m_nOverlapPos<=nPos && m_nOverlapPos+4096>nPos)
	{
		// we have overlap with overlap buffer
		int nOvOffs = max(nPos, m_nOverlapPos) - m_nOverlapPos;
		int nOvSize = min(nPos+nSize,m_nOverlapPos+4096) - m_nOverlapPos;
		if (0!=memcmp(m_pOverlapBuff->buffer()+nOvOffs, pData, nOvSize))
			return SR_NoMatch;
		if (nPos+nSize<=m_nOverlapPos+4096)
		{
			// we have to "simulate" callback
			m_pDownload->OnDataWritten(pDownloader, nPos0, nSize0, pBuff);
			return SR_Success;
		}
		int nBuffOfs = nPos - m_nOverlapPos + 4096;
		nPos += nBuffOfs;
		nSize -= nBuffOfs;
		pData += nBuffOfs;
		ASSERT(nSize>0);
		pBuff->offset(nBuffOfs);
		// passed successfully overlap check -- set 'continuality' checking
		m_nPartSizeQueued = nPos;
	}
	// add to the map if there is no such a pos already
	if (m_mapPos.find(nPos)!=m_mapPos.end())
	{
		TRACE("MDownFileMng::StoreDataAt(): repeated write request!");
		return SR_Error; // TODO: log this!
	}
	if (nPos>m_nPartSizeQueued)
	{
		TRACE("MDownFileMng::StoreDataAt(): tried to create gaps in the file!");
		return SR_Error; // TODO: log this!
	}
	#warning REMOVE THIS
	if (nPos!=m_nPartSizeQueued &&
		(m_nPartSizeQueued != m_nPartSize || m_nPartSize > 4096) )
	{
		TRACE4("MDownFileMng::StoreDataAt(): Warning -- nPos!=m_nPartSizeQueued ",nPos,"!=",m_nPartSizeQueued);
		// we'll deny storage if overlap buffer is gone now
		if (m_nOverlapPos==0 || m_nOverlapPos>nPos || m_nOverlapPos+4096<=nPos)
		{
			TRACE("MDownFileMng::StoreDataAt(): the other save came too late, we cannot verify it");
			return SR_Stop;
		}
		// we will also not save if the queed size is bigger than it would be
		if (m_nPartSizeQueued >= nPos0+nSize0)
		{
			TRACE("MDownFileMng::StoreDataAt(): the other save is too small");
			return SR_Stop;
		}
	}
	// now we have somebody passed the verify buffer
	if (m_pActiveDownloader==NULL)
		m_pActiveDownloader = pDownloader; // this downloader will take over for now, others go home
	SStoreHelper sh;
	sh.pD = pDownloader;
	sh.pB = pBuff;
	sh.nRequestedPos = nPos0;
	sh.nRequestedSize = nSize0;
	sh.nStoredPos = nPos;
	sh.nStoredSize = nSize;
	pBuff->AddRef();
	m_mapPos[nPos] = sh;
	// call async write
	if (m_pFile->WriteSeek(nPos, SEEK_SET, pBuff, nSize)>=0)
	{
		m_nPartSizeQueued = nPos0+nSize0;
		return SR_Success;
	}
	// WriteSeek failed
	TRACE("MDownFileMng::StoreDataAt(): WriteSeek() failed!");
	//m_pDownload->OnFileError(0);
	pBuff->Release();
	m_mapPos.erase(nPos); //map<int,SStoreHelper>::iterator itm = m_mapPos.find(rs.nFilePos-rs.nBufSize);
	return SR_Error;
}

void MDownFileMng::OnDisconnect(MDownloader* pDownloader)
{
	if (m_pActiveDownloader &&
		m_pActiveDownloader == pDownloader)
			m_pActiveDownloader = NULL; // active downloader was disconnected
}

bool MDownFileMng::RequestStartPoint(MDownloader* pDownloader)
{
	// in the current incarnation just read the verify buffer
	// if its not read already
	if (m_nOverlapPos)
	{
		ASSERT(m_pOverlapBuff);
		m_pDownload->OnSetStartPoint(pDownloader, m_nOverlapPos);
		return true;
	}
	if (m_nPartSize <= 16384)
	{
		m_pDownload->OnSetStartPoint(pDownloader, 0);
		return true;
	}
	//
	m_listStartRequesters.push_back(pDownloader);
	m_pOverlapBuff =  new MAFBufferTrivial(4096);
	ASSERT(m_pOverlapBuff);
	m_nOverlapPos = m_nPartSize - 4096;
	m_pFile->ReadSeek(m_nOverlapPos, SEEK_SET, m_pOverlapBuff, 4096);
	return true;
}

///////////////////////////////////////////////////////////////////////////
MDownloader::MDownloader(MGnuDownload* pDownl, MGnuDirector* pDirector, const Result& result) : m_result(result)
{
	m_pDownload = pDownl;
	m_pDirector = pDirector;
	//
	m_nStatus = TRANSFER_NEW;
	m_nDisconnectReason = REASON_UNDEFINED;
	m_nLastStatusChange = xtime();
	m_nLastActive = m_nLastStatusChange;
	m_nConErrors = 0;
	m_nPushTimeouts = 0;
	m_nFilePos = 0;
	m_nSecBytesReceived = 0;
	m_dwRate = 0;
	m_nSecsUnderLimit = 0;
	m_bServerIsUp = false;
	m_bForceUriRes = false;
	// double buffering
	m_pReceiveBuff = NULL;
	m_pStoreBuff = NULL;
}

MDownloader::~MDownloader()
{
	if (m_pReceiveBuff)
		m_pReceiveBuff->Release();
	if (m_pStoreBuff)
		m_pStoreBuff->Release();
}

void MDownloader::OnTimer(int nTimeStamp)
{
	MGnuPreferences* pPrefs = m_pDirector->GetPrefs();
	int nSecInactive = nTimeStamp - m_nLastActive;
	if (nSecInactive < 0)
		nSecInactive = 0;
	int nSecStable = nTimeStamp - m_nLastStatusChange;
	if (nSecStable < 0)
		nSecStable = 0;
	// Bytes
	switch (m_nStatus)
	{
		case TRANSFER_QUEUED: //check if its time for us to start
			{
				bool Ready = true;
				if (pPrefs->m_nMaxDownloads >0 && m_pDirector->CountActiveDownloads() >= pPrefs->m_nMaxDownloads ||
					pPrefs->m_nMaxPerHostDownloads >0 && m_pDirector->CountActiveDownloads(HOST(m_result.Host, m_result.Port), NULL) >= pPrefs->m_nMaxPerHostDownloads)
					Ready = false;
				if(Ready)
				{
					Initiate();
				}
			}
			break;
		case TRANSFER_CONNECTING:
			// check for timeouts
			if(nSecInactive > pPrefs->m_dwConnectTimeout)
			{
				ForceDisconnect(REASON_NO_RESPONSE, TRANSFER_PUSH);
				if (!m_pDirector->Route_LocalPush(m_result));
				{
					ForceDisconnect(REASON_NO_RESPONSE, m_nPushTimeouts>=20 ? TRANSFER_CLOSED : TRANSFER_FAILED); // there wont be a response
				}
			}
			break;
		case TRANSFER_PUSH: // waiting for a push -- check for timeout
			if(nSecInactive > pPrefs->m_dwPushTimeout)
			{
				ForceDisconnect(REASON_NO_RESPONSE, m_nPushTimeouts>=20 ? TRANSFER_CLOSED : TRANSFER_FAILED);
			}
			break;
		case TRANSFER_RECEIVING: // check for min speed limit
			if(pPrefs->m_dMinDownSpeed > 0 && nSecStable>pPrefs->m_dwSpeedTimeout)
			{
				// Check if its under the bandwidth limit
				if(m_dwRate < pPrefs->m_dMinDownSpeed &&
				   m_pDownload->GetNumAlternatives() > 1)
					m_nSecsUnderLimit++; // only do it if we have alternate locations
				else
					m_nSecsUnderLimit = 0;
				if(m_nSecsUnderLimit > 30)
				{
					ForceDisconnect(REASON_BELOW_MINIMUM_SPEED, TRANSFER_FAILED);
				}
			}
			// continue here
		case TRANSFER_NEGOTIATING: // Check for dead transfer
		case TRANSFER_CONNECTED:
			if(nSecInactive > pPrefs->m_dwTransferTimeout)
			{
				ForceDisconnect(REASON_NO_RESPONSE, TRANSFER_FAILED); // negotiation should not take long!
			}
			break;
		case TRANSFER_COOLDOWN:
			break;
		case TRANSFER_NEW:
			break;
		case TRANSFER_FAILED:
			break;
		case TRANSFER_CLOSED:
			break;
		default:
			TRACE2("TRANSFER_STATUS = ", m_nStatus);
	}
	// calculate rate
	m_dwRate = (m_dwRate*15 + m_nSecBytesReceived)>>4;
	m_nSecBytesReceived = 0;
}

void MDownloader::ForceDisconnect(int nReason, int nStatus)
{
	#warning Update Error Counters here! (and remove it from the other places)
	// update error counters
	switch (m_nStatus){
		case TRANSFER_CONNECTING:
			m_nConErrors++;
			break;
		case TRANSFER_PUSH:
			m_nPushTimeouts++;
			break;
	}

	// this will delay retrying on the ip-by-ip basis
	// IP is not the best ID for pushes, but still should work somehow...
	m_pDirector->OnDownloadHostBusy(GetHostIP(), xtime() + m_pDirector->GetPrefs()->m_nRetryWait);
	
	// lets check if we have not-yet-stored data in the receive buffer
	if (TRANSFER_RECEIVING == m_nStatus &&
		m_pReceiveBuff &&
		m_pReceiveBuff->contains())
	{
		ASSERT(m_pStoreBuff);
		if (REASON_STORAGE_DENIED != nReason &&
			REASON_WRITE_ERROR != nReason)
		{
			// we have to be a bit clever to protect ourseves
			// against repeated saves

			// TODO: do it more properly!
			if ( SR_AsyncError == m_pDownload->StoreDataAt(this, m_nFilePos+m_pStoreBuff->contains(), m_pReceiveBuff, m_pReceiveBuff->contains()) )
			{
				// this error may occur when the previous store rquest is not yet processed by the disk-write thread
				// the only way to fix thi without rewriting the whole of MAsyncFile is to sleep for a second and try again
				struct timespec rqtp, rmtp;
				rqtp.tv_sec = 1;
				rqtp.tv_nsec = rmtp.tv_sec = rmtp.tv_nsec = 0;
				safe_nanosleep(&rqtp, &rmtp);
				if (SR_Success != m_pDownload->StoreDataAt(this, m_nFilePos+m_pStoreBuff->contains(), m_pReceiveBuff, m_pReceiveBuff->contains()) )
				{
					POST_EVENT( MStringEvent(
						ET_ERROR,
						ES_IMPORTANT,
						"file:`" + m_result.Name + "' host: " + Ip2Str(m_result.Host),
						MSG_DOWNLOAD_STORE_WHILE_CLOSE_FAILED,
						MSGSRC_DOWNLOAD
					));
					if (nStatus == TRANSFER_COMPLETED)
						nStatus = TRANSFER_FAILED;
				}
			}
			
			m_pReceiveBuff->Release();
			m_pReceiveBuff = NULL;
		}
	}
	// release buffers to ensure data integrity
	if (m_pReceiveBuff)
	{
		m_pReceiveBuff->Release();
		m_pReceiveBuff = NULL;
	}
	if (m_pStoreBuff)
	{
		m_pStoreBuff->Release();
		m_pStoreBuff = NULL;
	}
	// notify MGnuDownload that there will be no more data from this Downloader
	if (TRANSFER_RECEIVING == m_nStatus)
		m_pDownload->OnDisconnect(this);
	//
	m_nDisconnectReason = nReason;
	SetStatus(nStatus);
	//
	Close(); // what if 'OnClose' will be called?
}

void MDownloader::Initiate()
{
	// call Connect or request push
	// Attempt to connect
	if(m_hSocket == INVALID_SOCKET)
		if(!Create(0, SOCK_STREAM, FD_READ|FD_CONNECT|FD_CLOSE))
		{
			POST_ERROR(ES_IMPORTANT, CString("Download Socket Create Error: ") + strerror(GetLastError()));
			SetStatus(TRANSFER_CLOSED);
			return;
		}
	SetStatus(TRANSFER_CONNECTING);
	SetLastError(0);
	bool bOkForConnect = m_nConErrors<3 && m_pDirector->IsOkForDirectConnect(m_result.Host);
	if(!bOkForConnect || !Connect(Ip2Str(m_result.Host).c_str(), m_result.Port))
	{
		if(!bOkForConnect || (GetLastError() != EWOULDBLOCK && GetLastError() != EINPROGRESS && GetLastError() != EAGAIN) )
		{
			SetStatus(TRANSFER_PUSH);
			if (!m_pDirector->Route_LocalPush(m_result))
			{
				// there wont be a response
				ForceDisconnect(REASON_NO_RESPONSE, m_nPushTimeouts>=20 ? TRANSFER_CLOSED : TRANSFER_FAILED);
			}
			Close();
		}
	}
}

void MDownloader::Enqueue()
{
	ASSERT(m_hSocket == INVALID_SOCKET); // not connected
	SetStatus(TRANSFER_QUEUED);
}

void MDownloader::UpdateResult(const Result& newerRes)
{
	if (m_result.ChangeTime != newerRes.ChangeTime)
	{
		m_result.ChangeTime = newerRes.ChangeTime;
		// the result was found again -- give it another chance
		m_nPushTimeouts = 0;
	}
	m_result.PushID   = newerRes.PushID;
	m_result.OriginID = newerRes.OriginID;
}

bool MDownloader::OnIncomingPush(const CString& sFileName, int nIndex, SOCKET hPushSock)
{
	if (TRANSFER_PUSH != m_nStatus)
		return false;
	if (nIndex != m_result.FileIndex)
		return false;
	if (sFileName != m_result.NameLower) // make more advanced checks to increase success rate
	                                     // like remove path or whatever...
		return false;
	// we have right incoming push!
	ASSERT(m_hSocket==INVALID_SOCKET);
	Attach(hPushSock, FD_READ|FD_CLOSE);
	// do similar things as in OnConnect();
	SetStatus(TRANSFER_CONNECTED);
	m_nSecBytesReceived = 0;
	m_dwRate = 0;
	m_nSecsUnderLimit = 0;
	ModifySelectFlags(FD_READ, 0);
	if (m_pDownload->RequestStartPoint(this))
		return true;
	ForceDisconnect(REASON_INTERNAL, TRANSFER_COOLDOWN);
	return true; // push request was processed, we just dont need it
}

void MDownloader::OnConnect(int nErrorCode)
{
	// we have established a connection somehow -- now proceed with
	// requesting start point for the download
	SetStatus(TRANSFER_CONNECTED);
	m_nSecBytesReceived = 0;
	m_dwRate = 0;
	m_nSecsUnderLimit = 0;
	ModifySelectFlags(FD_READ, 0);
	if (m_pDownload->RequestStartPoint(this))
		return;
	ForceDisconnect(REASON_INTERNAL, TRANSFER_COOLDOWN);
}

bool MDownloader::OnSetStartPoint(int nFilePos)
{
	// return true if start point was accepted
	if (TRANSFER_CONNECTED == m_nStatus)
	{
		m_nFilePos = nFilePos;
		SendRequest();
		return true;
	}
	return false;
}

void MDownloader::OnDataWritten(DWORD dwPos, DWORD dwSize, MDownloadBuffer* pB)
{
	if (TRANSFER_RECEIVING!=m_nStatus)
		return;
	// sanity check -- the file should be continuous
	if ( dwPos  != m_nFilePos )
	{
		TRACE("MDownloader::OnDataWritten() received wrong parameters");
		ForceDisconnect(REASON_WRITE_ERROR, TRANSFER_CLOSED);
		return;
	}
	m_nFilePos += dwSize;	
	// if we are storing something else than m_pStoreBuff => it is called from ForceDisconnect();
	if ( m_pStoreBuff == NULL ||
		 m_pStoreBuff != pB   )
	{
		// TODO: add pBuff to parameters to know for sure
		TRACE("MDownloader::OnDataWritten() stored something else than m_pStoreBuff -- must be disconnecting");
		return;
	}
	// all seems to be OK
	m_pStoreBuff->clear();
	
	if (m_pReceiveBuff == NULL || !m_pReceiveBuff->isFull())
		return;
	TRACE("Enabling OnReceive");
	ModifySelectFlags(FD_READ, 0);
	swap(m_pReceiveBuff, m_pStoreBuff);
	StoreStoreBuff();
}

void MDownloader::StoreStoreBuff()
{
	StoreResult sr = m_pDownload->StoreDataAt(this, m_nFilePos, m_pStoreBuff, m_pStoreBuff->contains());
	if (SR_Success != sr)
	{
		switch (sr){
			case SR_DiskError:
				ForceDisconnect(REASON_WRITE_ERROR, TRANSFER_CLOSED);
				POST_EVENT( MStringEvent(
					ET_ERROR,
					ES_IMPORTANT,
					"file:`" + m_result.Name,
					MSG_DOWNLOAD_FILE_ERROR,
					MSGSRC_DOWNLOAD
				));
				break;
			case SR_NoMatch:
				ForceDisconnect(REASON_FILES_ARE_DIFFERENT, TRANSFER_CLOSED);
				POST_EVENT( MStringEvent(
					ET_ERROR,
					ES_JUSTINCASE,
					"file:`" + m_result.Name + "' host: " + Ip2Str(m_result.Host),
					MSG_DOWNLOAD_FILE_MISSMATCH,
					MSGSRC_DOWNLOAD
				));
				break;
			case SR_Stop:
				ForceDisconnect(REASON_INTERNAL, TRANSFER_FAILED);
				break;
			case SR_Error:
			default:
				ForceDisconnect(REASON_STORAGE_DENIED, TRANSFER_CLOSED);
				POST_EVENT( MStringEvent(
					ET_ERROR,
					ES_IMPORTANT,
					"file:`" + m_result.Name,
					MSG_DOWNLOAD_STORAGE_DENIED,
					MSGSRC_DOWNLOAD
				));
				break;
		}
	}
}

void MDownloader::OnReceive(int nErrorCode)
{
	if (m_pReceiveBuff==NULL)
		m_pReceiveBuff = new MDownloadBuffer();
	if (m_pStoreBuff==NULL)
		m_pStoreBuff = new MDownloadBuffer();
	ASSERT(m_pReceiveBuff);
	ASSERT(m_pStoreBuff);
	//
	m_nLastActive = xtime();
	
	int nReceived = Receive(m_pReceiveBuff->space(), m_pReceiveBuff->remains());
    //printf("recieved %d bytes, errno = %d\n", BuffLength, errno);
	// Check for errors
	switch (nReceived)
	{
	case 0:
		//ASSERT(m_mutex.locked());
		ForceDisconnect(REASON_NO_DATA, TRANSFER_FAILED);
		return;
	case SOCKET_ERROR:
		//ASSERT(m_mutex.locked());
		ForceDisconnect(REASON_SOCKET_ERROR, TRANSFER_FAILED);
		return;
	}
	m_pReceiveBuff->increment(nReceived);
	m_nSecBytesReceived += nReceived;
	
	// Connected and downloading
	if(TRANSFER_RECEIVING == m_nStatus)
	{
		// do something about the buffers and if the receive buffer is full
		// and store buffer is empty -- store current receive buffer and swap those
		// otherwise, if the receive buffer is full -- just block further receives
		
		// check for compltetion
		if (m_result.Size <= m_pReceiveBuff->contains() + m_pStoreBuff->contains() + m_nFilePos)
		{
			if (m_result.Size < m_pReceiveBuff->contains() + m_pStoreBuff->contains() + m_nFilePos)
			{
				// shall we trim the file to the declared size?
				TRACE3("The remote peer sends us more data than we asked for... ", m_pReceiveBuff->contains() + m_pStoreBuff->contains() + m_nFilePos - m_result.Size, " additional bytes were received");
				m_pReceiveBuff->trim(m_result.Size - (m_pStoreBuff->contains() + m_nFilePos) );
			}
			ForceDisconnect(REASON_COMPLETED, TRANSFER_COMPLETED);
			return;
		}
		if (!m_pReceiveBuff->isFull())
			return;
		// if (!m_pStoreBuff->isEmpty()) is no longer a good indicator for the "uncompleted"
		// store requests because the buffer state is now reset by the store thread
		//if (m_pStoreBuff->GetRefs()>1) // this is better IMHO
		if (!m_pStoreBuff->isEmpty())
		{
			// block OnReceive
			TRACE("Disabling OnReceive");
			ModifySelectFlags(0, FD_READ);
			return;
		}
		swap(m_pReceiveBuff, m_pStoreBuff);
		StoreStoreBuff();
		return;
	}
	// Server sending us download header
	else if(TRANSFER_NEGOTIATING == m_nStatus)
	{
		// parse HTTP header
		// look for "\r\n\r\n"
		int nEndPos = -1;
		for(int i = 0; i <= m_pReceiveBuff->contains()-4; ++i)  // FIXED! it could be that we run out of buffer here
		{
			if( memcmp(m_pReceiveBuff->data()+i, "\r\n\r\n", 4) == 0)
			{
				nEndPos = i + 4; // FIXED! because of that
				break;
			}
		}
		if (nEndPos<0)
		{
			if (m_pReceiveBuff->isFull())
			{
				// looks like there was no header
				ForceDisconnect(REASON_BAD_HEADER, TRANSFER_FAILED);
			}
			// else give it a chance to collect more data
			return;
		}
		CString sHeader(m_pReceiveBuff->data(), nEndPos);
		// make it lower case
		sHeader.make_lower();
		// parse the header
		list<CString> lines;
		if (split_str(sHeader, '\n', lines)<=0) // this will leave '\r'-s at the end, but who would care?
		{
			ForceDisconnect(REASON_BAD_HEADER, TRANSFER_FAILED);
			return;
		}
		// first line should start with HTTP
		CString sLine = lines.front();
		int nPos;
		if ((nPos=sLine.find("http"))<0)
		{
			ForceDisconnect(REASON_BAD_HEADER, TRANSFER_FAILED);
			return;
		}
		// get the code
		CString sCode = sLine.substr(sLine.find(' ', nPos+4)+1);
		int nCode = atol(sCode.c_str());
		// check codes
		// Success code
		if(200 <= nCode && nCode < 300)
		{
			DWORD dwServerSize = 0;
			DWORD dwServerCompleted = 0;
			DWORD dwSmallSize = 0;
			int   nContLengthPos = sHeader.find("\r\ncontent-length:");
			int   nContRangePos  = sHeader.find("\r\ncontent-range:");
			// content-length
			if(nContLengthPos >= 0 && nContRangePos == -1)
			{
				CString sLen = sHeader.substr(nContLengthPos+strlen("content-length:")+2);
				dwServerSize = atol(sLen.c_str());
				if(m_nFilePos != 0)
				{
					ForceDisconnect(REASON_NO_RESUME_SUPPORT, TRANSFER_CLOSED);
				}
				else if(dwServerSize == m_result.Size)
				{
					// copy the rest of the buffer into the second buffer
					// and clear the first one
					ASSERT(m_pStoreBuff->isEmpty());
					ASSERT(m_pReceiveBuff->contains()>=nEndPos);
					memcpy(m_pStoreBuff->data(),m_pReceiveBuff->data()+nEndPos, m_pReceiveBuff->contains()-nEndPos);
					m_pStoreBuff->increment(m_pReceiveBuff->contains()-nEndPos);
					m_pReceiveBuff->clear();
					swap(m_pStoreBuff, m_pReceiveBuff);
					// and here we go!
					// its also good to reset the "LastError"
					m_nDisconnectReason = REASON_EMPTY;
					SetStatus(TRANSFER_RECEIVING);
				}
				else
				{
					ForceDisconnect(REASON_WRONG_FILE_SIZE, TRANSFER_CLOSED);
				}
			}
			// content-range
			else if(nContRangePos >= 0)
			{
				CString sHelp = sHeader.substr(nContRangePos+strlen("content-range:")+2);
				sHelp = sHelp.substr(0, sHelp.find("\r\n"));
				CString sRange = sHelp.substr(sHelp.find("bytes")+6);
				dwServerCompleted = atol(sRange.c_str());
				sHelp = sRange.substr(sRange.find("-")+1);
				dwSmallSize = atol(sHelp.c_str());
				if ((nPos = sHelp.find("/"))>=0)
				{
					dwServerSize = atoi(sHelp.substr(nPos+1).c_str());
				}
				else if (nContLengthPos>=0)
				{
					CString sLen = sHeader.substr(nContLengthPos+strlen("content-length:")+2);
					dwServerSize = dwSmallSize;
					dwSmallSize = atol(sLen.c_str());
				}
				if (dwServerSize == 0)
					dwServerSize = dwSmallSize;
				if(dwServerCompleted != m_nFilePos)
				{
					ForceDisconnect(REASON_NO_RESUME_SUPPORT, TRANSFER_CLOSED);
				}
				else if(dwServerSize == m_result.Size)
				{
					// copy the rest of the buffer into the second buffer
					// and clear the first one
					ASSERT(m_pStoreBuff->isEmpty());
					ASSERT(m_pReceiveBuff->contains()>=nEndPos);
					memcpy(m_pStoreBuff->data(),m_pReceiveBuff->data()+nEndPos, m_pReceiveBuff->contains()-nEndPos);
					m_pStoreBuff->increment(m_pReceiveBuff->contains()-nEndPos);
					m_pReceiveBuff->clear();
					swap(m_pStoreBuff, m_pReceiveBuff);
					// and here we go!
					// its also good to reset the "LastError"
					m_nDisconnectReason = REASON_EMPTY;
					SetStatus(TRANSFER_RECEIVING);
				}
				else
				{
					ForceDisconnect(REASON_WRONG_FILE_SIZE, TRANSFER_CLOSED);
				}
			}
			else
			{
				ForceDisconnect(REASON_BAD_HEADER, TRANSFER_FAILED);
			}
		}
		// Server error code
		else if(400 <= nCode && nCode < 500)
		{
			if (m_bForceUriRes || m_result.Sha1.empty())
				ForceDisconnect(REASON_FILE_NOT_FOUND, TRANSFER_CLOSED);
			else
			{
				ForceDisconnect(REASON_FILE_NOT_FOUND, TRANSFER_FAILED);
				m_bForceUriRes = true;
			}
		}
		// Server error code
		else if(500 <= nCode && nCode < 600)
		{
			ForceDisconnect(REASON_SERVER_BUSY, TRANSFER_FAILED);
			// TODO: parse queuing header
			//m_pDirector->OnDownloadHostBusy(GetHostIP(), xtime() + m_pPrefs->m_nRetryWait); // IP is not the best ID for pushes..
		}
		else
		{
			ForceDisconnect(REASON_UNKNOWN_SERVER_ERROR, TRANSFER_FAILED);
		}
	}
	// We are not in a connected or receiving state
	else
	{
		ForceDisconnect(REASON_WRONG_STATE, TRANSFER_CLOSED);
	}
}

void MDownloader::OnClose(int nErrorCode)
{
	// 'remainders' in the buffers will be stored by ForceDisconnect
	ForceDisconnect(REASON_REMOTELY_CANCELED, m_nPushTimeouts>=20 ? TRANSFER_CLOSED : TRANSFER_FAILED);
}

void MDownloader::SendRequest()
{
	// called from OnSetStartPoint
	//TRACE("SendRequest");
	m_bServerIsUp = true;
	m_nConErrors = 0;
	m_nPushTimeouts = 0;  // successful connection 'breaks' the sequence
	// create the request
	CString sGetRequest;
	CString sGetFormat;
	if (m_result.Vendor == "BEAR" && m_result.Sha1.size()==32)
		m_bForceUriRes = true;
	if (m_bForceUriRes)
	{
		sGetFormat = CString("GET /uri-res/N2R?urn:sha1:%s HTTP/1.1\r\n") +
							 "User-Agent: Mutella-" + VERSION + "\r\n" +
							 "Range: bytes=%d-\r\n" +
							 "X-Gnutella-Content-URN: urn:sha1:%s\r\n\r\n";
		sGetRequest.format(sGetFormat.c_str(), m_result.Sha1.c_str(), m_nFilePos, m_result.Sha1.c_str());
	}
	else
	{
		sGetFormat = CString("GET /get/%ld/%s HTTP/1.0\r\n") +
							 "User-Agent: Mutella-" + VERSION + "\r\n" +
							 "Range: bytes=%d-\r\n\r\n";
		sGetRequest.format(sGetFormat.c_str(), m_result.FileIndex, m_result.Name.c_str(), m_nFilePos);
	}
	
	Send(sGetRequest.c_str(), sGetRequest.length());
	
	SetStatus(TRANSFER_NEGOTIATING);

}

void MDownloader::SetStatus(int nNewStatus)
{
	m_nLastStatusChange = xtime();
	m_nLastActive = m_nLastStatusChange;
	m_nStatus = nNewStatus;
}

void MDownloader::CopyTo(SGnuDownloadItem& item)
{
	item.ipHost = m_result.Host;;
	item.nPort = m_result.Port;
	item.sVendor = m_result.Vendor;
	item.nStatus = m_nStatus;
	item.nLastStatusChange = m_nLastStatusChange;
	item.nLastActive = m_nLastActive;
	item.nDisconnectReason = m_nDisconnectReason;
	item.dwRate = m_dwRate;
	item.bServerIsUp = m_bServerIsUp;
	item.nConErrors = m_nConErrors;
	item.nPushTimeouts = m_nPushTimeouts;
}

////////////////////////////////////////////////////////////////////
// sort callback to sort results by speed
bool bySpeed(const Result* pr1, const Result* pr2)
{
	return pr1->Speed > pr2->Speed;
}

////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////
// MGnuDownload implementation

MGnuDownload::MGnuDownload(MGnuDirector* pDir, const ResultVec& results) : m_mutex(true)
{
	m_pDirector  = pDir;
	m_pPrefs     = m_pDirector->GetPrefs();
	
	m_dwBytesCompleted = 0;
	m_dwResumeStartPosition = 0;
	m_dRate = 0;
	
	// timers
	m_nCreationTime   = xtime();
	//m_nSecInactive    = 0;
	//
	m_nLastDisconnectReason = REASON_UNDEFINED;
	m_nStatus           = TRANSFER_NEW;
	
	m_dwSearchID = 0;
	m_bDelSearch = false;
	
	m_dwID = 0;
	
	ASSERT(results.size());
	m_sName        = results[0].Name;
	m_dwFileLength = results[0].Size;
	// sort results vector so that we start connection to the fastest hosts
	vector<const Result*> res_sorted;
	res_sorted.reserve(results.size());
	for (ResultVec::const_iterator it = results.begin(); it != results.end(); ++it)
		res_sorted.push_back(&(*it));
	sort(res_sorted.begin(), res_sorted.end(), bySpeed);
	//
	for (vector<const Result*>::const_iterator its = res_sorted.begin(); its != res_sorted.end(); ++its)
	{
		// create Downloader objects here
		MDownloader* pDl = new MDownloader(this, m_pDirector, **its);
		ASSERT(pDl);
		m_vecDownloaders.push_back(pDl);
		if ( m_vecDownloaders.size() >= MAX_DOWNLOADERS)
			break;
	}
	m_pFileMng = new MDownFileMng(this, m_pPrefs);
	ASSERT(m_pFileMng);
}

MGnuDownload::~MGnuDownload()
{
	MLock lock(m_mutex);
	// SHIT, again we are to expect problems with syncronisation here!
	delete m_pFileMng;
	//
	for (vector<MDownloader*>::iterator itd = m_vecDownloaders.begin(); itd != m_vecDownloaders.end(); ++itd)
		delete (*itd);
}

void MGnuDownload::GenerateID()
{
	m_dwID = GenID();
}

void MGnuDownload::Start()
{
	MLock lock(m_mutex);
	SetStatus(TRANSFER_OPENING);
	m_pFileMng->RequestOpen(m_sName, m_dwFileLength);
}

void MGnuDownload::Stop(bool bDeleteFile, bool bDeleteSearch)
{
	MLock lock(m_mutex);
	// this may and most probably will be called from another thread
	m_nLastDisconnectReason = REASON_STOPPED;
	ForceDisconnectAll();
	SetStatus(TRANSFER_CLOSING);
	if (bDeleteFile)
		m_pFileMng->RequestDelete();
	m_pFileMng->RequestClose();
	//if (bDeleteSearch && m_dwSearchID)
	//	m_pDirector->RemoveSearchByID(m_dwSearchID);
	m_bDelSearch = bDeleteSearch;
}

void MGnuDownload::ForceDisconnectAll()
{
	//ASSERT(m_mutex.locked());
	for (vector<MDownloader*>::iterator itd = m_vecDownloaders.begin(); itd != m_vecDownloaders.end(); ++itd)
		(*itd)->ForceDisconnect(REASON_INTERNAL, TRANSFER_CLOSED);
}

int MGnuDownload::CountActiveConnections(const HOST* pHost)
{
	MLock lock(m_mutex);
	int nCount = 0;
	for (vector<MDownloader*>::iterator itd = m_vecDownloaders.begin(); itd != m_vecDownloaders.end(); ++itd)
		if (((*itd)->GetStatus()==TRANSFER_RECEIVING   ||
			 (*itd)->GetStatus()==TRANSFER_NEGOTIATING ||
			 (*itd)->GetStatus()==TRANSFER_CONNECTED   ||
			 (*itd)->GetStatus()==TRANSFER_CONNECTING  ||
			 (*itd)->GetStatus()==TRANSFER_PUSH ) &&
			(pHost==NULL ||
			((*itd)->GetResult().Host==pHost->ip     &&
			 (*itd)->GetResult().Port==pHost->port   )))
			++nCount;
	return nCount;
}

void MGnuDownload::OnTimer(int nTimeStamp)
{
	MLock lock(m_mutex);
	if (m_nStatus != TRANSFER_OPEN)
		return;
	// first give Downloaders a chance to update their status
	for (vector<MDownloader*>::iterator itd = m_vecDownloaders.begin(); itd != m_vecDownloaders.end(); ++itd)
		(*itd)->OnTimer(nTimeStamp);
	// move "TRANSFER_CLOSED" downloaders to the 'bad' list
	for (int i = 0; i<m_vecDownloaders.size(); ++i)
	{
		if (m_vecDownloaders[i]->GetStatus() == TRANSFER_CLOSED)
		{
			m_vecBadResults.push_back(m_vecDownloaders[i]->GetResult());
			delete m_vecDownloaders[i];
			m_vecDownloaders.erase(m_vecDownloaders.begin()+i);
			--i;
		}
	}
	// update results list and downloaders list
	if ( m_vecDownloaders.size() < MAX_DOWNLOADERS &&           // dont create too many downloaders
		 (nTimeStamp % 15 == 0 || m_vecDownloaders.size()==0) ) // this might be quite demanding, so dont do that so often
	{
		UpdateResults();
		if (m_vecDownloaders.size()==0)
		{
			m_nLastDisconnectReason = REASON_TRANSFER_FAILED;
			//ForceDisconnectAll();
			SetStatus(TRANSFER_FAILED); // how we preserve status?
			// close fileman
			#warning status will be lost here
			SetStatus(TRANSFER_CLOSING);
			m_pFileMng->RequestClose();
			return;
		}
	}
	// calculate current rate
	m_dRate = 0;
	for (vector<MDownloader*>::iterator itd = m_vecDownloaders.begin(); itd != m_vecDownloaders.end(); ++itd)
		if ((*itd)->GetStatus()==TRANSFER_RECEIVING)
			m_dRate += (*itd)->GetRate();
	// update completed size
	m_dwBytesCompleted = m_pFileMng->GetSizeCompleted();
	// and schedule further actions
	if (m_pFileMng->NeedMoreSockets())
	{
		for (vector<MDownloader*>::iterator itd = m_vecDownloaders.begin(); itd != m_vecDownloaders.end(); ++itd)
		switch ((*itd)->GetStatus())
		{
			case TRANSFER_FAILED:
			case TRANSFER_COOLDOWN:
				//if ((*itd)->GetStatusChangeTime()+m_pPrefs->m_nRetryWait>=nTimeStamp)
				//	break;
			case TRANSFER_NEW:
				if (!m_pDirector->IsDownloadHostBusy((*itd)->GetHostIP()))
					(*itd)->Enqueue();
		}
	}
}

int  MGnuDownload::LookUp(IP ipHost, const CString& sFileName, int nIndex)
{
	MLock lock(m_mutex);
	// can be called from the different thread
	// searches the results list and returns the index or -1 if fails
	int i = 0;
	for (vector<MDownloader*>::iterator itd = m_vecDownloaders.begin(); itd != m_vecDownloaders.end(); ++itd)
	{
		if ((*itd)->GetResult().Host == ipHost      &&
			(*itd)->GetResult().FileIndex == nIndex &&
			((*itd)->GetResult().NameLower == sFileName ||
			 (*itd)->GetResult().Name == sFileName      ))
			return i;
		++i;
	}
	return -1;
}

DWORD MGnuDownload::GetSizeCompleted()
{
	return m_pFileMng->GetSizeCompleted();
}

DWORD MGnuDownload::GetSizeReceived()
{
	return m_pFileMng->GetSizeCompleted() - m_dwResumeStartPosition;
}

/*
struct SGnuDownloadItem
{
	// Node info
	IP		  ipHost;
	WORD      nPort;
	CString	  sVendor;
	// Download info
	int       nStatus;
	int       nLastStatusChange;
	int       nLastActive;
	int       nDisconnectReason;
	//
	bool      bServerIsUp;   // true if we recieved anything from it ever
	int       nConErrors;    // counter of consequitive connection errors
	int       nPushTimeouts; // counter of consequitive push timeouts
};

struct SGnuDownload
{
	// Download Properties
	CString m_sName;
	DWORD   m_dwSearchID;
	// File info
	DWORD   m_dwFileLength;
	DWORD   m_dwBytesCompleted;
	// source addresses, etc
	double m_dRate;          // real download rate in bytes per second
	//
	int    m_nActiveConnections;
	int    m_nCreationTime;
	// status
	int    m_nLastDisconnectReason;
	int    m_nStatus;
	// ID for UI
	DWORD  m_dwID;
	// Download items -- individual hosts
	DownloadItemVec m_vecItems;
	// few static methods
	static LPCSTR GetErrorString(int nCode);
	static LPCSTR GetStatusString(int nStatus);
};
*/

bool sortDownloadItems(const SGnuDownloadItem& i1, const SGnuDownloadItem& i2)
{
	// Sort by status then by rate, than by activity time, etc
	if (i1.nStatus<i2.nStatus)
		return true;
	if (i1.nStatus>i2.nStatus)
		return false;
	if (i1.nLastStatusChange>i2.nLastStatusChange)
		return true;
	if (i1.nLastStatusChange<i2.nLastStatusChange)
		return false;
	if (i1.dwRate>i2.dwRate)
		return true;
	if (i1.dwRate<i2.dwRate)
		return false;
	// equivalent elements return false
	return false;
}

void MGnuDownload::CopyTo(SGnuDownload& copy)
{
	MLock lock(m_mutex);
	// safely copies object into the structure
	ASSERT(m_vecItems.size() == 0);
	copy = *this;
	copy.m_dwBytesCompleted = m_pFileMng->GetSizeCompleted();
	copy.m_nBadResults = m_vecBadResults.size();
	// copy and sort items
	int n = m_vecDownloaders.size();
	copy.m_vecItems.reserve(n);
	copy.m_vecItems.resize(n);
	for (int i = 0; i<n; ++i)
	{
		m_vecDownloaders[i]->CopyTo(copy.m_vecItems[i]);
	}
	sort(copy.m_vecItems.begin(), copy.m_vecItems.end(), sortDownloadItems);
}

int MGnuDownload::GetNumAlternatives()
{
	MLock lock(m_mutex);
	return m_vecDownloaders.size();
}

void MGnuDownload::SetStatus(int nNewStatus)
{
	m_nStatus = nNewStatus;
	// POST_MESSAGE
	#warning POST_MESSAGE here
}

double MGnuDownload::GetRate()
{
	return m_dRate;
}

int MGnuDownload::DisconnReason()
{
	return m_nLastDisconnectReason;
}

CString MGnuDownload::GetFileName()
{
	MLock lock(m_mutex);
	return m_sName;
}

bool MGnuDownload::IsPartOf(const CString& path)
{
	MLock lock(m_mutex);
	if (m_pFileMng)
		return m_pFileMng->GetPartPath()==path;
	return false;
}

void MGnuDownload::SetFileName(const CString& sNewName)
{
	MLock lock(m_mutex);
	// this does not move the file. in fact it only makes sence to call before "Start()"
	m_sName = sNewName;
}

bool MGnuDownload::Rename(const CString& sNewName)
{
	MLock lock(m_mutex);
	// renames the partial file, sets new target name and updates automatic search
	// might be called from another thread
	ASSERT(m_pFileMng);
	if(!m_pFileMng->RequestRename(sNewName))
		return false;
	// change our name
	m_sName=sNewName;
	m_sName.make_lower();
	// modify automatic search...
	MGnuSearch* pSearch = NULL;
	if (0 != m_dwSearchID &&
		NULL != (pSearch = m_pDirector->GetSearchByID(m_dwSearchID)) )
	{
		CString sSearch = MakeSearchOfFilename(m_sName);
		pSearch->m_mutex.lock();
		pSearch->SetSearchString(sSearch, false);
		pSearch->m_Filename = m_sName;
		pSearch->m_mutex.unlock();
	}
	return true;
}

bool MGnuDownload::AddSearch(bool bAutoget)
{
	//ASSERT(m_mutex.locked());
	// adds search for alternate locations
	MGnuSearch* pSearch = NULL;
	//
	if ( (0==m_dwSearchID) || NULL == (pSearch = m_pDirector->GetSearchByID(m_dwSearchID)) )
	{
		// search with the given ID does not exists or
		// Id is just == 0
		// try to look up search by the name
		CString sSearch = MakeSearchOfFilename(m_sName);
		if (NULL == (pSearch = m_pDirector->LookUpSearch(sSearch, m_dwFileLength)) )
			pSearch = m_pDirector->AddSearch(sSearch.c_str(), ST_ALTERNATIVE, m_dwFileLength, LIMIT_EXACTLY);
		//
		if (pSearch)
		{
			MLock lock(pSearch->m_mutex);
			pSearch->m_Filename = m_sName;
			m_dwSearchID = pSearch->m_dwID;
		}
	}
	if (bAutoget && pSearch)
	{
		pSearch->m_bAutoget = true;
		// clear the search for any case
		pSearch->Clear();
	}
	return pSearch!=NULL;
}

void MGnuDownload::UpdateResults()
{
	//ASSERT(m_mutex.locked());
	// update list of downloaders if we have new hits on our "alternative" search
	if (m_dwSearchID)
	{
		// first we have to look up the appropriate search
		MGnuSearch* pSearch = pSearch = m_pDirector->GetSearchByID(m_dwSearchID);
		if (!pSearch || !pSearch->m_bUpdated)
		{
			if (!pSearch)
				AddSearch(false);
			return;
		}
		// lock search mutex for any case
		MLock lock(pSearch->m_mutex);
		// now we have to match and update
		// but first reset the 'updated' flag on the search
		pSearch->m_bUpdated = false;
		// sort results vector so that we start updating from the fastest hosts
		vector<Result*> res_sorted;
		res_sorted.reserve(pSearch->m_mapResults.size());
		for (map<DWORD,Result>::iterator it = pSearch->m_mapResults.begin(); it != pSearch->m_mapResults.end(); ++it)
			res_sorted.push_back(&(it->second));
		sort(res_sorted.begin(), res_sorted.end(), bySpeed);
		//
		for (vector<Result*>::iterator its = res_sorted.begin(); its != res_sorted.end(); ++its)
		{
			Result& SR = **its;
			bool bFound = false;
			for (vector<MDownloader*>::iterator itd = m_vecDownloaders.begin(); itd != m_vecDownloaders.end(); ++itd)
			{
				if (SR.Port        == (*itd)->GetResult().Port        &&
					SR.FileIndex   == (*itd)->GetResult().FileIndex   &&
					SR.Host.S_addr == (*itd)->GetResult().Host.S_addr &&
					SR.NameLower   == (*itd)->GetResult().NameLower   )
				{
					// update current result
					//TRACE("UpdateResults: updating existing");
					(*itd)->UpdateResult(SR);
					bFound = true;
					break;
				}
			}
			if (!bFound)
			{
				// look the result up in the m_BadResults list and see if it has been updated.
				bool bShouldAdd = true;
				//
				for (vector<Result>::iterator itBR = m_vecBadResults.begin(); itBR != m_vecBadResults.end(); ++itBR )
				{
					if (SR.Port        == itBR->Port        &&
					    SR.FileIndex   == itBR->FileIndex   &&
					    SR.Host.S_addr == itBR->Host.S_addr &&
					    SR.NameLower   == itBR->NameLower   )
					{
						// check if it was updated
						if (itBR->ChangeTime != SR.ChangeTime)
						{
							// it is not gona be bad anymore
							m_vecBadResults.erase(itBR);
							break;
						}
						else
						{
							bShouldAdd = false;
							break;
						}
					}
				}
				if ( bShouldAdd && m_vecDownloaders.size() < MAX_DOWNLOADERS )
				{
					//TRACE("UpdateResults: Adding result");
					// create Downloader objects here
					MDownloader* pDl = new MDownloader(this, m_pDirector, SR);
					ASSERT(pDl);
					m_vecDownloaders.push_back(pDl);
				}
			}
		}
	}
	else
	{
		//
		AddSearch(false);
	}
}

bool MGnuDownload::OnIncomingPush(const CString& sFileName, int nIndex, SOCKET hPushSock)
{
	MLock lock(m_mutex);
	// returns 'true' if the push was accepted
	for (vector<MDownloader*>::iterator itd = m_vecDownloaders.begin(); itd != m_vecDownloaders.end(); ++itd)
		if ((*itd)->OnIncomingPush(sFileName, nIndex, hPushSock))
			return true;
	return false;
}

bool MGnuDownload::OnSetStartPoint(MDownloader* pDownloader, int nFilePos)
{
	MLock lock(m_mutex);
	// return true if start point was accepted
	for (vector<MDownloader*>::iterator itd = m_vecDownloaders.begin(); itd != m_vecDownloaders.end(); ++itd)
		if (*itd == pDownloader)
			return (*itd)->OnSetStartPoint(nFilePos);
	return false;
}

void MGnuDownload::OnFileManReady()
{
	MLock lock(m_mutex);
	//TRACE("MGnuDownload::OnFileManReady");
	if (TRANSFER_OPENING == m_nStatus)
	{
		m_dwResumeStartPosition = m_pFileMng->GetSizeCompleted();
		SetStatus(TRANSFER_OPEN);
		// 'initiate' Downloaders here -- this will make them to try to connect
		for (vector<MDownloader*>::iterator it = m_vecDownloaders.begin(); it != m_vecDownloaders.end(); ++it)
		{
			(*it)->Initiate(); // calls "Connect" or requests push
		}
	}
	else
	{
		TRACE("Unexpected MGnuDownload::OnFileManReady()");
	}
}

void MGnuDownload::OnDataWritten(MDownloader* pDownloader, DWORD dwPos, DWORD dwSize, MDownloadBuffer* pB)
{
	MLock lock(m_mutex);
	//
	for (vector<MDownloader*>::iterator itd = m_vecDownloaders.begin(); itd != m_vecDownloaders.end(); ++itd)
		if (*itd == pDownloader)
		{
			// if it's the 'first success' than we should add the Sha1 to the search
			const CString& sha1 = pDownloader->GetResult().Sha1;
			if (sha1.length() == 32 && m_setSha1.find(sha1) == m_setSha1.end())
			{
				// TODO: modify the file name or somehow make this set persistent
				m_setSha1.insert(sha1);
				// modfy the auto-search
				MGnuSearch* pSearch = NULL;
				if (0 != m_dwSearchID &&
					NULL != (pSearch = m_pDirector->GetSearchByID(m_dwSearchID)) )
				{
					pSearch->AddSha1Hash(sha1);
				}
			}
			pDownloader->OnDataWritten(dwPos, dwSize, pB);
			return;
		}
}

void MGnuDownload::OnFileError(int nError)
{
	MLock lock(m_mutex);
	//
	SetStatus(TRANSFER_FILE_ERROR);
	// disconnect all
	m_nLastDisconnectReason = REASON_FILE_ERROR;
	ForceDisconnectAll();
	// close fileman
	#warning status will be lost here
	SetStatus(TRANSFER_CLOSING);
	m_pFileMng->RequestClose();
	// post a message
	POST_EVENT( MStringEvent(
		ET_ERROR,
		ES_IMPORTANT,
		CString("filename=`") + m_pFileMng->GetPartPath() + "' error: " + strerror(nError),
		MSG_DOWNLOAD_FILE_ERROR,
		MSGSRC_DOWNLOAD
	));
}

void MGnuDownload::OnFileManClosed()
{
	MLock lock(m_mutex);
	//
	if (TRANSFER_CLOSING == m_nStatus)
	{
		TRACE("MGnuDownload::OnFileManClosed()");
	}
	else
	{
		TRACE("Unexpected MGnuDownload::OnFileManClosed()");
	}
	SetStatus(TRANSFER_CLOSED);
}

void MGnuDownload::OnFileCompleted()
{
	MLock lock(m_mutex);
	// disconnect all
	TRACE("MGnuDownload::OnFileCompleted() is not fully implemented");
	m_nLastDisconnectReason = REASON_COMPLETED;
	ForceDisconnectAll();
	SetStatus(TRANSFER_COMPLETED); // how we preserve status?
	m_bDelSearch = true;
	// close fileman
	#warning status will be lost here
	SetStatus(TRANSFER_CLOSING);
	m_pFileMng->RequestClose();
	//
	POST_EVENT( MStringEvent(
		ET_MESSAGE,
		ES_GOODTOKNOW,
		CString("`") + m_sName + "'",
		MSG_DOWNLOAD_COMPLETED,
		MSGSRC_DOWNLOAD
	));
}

void MGnuDownload::OnSegmentCompleted(MDownloader* pDownloader, bool bSuccess)
{
	MLock lock(m_mutex);
	#warning MGnuDownload::OnSegmentCompleted(MDownloader* pDownloader, bool bSuccess) is not implemented
	assert(0);
	// Downloader should stop here
	// shell we give it a new task?
}

void MGnuDownload::OnDisconnect(MDownloader* pDownloader)
{
	// notify the manager
	m_pFileMng->OnDisconnect(pDownloader);
}

bool MGnuDownload::RequestStartPoint(MDownloader* pDownloader)
{
	// if returns false caller should close and be ready to go home
	return m_pFileMng->RequestStartPoint(pDownloader);
}

StoreResult MGnuDownload::StoreDataAt(MDownloader* pDownloader, int nPos, MDownloadBuffer* pBuff, int nSize)
{
	// if returns something other than SR_Success caller should close and be ready to go home
	return m_pFileMng->StoreDataAt(pDownloader, nPos, pBuff, nSize);
}

LPCSTR SGnuDownload::GetErrorString(int nCode)
{
	switch(nCode)
	{
		case REASON_UNDEFINED:
			return "UNDEFINED";
		case REASON_EMPTY:
			return "";
		case REASON_STOPPED:
			return "STOPPED";
		case REASON_NO_DATA:
			return "NO DATA";
		case REASON_SOCKET_ERROR:
			return "BAD SOCKET";
		case REASON_NO_RESUME_SUPPORT:
			return "NO RESUME";
		case REASON_WRONG_FILE_SIZE:
			return "WRONG SIZE";
		case REASON_FILE_NOT_FOUND:
			return "NO FILE";
		case REASON_SERVER_BUSY:
			return "BUSY";
		case REASON_UNKNOWN_SERVER_ERROR:
			return "UNKNOWN";
		case REASON_WRONG_STATE:
			return "WRONG STATE";
		case REASON_CONNECTION_ERROR:
			return "CONN ERROR";
		case REASON_REMOTELY_CANCELED:
			return "REM. CANCELED";
		case REASON_FAILED_TO_OPEN_FILE:
			return "CANNOT OPEN";
		case REASON_PARTIAL_FILESIZE_DIFFERENT:
			return "PART. SIZE";
		case REASON_HOST_OCCUPIED:
			return "OCCUPIED";
		case REASON_FILES_ARE_DIFFERENT:
			return "PART. MISMATCH";
		case REASON_WRITE_ERROR:
			return "WRITE ERROR";
		case REASON_FILE_ERROR:
			return "DISK ERROR";
		case REASON_COMPLETED:
			return "COMPLETED";
		case REASON_CORRUPT:
			return "CORRUPT";
		case REASON_NO_RESPONSE:
			return "NO RESPONSE";
		case REASON_BELOW_MINIMUM_SPEED:
			return "TOO SLOW";
		case REASON_INTERNAL:
			return "INTERNAL";
		case REASON_BAD_HEADER:
			return "BAD HEADER";
		case REASON_STORAGE_DENIED:
			return "NO STORAGE";
		case REASON_TRANSFER_FAILED:
			return "FAILED";
	}
	return         "WRONG CODE";
}

/*enum DisconnectReason {
	REASON_UNDEFINED,
	REASON_EMPTY,
	REASON_STOPPED,
	REASON_NO_DATA,
	REASON_SOCKET_ERROR,
	REASON_NO_RESUME_SUPPORT,
	REASON_WRONG_FILE_SIZE,
	REASON_FILE_NOT_FOUND,
	REASON_SERVER_BUSY,
	REASON_UNKNOWN_SERVER_ERROR,
	REASON_WRONG_STATE,
	REASON_CONNECTION_ERROR,
	REASON_REMOTELY_CANCELED,
	REASON_FAILED_TO_OPEN_FILE,
	REASON_PARTIAL_FILESIZE_DIFFERENT,
	REASON_HOST_OCCUPIED,
	REASON_FILES_ARE_DIFFERENT,
	REASON_WRITE_ERROR,
	REASON_FILE_ERROR,
	REASON_COMPLETED,
	REASON_CORRUPT,
	REASON_NO_RESPONSE,
	REASON_BELOW_MINIMUM_SPEED,
	REASON_INTERNAL,
	REASON_BAD_HEADER,
	REASON_STORAGE_DENIED,
	REASON_TRANSFER_FAILED*/


LPCSTR SGnuDownload::GetStatusString(int nStatus)
{
	switch(nStatus)
	{
		case TRANSFER_COMPLETED:
			return "COMPL";
		case TRANSFER_SEGM_COMPLETED:
			return "S-COMPL";
		case TRANSFER_RECEIVING:
			return "RECV";
		case TRANSFER_SENDING:
			return "SEND";
		case TRANSFER_NEGOTIATING:
			return "NEGOT";
		case TRANSFER_CONNECTED:
			return "CONN";
		case TRANSFER_PUSH_CONNECTING: // used by ulpoad only
		case TRANSFER_PUSH_CONNECTED:  // used by ulpoad only
			return "PUSH";
		case TRANSFER_CONNECTING:
			return "CONN..";
		case TRANSFER_PUSH:            // used by download only
			return "PUSH-W";           // meaning: waiting for a push
		case TRANSFER_QUEUED:
			return "QUEUED";
		case TRANSFER_NEW:
			return "NEW";
		case TRANSFER_COOLDOWN:
			return "DELAY";
		case TRANSFER_FAILED:          // used by download only
			return "WAIT";            // meaning: there was a error, but retry is possible
		case TRANSFER_OPEN:
			return "OPEN";
		case TRANSFER_OPENING:
			return "OPENING";
		case TRANSFER_INTERNAL_ERROR:
			return "ERROR";
		case TRANSFER_FILE_ERROR:
			return "DSK-ERR";
		case TRANSFER_CLOSING:
			return "CLOSING";
		case TRANSFER_CLOSED:
			return "CLOSED";
	}
	return         "WRONG";
}

/*enum TransferStatus {
	TRANSFER_COMPLETED,
	TRANSFER_SEGM_COMPLETED,
	TRANSFER_RECEIVING,
	TRANSFER_SENDING,
	TRANSFER_NEGOTIATING,
	TRANSFER_CONNECTED,
	TRANSFER_PUSH_CONNECTED,  // used by ulpoad only
	TRANSFER_PUSH_CONNECTING, // used by ulpoad only
	TRANSFER_CONNECTING,
	TRANSFER_PUSH,
	TRANSFER_QUEUED,
	TRANSFER_NEW,
	TRANSFER_COOLDOWN,
	TRANSFER_FAILED,
	TRANSFER_OPEN,
	TRANSFER_OPENING,        // used by MGnuDownload only
	TRANSFER_INTERNAL_ERROR,
	TRANSFER_FILE_ERROR,	
	TRANSFER_CLOSING,        // used by MGnuDownload only
	TRANSFER_CLOSED
};*/

