/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 event.cpp  -  Event and event distribution for messages (Event Log)

    begin                : Sun Jan 6 2002
    copyright            : (C) 2002 by 
    email                : 
 ***************************************************************************/
 
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "common.h"
#include "event.h"

// general event class -- abstract
MEvent::MEvent(int type, int severity, DWORD ID /*=0*/, DWORD SourceID /*=0*/) : m_nType(type), m_nSeverity(severity), m_dwID(ID), m_dwSourceID(SourceID)
{
	m_nTime = xtime();
	m_pNotifier = NULL;
}

MEvent::~MEvent()
{
	if (m_pNotifier)
	{
		m_pNotifier->mutex.lock();
		m_pNotifier->wc.wakeAll();
		m_pNotifier->mutex.unlock();
	}
}

// simple string event
MStringEvent::MStringEvent(int type, int severity, const CString& value, DWORD ID /*=0*/, DWORD SourceID /*=0*/) :
	TSimpleEvent<CString>(type,severity,value, ID, SourceID)
{
}

int MStringEvent::GetMemUsage()
{
	return sizeof(MStringEvent)+m_value.buffersize();
}

CString MStringEvent::Format()
{
	return m_value;
}

// event dispatcher
MEventDispatcher::MEventDispatcher()
{
}

MEventDispatcher::~MEventDispatcher()
{
}

MEventDispatcher MEventDispatcher::ms_EventDispatcher;
MEventDispatcher& ED()
{
	return MEventDispatcher::ms_EventDispatcher;
}

void MEventDispatcher::PostEvent(MEvent* pEvent)
{
	ASSERT(pEvent);
	if (!pEvent)
		return;
	m_mutex.lock();
	for (list<MEventReceiver*>::iterator it = m_receivers.begin(); it != m_receivers.end(); ++it)
		if ((*it)->IsOfInterest(pEvent))
			(*it)->OnEvent(pEvent, false);
	m_mutex.unlock();
	// this will free the memory
	pEvent->Release();
}

bool MEventDispatcher::SendEvent(MEvent* pEvent)
{
	// one should be really carefull with this function:
	// because it only returns after the event object is deleted
	ASSERT(pEvent);
	if (!pEvent)
		return false;
	bool bDelivered = false;
	// prepare 'feedback'
	SEventNotifier en;
	pEvent->m_pNotifier = &en;
	// broadcast the message
	m_mutex.lock();
	ASSERT(pEvent->GetRefs()==1);// Send() will hang otherwise
	//
	for (list<MEventReceiver*>::iterator it = m_receivers.begin(); it != m_receivers.end(); ++it)
		if ((*it)->IsOfInterest(pEvent))
		{
			(*it)->OnEvent(pEvent, true);
			bDelivered = true;
		}
	m_mutex.unlock();
	// now do the waiting thing NOTE: it could be done with the internal MRCObject's mutex
	en.mutex.lock();
	if (pEvent->GetRefs()>1)
	{
		// we still have plenty of references
		pEvent->Release();
		while (!en.wc.wait(&en.mutex)) {}
		en.mutex.unlock();
	}
	else
	{
		// nothing really to wait for -- everybody have released the object
		en.mutex.unlock();
		pEvent->Release();
	}
	return bDelivered;
}

void MEventDispatcher::AddReceiver(MEventReceiver* pReceiver)
{
	// possible deadlock if called from OnEvent();
	m_mutex.lock();
	m_receivers.push_back(pReceiver);
	m_mutex.unlock();
}

void MEventDispatcher::RemoveReceiver(MEventReceiver* pReceiver)
{
	// possible deadlock if called from OnEvent();
	m_mutex.lock();
	m_receivers.remove(pReceiver);
	m_mutex.unlock();
}

// event receivers
MEventReceiver::~MEventReceiver()
{
	ED().RemoveReceiver(this);
}

MAsyncEventReceiver::MAsyncEventReceiver(int nMaxTime, int nMaxMem /*=INT_MAX*/) : m_nMaxTime(nMaxTime), m_nMaxMem(nMaxMem)
{
	m_nMemUsed = 0;
	m_nMissedEvents = 0;
	m_hPollingThread = (HANDLE)-1; // i'd assume -1 is good "free" marker
	m_hThreadLock = (HANDLE)-1; // i'd assume -1 is good "free" marker
	m_nFrontLocks = 0;
}

MAsyncEventReceiver::~MAsyncEventReceiver()
{
}

void MAsyncEventReceiver::SetPollingThread()
{
	m_mutex.lock();
	m_hPollingThread = MThread::currentThread();
	m_mutex.unlock();
}

void MAsyncEventReceiver::OnEvent(MEvent* pEvent, bool bSend)
{
	m_mutex.lock();
	m_queue.push_back(pEvent);
	m_nMemUsed += pEvent->GetMemUsage();
	// apply limits
	if (!m_nFrontLocks)
		FlushToFit();
#ifdef _DEBUG
	else
		TRACE("MAsyncEventReceiver::OnEvent(): front is locked!"); // this is needed to see if somebody forgets to unlock
#endif //_DEBUG
	// make Poll() work
	m_wait.wakeAll();
	m_mutex.unlock();
	
}

void MAsyncEventReceiver::FlushToFit()
{
	ASSERT(m_mutex.locked());
	int nTime = (m_nMaxTime != INT_MAX) ? (xtime()-m_nMaxTime) : 0;
	while(m_queue.size()>1 &&
		  ( m_nMemUsed>m_nMaxMem ||
		    m_queue.front()->GetTime()<nTime ))
	{
		m_nMemUsed -= m_queue.front()->GetMemUsage();
		if (m_priorityEvents.front() == m_queue.front())
			m_priorityEvents.pop();
		m_queue.pop_front();
		++m_nMissedEvents;
	}
}

bool MAsyncEventReceiver::Poll(u_long time /*=ULONG_MAX*/)
{
	MLock lock(m_mutex, m_hThreadLock!=MThread::currentThread());
	if (m_hPollingThread!=(HANDLE)-1 && m_hPollingThread!=MThread::currentThread())
		return false;
	if (m_queue.size())
		return true;
	return m_wait.wait(&m_mutex, time);
}

bool MAsyncEventReceiver::HaveEvents()
{
	MLock lock(m_mutex, m_hThreadLock!=MThread::currentThread());
	return m_queue.size();
}

int MAsyncEventReceiver::QueuedEvents()
{
	MLock lock(m_mutex, m_hThreadLock!=MThread::currentThread());
	return m_queue.size();
}

int MAsyncEventReceiver::MissedEvents()
{
	MLock lock(m_mutex, m_hThreadLock!=MThread::currentThread());
	return m_nMissedEvents;
}

void MAsyncEventReceiver::ResetMissedEvents()
{
	MLock lock(m_mutex, m_hThreadLock!=MThread::currentThread());
	m_nMissedEvents = 0;
}

int MAsyncEventReceiver::PriorityEvents()
{
	MLock lock(m_mutex, m_hThreadLock!=MThread::currentThread());
	return m_priorityEvents.size();
}

MEvent* MAsyncEventReceiver::LockFront()
{
	MLock lock(m_mutex, m_hThreadLock!=MThread::currentThread());
	m_nFrontLocks++;
	if (m_queue.size())
		return m_queue.front();
	return NULL;
}

MEvent* MAsyncEventReceiver::UnsafeFront()
{
	MLock lock(m_mutex, m_hThreadLock!=MThread::currentThread());
	ASSERT(m_nFrontLocks);
	if (m_queue.size())
		return m_queue.front();
	return NULL;
}

void MAsyncEventReceiver::UnlockFront()
{
	MLock lock(m_mutex, m_hThreadLock!=MThread::currentThread());
	ASSERT(m_nFrontLocks);
	m_nFrontLocks--;
	if (!m_nFrontLocks)
		FlushToFit();
}

TSmartPtr<MEvent> MAsyncEventReceiver::SafeFront()
{
	MLock lock(m_mutex, m_hThreadLock!=MThread::currentThread());
	if (m_queue.size())
	{
		ASSERT(m_queue.front());
		return m_queue.front();
	}
	return NULL;
}

void MAsyncEventReceiver::Pop()
{
	MLock lock(m_mutex, m_hThreadLock!=MThread::currentThread());
	if (!m_queue.size())
		return;
	m_nMemUsed -= m_queue.front()->GetMemUsage();
	if (m_priorityEvents.size() && m_priorityEvents.front() == m_queue.front())
		m_priorityEvents.pop();
	m_queue.pop_front();
}

void MAsyncEventReceiver::CopyQueue(queue_type& copy)
{
	MLock lock(m_mutex, m_hThreadLock!=MThread::currentThread());
	copy = m_queue;
}

