/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_utilities.h 
 * @brief Some common utility functions.
 * 
 */

#ifndef __GC_GL_UTILITIES_H__
#define __GC_GL_UTILITIES_H__

#include <png.h>

#include "myx_gc.h"
#include "myx_gc_datatypes.h"

//----------------------------------------------------------------------------------------------------------------------

typedef enum tagColorType
{
  COLOR_TYPE_PALETTE =     PNG_COLOR_TYPE_PALETTE,
  COLOR_TYPE_GRAY =        PNG_COLOR_TYPE_GRAY,
  COLOR_TYPE_GRAY_ALPHA =  PNG_COLOR_TYPE_GRAY_ALPHA,
  COLOR_TYPE_RGB =         PNG_COLOR_TYPE_RGB,
  COLOR_TYPE_RGB_ALPHA =   PNG_COLOR_TYPE_RGB_ALPHA
} TColorType;

typedef struct tagTImage
{
  unsigned int Width;     // The width of the image in pixels.
  unsigned int Height;    // The height of the image in pixels.
  unsigned char* Data;    // The image data.
  TColorType ColorType;    // The color format of the image (RGB, gray, palette etc.)
                           // Note: Palette images are not supported.
  unsigned int Channels;  // Bytes per pixel.
  GLenum Format;           // OpenGL color format specifier. Set by the image user.
} TImage;

//----------------------------------------------------------------------------------------------------------------------

/** Converts the given string, which is supposed to be an UTF-8 encoded text into an UTF-16 string. */
GENERIC_CANVAS_API wstring utf8ToUtf16(const string& Source);

/** Converts the given UTF-16 string into an UTF-8 string. */
GENERIC_CANVAS_API string utf16ToUtf8(const wstring& Source);

/** Converts the given string into an ANSI string using the current system locale. */
GENERIC_CANVAS_API string utf16ToANSI(const wstring& Source);

/** Converts the given string, which is supposed to be an UTF-8 encoded text into an ANSI string using the current system locale. */
GENERIC_CANVAS_API string utf8ToANSI(const string& Source);

/** Loads the given PNG image from disk. */
GENERIC_CANVAS_API TImage* loadPNG(const string& Filename);

/** Releases the given image. */
GENERIC_CANVAS_API void freeImage(TImage* Image);

/** Returns the current working folder. */
GENERIC_CANVAS_API string getCurrentDir(void);

/** Sets the current working folder. */
GENERIC_CANVAS_API void setCurrentDir(const string& Folder);

/** Extracts the drive and path from the given file name. */
GENERIC_CANVAS_API string extractFilePath(const string& Filename);

/** Platform neutral file open function. */
FILE* openFile(string Filename, const char* OpenMode);

/** Sorts the given bounds so that left <= right and bottom <= top. */
TBoundingBox sortBounds(const TBoundingBox& Bounds);

/** Determines whether both bounds overlap. */
bool boundsIntersect(const TBoundingBox& Bounds1, const TBoundingBox& Bounds2);

/** Determines whether bounds are empty. */
bool boundsAreEmpty(const TBoundingBox& Bounds);

/** Checks if a given point is within the given bounds. */
bool boundsContainPoint(const TBoundingBox& Bounds, const float X, const float Y);

/** Converts a container name to an identifier suitable for quick lookup. */
GENERIC_CANVAS_API TContainerID getContainerID(const string& container);

/** Returns an identifier for a given property name. */
GENERIC_CANVAS_API TPropertyID getPropertyID(const string& property);

/** Returns the index value for a given property. */
int getEntryIndex(string& Path);

/** Converts a (byte) color to a string. */
string colorToString(GLubyte* Color);

/** Converts a (float) color to a string. */
string colorToString(GLfloat* Color);

// Adds colors to the named color table.
void registerSystemColors(const TColorMap &ColorMap);

// Find a color by name.
bool colorByName(string Name, GLubyte* Color);

/** Converts a string to color with byte members. */
int stringToColor(string ColorString, GLubyte* Color);

/** Converts a string to color with float members. */
int stringToColor(string ColorString, GLfloat* Color);

/** Conversion functions for GC variants and simple values. */
GENERIC_CANVAS_API string variantToString(const TGCVariant& Variant);
GENERIC_CANVAS_API int variantToInt(const TGCVariant& Variant);
GENERIC_CANVAS_API float variantToFloat(const TGCVariant& Variant);
GENERIC_CANVAS_API bool variantToBool(const TGCVariant& Variant);
GENERIC_CANVAS_API TGCVariant variant(const string& Value);
GENERIC_CANVAS_API TGCVariant variant(const char* Value);
GENERIC_CANVAS_API TGCVariant variant(const int Value);
GENERIC_CANVAS_API TGCVariant variant(const float Value);
GENERIC_CANVAS_API TGCVariant variant(const bool Value);

/** Matrix code */
static void matrixMultiply(TMatrix product, const TMatrix a, const TMatrix b);
void matrixRotate(TMatrix mat, GLfloat angle, GLfloat x, GLfloat y, GLfloat z);
void matrixScale(TMatrix mat, GLfloat x, GLfloat y, GLfloat z);
void matrixTranslate(TMatrix mat, GLfloat x, GLfloat y, GLfloat z);
TVertex matrixTransform(TMatrix M, TVertex V);

//----------------------------------------------------------------------------------------------------------------------

/**
 * The bounding box computer is a neat little helper class to construct a final bound box out of an arbitrary number of
 * other boxes as well as (lists of) points.
 */
class CBoundingBoxComputer
{
private:
  TBoundingBox FCurrentBox;
protected:
  void includeVertex(const TVertex& Vertex);
  TVertex transform(TMatrix Matrix, const float& X, const float& Y);
public:
  CBoundingBoxComputer(void);
  CBoundingBoxComputer(const TBoundingBox& InitialBox);

  TBoundingBox boundingBox();
  void include(TMatrix Matrix, TBoundingBox* NewBox);
  void include(TMatrix Matrix, const TVertex& Vertex);
  void include(TMatrix Matrix, const float& X, const float& Y);
};
 
//----------------------------------------------------------------------------------------------------------------------

#endif // __GC_GL_UTILITIES_H__
