unit GenericCanvasViewer;

// Copyright (C) 2003, 2004 MySQL AB
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//----------------------------------------------------------------------------------------------------------------------
//
//  Implementation of a viewer for the MySQL generic canvas used to visualize object models.
//
//----------------------------------------------------------------------------------------------------------------------

interface

{$A8} // We need 8 byte alignment as used by C++.
{$Z4} // Enumeration types must be double word sized.

uses
  Windows, Messages, Controls, Classes, Graphics, Sysutils, Forms;

const
  ScrollTimer = 1;           // The id of the time for scrolling.
  FPSBufferSize = 30;        // The number of values to keep for sliding average computation of the speed.

type
  TGenericCanvas = class;
  TGenericCanvasViewer = class;
  TFigure = class;
  TGCListener = class;
  TFigureInstance = class;

  TRCOptions = set of (
    opDoubleBuffered,
    opGDI,
    opStereo
  );

  PViewport = ^TViewport;
  TViewport = record
    Left, Top,
    Width, Height: Integer;
  end;

  TBoundingBox = record
    UpperLeftX: Single;
    UpperLeftY: Single;
    UpperLeftZ: Single;
    LowerRightX: Single;
    LowerRightY: Single;
    LowerRightZ: Single;
  end;


  TPointD = record
    X, Y: Double;
  end;

  TGCError = (
    GC_NO_ERROR = 0,
    GC_CANT_OPEN_FILE,
    GC_XML_PARSE_ERROR,
    GC_XML_INVALID_DOCUMENT,
    GC_XML_EMPTY_DOCUMENT,
    GC_OBJECT_NOT_FOUND,
    GC_CANT_READ_FROM_FILE,
    GC_CHARSET_CONVERSION_ERROR,
    GC_CHARSET_WRONG_CHARSET_SPECIFIED
  );

  TGCChangeReason =
  (
    // Selection related changes.
    GC_CHANGE_SELECTION_ADD,             // One or more figure instances were added to the current selection.
    GC_CHANGE_SELECTION_CLEAR,           // The current selection was cleared.
    GC_CHANGE_SELECTION_REMOVE,          // One or more figure instances were removed from the current selection.
    GC_CHANGE_SELECTION_CHANGE,          // One or more figure instances were added to or removed from the current selection.

    // Canvas related changes.
    GC_CHANGE_CANVAS_REFRESH,            // Used to indicate that the view must update the visual representation.
    GC_CHANGE_CANVAS_PROPERTY,           // The value of a property has been changed.
    GC_CHANGE_CANVAS_ADD_VIEW,           // A new view was added.
    GC_CHANGE_CANVAS_ADD_LAYER,          // A new layer was added.
    GC_CHANGE_CANVAS_SWITCH_VIEW,        // Another view was activated.
    GC_CHANGE_CANVAS_REMOVE_VIEW,        // A view was removed.
    GC_CHANGE_CANVAS_REMOVE_LAYER,       // A layer was removed.
    GC_CHANGE_CANVAS_CLEAR_CONTENT,      // All figures have been removed.
    GC_CHANGE_CANVAS_CLEAR_LAYOUTS,      // All layout definitions have been removed.
    GC_CHANGE_CANVAS_CLEAR_STYLES,       // All styles have been removed.

    // Model related changes.
    GC_CHANGE_MODEL_PROPERTY,            // The value of a property has been changed.
    GC_CHANGE_MODEL_ADD_FIGURE,          // A new figure was added.
    GC_CHANGE_MODEL_REMOVE_FIGURE,       // A figure was removed.
    GC_CHANGE_MODEL_ADD_STYLE,           // A new style was added.
    GC_CHANGE_MODEL_REMOVE_STYLE,        // A style was removed.

    // Caption element related changes.
    GC_CHANGE_CAPTION_PROPERTY,          // The value of a property has been changed.

    // Figure element related changes.
    GC_CHANGE_ELEMENT_PROPERTY,         // The value of a property has been changed.

    // Figure related changes.
    GC_CHANGE_FIGURE_PROPERTY,           // The value of a property has been changed.

    // Figure instance related changes.
    GC_CHANGE_FINSTANCE_PROPERTY,        // The value of a property has been changed.

    // View related changes.
    GC_CHANGE_VIEW_PROPERTY,             // The value of a property has been changed.
    GC_CHANGE_VIEW_ADD_LAYER,            // A new layer was added to a view.
    GC_CHANGE_VIEW_REMOVE_LAYER,         // A layer was removed.

    // Layer related changes.
    GC_CHANGE_LAYER_CLEAR,               // All figure instances on the layer are removed.
    GC_CHANGE_LAYER_VISIBILITY,          // The visibility of a layer has been changed.
    GC_CHANGE_LAYER_PROPERTY,            // The value of a property has been changed.
    GC_CHANGE_LAYER_ADD_INSTANCE,        // A new figure instance was added.
    GC_CHANGE_LAYER_REMOVE_INSTANCE,     // A figure instance was removed.
    GC_CHANGE_LAYER_ADD_GROUP,           // A new group was added to a view.
    GC_CHANGE_LAYER_REMOVE_GROUP         // A group was removed.
  );

  TGCDirection =
  (
    GC_SI_NONE,
    GC_SI_ON_OBJECT,
    GC_SI_NORTH,
    GC_SI_NORTH_EAST,
    GC_SI_EAST,
    GC_SI_SOUTH_EAST,
    GC_SI_SOUTH,
    GC_SI_SOUTH_WEST,
    GC_SI_WEST,
    GC_SI_NORTH_WEST
  );

  TRubberbandStyle =
  (
    GC_RBSTYLE_SOLID_THIN,             // A simple black rectangle with a one pixel wide border.
    GC_RBSTYLE_SOLID_THICK,            // A simple black rectangle with a 3 pixel wide border.
    GC_RBSTYLE_DOTTED_THIN,            // A simple black rectangle with a one pixel wide dotted border.
    GC_RBSTYLE_DOTTED_THICK,           // A simple black rectangle with a 3 pixel wide dotted border.
    GC_RBSTYLE_BLENDED_CHECKERBOARD,   // A filled rectangle with a one pixel border and a translucent interior.
                                       // The system's selection color is used. The interior is a checker board.
    GC_RBSTYLE_BLENDED_DIAGONALS       // A filled rectangle with a one pixel border and a translucent interior.
                                       // The system's selection color is used. The interior consists of diagonal bands.
  );

  // TRBSelectionAction (rubber band selection action) determines how to manipulate the selection state of figure
  //instances with regard to their bounding box intersecting with the rubber band.

  TRBSelectionAction =
  (
    GC_RBACTION_NONE,               // Don't touch the selection state of any figure instance.
                                    // Usually used for non-selecting rubber bands (e.g. for figure creation).
    GC_RBACTION_SELECT,             // Always select figure instances if their bounding box intersects. Keep selected
                                    // instances as their are if the do not intersect anymore.
                                    // Usually used for rubber bands with pressed shift key modifier.
    GC_RBACTION_SELECT_REMOVE,      // Select figure instances if they intersect, unselect those, which do not intersect.
                                    // Most common rubber band selection mode.
    GC_RBACTION_TOGGLE              // Revert the selection state of figure instances, which intersect. Don't touch the others.
                                    // Usually used for rubber bands with pressed control key modifier.
  );

  TFactor = array[0..2] of Single;
  TAxis = array[0..2] of Single;

  TGCVariant = record
    // You should not use this variant!
  end;

  TGCBase = class;

  // The general listener class is used to notify users of the canvas about general events like repaints and errors.
  TGCListener = class
    FViewer: TGenericCanvasViewer;
  public
    constructor Create(Viewer: TGenericCanvasViewer);

  public
    procedure OnChange(Sender, Origin: TGCBase; Reason: TGCChangeReason); virtual; cdecl;
    procedure OnDestroy(Sender: TGCBase); virtual; cdecl;
    procedure OnError(Sender: TGCBase; const Message: PChar); virtual; cdecl;
  end;

  TGCBase = class
  public
    procedure _CGCBase; virtual; cdecl; abstract; // The destructor of the class. Never call that.

    procedure AddListener(Listener: TGCListener); virtual; cdecl; abstract;
    procedure Change(Origin: TGCBase; Reason: TGCChangeReason); virtual; cdecl; abstract;
    function Destroying: Boolean; virtual; cdecl; abstract;
    procedure Error(Origin: TGCBase; Message: PChar); virtual; cdecl; abstract;
    function GetClassName: PChar; virtual; cdecl; abstract;
    function Property_(Name: PChar; Index: Cardinal): TGCVariant; overload; virtual; cdecl; abstract;
    procedure Property_(Name: PChar; Index: Cardinal; const Value: TGCVariant); overload; virtual; cdecl; abstract;
    procedure Release; virtual; cdecl; abstract;
    procedure RemoveListener(Listener: TGCListener); virtual; cdecl; abstract;
  end;

  TFigure = class(TGCBase)
  public
    procedure Bounds(var Bounds: TBoundingBox); virtual; cdecl; abstract;
    procedure Render; virtual; cdecl; abstract;
    procedure Rotate(Angle: Single; Rx, Ry, Rz: Single); overload; virtual; cdecl; abstract;
    procedure Rotate(Angle: Single; Axis: TAxis); overload; virtual; cdecl; abstract;
    procedure Scale(sX, sY, sZ: Single; Accumulative: Boolean); overload; virtual; cdecl; abstract;
    procedure Scale(const Factor: TFactor; Accumulative: Boolean);  overload; virtual; cdecl; abstract;
    procedure Translate(Tx, Ty, Tz: Single; Accumulative: Boolean); overload; virtual; cdecl; abstract;
    procedure Translate(const Factor: TFactor; Accumulative: Boolean); overload; virtual; cdecl; abstract;
  end;

  TFigureInstance = class(TGCBase)
  public
    procedure Bounds(var Bounds: TBoundingBox); virtual; cdecl; abstract;
    function ContainsPoint(const X, Y: Single): Boolean; virtual; cdecl; abstract;
    function Overlaps(const Box: TBoundingBox): Boolean; virtual; cdecl; abstract;
    procedure Render; virtual; cdecl; abstract;
    procedure Rotate(Angle, Rx, Ry, Rz: Single); virtual; cdecl; abstract;
    procedure RotateV(Angle: Single; const Axis:TFactor); virtual; cdecl; abstract;
    procedure Scale(Sx, Sy, Sz: Single; Accumulative: Boolean = False); overload; virtual; cdecl; abstract;
    procedure Scale(const Factor: TFactor; Accumulative: Boolean = False); overload; virtual; cdecl; abstract;
    function Selected: Boolean; virtual; cdecl; abstract;
    procedure Translate(Tx, Ty, Tz: Single;  Accumulative: Boolean = False); overload; virtual; cdecl; abstract;
    procedure Translate(const Factor: TFactor; Accumulative: Boolean = False); overload; virtual; cdecl; abstract;
  end;

  TLayer = class(TGCBase)
  protected
    procedure RenderLayerContent; virtual; cdecl; abstract;
    procedure ValidateLayerContent; virtual; cdecl; abstract;
  public
    procedure AddInstance(Instance: TFigureInstance); virtual; cdecl; abstract;
    procedure BeginUpdate; virtual; cdecl; abstract;
    function Canvas: TGenericCanvas; overload; virtual; cdecl; abstract;
    procedure Canvas(NewCanvas: TGenericCanvas); overload; virtual; cdecl; abstract;
    procedure Clear; virtual; cdecl; abstract;
    function CreateInstance(Figure: TFigure): TFigureInstance; virtual; cdecl; abstract;
    function Enabled: Boolean; overload; virtual; cdecl; abstract;
    procedure Enabled(IsEnabled: Boolean); overload; virtual; cdecl; abstract;
    procedure EndUpdate; virtual; cdecl; abstract;
    function IsUpdating: Boolean; virtual; cdecl; abstract;
    procedure RemoveInstance(Instance: TFigureInstance); virtual; cdecl; abstract;
    procedure Render; virtual; cdecl; abstract;
    procedure Translate(Tx, Ty, Tz: Single; Accumulative: Boolean = False); overload; virtual; cdecl; abstract;
    procedure Translate(const Factor: TFactor; Accumulative: Boolean = False); overload; virtual; cdecl; abstract;
    procedure Scale(Sx, Sy, Sz: Single; Accumulative: Boolean = False); overload; virtual; cdecl; abstract;
    procedure Scale(const Factor: TFactor; Accumulative: Boolean = False); overload; virtual; cdecl; abstract;
    function Visible: Boolean; overload; virtual; cdecl; abstract;
    procedure Visible(IsVisible: Boolean); overload; virtual; cdecl; abstract;
  end;

  THitResults = class
  public
    procedure _CHitResults; virtual; cdecl; abstract; // The destructor of the class. Never call that.

    function Count: Integer; virtual; cdecl; abstract;
    function HasNext: Boolean; virtual; cdecl; abstract;
    function Next: TFigureInstance; virtual; cdecl; abstract;
    procedure Release; virtual; cdecl; abstract;
    procedure Reset; virtual; cdecl; abstract;
  end;

  // The TFigureInstanceEnumerator class is for quick access to all figure instances on all (common) layers.
  // Enumeration happens depth-first. That means for each layer first all instances are enumerated before the next layer
  // is taken.
  TFigureInstanceEnumerator = class
  public
    function HasNext: Boolean; virtual; cdecl; abstract;
    function Next: TFigureInstance; virtual; cdecl; abstract;
    procedure Release; virtual; cdecl; abstract;
    procedure Reset; virtual; cdecl; abstract;
  end;

  TGridLayer = class
  end;

  TView = class(TGCBase)
  public
    procedure AddLayer(Layer: TLayer); virtual; cdecl; abstract;
    procedure Color(Red, Green, Blue, Alpha: Single); overload; virtual; cdecl; abstract;
    procedure Color(NewColor: array of Single); overload; virtual; cdecl; abstract;
    function Contains(Layer: TLayer): Boolean; virtual; cdecl; abstract;
    function GetHitTestInfo(X, Y: Single; SingleHit: Boolean): THitResults; virtual; cdecl; abstract;
    function Grid: TGridLayer; virtual; cdecl; abstract;
    function Jitter: Single; overload; virtual; cdecl; abstract;
    procedure Jitter(Value: Single); overload; virtual; cdecl; abstract;
    function OffsetX: Single; overload; virtual; cdecl; abstract;
    procedure OffsetX(Value: Single); overload; virtual; cdecl; abstract;
    function OffsetY: Single; overload; virtual; cdecl; abstract;
    procedure OffsetY(Value: Single); overload; virtual; cdecl; abstract;
    procedure RemoveLayer(Layer: TLayer); virtual; cdecl; abstract;
    function Viewport: TViewport; overload; virtual; cdecl; abstract;
    procedure Viewport(const NewViewport: TViewport); overload; virtual; cdecl; abstract;
    function ZoomX: Single; overload; virtual; cdecl; abstract;
    procedure ZoomX(Value: Single); overload; virtual; cdecl; abstract;
    function ZoomY: Single; overload; virtual; cdecl; abstract;
    procedure ZoomY(Value: Single); overload; virtual; cdecl; abstract;
  end;

  TGenericCanvas = class(TGCBase)
  public
    procedure AddLayer(const Layer: TLayer); virtual; cdecl; abstract;
    function AddLayoutsFromFile(FileName: PChar): TGCError; virtual; cdecl; abstract;
    function AddStylesFromFile(FileName: PChar): TGCError; virtual; cdecl; abstract;
    procedure AddToSelection(Instance: TFigureInstance); virtual; cdecl; abstract;
    procedure BeginUpdate; virtual; cdecl; abstract;
    procedure CheckError; virtual; cdecl; abstract;
    procedure ClearContent; virtual; cdecl; abstract;
    procedure ClearLayouts; virtual; cdecl; abstract;
    procedure ClearSelection; virtual; cdecl; abstract;
    procedure ClearStyles; virtual; cdecl; abstract;
    function CreateFigure(TypeName, ClassName: PChar): TFigure; virtual; cdecl; abstract;
    function CreateLayer(const Name: PChar; AddToCurrentView: Boolean): TLayer; virtual; cdecl; abstract;
    function CreateView(Name: PChar): TView; virtual; cdecl; abstract;
    function CurrentView: TView; overload; virtual; cdecl; abstract;
    procedure CurrentView(View: TView); overload; virtual; cdecl; abstract;
    function DoAction(Instance: TFigureInstance; const X, Y: Single): Boolean; virtual; cdecl; abstract;
    procedure EndUpdate; virtual; cdecl; abstract;
    function GetFigureInstanceEnumerator: TFigureInstanceEnumerator; virtual; cdecl; abstract;
    function GetSelectionInfo(X, Y: Single): TGCDirection; virtual; cdecl; abstract;
    procedure InvalidateSelectionBounds(Instance: TFigureInstance); virtual; cdecl; abstract;
    function IsUpdating: Boolean; virtual; cdecl; abstract;
    function LayerByName(Name: PChar): TLayer; virtual; cdecl; abstract;
    procedure RemoveFromSelection(Instance: TFigureInstance); virtual; cdecl; abstract;
    procedure RemoveLayer(Layer: TLayer); virtual; cdecl; abstract;
    procedure RemoveView(View: TView); virtual; cdecl; abstract;
    procedure Render; virtual; cdecl; abstract;
    procedure ResizeFiguresStart(X, Y: Integer; Direction: TGCDirection); virtual; cdecl; abstract;
    procedure ResizeFiguresStop; virtual; cdecl; abstract;
    procedure ResizeFiguresTo(X, Y: Integer); virtual; cdecl; abstract;
    procedure RubberbandResize(X, Y: Integer; Action: TRBSelectionAction); virtual; cdecl; abstract;
    procedure RubberbandStart(Style: TRubberbandStyle; X, Y: Integer; ClearSelection: Boolean); virtual; cdecl; abstract;
    procedure RubberbandStop; virtual; cdecl; abstract;
    procedure ShowSelection(Visible: Boolean); virtual; cdecl; abstract;
    function ViewByName(Name: PChar): TView; virtual; cdecl; abstract;
  end;

  TErrorEvent = procedure(Sender: TObject; const Message: string) of object;
  TChangeEvent = procedure(Sender, Source: TObject; Reason: TGCChangeReason) of object;

  TCanvasViewerOptions = set of
  (
    cvoAlwaysShowScrollbars,           // Always show the scrollbars, even if not needed for scrolling.
    cvoAutoCenterCanvas,               // Center viewer content (according to base size and zoom factor) automatically.
    cvoAutoCenterZoom                  // When zooming make the zoom center point also the window center.
  );

  // Flags that control the inner working of the viewer.
  TGCViewerStates = set of
  (
    vsDrawSelPending,         // Selection with a selection rectangle is about to start.
    vsDrawSelecting,          // Selection with a selection rectangle is in progress.
    vsGrabPanning,            // Panning using Ctlr+Alt+Space is in progress.
    vsGrabPanningPending,     // Panning using Ctlr+Alt+Space is about to begin (left mouse button must yet be down).
    vsRubberband,             // A rubber band is currently shown.
    vsScrollPending,          // Auto scrolling is about to start.
    vsScrolling,              // Auto scrolling (also wheel panning) is in progress. 
    vsWheelPanning,           // Wheel mouse panning is active or soon will be.
    vsWheelScrolling,         // Wheel mouse scrolling is active or soon will be.
    vsZoomInPending,          // Ctrl+Space is pressed. Next mouse down zooms one step in.
    vsZoomOutPending          // Ctrl+Alt+Space is pressed. Next mouse down zooms one step out.
  );

  // Certain keyboard and mouse states must be tracked because they trigger special behavior.
  TGCKeyStates = set of
  (
    ksAlt,              // Alt key is down.
    ksCtrl,             // Ctrl key is down.
    ksLeftButtonDown,   // Left mouse button is down.
    ksMiddleButtonDown, // Middle mouse button is down.
    ksRightButtonDown,  // Right mouse button is down.
    ksShift,            // Shift key is down.
    ksSpace             // Space key is down.
  );

    // Limits the speed interval which can be used for auto scrolling (milliseconds).
  TAutoScrollInterval = 1..1000;

  // Auto scroll directions.
  TScrollDirections = set of
  (
    sdLeft,
    sdUp,
    sdRight,
    sdDown
  );

  TGenericCanvasViewer = class(TWinControl)
  private
    FCanvas: TGenericCanvas;
    FDeviceContext: HDC;
    FListener: TGCListener;
    FOffset: TPoint;
    FOptions: TCanvasViewerOptions;
    FRenderingContext: HGLRC;
    FViewport: TViewport;
    FZoom: TPointD;                              // The zoom factors of the canvas.
    FBaseSize: TPoint;                           // The size of the canvas at 100% zoom.
    FLastClick: TPoint;                          // The position of the last mouse down action.
    FStates: TGCViewerStates;                    // Flags controlling how things are processed in the viewer.
    FKeyStates: TGCKeyStates;                    // Tracker for certain keyboard/mouse states which are important for the viewer.
    FCurrentZoomStep: Integer;                   // The current position in the zoom factor array.

    // Auto scrolling.
    FAutoScrollInterval: TAutoScrollInterval;    // Determines speed of auto scrolling.
    FScrollDirections: TScrollDirections;        // Directions to scroll client area into depending on mouse position.
    FAutoScrollDelay: Cardinal;                  // Amount of milliseconds to wait until autos crolling becomes active.
    FHorizontalScrollStep: Integer;              // Amount of pixel to scroll horizontally in one timer step.
    FVerticalScrollStep: Integer;                // Amount of pixel to scroll vertically in one timer step.

    // Wheel panning/scrolling support.
    FPanningWindow: HWND;                        // Helper window for wheel panning
    FPanningCursor: TCursor;                     // Current wheel panning cursor.
    FPanningImage: TBitmap;                      // A little 32x32 bitmap to indicate the panning reference point.

    // Speed computation.
    FFPS: Single;
    FTimeBuffer: array[0..FPSBufferSize - 1] of Single;
    FCounterFrequency: Int64;
    FCurrentTimeBufferIndex: Integer;
    FTotalTime: Single;

    FOnChange: TChangeEvent;
    FOnError: TErrorEvent;
    FOnSetup: TNotifyEvent;
    FOnZoomChange: TNotifyEvent;
    function GetCanvas: TGenericCanvas;
    procedure SetBaseSizeX(const Value: Integer);
    procedure SetBaseSizeY(const Value: Integer);
    procedure SetOffsetX(Value: Integer);
    procedure SetOffsetY(Value: Integer);
    procedure SetOptions(const Value: TCanvasViewerOptions);
    procedure SetZoomX(const Value: Double);
    procedure SetZoomY(const Value: Double);

    procedure CMMouseWheel(var Message: TCMMouseWheel); message CM_MOUSEWHEEL;

    procedure WMEraseBkgnd(var Message: TWMEraseBkgnd); Message WM_ERASEBKGND;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure WMHScroll(var Message: TWMHScroll); message WM_HSCROLL;
    procedure WMKeyDown(var Message: TWMKeyDown); message WM_KEYDOWN;
    procedure WMKeyUp(var Message: TWMKeyUp); message WM_KEYUP;
    procedure WMKillFocus(var Msg: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMLButtonDown(var Message: TWMLButtonDown); message WM_LBUTTONDOWN;
    procedure WMLButtonUp(var Message: TWMLButtonUp); message WM_LBUTTONUP;
    procedure WMMButtonDown(var Message: TWMMButtonDown); message WM_MBUTTONDOWN;
    procedure WMMButtonUp(var Message: TWMMButtonUp); message WM_MBUTTONUP;
    procedure WMMouseMove(var Message: TWMMouseMove); message WM_MOUSEMOVE;
    procedure WMPaint(var Message: TWMPaint); Message WM_PAINT;
    procedure WMRButtonDown(var Message: TWMRButtonDown); message WM_RBUTTONDOWN;
    procedure WMRButtonUp(var Message: TWMRButtonUp); message WM_RBUTTONUP;
    procedure WMSetCursor(var Message: TWMSetCursor); message WM_SETCURSOR;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    procedure WMSysCommand(var Message: TWMSysCommand); message WM_SYSCOMMAND;
    procedure WMSysKeyDown(var Message: TWMSysKeyDown); message WM_SYSKEYDOWN;
    procedure WMSysKeyUp(var Message: TWMSysKeyUp); message WM_SYSKEYUP;
    procedure WMTimer(var Message: TWMTimer); message WM_TIMER;
    procedure WMVScroll(var Message: TWMVScroll); message WM_VSCROLL;
  protected
    procedure AdjustPanningCursor(X, Y: Integer); virtual;
    function CanAutoScroll: Boolean; virtual;
    function CanvasExceedsClientArea: Boolean; virtual;
    procedure CheckKeys(var Message: TWMKey; Shift: TShiftState); virtual;
    procedure CheckNavigationStates;
    procedure CreateParams(var Params: TCreateParams); override;
    function CreateRenderingContext(DC: HDC; Options: TRCOptions; ColorBits, StencilBits, AccumBits, AuxBuffers: Integer;
      var Palette: HPALETTE): HGLRC; virtual;
    procedure CreateWnd; override;
    function CursorFromStates: TCursor; virtual;
    procedure DestroyContexts; virtual;
    procedure DestroyWnd; override;
    function DetermineScrollDirections(X, Y: Integer): TScrollDirections;
    procedure DoAutoScroll(X, Y: Integer); virtual;
    procedure DoChange(AObject: TObject; Reason: TGCChangeReason); virtual;
    procedure DoError(const Message: string); virtual;
    function DoFigureAction(X, Y: Integer; Shift: TShiftState): Boolean; virtual;
    procedure DoTimerScroll;
    procedure DoZoomChange(X, Y: Integer); virtual;
    procedure PanningWindowProc(var Message: TMessage); virtual;
    procedure StartWheelPanning; virtual;
    procedure StopTimer(ID: Integer); virtual;
    procedure StopWheelPanning; virtual;
    procedure UpdateScrollbars; virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    function IsMouseSelecting: Boolean;
    function LoadLayouts(FileName: string): TGCError;
    function LoadStyles(FileName: string): TGCError;
    procedure RenderScene;
    procedure SetZoom(X, Y: Double; ZoomCenter: TPoint); virtual;
    procedure ZoomIn(X, Y: Integer);
    procedure ZoomOut(X, Y: Integer);

    property BaseSizeX: Integer read FBaseSize.X write SetBaseSizeX;
    property BaseSizeY: Integer read FBaseSize.Y write SetBaseSizeY;
    property Canvas: TGenericCanvas read GetCanvas;
    property DeviceContext: HDC read FDeviceContext;
    property Font;
    property FPS: Single read FFPS;
    property MouseCapture;
    property OffsetX: Integer read FOffset.X write SetOffsetX default 0;
    property OffsetY: Integer read FOffset.Y write SetOffsetY default 0;
    property RenderingContext: HGLRC read FRenderingContext;
  published
    property Align;
    property Anchors;
    property AutoScrollDelay: Cardinal read FAutoScrollDelay write FAutoScrollDelay default 1000;
    property AutoScrollInterval: TAutoScrollInterval read FAutoScrollInterval write FAutoScrollInterval default 1;
    property Constraints;
    property DragCursor;
    property DragMode;
    property Enabled;
    property HelpContext;
    property Hint;
    property Options: TCanvasViewerOptions read FOptions write SetOptions default [cvoAutoCenterCanvas];
    property PopupMenu;
    property Visible;
    property ZoomX: Double read FZoom.X write SetZoomX;
    property ZoomY: Double read FZoom.Y write SetZoomY;

    property OnChange: TChangeEvent read FOnChange write FOnChange;
    property OnClick;
    property OnDragDrop;
    property OnDragOver;
    property OnError: TErrorEvent read FOnError write FOnError;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnSetup: TNotifyEvent read FOnSetup write FOnSetup;
    property OnZoomChange: TNotifyEvent read FOnZoomChange write FOnZoomChange;
  end;

//----------------------------------------------------------------------------------------------------------------------

implementation
                                                                                             
{$R ..\Resources\GenericCanvasViewer.res}

uses
  Math;

const
  GCLibrary = 'libmysqlgc.dll';

  ScrollMasks: array[Boolean] of Cardinal = (0, SIF_DISABLENOSCROLL);
  MaxZoomSteps = 18;
  ZoomSteps: array[0..MaxZoomSteps] of Single = (
    0.6125 / 12, 0.675 / 12, 0.75 / 12, 1 / 12, 2 / 12, 3 / 12, 4 / 12, 6 / 12, 8 / 12, 1, 2, 3, 4, 5, 6, 7, 8, 12, 16
  );

  crZoomIn         = TCursor(101);
  crZoomOut        = TCursor(102);
  crHandOpen       = TCursor(103);
  crhandClosed     = TCursor(104);
  crMoveAll        = TCursor(105);
  crMoveEast       = TCursor(106);
  crMoveEastWest   = TCursor(107);
  crMoveNorth      = TCursor(108);
  crMoveNorthEast  = TCursor(109);
  crMoveNorthSouth = TCursor(110);
  crMoveNorthWest  = TCursor(111);
  crMoveSouth      = TCursor(112);
  crMoveSouthEast  = TCursor(113);
  crMoveSouthWest  = TCursor(114);
  crMoveWest       = TCursor(115);

function CreateGenericCanvas(Context: HDC; Name: PChar): TGenericCanvas; cdecl; external GCLibrary;

//----------------------------------------------------------------------------------------------------------------------

function SetupPalette(DC: HDC; PFD: TPixelFormatDescriptor): HPalette;

var
  nColors,
  I: Integer;
  LogPalette: TMaxLogPalette;
  RedMask,
  GreenMask,
  BlueMask: Byte;

begin
  nColors := 1 shl Pfd.cColorBits;
  LogPalette.palVersion := $300;
  LogPalette.palNumEntries := nColors;
  RedMask := (1 shl Pfd.cRedBits  ) - 1;
  GreenMask := (1 shl Pfd.cGreenBits) - 1;
  BlueMask := (1 shl Pfd.cBlueBits ) - 1;
  with LogPalette, PFD do
    for I := 0 to nColors - 1 do
    begin
      palPalEntry[I].peRed := (((I shr cRedShift  ) and RedMask  ) * 255) div RedMask;
      palPalEntry[I].peGreen := (((I shr cGreenShift) and GreenMask) * 255) div GreenMask;
      palPalEntry[I].peBlue := (((I shr cBlueShift ) and BlueMask ) * 255) div BlueMask;
      palPalEntry[I].peFlags := 0;
    end;

  Result := CreatePalette(PLogPalette(@LogPalette)^);
  if Result <> 0 then
  begin
    SelectPalette(DC, Result, False);
    RealizePalette(DC);
  end
  else
    RaiseLastOSError;
end;

//----------------- TGCListener ----------------------------------------------------------------------------------------

constructor TGCListener.Create(Viewer: TGenericCanvasViewer);

begin
  FViewer := Viewer;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGCListener.OnDestroy(Sender: TGCBase);

begin

end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGCListener.OnChange(Sender, Origin: TGCBase; Reason: TGCChangeReason);

// Triggered by the canvas for regular changes.

begin
  FViewer.DoChange(Origin, Reason);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGCListener.OnError(Sender: TGCBase; const Message: PChar);

begin
  FViewer.DoError(Message);
end;

//----------------- TGenericCanvasViewer -------------------------------------------------------------------------------

constructor TGenericCanvasViewer.Create(AOwner: TComponent);

begin
  inherited;

  // We need csMenuEvents here to make the VCL send us the WM_SYSCOMMAND message, which we need to disable Alt+Space.
  ControlStyle := ControlStyle + [csMenuEvents];
  Width := 150;
  Height := 150;
  FListener := TGCListener.Create(Self);
  FZoom.X := 1;
  FZoom.Y := 1;
  FOptions := [cvoAutoCenterCanvas];
  FCurrentZoomStep := 3;
  FAutoScrollInterval := 1;
  FAutoScrollDelay := 1000;
  FHorizontalScrollStep := 10;
  FVerticalScrollStep := 10;

  QueryPerformanceFrequency(FCounterFrequency);
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TGenericCanvasViewer.Destroy;

begin
  StopWheelPanning;
  
  if Assigned(FCanvas) then
  begin
    FCanvas.RemoveListener(FListener);
    FCanvas.Release;
  end;
  FListener.Free;

  DestroyContexts;

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGenericCanvasViewer.GetCanvas: TGenericCanvas;

begin
  HandleNeeded;
  Result := FCanvas;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.SetBaseSizeX(const Value: Integer);

begin
  if FBaseSize.X <> Value then
  begin
    FBaseSize.X := Value;
    if HandleAllocated then
      UpdateScrollbars;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.SetBaseSizeY(const Value: Integer);

begin
  if FBaseSize.Y <> Value then
  begin
    FBaseSize.Y := Value;
    if HandleAllocated then
      UpdateScrollbars;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.SetOffsetX(Value: Integer);

begin
  if FOffset.X <> Value then
  begin
    FOffset.X := Value;
    if HandleAllocated then
      UpdateScrollbars;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.SetOffsetY(Value: Integer);

begin
  if FOffset.Y <> Value then
  begin
    FOffset.Y := Value;
    if HandleAllocated then
      UpdateScrollbars;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.SetOptions(const Value: TCanvasViewerOptions);

begin
  FOptions := Value;
  UpdateScrollbars;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.SetZoomX(const Value: Double);

begin
  SetZoom(Value, FZoom.Y, Point(ClientWidth div 2, ClientHeight div 2));
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.SetZoomY(const Value: Double);

begin
  SetZoom(FZoom.X, Value, Point(ClientWidth div 2, ClientHeight div 2));
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.CMMouseWheel(var Message: TCMMouseWheel);

var
  ScrollCount: Integer;
  ScrollLines: Integer;

begin
  StopWheelPanning;
  
  inherited;

  if Message.Result = 0  then
  begin
    with Message do
    begin
      Result := 1;
      if ssAlt in ShiftState then
      begin
        if WheelDelta > 0 then
          ZoomOut(ClientWidth div 2, ClientHeight div 2)
        else
          ZoomIn(ClientWidth div 2, ClientHeight div 2);
      end
      else
        if ssCtrl in ShiftState then
        begin
          if WheelDelta > 0 then
            ZoomOut(XPos, YPos)
          else
            ZoomIn(XPos, YPos);
        end
        else

      if FBaseSize.Y * FZoom.X >  ClientHeight then
      begin
        // Scroll vertically if there's something to scroll...
        SystemParametersInfo(SPI_GETWHEELSCROLLLINES, 0, @ScrollLines, 0);
        if Cardinal(ScrollLines) = WHEEL_PAGESCROLL then
          ScrollCount := WheelDelta div WHEEL_DELTA
        else
          ScrollCount := ScrollLines * WheelDelta div WHEEL_DELTA;
        OffsetY := FOffset.Y + ScrollCount * FVerticalScrollStep;
      end
      else
      begin
        // ...else scroll horizontally.
        ScrollCount := WheelDelta div WHEEL_DELTA;
        OffsetX := FOffset.X + ScrollCount * FHorizontalScrollStep;
      end;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMEraseBkgnd(var Message: TWMEraseBkgnd);

begin
  Message.Result := 1;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMGetDlgCode(var Message: TWMGetDlgCode);

begin
  Message.Result := DLGC_WANTCHARS or DLGC_WANTARROWS;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMKeyDown(var Message: TWMKeyDown);

var
  Shift: TShiftState;

begin
  inherited;

  Shift := KeyDataToShiftState(Message.KeyData);
  CheckKeys(Message, Shift);

  case Message.CharCode of
    VK_HOME:
      begin
        OffsetX := 0;
        OffsetY := 0;
      end;
    VK_END:
      begin
        OffsetX := -MaxInt;
        OffsetY := -MaxInt;
      end;
    VK_PRIOR:
      OffsetY := FOffset.Y + ClientHeight;
    VK_NEXT:
      OffsetY := FOffset.Y - ClientHeight;
    VK_UP:
      OffsetY := FOffset.Y + FVerticalScrollStep;
    VK_DOWN:
      OffsetY := FOffset.Y - FVerticalScrollStep;
    VK_LEFT:
      OffsetX := FOffset.X + FHorizontalScrollStep;
    VK_RIGHT:
      OffsetX := FOffset.X - FHorizontalScrollStep;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMKeyUp(var Message: TWMKeyUp);

var
  Shift: TShiftState;

begin
  inherited;

  Shift := KeyDataToShiftState(Message.KeyData);
  CheckKeys(Message, Shift);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMKillFocus(var Msg: TWMKillFocus);

begin
  StopWheelPanning;

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMHScroll(var Message: TWMHScroll);

  //--------------- local functions -------------------------------------------

  function GetRealScrollPosition: Integer;

  var
    SI: TScrollInfo;
    Code: Integer;

  begin
    SI.cbSize := SizeOf(TScrollInfo);
    SI.fMask := SIF_TRACKPOS;
    Code := SB_HORZ;
    GetScrollInfo(Handle, Code, SI);
    Result := SI.nTrackPos;
  end;

  //--------------- end local functions ---------------------------------------

begin
  case Message.ScrollCode of
    SB_BOTTOM:
      OffsetX := Round(-(FZoom.X - 1) * ClientWidth);
    SB_ENDSCROLL:
      UpdateScrollBars;
    SB_LINELEFT:
      OffsetX := FOffset.X + FHorizontalScrollStep;
    SB_LINERIGHT:
      OffsetX := FOffset.X - FHorizontalScrollStep;
    SB_PAGELEFT:
      OffsetX := FOffset.X + ClientWidth;
    SB_PAGERIGHT:
      OffsetX := FOffset.X - ClientWidth;
    SB_THUMBPOSITION,
    SB_THUMBTRACK:
      OffsetX := -GetRealScrollPosition;
    SB_TOP:
      OffsetX := 0;
  end;

  Message.Result := 0;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMLButtonUp(var Message: TWMLButtonUp);

begin
  Exclude(FKeyStates, ksLeftButtonDown);
  CheckNavigationStates;

  if vsRubberband in FStates then
  begin
    Exclude(FStates, vsRubberband);
    FCanvas.RubberbandStop;
  end;
  
  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMLButtonDown(var Message: TWMLButtonDown);

var
  Shift: TShiftState;
  HadAction: Boolean;
  
begin
  if [vsWheelPanning, vsWheelScrolling] * FStates <> [] then
    StopWheelPanning
  else
  begin
    if not Focused then
      SetFocus;

    with Message do
    begin
      FLastClick.X := XPos;
      FLastClick.Y := YPos;
      Include(FKeyStates, ksLeftButtonDown);
      CheckNavigationStates;

      if vsZoomInPending in FStates then
        ZoomIn(XPos, YPos)
      else
        if vsZoomOutPending in FStates then
          ZoomOut(XPos, YPos)
        else
          if not (vsGrabPanning in FStates) then
          begin
            Shift := KeysToShiftState(Keys) * [ssCtrl, ssShift];
            HadAction := DoFigureAction(XPos, YPos, Shift);
            if not HadAction then
            begin
              Include(FStates, vsRubberband);
              FCanvas.RubberbandStart(GC_RBSTYLE_BLENDED_DIAGONALS, XPos, YPos, Shift = []);
            end;
          end;
    end;

    inherited;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMMButtonDown(var Message: TWMMButtonDown);

begin
  inherited;

  Include(FKeyStates, ksMiddleButtonDown);

  // Start wheel panning or scrolling if not already active, allowed and scrolling is useful at all.
  if ([vsWheelScrolling, vsWheelPanning] * FStates = []) and CanvasExceedsClientArea then
  begin
    FLastClick := SmallPointToPoint(Message.Pos);
    StartWheelPanning;
  end
  else
    StopWheelPanning;
  CheckNavigationStates;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMMButtonUp(var Message: TWMMButtonUp);

begin
  Exclude(FKeyStates, ksMiddleButtonDown);

  // If wheel panning/scrolling is active and the mouse has not yet been moved then the user starts wheel auto scrolling.
  // Indicate this by removing the panning flag. Otherwise (the mouse has moved meanwhile) stop panning.
  if [vsWheelPanning, vsWheelScrolling] * FStates <> [] then
  begin
    if vsWheelScrolling in FStates then
      Exclude(FStates, vsWheelPanning)
    else
      StopWheelPanning;
  end
  else
    inherited;
    
  CheckNavigationStates;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMMouseMove(var Message: TWMMouseMove);

var
  dX, dY: Integer;
  Action: TRBSelectionAction;

begin
  with Message do
  begin
    // If both wheel panning and auto scrolling are pending then the user moved the mouse while holding down the
    // middle mouse button. This means panning is being used, hence remove the wheel scroll flag.
    if [vsWheelPanning, vsWheelScrolling] * FStates = [vsWheelPanning, vsWheelScrolling] then
    begin
      if ((Abs(FLastClick.X - XPos) >= Mouse.DragThreshold) or (Abs(FLastClick.Y - YPos) >= Mouse.DragThreshold)) then
        Exclude(FStates, vsWheelScrolling);
    end;

    if CanAutoScroll then
      DoAutoScroll(XPos, YPos);
    if [vsWheelPanning, vsWheelScrolling] * FStates <> [] then
      AdjustPanningCursor(XPos, YPos)
    else
    begin
      if vsGrabPanning in FStates then
      begin
        dX := XPos - FlastClick.X;
        FLastClick.X := XPos;
        dY := YPos -FlastClick.Y;
        FLastClick.Y := YPos;

        Inc(FOffset.X, dX);
        Inc(FOffset.Y, dY);
        UpdateScrollbars;
      end
      else
        if vsRubberband in FStates then
        begin
          if ksCtrl in FKeyStates then
            Action := GC_RBACTION_TOGGLE
          else
            if ksShift in FKeyStates then
              Action := GC_RBACTION_SELECT
            else
              Action := GC_RBACTION_SELECT_REMOVE;
          FCanvas.RubberbandResize(XPos, YPos, Action);
        end;
    end;
  end;

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMPaint(var Message: TWMPaint);

var
  PS: TPaintStruct;

begin
  BeginPaint(Handle, PS);
  RenderScene;
  EndPaint(Handle, PS);
  Message.Result := 0;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMRButtonDown(var Message: TWMRButtonDown);

begin
  if [vsWheelPanning, vsWheelScrolling] * FStates <> [] then
    StopWheelPanning
  else
  begin
    inherited;

    Include(FKeyStates, ksRightButtonDown);
    CheckNavigationStates;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMRButtonUp(var Message: TWMRButtonUp);

begin
  Exclude(FKeyStates, ksRightButtonDown);
  CheckNavigationStates;

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMSetCursor(var Message: TWMSetCursor);

// Sets a new cursor for the viewer.

var
  NewCursor: TCursor;
  P: TPoint;
  Direction: TGCDirection;

begin
  with Message do
  begin
    if (CursorWnd = Handle) and not (csDesigning in ComponentState) and Assigned(FCanvas)
      and ([vsWheelPanning, vsWheelScrolling] * FStates = []) then
    begin
      // Apply own cursors only if there is no global cursor set.
      if Screen.Cursor = crDefault then
      begin
        NewCursor := CursorFromStates;
        if NewCursor = crDefault then
        begin
          GetCursorPos(P);
          with ScreenToClient(P) do
            Direction := FCanvas.GetSelectionInfo(X, Y);

          case Direction of
            GC_SI_ON_OBJECT:
              NewCursor := crSizeAll;
            GC_SI_SOUTH,
            GC_SI_NORTH:
              NewCursor := crSizeNS;
            GC_SI_NORTH_EAST,
            GC_SI_SOUTH_WEST:
              NewCursor := crSizeNESW;
            GC_SI_EAST,
            GC_SI_WEST:
              NewCursor := crSizeWE;
            GC_SI_SOUTH_EAST,
            GC_SI_NORTH_WEST:
              NewCursor := crSizeNWSE;
          else
            NewCursor := crDefault;
          end;
        end;
        Windows.SetCursor(Screen.Cursors[NewCursor]);
        Message.Result := 1;
      end
      else
        inherited;
    end
    else
      inherited;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMSize(var Message: TWMSize);

begin
  inherited;
  
  // Update viewport (here always the entire window).
  with FViewport do
  begin
    Width := Message.Width;
    Height := Message.Height;
    if Height = 0 then
      Height := 1;
  end;

  if Assigned(FCanvas) then
  begin
    FHorizontalScrollStep := ClientWidth div 20;
    FVerticalScrollStep := ClientHeight div 20;

    if FCanvas.CurrentView <> nil then
      FCanvas.CurrentView.Viewport(FViewport);
    UpdateScrollbars;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMSysCommand(var Message: TWMSysCommand);

begin
  if Message.CmdType = SC_KEYMENU then
    Message.Result := 1
  else
    inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMSysKeyDown(var Message: TWMSysKeyDown);

var
  Shift: TShiftState;

begin
  Shift := KeyDataToShiftState(Message.KeyData);
  CheckKeys(Message, Shift);

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMSysKeyUp(var Message: TWMSysKeyUp);

var
  Shift: TShiftState;

begin
  Shift := KeyDataToShiftState(Message.KeyData);
  CheckKeys(Message, Shift);
  
  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMTimer(var Message: TWMTimer);

// Centralized timer handling happens here.

begin
  with Message do
  begin
    case TimerID of
      ScrollTimer:
        begin
          if vsScrollPending in FStates then
          begin
            Application.CancelHint;
            // Scroll delay has elapsed, set to normal scroll interval now.
            SetTimer(Handle, ScrollTimer, FAutoScrollInterval, nil);
            FStates := FStates + [vsScrolling] - [vsScrollPending];
          end;
          DoTimerScroll;
        end;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.WMVScroll(var Message: TWMVScroll);

  //---------------------------------------------------------------------------

  function GetRealScrollPosition: Integer;

  var
    SI: TScrollInfo;

  begin
    SI.cbSize := SizeOf(TScrollInfo);
    SI.fMask := SIF_TRACKPOS;
    GetScrollInfo(Handle, SB_VERT, SI);
    Result := SI.nTrackPos;
  end;

  //---------------------------------------------------------------------------

begin
  case Message.ScrollCode of
    SB_BOTTOM:
      OffsetY := Round(-(FZoom.Y - 1) * ClientHeight);
    SB_ENDSCROLL:
      UpdateScrollBars;
    SB_LINEUP:
      OffsetY := FOffset.Y + FVerticalScrollStep;
    SB_LINEDOWN:
      OffsetY := FOffset.Y - FVerticalScrollStep;
    SB_PAGEUP:
      OffsetY := FOffset.Y + ClientHeight;
    SB_PAGEDOWN:
      OffsetY := FOffset.Y - ClientHeight;
    SB_THUMBPOSITION,
    SB_THUMBTRACK:
      OffsetY := -GetRealScrollPosition;
    SB_TOP:
      OffsetY := 0;
  end;
  Message.Result := 0;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.AdjustPanningCursor(X, Y: Integer);

// Triggered by a mouse move when wheel panning/scrolling is active.
// Loads the proper cursor which indicates into which direction scrolling is done.

var
  NewCursor: TCursor;
  ScrollHorizontal,
  ScrollVertical: Boolean;

begin
  ScrollHorizontal := Round(FZoom.X * FBaseSize.X) > ClientWidth;
  ScrollVertical := Round(FZoom.Y * FBaseSize.Y) > ClientHeight;

  if (Abs(X - FLastClick.X) < 8) and (Abs(Y - FLastClick.Y) < 8) then
  begin
    // Mouse is in the neutral zone.
    if ScrollHorizontal then
    begin
      if ScrollVertical then
        NewCursor := crMoveAll
      else
        NewCursor := crMoveEastWest;
    end
    else
      NewCursor := crMoveNorthSouth;
  end
  else
  begin
    // One of 8 directions applies: north, north-east, east, south-east, south, south-west, west and north-west.
    // Check also if scrolling in the particular direction is possible.
    if ScrollVertical and ScrollHorizontal then
    begin
      // All directions allowed.
      if X - FLastClick.X < -8 then
      begin
        // Left hand side.
        if Y - FLastClick.Y < -8 then
          NewCursor := crMoveNorthWest
        else
          if Y - FLastClick.Y > 8 then
            NewCursor := crMoveSouthWest
          else
            NewCursor := crMoveWest;
      end
      else
        if X - FLastClick.X > 8 then
        begin
          // Right hand side.
          if Y - FLastClick.Y < -8 then
            NewCursor := crMoveNorthEast
          else
            if Y - FLastClick.Y > 8 then
              NewCursor := crMoveSouthEast
            else
              NewCursor := crMoveEast;
        end
        else
        begin
          // Up or down.
          if Y < FLastClick.Y then
            NewCursor := crMoveNorth
          else
            NewCursor := crMoveSouth;
        end;
    end
    else
      if ScrollHorizontal then
      begin
        // Only horizontal movement allowed.
        if X < FLastClick.X then
          NewCursor := crMoveWest
        else
          NewCursor := crMoveEast;
      end
      else
      begin
        // Only vertical movement allowed.
        if Y < FLastClick.Y then
          NewCursor := crMoveNorth
        else
          NewCursor := crMoveSouth;
      end;
  end;

  if FPanningCursor <> NewCursor then
  begin
    FPanningCursor := NewCursor;
    Windows.SetCursor(Screen.Cursors[FPanningCursor]);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGenericCanvasViewer.CanAutoScroll: Boolean;

// Determines if auto scrolling is currently allowed.

var
  IsDropTarget: Boolean;
  IsDrawSelecting: Boolean;
  IsWheelPanning: Boolean;

begin
  // Do auto scroll only if there is a draw selection in progress or the tree is the current drop target or
  // wheel panning/scrolling is active.
  IsDropTarget := False; //Assigned(FDragManager) and DragManager.IsDropTarget;
  IsDrawSelecting := [vsDrawSelPending, vsDrawSelecting] * FStates <> [];
  IsWheelPanning := [vsWheelPanning, vsWheelScrolling] * FStates <> [];
  Result := IsDrawSelecting or IsDropTarget or IsWheelPanning;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGenericCanvasViewer.CanvasExceedsClientArea: Boolean;

// Determines if the canvas (with the current zoom state) is larger than the client area (in any direction) and
// returns True if so.

begin
  Result := (Round(FZoom.X * FBaseSize.X) > ClientWidth) or (Round(FZoom.Y * FBaseSize.Y) > ClientHeight);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.CheckKeys(var Message: TWMKey; Shift: TShiftState);

// Checks the given key message record and updates the internal navigation states.

var
  Down: Boolean;
  NeedStateCheck: Boolean;

begin
  NeedStateCheck := False;
  Down := (Message.KeyData and $80000000) = 0;

  // Modifier keys are usually not passed in if they have been released.
  if (ssCtrl in Shift) <> (ksCtrl in FKeyStates) then
  begin
    NeedStateCheck := True;
    if ssCtrl in Shift then
      Include(FKeyStates, ksCtrl)
    else
      Exclude(FKeyStates, ksCtrl);
  end;

  if (ssShift in Shift) <> (ksShift in FKeyStates) then
  begin
    NeedStateCheck := True;
    if ssShift in Shift then
      Include(FKeyStates, ksShift)
    else
      Exclude(FKeyStates, ksShift);
  end;

  if (ssAlt in Shift) <> (ksAlt in FKeyStates) then
  begin
    NeedStateCheck := True;
    if ssAlt in Shift then
      Include(FKeyStates, ksAlt)
    else
      Exclude(FKeyStates, ksAlt);
  end;

  // Normal keys are usually also passed in if they have been released.
  case Message.CharCode of
    VK_SPACE:
      begin
        if Down <> (ksSpace in FKeyStates) then
        begin
          NeedStateCheck := True;
          if Down then
            Include(FKeyStates, ksSpace)
          else
            Exclude(FKeyStates, ksSpace);
        end;
      end;
  end;

  if NeedStateCheck then
    CheckNavigationStates;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.CheckNavigationStates;

// Updates the internal working states depending on keyboard and mouse states.

var
  NewCursor: TCursor;

begin
  FStates := FStates - [vsZoomInPending, vsZoomOutPending, vsGrabPanningPending, vsGrabPanning];

  if ksSpace in FKeyStates then
  begin
    // Possible states: grab panning (pending), zoom in/out.
    // If neither ctrl nor alt nor shift are pressed then we are in grab panning mode.
    if (FKeyStates - [ksSpace, ksLeftButtonDown]) = [] then
    begin
      if FKeyStates = [ksSpace] then
        Include(FStates, vsGrabPanningPending)
      else
        Include(FStates, vsGrabPanning);
    end
    else
    begin
      if ksAlt in FKeyStates then
        Include(FStates, vsZoomOutPending)
      else
        if ksCtrl in FKeyStates then
          Include(FStates, vsZoomInPending);
    end;
  end;

  // Set new cursor if necessary.
  if (Screen.Cursor = crDefault) and ([vsWheelPanning, vsWheelScrolling] * FStates = []) then
  begin
    NewCursor := CursorFromStates;
    Windows.SetCursor(Screen.Cursors[NewCursor]);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.CreateParams(var Params: TCreateParams);

begin
  inherited;

  with Params do
  begin
    Style := Style or WS_CLIPCHILDREN or WS_CLIPSIBLINGS;
    WindowClass.Style :=  CS_VREDRAW or CS_HREDRAW or CS_OWNDC;
  end
end;

//----------------------------------------------------------------------------------------------------------------------

function TGenericCanvasViewer.CreateRenderingContext(DC: HDC; Options: TRCOptions; ColorBits, StencilBits, AccumBits,
  AuxBuffers: Integer; var Palette: HPALETTE): HGLRC;

// Set the OpenGL properties required to draw to the given canvas and create a rendering context for it.

const
  MemoryDCs = [OBJ_MEMDC, OBJ_METADC, OBJ_ENHMETADC];

var
  PFDescriptor: TPixelFormatDescriptor;
  PixelFormat: Integer;
  AType: DWORD;

begin
  FillChar(PFDescriptor, SizeOf(PFDescriptor), 0);
  with PFDescriptor do
  begin
    nSize := SizeOf(PFDescriptor);
    nVersion := 1;
    dwFlags := PFD_SUPPORT_OPENGL;
    AType := GetObjectType(DC);
    if AType = 0 then
      RaiseLastOSError;

    if AType in MemoryDCs then
      dwFlags := dwFlags or PFD_DRAW_TO_BITMAP
    else
      dwFlags := dwFlags or PFD_DRAW_TO_WINDOW;
    if opDoubleBuffered in Options then
      dwFlags := dwFlags or PFD_DOUBLEBUFFER;
    if opGDI in Options then
      dwFlags := dwFlags or PFD_SUPPORT_GDI;
    if opStereo in Options then
      dwFlags := dwFlags or PFD_STEREO;
    iPixelType := PFD_TYPE_RGBA;
    cColorBits := ColorBits;
    cDepthBits := 32;
    cStencilBits := StencilBits;
    cAccumBits := AccumBits;
    cAuxBuffers := AuxBuffers;
    iLayerType := PFD_MAIN_PLANE;
  end;

  PixelFormat := ChoosePixelFormat(DC, @PFDescriptor);
  if PixelFormat = 0 then
    RaiseLastOSError;

  // NOTE: It is not allowed to change a pixel format of a device context once it has been set.
  //       Hence you may create more than one rendering context for one single device only if it
  //       uses the same pixel format as the first created RC.
  if GetPixelFormat(DC) <> PixelFormat then
  begin
    if not SetPixelFormat(DC, PixelFormat, @PFDescriptor) then
      RaiseLastOSError;
  end;

  // Check the properties we just set.
  DescribePixelFormat(DC, PixelFormat, SizeOf(PFDescriptor), PFDescriptor);
  with PFDescriptor do
    if (dwFlags and PFD_NEED_PALETTE) <> 0 then
      Palette := SetupPalette(DC, PFDescriptor)
    else
      Palette := 0;

  Result := wglCreateContext(DC);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.CreateWnd;

var
  Dummy: HPALETTE;

begin
  inherited;

  FDeviceContext := GetDC(Handle);
  FRenderingContext := CreateRenderingContext(FDeviceContext, [opDoubleBuffered], 32, 0, 0, 0, Dummy);
  if FRenderingContext = 0 then
    raise Exception.Create('Creation of OpenGL rendering context failed.'#10 + SysErrorMessage(GetLastError));

  // Activate the new rendering context. It will stay active for the entire lifetime of the window.
  // We have a private DC that ensures our settings do not get destroyed between paint calls.
  wglMakeCurrent(FDeviceContext, FRenderingContext);

  FCanvas := CreateGenericCanvas(FDeviceContext, '');
  FCanvas.AddListener(FListener);
  Canvas.CreateView('Default');

  // Set the viewport to the entire window size.
  with FViewPort do
  begin
    Left := 0;
    Top := 0;
    Width := Self.Width;
    Height := Self.Height;
    if FCanvas.CurrentView <> nil then
      FCanvas.CurrentView.Viewport(FViewport);
  end;

  if FBaseSize.X = 0 then
    FBaseSize.X := Width;
  if FBaseSize.Y = 0 then
  FBaseSize.Y := Height;

  DoZoomChange(ClientWidth div 2, ClientHeight div 2);
  UpdateScrollbars;
  if Assigned(FOnSetup) then
    FOnSetup(Self);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGenericCanvasViewer.CursorFromStates: TCursor;

// Checks the current state and returns an appropriate cursor for it.
// Default is crDefault if none of the special states is currently active.

begin
  Result := crDefault;
  if vsZoomInPending in FStates then
    Result := crZoomIn
  else
    if vsZoomOutPending in FStates then
      Result := crZoomOut
    else
      if vsGrabPanningPending in FStates then
        Result := crHandOpen
      else
        if vsGrabPanning in FStates then
          Result := crHandClosed;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.DestroyContexts;

begin
  wglMakeCurrent(0, 0);
  wglDeleteContext(FRenderingContext);
  FRenderingContext := 0;
  if HandleAllocated then
    ReleaseDC(Handle, FDeviceContext);
  FDeviceContext := 0;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.DestroyWnd;

begin
  DestroyContexts;

  // Caution: Never call free or destroy. This is a C++ class we have here. We don't own it. Release takes care.
  FCanvas.Release;
  FCanvas := nil;

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGenericCanvasViewer.DetermineScrollDirections(X, Y: Integer): TScrollDirections;

// Determines which direction the client area must be scrolled depending on the given position.

begin
  Result:= [];

  if CanAutoScroll then
  begin
    // Calculation for wheel panning/scrolling is a bit different to normal auto scroll.
    if [vsWheelPanning, vsWheelScrolling] * FStates <> [] then
    begin
      if (X - FLastClick.X) < -8 then
        Include(Result, sdLeft);
      if (X - FLastClick.X) > 8 then
        Include(Result, sdRight);

      if (Y - FLastClick.Y) < -8 then
        Include(Result, sdUp);
      if (Y - FLastClick.Y) > 8 then
        Include(Result, sdDown);
    end
    else
    begin
      if (X < FHorizontalScrollStep) and (FOffset.X < 0) then
        Include(Result, sdLeft);
      if (ClientWidth - FOffset.X < Round(FZoom.X * FBaseSize.X)) and (X > ClientWidth - FHorizontalScrollStep) then
        Include(Result, sdRight);

      if (Y < FVerticalScrollStep) and (FOffset.Y < 0) then
        Include(Result, sdUp);
      if (ClientHeight - FOffset.Y < Round(FZoom.Y * FBaseSize.Y)) and (Y > ClientHeight - FVerticalScrollStep) then
        Include(Result, sdDown);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.DoAutoScroll(X, Y: Integer);

begin
  FScrollDirections := DetermineScrollDirections(X, Y);

  if FStates * [vsWheelPanning, vsWheelScrolling] = [] then
  begin
    if FScrollDirections = [] then
    begin
      if ((FStates * [vsScrollPending, vsScrolling]) <> []) then
      begin
        StopTimer(ScrollTimer);
        FStates := FStates - [vsScrollPending, vsScrolling];
      end;
    end
    else
    begin
      // start auto scroll if not yet done
      if (FStates * [vsScrollPending, vsScrolling]) = [] then
      begin
        FStates := FStates + [vsScrollPending];
        SetTimer(Handle, ScrollTimer, FAutoScrollDelay, nil);
      end;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.DoChange(AObject: TObject; Reason: TGCChangeReason);

// Triggered by the canvas for regular changes.

begin
  // Let the application first do what must be done.
  if Assigned(FOnChange) then
    FOnChange(Self, AObject, Reason);

  // Viewer internal handling of changes.
  case Reason of
    GC_CHANGE_CANVAS_REFRESH,
    GC_CHANGE_VIEW_PROPERTY:
      Invalidate;
    GC_CHANGE_CANVAS_SWITCH_VIEW: // A new view has been activated.
      begin
        if FCanvas.CurrentView <> nil then
        begin
          FCanvas.CurrentView.Viewport(FViewport);
          FOffset.X := Round(FCanvas.CurrentView.OffsetX);
          FOffset.Y := Round(FCanvas.CurrentView.OffsetY);
          FZoom.X := FCanvas.CurrentView.ZoomX;
          FZoom.Y := FCanvas.CurrentView.ZoomY;
        end;
        UpdateScrollbars;
      end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.DoError(const Message: string);

begin
  if Assigned(FOnError) then
    FOnError(Self, Message);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGenericCanvasViewer.DoFigureAction(X, Y: Integer; Shift: TShiftState): Boolean;

var
  Hits: THitResults;
  Instance: TFigureInstance;

begin
  Result := False;
  
  if Shift = [] then
    FCanvas.ClearSelection;
  Hits := FCanvas.CurrentView.GetHitTestInfo(X, Y, True);
  Hits.Reset;
  if Hits.HasNext then
  begin
    Instance := Hits.Next;
    Result := FCanvas.DoAction(Instance, X, Y);
  end;
  Hits.Release;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.DoTimerScroll;

var
  P,
  ClientP: TPoint;
  InRect,
  Panning: Boolean;
  R,
  ClipRect: TRect;
  DeltaX,
  DeltaY: Integer;
  IncrementX,
  IncrementY: Integer;

begin
  GetCursorPos(P);
  R := ClientRect;
  ClipRect := R;
  MapWindowPoints(Handle, 0, R, 2);
  InRect := PtInRect(R, P);
  ClientP := ScreenToClient(P);
  Panning := [vsWheelPanning, vsWheelScrolling] * FStates <> [];
  IncrementX := FHorizontalScrollStep;
  IncrementY := FVerticalScrollStep;

  if IsMouseSelecting or InRect or Panning then
  begin
    DeltaX := 0;
    DeltaY := 0;
    if sdUp in FScrollDirections then
    begin
      if Panning then
        DeltaY := FLastClick.Y - ClientP.Y - 8
      else
        if InRect then
          DeltaY := Min(IncrementY, ClientHeight)
        else
          DeltaY := Min(IncrementY, ClientHeight) * Abs(R.Top - P.Y);
      if FOffset.Y = 0 then
        Exclude(FScrollDirections, sdUp);
    end;

    if sdDown in FScrollDirections then
    begin
      if Panning then
        DeltaY := FLastClick.Y - ClientP.Y + 8
      else
        if InRect then
          DeltaY := -Min(IncrementY, ClientHeight)
        else
          DeltaY := -Min(IncrementY, ClientHeight) * Abs(P.Y - R.Bottom);
      if (ClientHeight - FOffset.Y) = Round(FZoom.Y * FBaseSize.Y) then
        Exclude(FScrollDirections, sdDown);
    end;

    if sdLeft in FScrollDirections then
    begin
      if Panning then
        DeltaX := FLastClick.X - ClientP.X - 8
      else
        if InRect then
          DeltaX := IncrementX
        else
          DeltaX := IncrementX * Abs(R.Left - P.X);
      if FOffset.X = 0 then
        Exclude(FScrollDirections, sdleft);
    end;

    if sdRight in FScrollDirections then
    begin
      if Panning then
        DeltaX := FLastClick.X - ClientP.X + 8
      else
        if InRect then
          DeltaX := -IncrementX
        else
          DeltaX := -IncrementX * Abs(P.X - R.Right);

      if (ClientWidth - FOffset.X) = Round(FZoom.X * FBaseSize.X) then
        Exclude(FScrollDirections, sdRight);
    end;

    OffsetX := OffsetX + DeltaX;
    OffsetY := OffsetY + DeltaY;

    if (FScrollDirections = []) and ([vsWheelPanning, vsWheelScrolling] * FStates = []) then
    begin
      StopTimer(ScrollTimer);
      FStates := FStates - [vsScrollPending, vsScrolling];
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.DoZoomChange(X, Y: Integer);

var
  NewWidth: Integer;
  NewHeight: Integer;

begin
  if Align <> alClient then
  begin
    NewWidth := Min(Round(FBaseSize.X * ZoomSteps[FCurrentZoomStep]), ClientWidth);
    NewHeight := Min(Round(FBaseSize.Y * ZoomSteps[FCurrentZoomStep]), ClientHeight);
    SetBounds(Left, Top, NewWidth, NewHeight);
  end;
  SetZoom(ZoomSteps[FCurrentZoomStep], ZoomSteps[FCurrentZoomStep], Point(X, Y));
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.PanningWindowProc(var Message: TMessage);

var
  PS: TPaintStruct;
  Canvas: TCanvas;

begin
  if Message.Msg = WM_PAINT then
  begin
    BeginPaint(FPanningWindow, PS);
    Canvas := TCanvas.Create;
    Canvas.Handle := PS.hdc;
    try
      Canvas.Draw(0, 0, FPanningImage);
    finally
      Canvas.Handle := 0;
      Canvas.Free;
      EndPaint(FPanningWindow, PS);
    end;
    Message.Result := 0;
  end
  else
    with Message do
      Result := DefWindowProc(FPanningWindow, Msg, wParam, lParam);
end;

//----------------------------------------------------------------------------------------------------------------------

var
  PanningWindowClass: TWndClass = (
    style: 0;
    lpfnWndProc: @DefWindowProc;
    cbClsExtra: 0;
    cbWndExtra: 0;
    hInstance: 0;
    hIcon: 0;
    hCursor: 0;
    hbrBackground: 0;
    lpszMenuName: nil;
    lpszClassName: 'GCPanningWindow'
  );

procedure TGenericCanvasViewer.StartWheelPanning;

// Called when wheel panning should start. A little helper window is created to indicate the reference position,
// which determines in which direction and how far wheel panning/scrolling will happen.

  //--------------- local function --------------------------------------------

  function CreateClipRegion: HRGN;

  // In order to avoid doing all the transparent drawing ourselves we use a
  // window region for the wheel window.
  // Since we only work on a very small image (32x32 pixels) this is acceptable.

  var
    Start, X, Y: Integer;
    Temp: HRGN;

  begin
    Assert(not FPanningImage.Empty, 'Invalid wheel panning image.');

    // Create an initial region on which we operate.
    Result := CreateRectRgn(0, 0, 0, 0);
    with FPanningImage, Canvas do
    begin
      for Y := 0 to Height - 1 do
      begin
        Start := -1;
        for X := 0 to Width - 1 do
        begin
          // Start a new span if we found a non-transparent pixel and no span is currently started.
          if (Start = -1) and (Pixels[X, Y] <> clFuchsia) then
            Start := X
          else
            if (Start > -1) and (Pixels[X, Y] = clFuchsia) then
            begin
              // A non-transparent span is finished. Add it to the result region.
              Temp := CreateRectRgn(Start, Y, X, Y + 1);
              CombineRgn(Result, Result, Temp, RGN_OR);
              DeleteObject(Temp);
              Start := -1;
            end;
        end;
        // If there is an open span then add this also to the result region.
        if Start > -1 then
        begin
          Temp := CreateRectRgn(Start, Y, Width, Y + 1);
          CombineRgn(Result, Result, Temp, RGN_OR);
          DeleteObject(Temp);
        end;
      end;
    end;

    // The resulting region is used as window region so we must not delete it.
    // Windows will own it after the assignment below.
  end;

  //--------------- end local function ----------------------------------------

var
  TempClass: TWndClass;
  ClassRegistered: Boolean;
  ImageName: string;

begin
  // Set both panning and scrolling flag. One will be removed shortly depending on whether the middle mouse button is
  // released before the mouse is moved or vice versa. The first case is referred to as wheel scrolling while the
  // latter is called wheel panning.
  StopTimer(ScrollTimer);
  FStates := FStates + [vsWheelPanning, vsWheelScrolling];

  // Register the helper window class.
  PanningWindowClass.hInstance := HInstance;
  ClassRegistered := GetClassInfo(HInstance, PanningWindowClass.lpszClassName, TempClass);
  if not ClassRegistered or (TempClass.lpfnWndProc <> @DefWindowProc) then
  begin
    if ClassRegistered then
      Windows.UnregisterClass(PanningWindowClass.lpszClassName, HInstance);
    Windows.RegisterClass(PanningWindowClass);
  end;
  // Create the helper window and show it at the given position without activating it.
  with ClientToScreen(FLastClick) do
    FPanningWindow := CreateWindowEx(WS_EX_TOOLWINDOW, PanningWindowClass.lpszClassName, nil, WS_POPUP, X - 16, Y - 16,
      32, 32, 0, 0, HInstance, nil);

  FPanningImage := TBitmap.Create;
  if Round(FZoom.X * FBaseSize.X) > ClientWidth then
  begin
    if Round(FZoom.Y * FBaseSize.Y) > ClientHeight then
      ImageName := 'GC_MoveAllBitmap'
    else
      ImageName := 'GC_MoveEastWestBitmap'
  end
  else
    ImageName := 'GC_MoveNorthSouthBitmap';
  FPanningImage.LoadFromResourceName(HInstance, ImageName);
  SetWindowRgn(FPanningWindow, CreateClipRegion, False);

  SetWindowLong(FPanningWindow, GWL_WNDPROC, Integer(Classes.MakeObjectInstance(PanningWindowProc)));
  ShowWindow(FPanningWindow, SW_SHOWNOACTIVATE);

  // Setup the panscroll timer and capture all mouse input.
  SetFocus;
  SetCapture(Handle);
  SetTimer(Handle, ScrollTimer, 10, nil);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.StopTimer(ID: Integer);

begin
  if HandleAllocated then
    KillTimer(Handle, ID);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.StopWheelPanning;

// Stops panning if currently active and destroys the helper window.

var
  Instance: Pointer;

begin
  if [vsWheelPanning, vsWheelScrolling] * FStates <> [] then
  begin
    // Release the mouse capture and stop the panscroll timer.
    StopTimer(ScrollTimer);
    ReleaseCapture;
    FStates := FStates - [vsWheelPanning, vsWheelScrolling];

    // Destroy the helper window.
    Instance := Pointer(GetWindowLong(FPanningWindow, GWL_WNDPROC));
    DestroyWindow(FPanningWindow);
    if Instance <> @DefWindowProc then
      Classes.FreeObjectInstance(Instance);
    FPanningWindow := 0;
    FPanningImage.Free;
    FPanningImage := nil;
    DeleteObject(FPanningCursor);
    FPanningCursor := 0;
    Windows.SetCursor(Screen.Cursors[Cursor]);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.UpdateScrollbars;

var
  ScrollInfo: TScrollInfo;
  UpdateCanvas: Boolean;
  ActualWidth,
  ActualHeight: Integer;

begin
  // Check if the virtual size is larger than the client area. In this case the offsets must correspond to the available
  // scroll range.
  ActualWidth := Round(FZoom.X * FBaseSize.X);
  if ActualWidth > ClientWidth then
  begin
    if FOffset.X > 0 then
      FOffset.X := 0;
    FOffset.X := Max(FOffset.X, Max(0, ClientWidth) - ActualWidth);
  end;
  ActualHeight := Round(FZoom.Y * FBaseSize.Y);
  if Actualheight > ClientHeight then
  begin
    if FOffset.Y > 0 then
      FOffset.Y := 0;
    FOffset.Y := Max(FOffset.Y, Max(0, ClientHeight) - ActualHeight);
  end;

  UpdateCanvas := False;

  // Horizontal scrollbar.
  ScrollInfo.cbSize := SizeOf(ScrollInfo);
  ScrollInfo.fMask := SIF_ALL or ScrollMasks[cvoAlwaysShowScrollbars in FOptions];

  ScrollInfo.nMax := ActualWidth;
  if (ActualWidth > ClientWidth) or (cvoAlwaysShowScrollbars in FOptions) then
  begin
    UpdateCanvas := True;
    ScrollInfo.nMin := 0;
    ScrollInfo.nPos := -FOffset.X;
    ScrollInfo.nPage := Max(0, ClientWidth);
    SetScrollInfo(Handle, SB_HORZ, ScrollInfo, True);
    ShowScrollBar(Handle, SB_HORZ, True);
  end
  else
  begin
    ShowScrollBar(Handle, SB_HORZ, False);

    // No scrollbars, so we center the main area in the viewer.
    if cvoAutoCenterCanvas in FOptions then
    begin
      UpdateCanvas := True;
      FOffset.X := Round((ClientWidth - ActualWidth) / 2);
    end;
  end;

  // Vertical scrollbar.
  ScrollInfo.nMax := ActualHeight;
  if (ActualHeight > ClientHeight) or (cvoAlwaysShowScrollbars in FOptions) then
  begin
    UpdateCanvas := True;
    ScrollInfo.nMin := 0;
    ScrollInfo.nPage := Max(0, ClientHeight);
    ScrollInfo.nPos := -FOffset.Y;
    SetScrollInfo(Handle, SB_VERT, ScrollInfo, True);
    ShowScrollBar(Handle, SB_VERT, True);
  end
  else
  begin
    ShowScrollBar(Handle, SB_VERT, False);

    // No scrollbars, so we center the main area in the viewer.
    if cvoAutoCenterCanvas in FOptions then
    begin
      UpdateCanvas := True;
      FOffset.Y := Round((ClientHeight - ActualHeight) / 2);
    end;
  end;

  if UpdateCanvas and (Canvas.CurrentView <> nil) then
  begin
    Canvas.CurrentView.OffsetX(FOffset.X);
    Canvas.CurrentView.OffsetY(FOffset.Y);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGenericCanvasViewer.IsMouseSelecting: Boolean;

begin
  Result := (vsDrawSelPending in FStates) or (vsDrawSelecting in FStates);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGenericCanvasViewer.LoadLayouts(FileName: string): TGCError;

begin
  Result := Canvas.AddlayoutsFromFile(PChar(FileName));
end;

//----------------------------------------------------------------------------------------------------------------------

function TGenericCanvasViewer.LoadStyles(FileName: string): TGCError;

begin
  Result := Canvas.AddStylesFromFile(PChar(FileName));
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.RenderScene;

// This is the actual paint routine. All initialization must already have been done.

var
  Start,
  Stop: Int64;
  Time: Single;
  I: Integer;

begin
  QueryPerformanceCounter(Start);

  Canvas.Render();
  SwapBuffers(FDeviceContext);

  // Compute rendering speed.
  QueryPerformanceCounter(Stop);
  Time := (Stop - Start) / FCounterFrequency;
  if FTimeBuffer[0] = 0 then
  begin
    // First run.
    for I := 0 to FPSBufferSize - 1 do
      FTimeBuffer[I] := Time;
    FTotalTime := FPSBufferSize * Time;
  end
  else
  begin
    FTotalTime := FTotalTime - FTimeBuffer[FCurrentTimeBufferIndex];
    FTimeBuffer[FCurrentTimeBufferIndex] := Time;
    FTotalTime := FTotalTime + FTimeBuffer[FCurrentTimeBufferIndex];
  end;
  Inc(FCurrentTimeBufferIndex);
  if FCurrentTimeBufferIndex = FPSBufferSize then
    FCurrentTimeBufferIndex := 0;
  FFPS := FPSBufferSize / FTotalTime;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.SetZoom(X, Y: Double; ZoomCenter: TPoint);

// Zooms the canvas around the given zoom center (zoom origin) by the given factors.
// The origin must be given in window coordinates.

var
  CanvasX,
  CanvasY: Integer;

begin
  // Convert the origin's window to canvas coordinates.
  CanvasX := ZoomCenter.X - FOffset.X;
  CanvasY := ZoomCenter.Y - FOffset.Y;

  // Compute the new coordinates as they are after applying the new zoom factors.
  CanvasX := Round(CanvasX * X / FZoom.X);
  CanvasY := Round(CanvasY * Y / FZoom.Y);

  FZoom.X := X;
  FZoom.Y := Y;

  if cvoAutoCenterZoom in FOptions then
  begin
    FOffset.X := (ClientWidth div 2) - CanvasX;
    FOffset.Y := (ClientHeight div 2) - CanvasY;
  end
  else
  begin
    FOffset.X := ZoomCenter.X - CanvasX;
    FOffset.Y := ZoomCenter.Y - CanvasY;
  end;

  if Canvas.CurrentView <> nil then
  begin
    Canvas.CurrentView.ZoomX(X);
    Canvas.CurrentView.ZoomY(Y);
  end;
  UpdateScrollbars;

  if Assigned(FOnZoomChange) then
    FOnZoomChange(Self);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.ZoomIn(X, Y: Integer);

begin
  if FCurrentZoomStep < MaxZoomSteps then
  begin
    Inc(FCurrentZoomStep);
    DoZoomChange(X, Y);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGenericCanvasViewer.ZoomOut(X, Y: Integer);

begin
  if FCurrentZoomStep > 0 then
  begin
    Dec(FCurrentZoomStep);
    DoZoomChange(X, Y);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure InitializeGlobalStructures;

begin
  // Mask out exceptions for invalid floating point conversions.
  Set8087CW($133F);
  
  Screen.Cursors[crZoomIn] := LoadCursor(HInstance, 'GC_ZoomInCursor');
  Screen.Cursors[crZoomOut] := LoadCursor(HInstance, 'GC_ZoomOutCursor');
  Screen.Cursors[crHandOpen] := LoadCursor(HInstance, 'GC_PanningHandOpenCursor');
  Screen.Cursors[crHandClosed] := LoadCursor(HInstance, 'GC_PanningHandClosedCursor');

  Screen.Cursors[crMoveAll] := LoadCursor(HInstance, 'GC_MoveAllCursor');
  Screen.Cursors[crMoveEast] := LoadCursor(HInstance, 'GC_MoveEastCursor');
  Screen.Cursors[crMoveEastWest] := LoadCursor(HInstance, 'GC_MoveEastWestCursor');
  Screen.Cursors[crMoveNorth] := LoadCursor(HInstance, 'GC_MoveNorthCursor');
  Screen.Cursors[crMoveNorthEast] := LoadCursor(HInstance, 'GC_MoveNorthEastCursor');
  Screen.Cursors[crMoveNorthSouth] := LoadCursor(HInstance, 'GC_MoveNorthSouthCursor');
  Screen.Cursors[crMoveNorthWest] := LoadCursor(HInstance, 'GC_MoveNorthWestCursor');
  Screen.Cursors[crMoveSouth] := LoadCursor(HInstance, 'GC_MoveSouthCursor');
  Screen.Cursors[crMoveSouthEast] := LoadCursor(HInstance, 'GC_MoveSouthEastCursor');
  Screen.Cursors[crMoveSouthWest] := LoadCursor(HInstance, 'GC_MoveSouthWestCursor');
  Screen.Cursors[crMoveWest] := LoadCursor(HInstance, 'GC_MoveWestCursor');
end;

//----------------------------------------------------------------------------------------------------------------------

procedure FinalizeGlobalStructures;

begin

end;

//----------------------------------------------------------------------------------------------------------------------

initialization
  InitializeGlobalStructures;
finalization
  FinalizeGlobalStructures;
end.



