/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_view.h 
 * @brief Implementation of the view class.
 * 
 */

#ifndef __GC_VIEW_H__
#define __GC_VIEW_H__

#include "myx_gc_datatypes.h"
#include "myx_gc_layer.h"

//----------------- Hit testing structures -----------------------------------------------------------------------------

typedef struct tagHitEntry
{
  CFigureInstance* Instance;
  double ZMin;
  double ZMax;
} THitEntry;

typedef vector<CFigureInstance*> THitEntries;
typedef vector<CFigureInstance*>::iterator THitEntryIterator;

/**
 * The CHitResult class is used to collect a number of figures that are located at a given point in the canvas.
 *
 * @note Never hold the given hit results record for a long time. The referenced figure instances may disappear at any time.
 */
class GENERIC_CANVAS_API CHitResults         
{
  friend class CGCView;
  friend class CLayer;
private:
  THitEntries FEntries;
  THitEntryIterator FCurrentEntry;
protected:
  void addHit(CFigureInstance* Instance);
public:
  CHitResults(void);
  virtual ~CHitResults(void);

  virtual int __cdecl count(void);
  virtual bool __cdecl hasNext(void);
  virtual CFigureInstance* __cdecl next(void);
  virtual void __cdecl release(void);
  virtual void __cdecl reset(void);
};

//----------------------------------------------------------------------------------------------------------------------

class CGenericCanvas;
class CGridLayer;
class CGCView;

class CLayerListener: public CGCListener
{
  friend class CGCView;
protected:
  CGCView* view;
public:
  virtual void __cdecl onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason);
  virtual void __cdecl onDestroy(CGCBase* object);
  virtual void __cdecl onError(CGCBase* sender, CGCBase* origin, const char* message);
};

/** 
 * A view implements an association between a set of layers and their visual representation on screen.
 * Views can have individual zoom and offset values, viewports and other properties.
 * There can always be only one active view. Views are managed by the canvas class.
 */
class GENERIC_CANVAS_API CGCView: public CGCBase
{
  friend class CGenericCanvas;
private:
  CGenericCanvas* FCanvas;   // The owning canvas.
  CGridLayer* FGrid;         // A grid rendered on top of all content.
  string FName;              // An identifier for this view (UTF-8 encoded).
  GLfloat FColor[4];         // The background color of the viewer.
  TGCViewport FViewport;     // Coordinates of the output area within the viewer window.
  CLayers FLayers;           // A list of layers that are visible in this view.
  float FZoomX;              // The current horizontal zoom factor. It's a normal scaling factor but for the 
                             // whole view (scales everything).
  float FZoomY;              // The current vertical zoom factor.
  float FMinimalZoom;        // The lesser of FZoomX and FZoomY.
  float FOffsetX;            // A translation offset to enable scrolling (panning) in a zoomed scene.
  float FOffsetY;            // The vertical translation offset.
  float FNearPlane;          // The cut off plane that is closest to the eye.
  float FFarPlane;           // The cut off plane that is farthest from the eye.
  float FJitter;             // Little offset to add to the viewport values to improve display quality on some boards.
  CLayerListener FListener;   // A listener to get notified if a layer is being destroyed.
protected:
  void activate(void);
  void applyTransformations(bool DoProjection);
  float feedbackZoom(void);
  void render(void);
public:
  CGCView(CGenericCanvas* Owner, string Name);
  virtual ~CGCView(void);

  virtual void __cdecl addLayer(CLayer* Layer);
  virtual void __cdecl color(GLfloat* NewColor);
  virtual void __cdecl color(float Red, float Green, float Blue, float Alpha);
  virtual bool __cdecl contains(CLayer* Layer);
  virtual CHitResults* __cdecl getHitTestInfoAt(const float X, const float Y, bool SingleHit);
  virtual CGridLayer* __cdecl grid(void) { return FGrid; };
  virtual void __cdecl jitter(float Value);
  virtual float __cdecl jitter(void);
  virtual void __cdecl offsetX(float Value);
  virtual float __cdecl offsetX(void);
  virtual void __cdecl offsetY(float Value);
  virtual float __cdecl offsetY(void);
  virtual TGCVariant __cdecl property(const char* name, unsigned int index);
  virtual void __cdecl property(const char* name, unsigned int index, const TGCVariant& value);
  virtual void __cdecl removeLayer(CLayer* Layer);
  virtual void __cdecl viewport(const TGCViewport& NewViewport);
  virtual TGCViewport __cdecl viewport(void) { return FViewport; };
  virtual void __cdecl zoomX(float Value);
  virtual float __cdecl zoomX(void);
  virtual void __cdecl zoomY(float Value);
  virtual float __cdecl zoomY(void);
};

//----------------------------------------------------------------------------------------------------------------------

#endif // __GC_VIEW_H__
