/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_model.cpp 
 * @brief Implementation of the model that manages the visual representation in the generic canvas.
 * 
 */

#include "myx_gc.h"

#include "myx_gc_font_manager.h"
#include "myx_gc_utilities.h"

//----------------------------------------------------------------------------------------------------------------------

// Converts the given string into a font weight value.
// Allowed values are: normal | bold | bolder | lighter | 100 | 200 | 300| 400 | 500 | 600 | 700 | 800 | 900 | inherit
int convertFontWeight(const string Weight)
{
  int Result = 0;
#ifdef USE_FONTCONFIG
  if (Weight == "normal")
    Result = FC_WEIGHT_NORMAL;
  else if (Weight == "bold")
    Result = FC_WEIGHT_BOLD;
  else if (Weight == "bolder")
    Result = FC_WEIGHT_EXTRABOLD;
  else if (Weight == "lighter")
    Result = FC_WEIGHT_LIGHT;
  else if (Weight == "inherit")
    Result = FC_WEIGHT_NORMAL;
  else
    Result = atoi(Weight.c_str());
#else // !USE_FONTCONFIG
  if (Weight == "normal")
    Result = 400;
  else
    if (Weight == "bold")
      Result = 700;
    else
      if (Weight == "bolder")
        Result = 800;
      else
        if (Weight == "lighter")
          Result = 300;
        else
          if (Weight == "inherit")
            Result = 400;
          else
            Result = atoi(Weight.c_str());
#endif

  return Result;
}

//----------------- CFontManager ---------------------------------------------------------------------------------------

static CFontManager* internalManager;
static int lockCount = 0;

// Returns the current font manager (there is always only one).
CFontManager* fontManager(void)
{
  return internalManager;
}

//----------------------------------------------------------------------------------------------------------------------

/*
 * Increases the lock count of the font manager. If the manager does not yet exist it is created.
 *
 */
void lockFontManager(void)
{
  if (internalManager == NULL)
  {
    internalManager = new CFontManager();
#ifdef USE_FONTCONFIG
    if (!FcInit())
      g_warning("Could not initialize Fontconfig");
#endif
  }
  ++lockCount;
}

//----------------------------------------------------------------------------------------------------------------------

// Returns the current font manager (there is always only one).
void unlockFontManager(void)
{
  if (lockCount > 0)
  {
    --lockCount;
    delete internalManager;
    internalManager = NULL;
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Constructor of the class.
 */
CFontManager::CFontManager(void)
{
  FUseLocalDisplayLists = true;     

#ifndef USE_FONTCONFIG
  // Fill our font file list with some predefined font files.
  FontFileEntry* entry;

  // "Arial" normal, bold, italic, bold italic.
  entry = new FontFileEntry;
  entry->entries[false][false] = "Arial.ttf"; entry->entries[false][true] = "Arialbd.ttf"; 
  entry->entries[true][false] = "Ariali.ttf";  entry->entries[true][true] = "Arialbi.ttf"; 
  entry->useCount = 5;
  FFiles["Arial"] = entry;
  FFiles["Arial Normal"] = entry;
  FFiles["Arial-Normal"] = entry;
  FFiles["Arial Standard"] = entry;
  FFiles["Arial-Standard"] = entry;

  // Courier New
  entry = new FontFileEntry;
  entry->entries[false][false] = "Cour.ttf"; entry->entries[false][true] = "Courbd.ttf"; 
  entry->entries[true][false] = "Couri.ttf";  entry->entries[true][true] = "Courbi.ttf"; 
  entry->useCount = 3;
  FFiles["Courier New"] = entry;
  FFiles["Courier New Standard"] = entry;
  FFiles["Courier"] = entry;

  // Garamond
  entry = new FontFileEntry;
  entry->entries[false][false] = "Gara.ttf"; entry->entries[false][true] = "Garabd.ttf"; 
  entry->entries[true][false] = "Garait.ttf";  entry->entries[true][true] = "Garabi.ttf";
  entry->useCount = 3;
  FFiles["Garamond"] = entry;
  FFiles["Garamond Standard"] = entry;
  FFiles["Garamond-Standard"] = entry;

  // Palatino Linotype
  entry = new FontFileEntry;
  entry->entries[false][false] = "Pala.ttf"; entry->entries[false][true] = "Palab.ttf"; 
  entry->entries[true][false] = "Palai.ttf";  entry->entries[true][true] = "Palabi.ttf";
  entry->useCount = 1;
  FFiles["Palation Linotype"] = entry;

  // Tahoma
  entry = new FontFileEntry;
  entry->entries[false][false] = "Tahoma.ttf"; entry->entries[false][true] = "Tahomabd.ttf"; 
  entry->entries[true][false] = "Tahomai.ttf";  entry->entries[true][true] = "Tahomabi.ttf";
  entry->useCount = 1;
  FFiles["Tahoma"] = entry;

  // Tahoma-Bold, not an own font but Adobe Illustrator denotes it so.
  entry = new FontFileEntry;
  entry->entries[false][false] = "Tahomabd.ttf"; entry->entries[false][true] = "Tahomabd.ttf"; 
  entry->entries[true][false] = "Tahomabdi.ttf";  entry->entries[true][true] = "Tahomabdi.ttf";
  entry->useCount = 1;
  FFiles["Tahoma-Bold"] = entry;

  // Times New Roman
  entry = new FontFileEntry;
  entry->entries[false][false] = "Times.ttf"; entry->entries[false][true] = "Timesbd.ttf"; 
  entry->entries[true][false] = "Timesi.ttf";  entry->entries[true][true] = "Timesbi.ttf";
  entry->useCount = 2;
  FFiles["Times New Roman"] = entry;
  FFiles["Times"] = entry;

  // Trebuchet MS
  entry = new FontFileEntry;
  entry->entries[false][false] = "Trebuc.ttf"; entry->entries[false][true] = "Trebucbd.ttf"; 
  entry->entries[true][false] = "Trebucit.ttf";  entry->entries[true][true] = "Trebucbi.ttf";
  entry->useCount = 2;
  FFiles["Trebuchet MS"] = entry;
  FFiles["Trebuchet"] = entry;

  // Verdana
  entry = new FontFileEntry;
  entry->entries[false][false] = "Verdana.ttf"; entry->entries[false][true] = "Verdanab.ttf"; 
  entry->entries[true][false] = "Verdanai.ttf";  entry->entries[true][true] = "Verdanaz.ttf";
  entry->useCount = 1;
  FFiles["Verdana"] = entry;
#endif // !USE_FONTCONFIG
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Destructor of the class. Does some clean up.
 */
CFontManager::~CFontManager(void)
{
  clear();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines platform dependantly the full path of a font file depending on some characteristics.
 *
 * @param Family The font family (e.g. Arial, Tahoma, Verdana).
 * @param Style The font style (normal, italic).
 * @param Weight The "boldness" of the font in the range of [100..900]. Currently values <= 400 are
 *               considered als normal font, everything else as bold.
 *
 * @return The full path and file name of a font that represents the given characteristics.
 *          A kind of intelligent replacement is done here, though. If there is no file with the given characteristics 
 *          (or cannot be found) then the one from the same family, but normal styles, is used instead. If there is no 
 *          entry for the family or even the standard style for a family cannot be found then Arial Standard is returned as 
 *          default. If this files is also not present on the system then the FTGL lib will throw an exception.
 * @note The returned file name is ANSI encoded as FTGL expects it so.
 */
string CFontManager::getFontFile(string Family, string Style, int Weight)
{
  string Filename;

#ifdef USE_FONTCONFIG
  FcPattern *pattern= FcPatternCreate();
  FcResult result;
  int slant;
  
  if (Style == "italic")
    slant= FC_SLANT_ITALIC;
  else if (Style == "oblique")
    slant= FC_SLANT_OBLIQUE;
  else
    slant= FC_SLANT_ROMAN;
  
  FcPatternAddString(pattern, FC_FAMILY, (FcChar8*)Family.c_str());
  FcPatternAddInteger(pattern, FC_SLANT, slant);
  FcPatternAddInteger(pattern, FC_WEIGHT, Weight);

  if (!FcConfigSubstitute(FcConfigGetCurrent(), pattern, FcMatchPattern))
  {
    g_warning("Error calling FcConfigSubstitute() on font '%s %s'",
              Family.c_str(), Style.c_str());
    FcPatternDestroy(pattern);
  }
  else
  {
    FcPattern *fpat;
    
    FcDefaultSubstitute(pattern);
  
    fpat= FcFontMatch(FcConfigGetCurrent(), pattern, &result);
    if (result != FcResultMatch)
      g_warning("Error looking up for font '%s %s'",
                Family.c_str(), Style.c_str());
    else
    {
      FcValue value;
      
      if (FcPatternGet(fpat, FC_FILE, 0, &value))
      {
        g_message("FILE --> %s", value.u.s);
      
        Filename= (char*)value.u.s;
      }
      FcPatternDestroy(fpat);
    }
    FcPatternDestroy(pattern);
  }
#else // !USE_FONTCONFIG
  FontFileEntry* entry;

  // Adobe illustrator (and probably others too) put the family name in quotes.
  if (Family[0] == '"' || Family[0] == '\'')
    Family = Family.substr(1, Family.size() - 2);
  FontFiles::iterator iterator = FFiles.find(Family);
  if (iterator == FFiles.end())
  {
    // Family is unknown, use Arial as default.
    entry = FFiles["Arial"];
  }
  else
  {
    entry = iterator->second;
  };

  bool Italic = (Style == "italic") || (Style == "oblique");
  bool Bold = Weight > 400;

  LPITEMIDLIST pidl;
  wchar_t Path[MAX_PATH];
  if (SHGetSpecialFolderLocation(0, CSIDL_FONTS, &pidl) == NOERROR)
  {
    if (SHGetPathFromIDList(pidl, Path))
      Filename = utf16ToANSI(Path);
    
    // We are required to free the returned pidl via the shell's IMalloc interface.
    LPMALLOC pMalloc;
    SHGetMalloc(&pMalloc);
    pMalloc->Free(pidl);
  }
  string FontFile = entry->entries[Italic][Bold];
  string Testfile = Filename + '/' + FontFile;

  FILE* File = fopen(Testfile.c_str(), "rb");
  if (File != NULL)
  {
    fclose(File);
    Filename = Testfile;
  }
  else
  {
    // There is no file for the required style. Try the base entry.
    FontFile = entry->entries[false][false];
    Testfile = Filename + '/' + FontFile;
    File = fopen(Testfile.c_str(), "rb");
    if (File != NULL)
    {
      fclose(File);
      Filename = Testfile;
    }
    {
      // That's bad luck. No standard file either. Use the default font.
      entry = FFiles["Arial"];
      Filename += '/' + entry->entries[false][false];
    };
  };
#endif // !USE_FONTCONFIG
  return Filename;
}

//----------------------------------------------------------------------------------------------------------------------

void CFontManager::boundingBox(const wstring& Output, string FontFamily, string FontStyle, int Weight, int FontSize, 
                               TBoundingBox* Box)
{
  FTFont* Font;
  string Key = createLookupKey(FontFamily, FontStyle, Weight);
  Fonts::iterator iterator = FFonts.find(Key);

  if (iterator == FFonts.end())
  {
    string FontFile = getFontFile(FontFamily, FontStyle, Weight);
    Font = new FTGLPixmapFont(FontFile.c_str());
    Font->UseDisplayList(FUseLocalDisplayLists);
    FFonts[Key] = Font;
  }
  else
    Font = iterator->second;

  if ((int)Font->FaceSize() != FontSize)
    Font->FaceSize(FontSize);

  Font->BBox(Output.c_str(), Box->upper.x, Box->upper.y, Box->upper.z, Box->lower.x, Box->lower.y, Box->lower.z);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Clears the font file list.
 */
void CFontManager::clear(void)
{
  for (Fonts::iterator iterator = FFonts.begin(); iterator != FFonts.end(); ++iterator)
  {
    FTFont* Font = iterator->second;
    delete Font;
  };
  FFonts.clear();

#ifndef USE_FONTCONFIG
  for (FontFiles::iterator iterator = FFiles.begin(); iterator != FFiles.end(); iterator++)
  {
    FontFileEntry* entry = iterator->second;
    if (--entry->useCount == 0)
      delete entry;
  };
  FFiles.clear();
#endif
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a string that can be used for lookup in the font list.
 * Either parameter can be NULL (or 0 for Weight) causing the manager to use the default values for each missing 
 * parameter. See header file for the list of default values.
 *
 * @param Family The font family (Arial, Courier etc.)
 * @param Style The font style.
 * @param Weight The boldness of the font (400 for normal).
 */
string CFontManager::createLookupKey(const string& Family, const string& Style, int Weight)
{
  char WeightString[30];

  // Construct a lookup string out of the font properties.
  string Key = Family + Style;
  #ifdef _WINDOWS
    // Microsoft has confirmed that there is no guaranteed NULL termination (see MSDN for details).
    // Hence we better add one.
    _snprintf(WeightString, sizeof(WeightString), "%i\0", Weight);
  #else
    snprintf(WeightString, sizeof(WeightString), "%i", Weight);
  #endif // #ifdef _WINDOWS
  Key += WeightString;

  return Key;
}

//----------------------------------------------------------------------------------------------------------------------

void CFontManager::textOut(const wstring& Output, string FontFamily, string FontStyle, int Weight, int FontSize)
{
  FTFont* Font;
  string Key = createLookupKey(FontFamily, FontStyle, Weight);
  Fonts::iterator iterator = FFonts.find(Key);

  if (iterator == FFonts.end())
  {
    string FontFile = getFontFile(FontFamily, FontStyle, Weight);
    Font = new FTGLPixmapFont(FontFile.c_str());
    Font->UseDisplayList(FUseLocalDisplayLists);
    
    FFonts[Key] = Font;
  }
  else
    Font = iterator->second;

  if ((int) Font->FaceSize() != FontSize)
    Font->FaceSize(FontSize);
  Font->Render(Output.c_str());
}

//----------------------------------------------------------------------------------------------------------------------

