/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_utilities.cpp
 * @brief Some common utility functions.
 * 
 */

#include "myx_gc.h"
#include "myx_gc_utilities.h"
#include "myx_gc_const.h"

//----------------------------------------------------------------------------------------------------------------------

/**
 * Converts the given string, which is supposed to be an UTF-8 encoded text into an UTF-16 string.
 *
 * @param Source Contains the source string encoded in UTF-8.
 * @return The converted string in UTF-16 encoding.
 */
wstring utf8ToUtf16(const string& Source)
{
  wstring Result;

#ifdef _WINDOWS
  const char* Raw = Source.c_str();
  int Size = MultiByteToWideChar(CP_UTF8, 0, Raw, -1, NULL, 0);
  wchar_t* Buffer = new wchar_t[Size];
  MultiByteToWideChar(CP_UTF8, 0, Raw, -1, Buffer, Size);
  Result += Buffer;
  delete[] Buffer;
#else
  gunichar2 *tmp;
  tmp= g_utf8_to_utf16(Source.c_str(), Source.length(), NULL, NULL, NULL);
  Result= (wchar_t*)tmp;
  g_free(tmp);
#endif

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Converts the given UTF-16 string into an UTF-8 string.
 *
 * @param Source Contains the source string encoded in UTF-16.
 * @return The converted string in UTF-8 encoding.
 */
string utf16ToUtf8(const wstring& Source)
{
  string Result;

#ifdef _WINDOWS
  const wchar_t* Raw = Source.c_str();
  int Size = WideCharToMultiByte(CP_UTF8, 0, Raw, -1, NULL, 0, NULL, NULL);
  char* Buffer = new char[Size];
  WideCharToMultiByte(CP_UTF8, 0, Raw, -1, Buffer, Size, NULL, NULL);
  Result += Buffer;
  delete[] Buffer;
#else
  gunichar2 *tmp;
  tmp= g_utf16_to_utf8((gunichar2*)Source.c_str(), Source.length(), NULL, NULL, NULL);
  Result= (char*)tmp;
  g_free(tmp);
#endif

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Converts the given string into an ANSI string.
 * @param Source Contains the source string encoded in UTF-16.
 * @return The converted string in ANSI encoding.
 * @note The current user locale is used to convert the Unicode string to ANSI.
 */
string utf16ToANSI(const wstring& Source)
{
  string Result;

#ifdef _WINDOWS
  const wchar_t* Raw = Source.c_str();
  int Size = WideCharToMultiByte(CP_ACP, 0, Raw, -1, NULL, 0, NULL, NULL);
  char* Buffer = new char[Size];
  WideCharToMultiByte(CP_ACP, 0, Raw, -1, Buffer, Size, NULL, NULL);
  Result += Buffer;
  delete[] Buffer;
#else
  gchar *tmp;
  tmp= g_utf16_to_utf8((gunichar2*)Source.c_str(), Source.length(), NULL, NULL, NULL);
  Result= tmp;
  g_free(tmp);
#endif

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Converts the given string, which is supposed to be an UTF-8 encoded text into an ANSI string.
 * @param Source Contains the source string encoded in UTF-8.
 * @return The converted string in ANSI encoding.
 * @note The current user locale is used to convert the Unicode string to ANSI.
 */
string utf8ToANSI(const string& Source)
{
#ifdef _WINDOWS
  return utf16ToANSI(utf8ToUtf16(Source));
#else
  return Source;
#endif
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief User defined file handling callback for handling file i/o in png lib.
 *        We have to make it so because of the FILE allocation and png being in a DLL on Windows.
 */
void pngRead(png_structp png_ptr, png_bytep data, png_size_t length)
{
  fread(data, length, 1, (FILE*) png_ptr->io_ptr);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Loads a png file.
 * @param Filename An ANSI encoded file name (can contain path info) to the png file.
 * @return Returns a pointer to a TImage structure containing the image data.
 * @note The return memory must be freed using FreeImage().
 */
TImage* loadPNG(const string& Filename)
{
  TImage *Image = NULL;
  FILE* File;
  png_structp PNG;
  png_infop Info;
  png_uint_32 Width, Height;
  int depth, junk, color_type;
  int Offset;

  File = fopen(Filename.c_str(), "rb");
  try
  {
    if (File) 
    {
      png_byte Magic[8];
      if (fread(Magic, sizeof(Magic), 1, File) != 1)
      {
        fprintf(stderr, "can't read '%s': %s\n", Filename.c_str(), strerror(errno));
        fclose(File);
        return NULL;
      };
      if (!png_check_sig(Magic, sizeof(Magic)))
      {
        fclose(File);
        return NULL;
      };

      PNG = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
      if (!PNG)
      {
        fclose(File);
        return NULL;
      };
      
      Info = png_create_info_struct(PNG);
      if (!Info) 
      {
        fclose(File);
        png_destroy_read_struct(&PNG, NULL, NULL);
        return NULL;
      };

      png_set_read_fn(PNG, File, &pngRead);

      png_init_io(PNG, File);
      png_set_sig_bytes(PNG, sizeof(Magic));

      png_read_info(PNG, Info);

      png_get_IHDR(PNG, Info, &Width, &Height, &depth, &color_type,
                  &junk, &junk, &junk);

      /* sanity check */
      if (Width < 1 || Height < 1) 
      {
        fclose(File);
        png_destroy_read_struct(&PNG, &Info, NULL);
        return NULL;
      }

      Image = (TImage*) malloc(sizeof(TImage));
      if (!Image)
      {
        fclose(File);
        png_destroy_read_struct(&PNG, &Info, NULL);
        return NULL;
      };

      Image->Data = (unsigned char*) malloc(Width * Height * Info->channels);
      if (!Image->Data) 
      {
        fclose(File);
        png_destroy_read_struct(&PNG, &Info, NULL);
        return NULL;
      };

      Image->Width = Width;
      Image->Height = Height;
      Image->ColorType = (TColorType) Info->color_type;
      Image->Channels = Info->channels;

      // normalize to 8bpp with alpha channel
      if (color_type == PNG_COLOR_TYPE_PALETTE && depth <= 8)
        png_set_expand(PNG);

      if (color_type == PNG_COLOR_TYPE_GRAY && depth <= 8)
        png_set_expand(PNG);

      if (png_get_valid(PNG, Info, PNG_INFO_tRNS))
        png_set_expand(PNG);

      if (depth == 16)
        png_set_strip_16(PNG);

      if (color_type == PNG_COLOR_TYPE_GRAY ||
          color_type == PNG_COLOR_TYPE_GRAY_ALPHA)
        png_set_gray_to_rgb(PNG);

      /* do the transforms */
      png_read_update_info(PNG, Info);

      Offset= 0;
      /* read Data */
      for (unsigned int I = 0; I < Height; I++)
      {
        png_read_row(PNG, &Image->Data[Offset], NULL);
        Offset += Info->channels * Width;
      }
      png_read_end(PNG, Info);
      png_destroy_read_struct(&PNG, &Info, NULL);

      fclose(File);
    };
  }
  catch (...)
  {
    if (Image)
      freeImage(Image);
    Image = NULL;
  };

  return Image;
}

//----------------------------------------------------------------------------------------------------------------------

void freeImage(TImage* Image)
{
  free(Image->Data);
  free(Image);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Returns the current working folder (ANSI encoded).
 * @return The current working folder.
 */
string getCurrentDir(void)
{
  char Buffer[MAX_PATH];

  getcwd(Buffer, sizeof(Buffer));

  return string(Buffer);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Sets the current working folder (folder name must be ANSI encoded).
 * @param Folder The new folder to be set.
 */
void setCurrentDir(const string& Folder)
{
  chdir(Folder.c_str());
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief ExtractFilePath extracts the drive and directory parts of the given filename. The resulting string is the 
          leftmost characters of FileName, up to and including the colon or backslash that separates the path information 
          from the name and extension. The resulting string is empty if FileName contains no drive and directory parts.
 * @param Filename The file name (ANSI encoded) of which the path is to be extracted.
 * @return The extracted path part (ANSI encoded).
 */
string extractFilePath(const string& Filename)
{
  gchar* Raw = g_path_get_dirname(Filename.c_str());

  string Result(Raw);
  g_free(Raw);

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Platform neutral file open routine.
 * @param Filename The name of file encode in UTF-8.
 * @param OpenMode The mode how to open the file (the same as used for fopen calls).
 * @return A pointer to a FILE structure if the file could be opened, NULL otherwise.
 */
FILE* openFile(string Filename, const char* OpenMode)
{
#ifdef _WINDOWS
  wstring Name = utf8ToUtf16(Filename);
  wstring Mode = utf8ToUtf16(string(OpenMode));
  return _wfopen(Name.c_str(), Mode.c_str());
#else
  FILE *file;
  char * local_filename;

  if (!(local_filename = g_filename_from_utf8(Filename.c_str(), -1, NULL, NULL, NULL)))
    return NULL;

  file = fopen(local_filename, OpenMode);

  g_free(local_filename);

  return file;
#endif // #ifdef _WINDOWS
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Helper method to sort left/right and bottom/top coordinates so that for left/top are always smaller than right/bottom
 * (origin is considered in the left-upper corner, +y pointing down).
 */
TBoundingBox sortBounds(const TBoundingBox& bounds)
{
  TBoundingBox result;

  result.upper.x = (bounds.upper.x <= bounds.lower.x) ? bounds.upper.x : bounds.lower.x;
  result.lower.x = (bounds.upper.x <= bounds.lower.x) ? bounds.lower.x : bounds.upper.x;
  result.upper.y = (bounds.upper.y <= bounds.lower.y) ? bounds.upper.y : bounds.lower.y;
  result.lower.y = (bounds.upper.y <= bounds.lower.y) ? bounds.lower.y : bounds.upper.y;

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Determines whether both bounds overlap. 
 *
 * @param Bounds1 One of the bounds to compare.
 * @param Bounds2 The other bounds to compare.
 * @return True if both bounds overlap each other, otherwise false.
 */
bool boundsIntersect(const TBoundingBox& Bounds1, const TBoundingBox& Bounds2)
{
  return (Bounds1.lower.x >= Bounds2.upper.x) && (Bounds1.upper.x <= Bounds2.lower.x) && 
    (Bounds1.lower.y >= Bounds2.upper.y) && (Bounds1.upper.y <= Bounds2.lower.y);
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Determines whether the given bounds include the given point.
 *
 * @param bounds The bounds to check the point against.
 * @param x The horizontal coordinate to check.
 * @param y The vertical coordinate to check.
 * @return True if the point is within the bounds, otherwise false.
 */
bool boundsContainPoint(const TBoundingBox& bounds, const float x, const float y)                  
{
  return (bounds.lower.x >= x) && (bounds.upper.x <= x) && (bounds.lower.y >= y) && (bounds.upper.y <= y);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Examines the given bounds and returns whether it is empty or not.
 *
 * @param bounds The bounds to examine.
 * @return True if the bounds are empty, false otherwise.
 */
bool boundsAreEmpty(const TBoundingBox& bounds)
{
  return (bounds.upper.x == bounds.lower.x) && (bounds.upper.y == bounds.lower.y);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Looks the given container name up and returns an identifier for it that can be used for quick lookup/handling.
 *
 * @param container The name of the container.
 * @return An identifier that specifies which container was given.
 */
TContainerID getContainerID(const string& container)
{
  TContainerID result = GC_CONTAINER_UNKNOWN;

#ifdef __GNUC__
  static map<string, TContainerID> containers;
#else
  static hash_map<string, TContainerID> containers;
#endif

  // Initialize container mapping if not yet done.
  if (containers.size() == 0)
  {
    containers[""] = GC_CONTAINER_UNKNOWN;
    containers["layers"] = GC_CONTAINER_LAYERS;
    containers["feedback"] = GC_CONTAINER_FEEDBACK;
    containers["views"] = GC_CONTAINER_VIEWS;
    containers["model"] = GC_CONTAINER_MODEL;
    containers["style"] = GC_CONTAINER_STYLE;
    containers["styles"] = GC_CONTAINER_STYLES;
    containers["layouts"] = GC_CONTAINER_LAYOUTS;
    containers["figure"] = GC_CONTAINER_FIGURE;
    containers["figures"] = GC_CONTAINER_FIGURES;
    containers["figureInstances"] = GC_CONTAINER_FIGURE_INSTANCES;
    containers["content"] = GC_CONTAINER_FIGURE_CONTENT;
    containers["scaling"] = GC_CONTAINER_SCALING;
    containers["translation"] = GC_CONTAINER_TRANSLATION;
    containers["rotation"] = GC_CONTAINER_ROTATION;
    containers["groups"] = GC_CONTAINER_GROUPS;
    containers["children"] = GC_CONTAINER_CHILDREN;
    containers["caption"] = GC_CONTAINER_CAPTION;
    containers["content"] = GC_CONTAINER_CONTENT;
  };

  if (containers.find(container) != containers.end())
    result = containers[container];

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Looks up the property name and returns an identifier for it.
 *
 * @param property The property name to look up.
 * @return An identifier that specifies which property was given.
 */
TPropertyID getPropertyID(const string& property)
{
#ifdef __GNUC__
  static map<string, TPropertyID> properties;
#else
  static hash_map<string, TPropertyID> properties;
#endif
  TPropertyID result = GC_PROPERTY_UNKNOWN;

  // Fill property mapping first if not yet done.
  if (properties.size() == 0)
  {
    // Simple properties.
    properties["width"] = GC_PROPERTY_WIDTH;
    properties["height"] = GC_PROPERTY_HEIGHT;
    properties["x"] = GC_PROPERTY_X;
    properties["y"] = GC_PROPERTY_Y;
    properties["z"] = GC_PROPERTY_Z;
    properties["name"] = GC_PROPERTY_NAME;
    properties["_id"] = GC_PROPERTY_ID;
    properties["description"] = GC_PROPERTY_DESCRIPTION;
    properties["zoomX"] = GC_PROPERTY_ZOOMX;
    properties["zoomY"] = GC_PROPERTY_ZOOMY;
    properties["color"] = GC_PROPERTY_COLOR;
    properties["jitter"] = GC_PROPERTY_JITTER;
    properties["angle"] = GC_PROPERTY_ANGLE;
    properties["visible"] = GC_PROPERTY_VISIBLE;
    properties["enabled"] = GC_PROPERTY_ENABLED;
    properties["selected"] = GC_PROPERTY_SELECTED;
    properties["layout"] = GC_PROPERTY_LAYOUT;
    properties["resizable"] = GC_PROPERTY_RESIZABLE;
    properties["expanded"] = GC_PROPERTY_EXPANDED;
    properties["constraintMinWidth"] = GC_PROPERTY_MIN_WIDTH;
    properties["constraintMinHeight"] = GC_PROPERTY_MIN_HEIGHT;
    properties["constraintMaxWidth"] = GC_PROPERTY_MAX_WIDTH;
    properties["constraintMaxHeight"] = GC_PROPERTY_MAX_HEIGHT;
    properties["text"] = GC_PROPERTY_TEXT;
    properties["fontFamily"] = GC_PROPERTY_FONT_FAMILY;
    properties["fontSize"] = GC_PROPERTY_FONT_SIZE;
    properties["fontWeight"] = GC_PROPERTY_FONT_WEIGHT;
    properties["fontStyle"] = GC_PROPERTY_FONT_STYLE;
    properties["verticalAlignment"] = GC_PROPERTY_ALIGNMENT_VERTICAL;
    properties["horizontalAlignment"] = GC_PROPERTY_ALIGNMENT_HORIZONTAL;
    properties["bidiMode"] = GC_PROPERTY_BIDI_MODE;
    properties["owner"] = GC_PROPERTY_OWNER;
  };

  if (properties.find(property) != properties.end())
    result = properties[property];

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Treats the given path as property name preceeded with a slash. The first (or only) subpath must be an integer number
 * denoting an index in a list.
 *
 * @param Path [in, out] The path to examine. It must start with a slash to be valid. On return it contains a new 
 *                       path without the index part (can be empty then).
 * @return The integer value extracted from the top path part.
 */
int getEntryIndex(string& Path)
{
  int Result = -1;

  if (Path.size() > 0 && Path[0] == '/')
  {
    // Is there another path separator after the initial one?
    string::size_type Position = Path.find_first_of('/', 1);
    string Index;
    if (Position != Path.npos)
    {
      Index = Path.substr(1, Position - 1);
      Path = Path.substr(Position);
    }
    else
    {
      Index = Path.substr(1);
      Path = "";
    };
    Result = atoi(Index.c_str());
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

static TColorMap predefinedColors;

/**
 * Converts a color to a string in the form #RRGGBB.
 *
 * @param Color The color to convert.
 * @return The given color as HTML string.
 */
string colorToString(GLfloat* Color)
{
  char buffer[10];

  #ifdef _WINDOWS
    // Microsoft has confirmed that there is no guaranteed NULL termination (see MSDN for details).
    // Hence we better add one.
    _snprintf(buffer, sizeof(buffer), "#%.2X%.2X%.2X\0", ROUND(Color[0] * 255), ROUND(Color[1] * 255), ROUND(Color[2] * 255));
  #else
    snprintf(buffer, sizeof(buffer), "#%.2X%.2X%.2X", ROUND(Color[0] * 255), ROUND(Color[1] * 255), ROUND(Color[2] * 255));
  #endif // #ifdef _WINDOWS

  return buffer;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Converts a color to a string in the form #RRGGBB.
 *
 * @param Color The color to convert.
 * @return The given color as HTML string.
 */
string colorToString(GLubyte* Color)
{
  char buffer[10];

  #ifdef _WINDOWS
    // Microsoft has confirmed that there is no guaranteed NULL termination (see MSDN for details).
    // Hence we better add one.
    _snprintf(buffer, sizeof(buffer), "#%.2X%.2X%.2X\0", Color[0], Color[1], Color[2]);
  #else
    snprintf(buffer, sizeof(buffer), "#%.2X%.2X%.2X", Color[0], Color[1], Color[2]);
  #endif // #ifdef _WINDOWS

  return buffer;
}

//----------------------------------------------------------------------------------------------------------------------

// Converts the two hex digits given by upper and lower to an unsigned byte.
unsigned char hexToByte(char upper, char lower)
{
  upper -= '0';
  if (upper > 9)
    upper -= 7;
  lower -= '0';
  if (lower > 9)
    lower -= 7;
  return upper * 16 + lower;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Searchs the predifined colors and tries to find one with the given name.
 *
 * @param Name The name of the color to find.
 * @param Color [out] The color data if it could be found.
 * @returns true if the color could be found, otherwise false;
 */
bool colorByName(string Name, GLubyte* Color)
{

  // Fill the color map if not already done.
  if (predefinedColors.size() == 0)
  {
    for (unsigned int I = 0; I < COLOR_COUNT; I++)
      predefinedColors[Colors[I].Name] = Colors[I].Color;

#ifdef _WINDOWS
    for (unsigned int I = 0; I < SYS_COLOR_COUNT; I++)
    {
      COLORREF Reference = GetSysColor(SystemColors[I].Value);
      SystemColors[I].Color[0] = GetRValue(Reference);
      SystemColors[I].Color[1] = GetGValue(Reference);
      SystemColors[I].Color[2] = GetBValue(Reference);
      SystemColors[I].Color[3] = 1;
      predefinedColors[SystemColors[I].Name] = SystemColors[I].Color;
    };
#endif // #ifdef _WINDOWS
  };

  TColorMapIterator iterator = predefinedColors.find(Name);
  bool Result = iterator != predefinedColors.end();
  if (Result)
  {
    Color[0] = iterator->second[0];
    Color[1] = iterator->second[1];
    Color[2] = iterator->second[2];
    Color[3] = iterator->second[3];
  };

  return Result;
}


//----------------------------------------------------------------------------------------------------------------------

/**
 * Registers predifined colors.
 *
 * @param ColorMap Colors to add to the predefined color list.
 */
void registerSystemColors(const TColorMap &ColorMap)
{
  for (TColorMapIterator iterator = ColorMap.begin(); iterator != ColorMap.end(); ++iterator)
    predefinedColors[iterator->first]= iterator->second;
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Converts a string to a color with byte members.
 * The allowed syntax for colors is (as given by the SVG specification) either an HTML like
 * value (e.g. #FFFFFF, #FFF) or a function like form (e.g. rgb(100, 255, 255), rgb(10%, 100%, 0%)).
 *
 * @param ColorString The string to parse.
 * @param [out] The color converted from the string. It must have room for at least 3 members.
 * @return 
 *   0 - If a color could be found and converted.
 *   1 - If a color could be found but a conversion error occured.
 *   2 - No color was given.
 *   3 - The special color "none" was found.
 */
int stringToColor(string ColorString, GLubyte* Color)
{
  int Result = 2;
  
  if (ColorString.size() > 0)
  {
    Color[0] = 0;
    Color[1] = 0; 
    Color[2] = 0; 
    const char* Head = ColorString.c_str();
    const char* Tail;

    // Start by skipping leading white spaces. We have only simple ASCII compatible strings here,
    // so we don't need to care for conversion from UTF-8.
    while ((*Head != '\0') && ((*Head == ' ') || (*Head == '\t')))
      Head++;

    if (*Head != '\0')
    {
      if (*Head == '#')
      {
        // HTML color.
        Head++;
        Tail = Head;
        while ((*Tail >= '0' && *Tail <= '9') || (*Tail >= 'a' && *Tail <= 'f') || (*Tail >= 'A' && *Tail <= 'F'))
          Tail++;
        switch (Tail - Head)
        {
          // Support only the two defined cases. Other values are simply ignored.
          case 3:
            {
              Color[0] = hexToByte(*Head, *Head);
              Head++;
              Color[1] = hexToByte(*Head, *Head);
              Head++;
              Color[2] = hexToByte(*Head, *Head);
              Result = 0;

              break;
            };
          case 6:
            {
              Tail = Head + 1;
              Color[0] = hexToByte(*Head, *Tail);
              Head += 2; Tail += 2;
              Color[1] = hexToByte(*Head, *Tail);
              Head += 2; Tail += 2;
              Color[2] = hexToByte(*Head, *Tail);
              Result = 0;

              break;
            };
        }
      }
      else
        if (strcmp(Head, "none") == 0)
        {
          // Do not fill at all.
          Result = 3;
        }
        else
        {
          bool isRGB = strncmp(Head, "rgb(", 4) == 0;
          if (isRGB)
          {
            // Found a function call like specification. Split the entries and look if they are absolute
            // or percentage values.
            int Index = 0;
            int Value;
            int ComponentCount = 3;
            Head += ComponentCount + 1;
            while (Index < ComponentCount)
            {
              Value = 0;
              
              // Skip leading white spaces.
              while ((*Head != '\0') && ((*Head == ' ') || (*Head == '\t')))
                Head++;
              while ((*Head >= '0') && (*Head <= '9'))
                Value = Value * 10 + (*Head++ - '0');

              if (Value < 0)
                Value = 0;
              if (*Head == '%')
              {
                if (Value > 100)
                  Value = 100;
                Value = (Value * 255) / 100;
                Head++;
              }
              else
              {
                if (Value > 255)
                  Value = 255;
              };

              Color[Index++] = Value;

              // Skip trailing white spaces.
              while ((*Head != '\0') && ((*Head == ' ') || (*Head == '\t')))
                Head++;

              // If there is no comma or closing parenthesis then there is something wrong.
              if (*Head == ')') 
                break;
              if (*Head != ',')
                return 1;

              Head++;
            };
            if ((*Head == ')') && (Index == ComponentCount))
              Result = 0;
            else
              Result = 1;
          }
          else
          {
            // Last chance are color names. Try to find the text in the color constants.
            if (colorByName(Head, Color))
              Result = 0;
            else
              Result = 1;
          };
        };
    };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Converts a string to color with float members.
 * The allowed syntax for colors is (as given by the SVG specification) either an HTML like
 * value (e.g. #FFFFFF, #FFF) or a function like form (e.g. rgb(100, 255, 255), rgb(10%, 100%, 0%)).
 *
 * @param ColorString The string to parse.
 * @param [out] The color converted from the string. It must have room for at least 3 members.
 * @return 
 *   0 - If a color could be found and converted.
 *   1 - If a color could be found but a conversion error occured.
 *   2 - No color was given.
 *   3 - The special color "none" was found.
 */
int stringToColor(string ColorString, GLfloat* Color)
{
  GLubyte ByteColor[3];
  int Result = stringToColor(ColorString, ByteColor);
  if (Result == 0)
  {
    Color[0] = (float) ByteColor[0] / 255;
    Color[1] = (float) ByteColor[1] / 255;
    Color[2] = (float) ByteColor[2] / 255;
  };
  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Converts a GC variant to a string.
 *
 * @param Variant The variant to convert.
 * @return The value of the variant as string.
 */
string variantToString(const TGCVariant& Variant)
{
  string Result;

  switch (Variant.type)
  {
    case GC_VAR_UNKNOWN:
      {
        Result = "unknown";
        break;
      };
    case GC_VAR_BOOL:
      {
        Result = Variant.b ? "true" : "false";
        break;
      };
    case GC_VAR_INT:
      {
        char buffer[12];
        Result = itoa(Variant.i, buffer, 10);
        break;
      };
    case GC_VAR_FLOAT:
      {
        char buffer[100];
        #ifdef _WINDOWS
          // Microsoft has confirmed that there is no guaranteed NULL termination (see MSDN for details).
          // Hence we better add one.
          _snprintf(buffer, sizeof(buffer), "%gi\0", Variant.f);
        #else
          snprintf(buffer, sizeof(buffer), "%g", Variant.f);
        #endif // #ifdef _WINDOWS

        Result = buffer;
        break;
      };
    case GC_VAR_STRING:
      {
        Result = Variant.s;
        break;
      };
    case GC_VAR_LIST:
      {
        Result = "list";
        break;
      };
    case GC_VAR_OBJECT:
      {
        Result = "object";
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Converts a GC variant to an integer.
 *
 * @param Variant The variant to convert.
 * @return The value of the variant as integer.
 */
int variantToInt(const TGCVariant& Variant)
{
  int Result = 0;

  switch (Variant.type)
  {
    case GC_VAR_UNKNOWN:
      {
        Result = 0;
        break;
      };
    case GC_VAR_BOOL:
      {
        Result = Variant.b ? 1 : 0;
        break;
      };
    case GC_VAR_INT:
      {
        Result = Variant.i;
        break;
      };
    case GC_VAR_FLOAT:
      {
        Result = ROUND(Variant.f);
        break;
      };
    case GC_VAR_STRING:
      {
        Result = atoi(Variant.s.c_str());
        break;
      };
    case GC_VAR_LIST:
    case GC_VAR_OBJECT:
      {
        Result = 0;
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Converts a GC variant to a float.
 *
 * @param Variant The variant to convert.
 * @return The value of the variant as float.
 */
float variantToFloat(const TGCVariant& Variant)
{
  float Result = 0;

  switch (Variant.type)
  {
    case GC_VAR_UNKNOWN:
      {
        Result = 0;
        break;
      };
    case GC_VAR_BOOL:
      {
        Result = (float) (Variant.b ? 1 : 0);
        break;
      };
    case GC_VAR_INT:
      {
        Result = (float) Variant.i;
        break;
      };
    case GC_VAR_FLOAT:
      {
        Result = Variant.f;
        break;
      };
    case GC_VAR_STRING:
      {
        Result = (float) atof(Variant.s.c_str());
        break;
      };
    case GC_VAR_LIST:
    case GC_VAR_OBJECT:
      {
        Result = 0;
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Converts a GC variant to a bool.
 *
 * @param Variant The variant to convert.
 * @return The value of the variant as bool.
 */
bool variantToBool(const TGCVariant& Variant)
{
  bool Result = false;

  switch (Variant.type)
  {
    case GC_VAR_UNKNOWN:
      {
        Result = false;
        break;
      };
    case GC_VAR_BOOL:
      {
        Result = Variant.b;
        break;
      };
    case GC_VAR_INT:
      {
        Result = Variant.i > 0;
        break;
      };
    case GC_VAR_FLOAT:
      {
        Result = Variant.f > 0;
        break;
      };
    case GC_VAR_STRING:
      {
        if (Variant.s == "true" || Variant.s == "yes" || Variant.s == "t" || Variant.s == "y")
          Result = true;
        else
        {
          float Value = (float) atof(Variant.s.c_str());
          Result = Value > 0;
        };
        break;
      };
    case GC_VAR_LIST:
    case GC_VAR_OBJECT:
      {
        Result = Variant.reference != NULL;
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a GC variant from the given value.
 * 
 * @param Value The value to wrap in the variant.
 * @return A new variant carrying the given value.
 */
TGCVariant variant(const string& Value)
{
  TGCVariant Result;

  Result.type = GC_VAR_STRING;
  Result.s = Value;

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a GC variant from the given value.
 * 
 * @param Value The value to wrap in the variant.
 * @return A new variant carrying the given value.
 */
TGCVariant variant(const char* Value)
{
  TGCVariant Result;

  Result.type = GC_VAR_STRING;
  Result.s = Value;

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a GC variant from the given value.
 * 
 * @param Value The value to wrap in the variant.
 * @return A new variant carrying the given value.
 */
TGCVariant variant(const int Value)
{
  TGCVariant Result;

  Result.type = GC_VAR_INT;
  Result.i = Value;

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a GC variant from the given value.
 * 
 * @param Value The value to wrap in the variant.
 * @return A new variant carrying the given value.
 */
TGCVariant variant(const float Value)
{
  TGCVariant Result;

  Result.type = GC_VAR_FLOAT;
  Result.f = Value;

return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a GC variant from the given value.
 * 
 * @param Value The value to wrap in the variant.
 * @return A new variant carrying the given value.
 */
TGCVariant variant(const bool Value)
{
  TGCVariant Result;

  Result.type = GC_VAR_BOOL;
  Result.b = Value;

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

// The following matrix code was taken from Mesa3D (http://www.mesa3d.org/).

#define A(row,col)  a[(col<<2)+row]
#define B(row,col)  b[(col<<2)+row]
#define P(row,col)  product[(col<<2)+row]

/**
 * Perform a full 4x4 matrix multiplication.
 *
 * @param a matrix.
 * @param b matrix.
 * @param product will receive the product of a and b.
 *
 * @warning Is assumed that product != b. product == a is allowed.
 *
 * @note KW: 4*16 = 64 multiplications
 * 
 * @author This function was taken from Mesa3D (http://www.mesa3d.org/).
 */
static void matrixMultiply(TMatrix product, const TMatrix a, const TMatrix b)
{
   GLint i;
   for (i = 0; i < 4; i++) {
      const GLfloat ai0=A(i,0),  ai1=A(i,1),  ai2=A(i,2),  ai3=A(i,3);
      P(i,0) = ai0 * B(0,0) + ai1 * B(1,0) + ai2 * B(2,0) + ai3 * B(3,0);
      P(i,1) = ai0 * B(0,1) + ai1 * B(1,1) + ai2 * B(2,1) + ai3 * B(3,1);
      P(i,2) = ai0 * B(0,2) + ai1 * B(1,2) + ai2 * B(2,2) + ai3 * B(3,2);
      P(i,3) = ai0 * B(0,3) + ai1 * B(1,3) + ai2 * B(2,3) + ai3 * B(3,3);
   }
}

#undef A
#undef B
#undef P

//----------------------------------------------------------------------------------------------------------------------

/**
 * Generate a 4x4 transformation matrix from glRotate parameters, and post-multiply the input matrix by it.
 *
 * @param mat The target matrix to multiply the rotation into.
 * @param angle The angle around which to rotate.
 * @param x The x coordinate of the axis to turn around.
 * @param y The y coordinate of the axis to turn around.
 * @param z The z coordinate of the axis to turn around.
 * @author This function was taken from Mesa3D (http://www.mesa3d.org/).
 */
void matrixRotate(TMatrix mat, GLfloat angle, GLfloat x, GLfloat y, GLfloat z)
{
   GLfloat xx, yy, zz, xy, yz, zx, xs, ys, zs, one_c, s, c;
   TMatrix m;
   GLboolean optimized;

   s = (GLfloat) sin( angle * DEG2RAD );
   c = (GLfloat) cos( angle * DEG2RAD );

   memcpy(m, Identity, sizeof(TMatrix));
   optimized = GL_FALSE;

#define M(row,col)  m[col * 4 + row]

   if (x == 0.0F) {
      if (y == 0.0F) {
         if (z != 0.0F) {
            optimized = GL_TRUE;
            /* rotate only around z-axis */
            M(0,0) = c;
            M(1,1) = c;
            if (z < 0.0F) {
               M(0,1) = s;
               M(1,0) = -s;
            }
            else {
               M(0,1) = -s;
               M(1,0) = s;
            }
         }
      }
      else if (z == 0.0F) {
         optimized = GL_TRUE;
         /* rotate only around y-axis */
         M(0,0) = c;
         M(2,2) = c;
         if (y < 0.0F) {
            M(0,2) = -s;
            M(2,0) = s;
         }
         else {
            M(0,2) = s;
            M(2,0) = -s;
         }
      }
   }
   else if (y == 0.0F) {
      if (z == 0.0F) {
         optimized = GL_TRUE;
         /* rotate only around x-axis */
         M(1,1) = c;
         M(2,2) = c;
         if (x < 0.0F) {
            M(1,2) = s;
            M(2,1) = -s;
         }
         else {
            M(1,2) = -s;
            M(2,1) = s;
         }
      }
   }

   if (!optimized) {
      const GLfloat mag = sqrt(x * x + y * y + z * z);

      if (mag <= 1.0e-4) {
         /* no rotation, leave mat as-is */
         return;
      }

      x /= mag;
      y /= mag;
      z /= mag;


      /*
       *     Arbitrary axis rotation matrix.
       *
       *  This is composed of 5 matrices, Rz, Ry, T, Ry', Rz', multiplied
       *  like so:  Rz * Ry * T * Ry' * Rz'.  T is the final rotation
       *  (which is about the x-axis), and the two composite transforms
       *  Ry' * Rz' and Rz * Ry are (respectively) the rotations necessary
       *  from the arbitrary axis to the x-axis then back.  They are
       *  all elementary rotations.
       *
       *  Rz' is a rotation about the z-axis, to bring the axis vector
       *  into the x-z plane.  Then Ry' is applied, rotating about the
       *  y-axis to bring the axis vector parallel with the x-axis.  The
       *  rotation about the x-axis is then performed.  Ry and Rz are
       *  simply the respective inverse transforms to bring the arbitrary
       *  axis back to it's original orientation.  The first transforms
       *  Rz' and Ry' are considered inverses, since the data from the
       *  arbitrary axis gives you info on how to get to it, not how
       *  to get away from it, and an inverse must be applied.
       *
       *  The basic calculation used is to recognize that the arbitrary
       *  axis vector (x, y, z), since it is of unit length, actually
       *  represents the sines and cosines of the angles to rotate the
       *  x-axis to the same orientation, with theta being the angle about
       *  z and phi the angle about y (in the order described above)
       *  as follows:
       *
       *  cos ( theta ) = x / sqrt ( 1 - z^2 )
       *  sin ( theta ) = y / sqrt ( 1 - z^2 )
       *
       *  cos ( phi ) = sqrt ( 1 - z^2 )
       *  sin ( phi ) = z
       *
       *  Note that cos ( phi ) can further be inserted to the above
       *  formulas:
       *
       *  cos ( theta ) = x / cos ( phi )
       *  sin ( theta ) = y / sin ( phi )
       *
       *  ...etc.  Because of those relations and the standard trigonometric
       *  relations, it is pssible to reduce the transforms down to what
       *  is used below.  It may be that any primary axis chosen will give the
       *  same results (modulo a sign convention) using thie method.
       *
       *  Particularly nice is to notice that all divisions that might
       *  have caused trouble when parallel to certain planes or
       *  axis go away with care paid to reducing the expressions.
       *  After checking, it does perform correctly under all cases, since
       *  in all the cases of division where the denominator would have
       *  been zero, the numerator would have been zero as well, giving
       *  the expected result.
       */

      xx = x * x;
      yy = y * y;
      zz = z * z;
      xy = x * y;
      yz = y * z;
      zx = z * x;
      xs = x * s;
      ys = y * s;
      zs = z * s;
      one_c = 1.0F - c;

      /* We already hold the identity-matrix so we can skip some statements */
      M(0,0) = (one_c * xx) + c;
      M(0,1) = (one_c * xy) - zs;
      M(0,2) = (one_c * zx) + ys;
/*    M(0,3) = 0.0F; */

      M(1,0) = (one_c * xy) + zs;
      M(1,1) = (one_c * yy) + c;
      M(1,2) = (one_c * yz) - xs;
/*    M(1,3) = 0.0F; */

      M(2,0) = (one_c * zx) - ys;
      M(2,1) = (one_c * yz) + xs;
      M(2,2) = (one_c * zz) + c;
/*    M(2,3) = 0.0F; */

/*
      M(3,0) = 0.0F;
      M(3,1) = 0.0F;
      M(3,2) = 0.0F;
      M(3,3) = 1.0F;
*/
   }
#undef M

   matrixMultiply(mat, mat, m);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Multiply a matrix with a general scaling matrix.
 *
 * @param mat matrix.
 * @param x x axis scale factor.
 * @param y y axis scale factor.
 * @param z z axis scale factor.
 *
 * Multiplies in-place the elements of mat by the scale factors.
 *
 * @author This function was taken from Mesa3D (http://www.mesa3d.org/).
 */
void matrixScale(TMatrix mat, GLfloat x, GLfloat y, GLfloat z)
{
  mat[0] *= x; mat[4] *= y; mat[8]  *= z;
  mat[1] *= x; mat[5] *= y; mat[9]  *= z;
  mat[2] *= x; mat[6] *= y; mat[10] *= z;
  mat[3] *= x; mat[7] *= y; mat[11] *= z;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Multiply a matrix with a translation matrix.
 *
 * @param mat matrix.
 * @param x translation vector x coordinate.
 * @param y translation vector y coordinate.
 * @param z translation vector z coordinate.
 *
 * Adds the translation coordinates to the elements of mat in-place.
 *
 * @author This function was taken from Mesa3D (http://www.mesa3d.org/).
*/
void matrixTranslate(TMatrix mat, GLfloat x, GLfloat y, GLfloat z)
{
  mat[12] = mat[0] * x + mat[4] * y + mat[8]  * z + mat[12];
  mat[13] = mat[1] * x + mat[5] * y + mat[9]  * z + mat[13];
  mat[14] = mat[2] * x + mat[6] * y + mat[10] * z + mat[14];
  mat[15] = mat[3] * x + mat[7] * y + mat[11] * z + mat[15];
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Multiplies the given vertex by matrix M and returns the result.
 *
 * @param M The matrix containing the transformation parameters.
 * @param V The vertex to transform.
 * @return The transformed vertex.
 */
TVertex matrixTransform(TMatrix M, TVertex V)
{
  TVertex result;

  result.x = V.x * M[0] + V.y * M[4] + V.z * M[8] + V.w * M[12];
  result.y = V.x * M[1] + V.y * M[5] + V.z * M[9] + V.w * M[13];
  result.z = V.x * M[2] + V.y * M[6] + V.z * M[10] + V.w * M[14];
  result.w = V.x * M[3] + V.y * M[7] + V.z * M[11] + V.w * M[15];

  return result;
}

//----------------- CBoundingBoxComputer -------------------------------------------------------------------------------

CBoundingBoxComputer::CBoundingBoxComputer(void)
{
  // The bound box initializes itself to all zero.
}

//----------------------------------------------------------------------------------------------------------------------

CBoundingBoxComputer::CBoundingBoxComputer(const TBoundingBox& InitialBox)
{
  FCurrentBox = InitialBox;
}

//----------------------------------------------------------------------------------------------------------------------

void CBoundingBoxComputer::includeVertex(const TVertex& Vertex)
{
  if (Vertex.x < FCurrentBox.upper.x)
    FCurrentBox.upper.x = Vertex.x;
  if (Vertex.x > FCurrentBox.lower.x)
    FCurrentBox.lower.x = Vertex.x;
  if (Vertex.y < FCurrentBox.upper.y)
    FCurrentBox.upper.y = Vertex.y;
  if (Vertex.y > FCurrentBox.lower.y)
    FCurrentBox.lower.y = Vertex.y;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Transforms the given coordinate using the provided matrix.
 *
 * @param Matrix The matrix with which to do the transformation. It is a usual 4x4 matrix, although only the upper-left 
 *               2x2 entries are used.
 * @param x The x coordinate.
 * @param y The y coordinate.
 * @return The transformed vertex.
 */
TVertex CBoundingBoxComputer::transform(TMatrix Matrix, const float& x, const float& y)
{
  TVertex Result;

  Result.x = x * Matrix[0] + y * Matrix[4];
  Result.y = x * Matrix[1] + y * Matrix[5];

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current bounding box.
 *
 * @return The current bounding box.
 */
TBoundingBox CBoundingBoxComputer::boundingBox()
{
  return FCurrentBox;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes the new box and merges it with the current bounding box. The new data is not required to be ordered.
 *
 * @param NewBox A new bounding box to merge in.
 */
void CBoundingBoxComputer::include(TMatrix Matrix, TBoundingBox* NewBox)
{
  float LLX;
  float URX;
  if (NewBox->upper.x < NewBox->lower.x)
  {
    LLX = NewBox->upper.x;
    URX = NewBox->lower.x;
  }
  else
  {
    LLX = NewBox->lower.x;
    URX = NewBox->upper.x;
  };

  float LLY;
  float URY;
  if (NewBox->upper.y < NewBox->lower.y)
  {
    LLY = NewBox->upper.y;
    URY = NewBox->lower.y;
  }
  else
  {
    LLY = NewBox->lower.y;
    URY = NewBox->upper.y;
  };

  float LLZ;
  float URZ;
  if (NewBox->upper.z < NewBox->lower.z)
  {
    LLZ = NewBox->upper.z;
    URZ = NewBox->lower.z;
  }
  else
  {
    LLZ = NewBox->lower.z;
    URZ = NewBox->upper.z;
  };

  TVertex V;
  if (Matrix != NULL)
    V = transform(Matrix, LLX, LLY);
  else
  {
    V.x = LLX;
    V.y = LLY;
  };
  includeVertex(V);
  if (Matrix != NULL)
    V = transform(Matrix, URX, URY);
  else
  {
    V.x = URX;
    V.y = URY;
  };
  includeVertex(V);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes the vertex and merges it with the current bounding box.
 *
 * @param Vertex The vertex to be included
 */
void CBoundingBoxComputer::include(TMatrix Matrix, const TVertex& Vertex)
{
  TVertex V;
  if (Matrix != NULL)
    V = transform(Matrix, Vertex.x, Vertex.y);
  else
    V = Vertex;
  includeVertex(V);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes the x and y values and merges them with the current bounding box.
 *
 * @param x The x coordinate of the point to include.
 * @param y The y coordinate of the point to include.
 */
void CBoundingBoxComputer::include(TMatrix Matrix, const float& x, const float& y)
{
  TVertex V;
  if (Matrix != NULL)
    V = transform(Matrix, x, y);
  else
  {
    V.x = x;
    V.y = y;
  };
  includeVertex(V);
}

//----------------------------------------------------------------------------------------------------------------------

