/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#include <myx_public_interface.h>
#include <myx_grt_public_interface.h>
#include <myx_grt_builtin_module_public_interface.h>

#include "myx_grt_mysql_transformation.h"
#include "myx_aux_functions.h"


static void generate_sql_create_statements_assets(MYX_GRT_VALUE *schema, const char *assets_name, void *data);
MYX_GRT_VALUE * execute_sql_statements(MYX_GRT_VALUE *param, void *data);

static char *get_sql_create_script_assets(MYX_GRT_VALUE *schema, const char *assets_name);

static char * get_sql_create_schema(MYX_GRT *grt, MYX_GRT_VALUE *schema);
static char * get_sql_create_table(MYX_GRT *grt, MYX_GRT_VALUE *table);
static char * get_sql_create_view(MYX_GRT *grt, MYX_GRT_VALUE *view);
static char * get_sql_create_procedure(MYX_GRT *grt, MYX_GRT_VALUE *procedure);

static char * get_sql_drop_schema(MYX_GRT *grt, MYX_GRT_VALUE *schema);
static char * get_sql_drop_table(MYX_GRT *grt, MYX_GRT_VALUE *table);
static char * get_sql_drop_view(MYX_GRT *grt, MYX_GRT_VALUE *view);
static char * get_sql_drop_procedure(MYX_GRT *grt, MYX_GRT_VALUE *procedure);



MYX_GRT_VALUE *generate_sql_create_statements(MYX_GRT_VALUE *param, void *data)
{
//  MYX_GRT *grt= (MYX_GRT *) data;

  if (!param)
    return myx_grt_function_create_error_result("You have to submit one object as parameter.", NULL);

  if ( (myx_grt_value_get_type(param) != MYX_DICT_VALUE) || 
    (strcmp2(myx_grt_dict_struct_get_name(param), "db.mysql.Catalog") != 0) )
    return myx_grt_function_create_error_result("The submitted parameter has to be a dict value of the struct db.mysql.catalog.", NULL);
  else
  {
    MYX_GRT_VALUE *schemata= myx_grt_dict_item_get_value(param, "schemata");
    unsigned int i;

    for (i= 0; i < myx_grt_list_item_count(schemata); i++)
    {
      MYX_GRT_VALUE *schema= myx_grt_list_item_get(schemata, i);
      MYX_GRT_VALUE *schema_create_string= 
        myx_grt_function_extract_value_from_result(
          get_sql_create(schema, data));

      myx_grt_dict_item_set_value(schema, "sql", schema_create_string);
      myx_grt_value_release(schema_create_string);

      generate_sql_create_statements_assets(schema, "tables", data);
      generate_sql_create_statements_assets(schema, "views", data);
      generate_sql_create_statements_assets(schema, "procedures", data);
    }
  }

  return myx_grt_function_create_result(NULL);
}

MYX_GRT_VALUE * execute_sql_statements(MYX_GRT_VALUE *param, void *data)
{
  // to be implemented
  return myx_grt_function_create_result(NULL);
}

static void generate_sql_create_statements_assets(MYX_GRT_VALUE *schema, const char *assets_name, void *data)
{
  MYX_GRT_VALUE *assets= myx_grt_dict_item_get_value(schema, assets_name);
  unsigned int i;

  if (assets)
  {
    for (i= 0; i < myx_grt_list_item_count(assets); i++)
    {
      MYX_GRT_VALUE *asset= myx_grt_list_item_get(assets, i);
      MYX_GRT_VALUE *asset_create_string= 
          myx_grt_function_extract_value_from_result(
            get_sql_create(asset, data));

      myx_grt_dict_item_set_value(asset, "sql", asset_create_string);
      myx_grt_value_release(asset_create_string);
    }
  }
}

MYX_GRT_VALUE *get_sql_create_script(MYX_GRT_VALUE *param, void *data)
{
//  MYX_GRT *grt= (MYX_GRT *) data;
  MYX_GRT_VALUE *result= NULL;
  char *sql;

  if (!param)
    return myx_grt_function_create_error_result("You have to submit one object as parameter.", NULL);

  if ( (myx_grt_value_get_type(param) != MYX_DICT_VALUE) || 
    (strcmp2(myx_grt_dict_struct_get_name(param), "db.mysql.Catalog") != 0) )
    return myx_grt_function_create_error_result("The submitted parameter has to be a dict value of the struct db.mysql.catalog.", NULL);
  else
  {
    MYX_GRT_VALUE *schemata= myx_grt_dict_item_get_value(param, "schemata");
    unsigned int i;

    sql= g_strdup(
        "-- ----------------------------------------------------------------------" _br
		    "-- SQL create script" _br
		    "-- ----------------------------------------------------------------------" _br 
        _br
        "SET FOREIGN_KEY_CHECKS = 0;" _br _br);

    for (i= 0; i < myx_grt_list_item_count(schemata); i++)
    {
      MYX_GRT_VALUE *schema= myx_grt_list_item_get(schemata, i);
      MYX_GRT_VALUE *schema_drop_string= get_sql_drop(schema, data);

      // add drop schema
      sql= str_g_append(sql, myx_grt_dict_item_get_as_string(schema_drop_string, "value"));
      sql= str_g_append(sql, _br _br);

      myx_grt_value_release(schema_drop_string);

      sql= str_g_append(sql, myx_grt_dict_item_get_as_string(schema, "sql"));
      sql= str_g_append(sql, _br _br);

      sql= str_g_append_and_free(sql,
          get_sql_create_script_assets(schema, "tables"));
      sql= str_g_append(sql, _br _br);

      sql= str_g_append_and_free(sql,
          get_sql_create_script_assets(schema, "views"));
      sql= str_g_append(sql, _br _br);

      sql= str_g_append_and_free(sql,
          get_sql_create_script_assets(schema, "procedures"));
      sql= str_g_append(sql, _br _br);
    }

    sql= str_g_append(sql, "SET FOREIGN_KEY_CHECKS = 1;" _br _br);
  }

  result= myx_grt_function_create_result(myx_grt_value_from_string(sql));
  g_free(sql);

  return result;
}

static char *get_sql_create_script_assets(MYX_GRT_VALUE *schema, const char *assets_name)
{
  char *sql= NULL;
  MYX_GRT_VALUE *assets= myx_grt_dict_item_get_value(schema, assets_name);
  unsigned int i;

  for (i= 0; i < myx_grt_list_item_count(assets); i++)
  {
    MYX_GRT_VALUE *asset= myx_grt_list_item_get(assets, i);

    sql= str_g_append(sql, myx_grt_dict_item_get_as_string(asset, "sql"));
    sql= str_g_append(sql, _br _br);
  }

  return sql;
}

MYX_GRT_VALUE *get_sql_create(MYX_GRT_VALUE *param, void *data)
{
  MYX_GRT *grt= (MYX_GRT *) data;
  MYX_GRT_VALUE *result= NULL;
  const char *struct_name;
  char *sql= NULL;

  if (!param)
    return myx_grt_function_create_error_result("You have to submit one object as parameter.", NULL);

  if (myx_grt_value_get_type(param) != MYX_DICT_VALUE)
    return myx_grt_function_create_error_result("The submitted parameter has to be a dict value.", NULL);

  struct_name= myx_grt_dict_struct_get_name(param);

  if (strcmp2(struct_name, "db.mysql.Schema") == 0)
    sql= get_sql_create_schema(grt, param);
  else if (strcmp2(struct_name, "db.mysql.Table") == 0)
    sql= get_sql_create_table(grt, param);
  else if (strcmp2(struct_name, "db.mysql.View") == 0)
    sql= get_sql_create_view(grt, param);
  else if (strcmp2(struct_name, "db.mysql.Procedure") == 0)
    sql= get_sql_create_procedure(grt, param);
  else
    return myx_grt_function_create_error_result("The struct of the submitted dict is of a supported type.", NULL);

  result= myx_grt_function_create_result(myx_grt_value_from_string(sql));
  g_free(sql);

  return result;    
}

MYX_GRT_VALUE *get_sql_drop(MYX_GRT_VALUE *param, void *data)
{
  MYX_GRT *grt= (MYX_GRT *) data;
  MYX_GRT_VALUE *result= NULL;
  const char *struct_name;
  char *sql= NULL;

  if (!param)
    return myx_grt_function_create_error_result("You have to submit one object as parameter.", NULL);

  if (myx_grt_value_get_type(param) != MYX_DICT_VALUE)
    return myx_grt_function_create_error_result("The submitted parameter has to be a dict value.", NULL);

  struct_name= myx_grt_dict_struct_get_name(param);

  if (strcmp2(struct_name, "db.mysql.Schema") == 0)
    sql= get_sql_drop_schema(grt, param);
  else if (strcmp2(struct_name, "db.mysql.Table") == 0)
    sql= get_sql_drop_table(grt, param);
  else if (strcmp2(struct_name, "db.mysql.View") == 0)
    sql= get_sql_drop_view(grt, param);
  else if (strcmp2(struct_name, "db.mysql.Procedure") == 0)
    sql= get_sql_drop_procedure(grt, param);
  else
    return myx_grt_function_create_error_result("The struct of the submitted dict is of a supported type.", NULL);

  result= myx_grt_function_create_result(myx_grt_value_from_string(sql));
  g_free(sql);

  return result;    
}

MYX_GRT_VALUE *get_sql_merge(MYX_GRT_VALUE *param, void *data)
{
//  MYX_GRT *grt= (MYX_GRT *) data;
//  MYX_GRT_VALUE *result= NULL;
  MYX_GRT_VALUE *existing= NULL;
  MYX_GRT_VALUE *altered= NULL;
//  char *sql= NULL;

  if (!((myx_grt_value_get_type(param) == MYX_LIST_VALUE) && (myx_grt_list_item_count(param) != 2)))
    return myx_grt_function_create_error_result("The submitted parameter has to be a list with two elements.", 
      "TransformationMySQL:getMergeSQL(GrtObject alteredObject, GrtObject existingObject);");

  altered= myx_grt_list_item_get(param, 0);
  if (myx_grt_value_get_type(altered) != MYX_DICT_VALUE)
    return myx_grt_function_create_error_result("The first parameter has to be a DICT.", NULL);

  existing= myx_grt_list_item_get(param, 1);

  if (existing)
  {
    // There is an existing object so make an ALTER statement

    return myx_grt_function_create_error_result("Not implemented yet.", NULL);

    /*if (sql)
    {
      result= myx_grt_function_create_result(myx_grt_value_from_string(sql));
      g_free(sql);

      return result;
    }*/
  }
  else
  {
    // There is no existing object, so make a CREATE statement
    return get_sql_create(altered, data);
  }
}

static char * get_sql_create_schema(MYX_GRT *grt, MYX_GRT_VALUE *schema)
{
  char *quote_char= g_strdup("`");
  const char *schema_name= myx_grt_dict_item_get_as_string(schema, "name");
  const char *charset= myx_grt_dict_item_get_as_string(schema, "defaultCharacterSetName");
  const char *collation= myx_grt_dict_item_get_as_string(schema, "defaultCollationName");

  char *sql= g_strdup_printf(
      "CREATE DATABASE %s%s%s" _br,
      quote_char, schema_name, quote_char);

  if ((charset) && (charset[0]))
  {
    sql = str_g_append_and_free(sql, g_strdup_printf("  CHARACTER SET %s", charset));

    if ((collation) && (collation[0]))
      sql = str_g_append_and_free(sql, g_strdup_printf(" COLLATE %s" _br, collation));
    else
      sql = str_g_append(sql, _br);
  }

  // remove last \n
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  sql[strlen(sql)-2]= 0;
#else
  sql[strlen(sql)-1]= 0;
#endif
  sql= str_g_append(sql, ";");

  g_free(quote_char);

  return sql;
}

static char * get_owner_name(MYX_GRT *grt, MYX_GRT_VALUE *obj, const char *quote_char)
{
  MYX_GRT_VALUE *owner= myx_grt_dict_item_get_reference_value(grt, obj, "owner");

  if (owner)
  {
    const char *owner_name= myx_grt_dict_item_get_as_string(owner, "name");

    return g_strdup_printf("%s%s%s.",
      quote_char, owner_name, quote_char);
  }
  else
    return g_strdup("");
}

static char * get_sql_create_table(MYX_GRT *grt, MYX_GRT_VALUE *table)
{
  char *quote_char= g_strdup("`");
  unsigned int i;
  const char *table_name= myx_grt_dict_item_get_as_string(table, "name");
  char *quoted_schema_name= get_owner_name(grt, table, quote_char);
  MYX_GRT_VALUE *columns= myx_grt_dict_item_get_value(table, "columns");
  MYX_GRT_VALUE *pk= myx_grt_dict_item_get_value(table, "primaryKey");
  MYX_GRT_VALUE *indices= myx_grt_dict_item_get_value(table, "indices");
  MYX_GRT_VALUE *fks= myx_grt_dict_item_get_value(table, "foreignKeys");
  char *sql= NULL;
  char *sql_cols= NULL;
  char *sql_pk= NULL;
  char *sql_indices= NULL;
  char *sql_fk= NULL;
  const char *opt_engine= myx_grt_dict_item_get_as_string(table, "tableEngine");
  const char *opt_next_auto_inc= myx_grt_dict_item_get_as_string(table, "nextAutoInc");
  const char *opt_password= myx_grt_dict_item_get_as_string(table, "password");
  int opt_delay_key_write= myx_grt_dict_item_get_as_int(table, "delayKeyWrite");
  const char *opt_charset= myx_grt_dict_item_get_as_string(table, "defaultCharacterSetName");
  const char *opt_collation= myx_grt_dict_item_get_as_string(table, "defaultCollationName");
  const char *opt_comment= myx_grt_dict_item_get_as_string(table, "tableComment");
  const char *opt_merge_union= myx_grt_dict_item_get_as_string(table, "mergeUnion");
  const char *opt_merge_insert= myx_grt_dict_item_get_as_string(table, "mergeInsert");
  const char *opt_data_dir= myx_grt_dict_item_get_as_string(table, "tableDataDir");
  const char *opt_index_dir= myx_grt_dict_item_get_as_string(table, "tableIndexDir");
  const char *opt_pack_keys= myx_grt_dict_item_get_as_string(table, "packKeys");
  const char *opt_raid_type= myx_grt_dict_item_get_as_string(table, "raidType");
  const char *opt_raid_chunks= myx_grt_dict_item_get_as_string(table, "raidChunks");
  const char *opt_raid_chunk_size= myx_grt_dict_item_get_as_string(table, "raidChunkSize");
  int opt_checksum= myx_grt_dict_item_get_as_int(table, "checksum");
  const char *opt_row_format= myx_grt_dict_item_get_as_string(table, "rowFormat");
  const char *opt_avg_row_length= myx_grt_dict_item_get_as_string(table, "avgRowLength");
  const char *opt_min_rows= myx_grt_dict_item_get_as_string(table, "minRows");
  const char *opt_max_rows= myx_grt_dict_item_get_as_string(table, "maxRows");

  // build create table sql
  sql= g_strdup_printf(
    "CREATE TABLE %s%s%s%s (" _br, 
    quoted_schema_name,
    quote_char, table_name, quote_char);

  // --------------------------------------------------------------------------------------------------------------------
  // get columns sql
  for (i= 0; i < myx_grt_list_item_count(columns); i++)
  {
    MYX_GRT_VALUE *column= myx_grt_list_item_get(columns, i);
    const char *is_nullable= (myx_grt_dict_item_get_as_int(column, "isNullable") == 1) ? "NULL" : "NOT NULL";
    const char *datatype= myx_grt_dict_item_get_as_string(column, "datatypeName");
    const char *datatype_explicit_params= myx_grt_dict_item_get_as_string(column, "datatypeExplicitParams");
    const char *datatype_unsigned= (myx_grt_dict_item_get_as_int(column, "unsigned") == 1) ? " UNSIGNED" : "";
    char *datatype_definition;
    char *column_comment;
    char *column_length= myx_grt_dict_item_get_formated_as_string(column, "length");
    char *column_precision= myx_grt_dict_item_get_formated_as_string(column, "precision");
    char *column_scale= myx_grt_dict_item_get_formated_as_string(column, "scale");

    // if there is an explicit parameter definition for the datatype, use it
    if (datatype_explicit_params && datatype_explicit_params[0])
      datatype_definition= g_strdup_printf("%s%s", datatype, datatype_explicit_params);
    else
    {
      // append length / precision / scale to datatype definition
      if ((strcmp2(datatype, "CHAR") == 0) ||
         (strcmp2(datatype, "VARCHAR") == 0))
      {
        datatype_definition= g_strdup_printf("%s(%s)", 
          datatype, column_length);
      }
      else if ((strcmp2(datatype, "TINYINT") == 0) ||
         (strcmp2(datatype, "SMALLINT") == 0) ||
         (strcmp2(datatype, "MEDIUMINT") == 0) ||
         (strcmp2(datatype, "INT") == 0) ||
         (strcmp2(datatype, "INTEGER") == 0) ||
         (strcmp2(datatype, "BIGINT") == 0))
      {
        if (strcmp2(column_precision, "0") != 0)
        {
          datatype_definition= g_strdup_printf("%s(%s)", 
            datatype, 
            column_precision);
        }
        else
          datatype_definition= g_strdup(datatype);
      }
      else if ((strcmp2(datatype, "REAL") == 0) ||
         (strcmp2(datatype, "DOUBLE") == 0) ||
         (strcmp2(datatype, "FLOAT") == 0) ||
         (strcmp2(datatype, "DECIMAL") == 0) ||
         (strcmp2(datatype, "NUMERIC") == 0))
      {
        datatype_definition= g_strdup_printf("%s(%s, %s)", 
          datatype,
          column_precision,
          column_scale);
      }
      else
        datatype_definition= g_strdup(datatype);
    }

    sql_cols= str_g_append_and_free(sql_cols, g_strdup_printf("  %s%s%s %s%s %s", 
      quote_char,
      myx_grt_dict_item_get_as_string(column, "name"), 
      quote_char,
      datatype_definition,
      datatype_unsigned,
      is_nullable));

    // add column auto increment
    if (myx_grt_dict_item_get_as_int(column, "autoIncrement") == 1)
      sql_cols= str_g_append(sql_cols, " AUTO_INCREMENT");

    // add column comment
    column_comment= g_strdup(myx_grt_dict_item_get_as_string(column, "columnComment"));
    if (column_comment && column_comment[0])
    {
      column_comment= str_g_replace(column_comment, "'", "\\'");
      sql_cols= str_g_append_and_free(sql_cols, g_strdup_printf(" COMMENT '%s'", column_comment));
    }

    sql_cols= str_g_append(sql_cols, "," _br);

    g_free(datatype_definition);
    g_free(column_comment);
    g_free(column_length);
    g_free(column_precision);
    g_free(column_scale);
  }

  // --------------------------------------------------------------------------------------------------------------------
  // get primary key sql
  if (pk)
  {
    MYX_GRT_VALUE *pk_columns= myx_grt_dict_item_get_value(pk, "columns");
    char *sql_pk_cols= NULL;

    for (i= 0; i < myx_grt_list_item_count(pk_columns); i++)
    {
      MYX_GRT_VALUE *column= myx_grt_list_item_get(columns, i);

      if (sql_pk_cols)
        sql_pk_cols= str_g_append(sql_pk_cols, ", ");

      sql_pk_cols= str_g_append_and_free(sql_pk_cols, g_strdup_printf("%s%s%s", 
          quote_char,
          myx_grt_dict_item_get_as_string(column, "name"),
          quote_char));
    }

    if (sql_pk_cols)
      sql_pk= str_g_append_and_free(sql_pk, g_strdup_printf("  PRIMARY KEY (%s)," _br, 
        sql_pk_cols));

    g_free(sql_pk_cols);
  }

  // --------------------------------------------------------------------------------------------------------------------
  // get indices sql
  if (indices)
  {
    for (i= 0; i < myx_grt_list_item_count(indices); i++)
    {
      MYX_GRT_VALUE *index= myx_grt_list_item_get(indices, i);
      const char *index_name= myx_grt_dict_item_get_as_string(index, "name");
      const char *unique= (myx_grt_dict_item_get_as_int(index, "unique") == 1) ? "UNIQUE " : "";
      unsigned int j;

      MYX_GRT_VALUE *index_columns= myx_grt_dict_item_get_value(index, "columns");
      char *sql_index_cols= NULL;

      for (j= 0; j < myx_grt_list_item_count(index_columns); j++)
      {
        MYX_GRT_VALUE *index_col= myx_grt_list_item_get(index_columns, j);
        int column_length= myx_grt_dict_item_get_as_int(index_col, "columnLength");
        MYX_GRT_VALUE *column_ref= myx_grt_dict_item_get_reference_value(grt, index_col, "referedColumn");
        const char *column_name= myx_grt_dict_item_get_as_string(column_ref, "name");

        if (sql_index_cols)
          sql_index_cols= str_g_append(sql_index_cols, ", ");

        sql_index_cols= str_g_append_and_free(sql_index_cols, g_strdup_printf("%s%s%s", 
            quote_char, column_name, quote_char));

        if (column_length > 0)
          sql_index_cols= str_g_append_and_free(sql_index_cols, g_strdup_printf("(%d)", 
            column_length));
      }

      sql_indices= str_g_append_and_free(sql_indices, 
        g_strdup_printf("  %sINDEX %s%s%s (%s)," _br, 
          unique,
          quote_char, index_name, quote_char,
          sql_index_cols));

      g_free(sql_index_cols);
    }
  }

  // --------------------------------------------------------------------------------------------------------------------
  // get foreign keys sql
  if (fks)
  {
    for (i= 0; i < myx_grt_list_item_count(fks); i++)
    {
      MYX_GRT_VALUE *fk= myx_grt_list_item_get(fks, i);
      const char *fk_name= myx_grt_dict_item_get_as_string(fk, "name");
      const char *fk_ref_schema_name= myx_grt_dict_item_get_as_string(fk, "referedTableSchemaName");
      const char *fk_ref_table_name= myx_grt_dict_item_get_as_string(fk, "referedTableName");

      MYX_GRT_VALUE *fk_columns= myx_grt_dict_item_get_value(fk, "columns");
      MYX_GRT_VALUE *fk_ref_columns= myx_grt_dict_item_get_value(fk, "referedColumnNames");
      const char *delete_rule= myx_grt_dict_item_get_as_string(fk, "deleteRule");
      const char *update_rule= myx_grt_dict_item_get_as_string(fk, "updateRule");
      char *sql_fk_cols= NULL;
      char *sql_fk_ref_cols= NULL;

      unsigned int j;

      for (j= 0; j < myx_grt_list_item_count(fk_columns); j++)
      {
        MYX_GRT_VALUE *column= myx_grt_list_item_get_reference_value(grt, fk_columns, j);

        if (sql_fk_cols)
          sql_fk_cols= str_g_append(sql_fk_cols, ", ");

        sql_fk_cols= str_g_append_and_free(sql_fk_cols, g_strdup_printf("%s%s%s", 
            quote_char,
            myx_grt_dict_item_get_as_string(column, "name"),
            quote_char));
      }

      for (j= 0; j < myx_grt_list_item_count(fk_ref_columns); j++)
      {
        if (sql_fk_ref_cols)
          sql_fk_ref_cols= str_g_append(sql_fk_ref_cols, ", ");

        sql_fk_ref_cols= str_g_append_and_free(sql_fk_ref_cols, g_strdup_printf("%s%s%s", 
            quote_char,
            myx_grt_list_item_get_as_string(fk_ref_columns, j),
            quote_char));
      }

      sql_fk= g_strdup_printf("  CONSTRAINT %s%s%s FOREIGN KEY %s%s%s (%s)" _br
        "    REFERENCES %s%s%s.%s%s%s (%s)" _br, 
        quote_char, fk_name, quote_char,
        quote_char, fk_name, quote_char,
        sql_fk_cols,
        quote_char, fk_ref_schema_name, quote_char,
        quote_char, fk_ref_table_name, quote_char,
        sql_fk_ref_cols);

      if (delete_rule)
        sql_fk= str_g_append_and_free(sql_fk, g_strdup_printf("    ON DELETE %s" _br, delete_rule));

      if (update_rule)
        sql_fk= str_g_append_and_free(sql_fk, g_strdup_printf("    ON UPDATE %s" _br, update_rule));

      sql_fk= str_g_append(sql_fk, "," _br);

      g_free(sql_fk_cols);
      g_free(sql_fk_ref_cols);
    }
  }

  if (sql_cols)
    sql= str_g_append_and_free(sql, sql_cols);

  if (sql_pk)
    sql= str_g_append_and_free(sql, sql_pk);

  if (sql_indices)
    sql= str_g_append_and_free(sql, sql_indices);

  if (sql_fk)
    sql= str_g_append_and_free(sql, sql_fk);

  // remove last ,
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  sql[strlen(sql)-3]= 0;
#else
  sql[strlen(sql)-2]= 0;
#endif

  sql= str_g_append(sql, _br ")" _br);

  // add table options
  if ((opt_engine) && (opt_engine[0]))
    sql= str_g_append_and_free(sql, g_strdup_printf("ENGINE = %s" _br, opt_engine));
  else
    sql= str_g_append(sql, "ENGINE=InnoDB");

  if ((opt_next_auto_inc) && (opt_next_auto_inc[0]))
    sql= str_g_append_and_free(sql, g_strdup_printf("AUTO_INCREMENT = %s" _br, opt_next_auto_inc));

  if ((opt_avg_row_length) && (opt_avg_row_length[0]))
    sql= str_g_append_and_free(sql, g_strdup_printf("AVG_ROW_LENGTH = %s" _br, opt_avg_row_length));

  if (opt_checksum == 1)
    sql= str_g_append(sql, "CHECKSUM = 1" _br);

  if ((opt_comment) && (opt_comment[0]))
  {
    char *table_comment= str_g_replace(g_strdup(opt_comment), "'", "\\'");
    sql= str_g_append_and_free(sql, g_strdup_printf("COMMENT = '%s'", table_comment));
    g_free(table_comment);
  }

  if ((opt_max_rows) && (opt_max_rows[0]))
    sql= str_g_append_and_free(sql, g_strdup_printf("MAX_ROWS = %s" _br, opt_max_rows));

  if ((opt_min_rows) && (opt_min_rows[0]))
    sql= str_g_append_and_free(sql, g_strdup_printf("MIN_ROWS = %s" _br, opt_min_rows));

  if ((opt_pack_keys) && (opt_pack_keys[0]))
    sql= str_g_append_and_free(sql, g_strdup_printf("PACK_KEYS = %s" _br, opt_pack_keys));

  if ((opt_password) && (opt_password[0]))
  {
    char *table_password= str_g_replace(g_strdup(opt_password), "'", "\\'");
    sql= str_g_append_and_free(sql, g_strdup_printf("PASSWORD = '%s'" _br, table_password));
    g_free(table_password);
  }

  if (opt_delay_key_write == 1)
    sql= str_g_append(sql, "DELAY_KEY_WRITE = 1" _br);

  if ((opt_row_format) && (opt_row_format[0]))
    sql= str_g_append_and_free(sql, g_strdup_printf("ROW_FORMAT = %s" _br, opt_row_format));

  if (opt_raid_type && opt_raid_type[0] && 
    opt_raid_chunks && opt_raid_chunks[0] && 
    opt_raid_chunk_size && opt_raid_chunk_size[0])
  {
    sql= str_g_append_and_free(sql, g_strdup_printf("RAID_TYPE = %s" _br, opt_raid_type));
    sql= str_g_append_and_free(sql, g_strdup_printf("  RAID_CHUNKS = %s" _br, opt_raid_chunks));
    sql= str_g_append_and_free(sql, g_strdup_printf("  RAID_CHUNKSIZE = %s" _br, opt_raid_chunk_size));
  }

  if ((opt_merge_union) && (opt_merge_union[0]))
    sql= str_g_append_and_free(sql, g_strdup_printf("UNION = %s" _br, opt_merge_union));

  if ((opt_merge_insert) && (opt_merge_insert[0]))
    sql= str_g_append_and_free(sql, g_strdup_printf("INSERT_METHOD = %s" _br, opt_merge_insert));

  if ((opt_merge_insert) && (opt_merge_insert[0]))
    sql= str_g_append_and_free(sql, g_strdup_printf("INSERT_METHOD = %s" _br, opt_merge_insert));

  if ((opt_data_dir) && (opt_data_dir[0]))
    sql= str_g_append_and_free(sql, g_strdup_printf("DATA DIRECTORY = %s" _br, opt_data_dir));

  if ((opt_index_dir) && (opt_index_dir[0]))
    sql= str_g_append_and_free(sql, g_strdup_printf("INDEX DIRECTORY = %s" _br, opt_index_dir));

  if ((opt_charset) && (opt_charset[0]))
  {
    sql= str_g_append_and_free(sql, g_strdup_printf("CHARACTER SET %s", opt_charset));

    if ((opt_collation) && (opt_collation[0]))
      sql= str_g_append_and_free(sql, g_strdup_printf(" COLLATE %s" _br, opt_collation));
    else
      sql= str_g_append(sql, _br);
  }

  // remove last \n
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  sql[strlen(sql)-2]= 0;
#else
  sql[strlen(sql)-1]= 0;
#endif

  sql= str_g_append(sql, ";");

  g_free(quote_char);
  g_free(quoted_schema_name);

  return sql;
}

static char * get_sql_create_view(MYX_GRT *grt, MYX_GRT_VALUE *view)
{
  char *quote_char= g_strdup("`");
  const char *view_name= myx_grt_dict_item_get_as_string(view, "name");
  char *quoted_schema_name= get_owner_name(grt, view, quote_char);
  const char *queryExpression= myx_grt_dict_item_get_as_string(view, "queryExpression");
  int with_check_condition= myx_grt_dict_item_get_as_int(view, "withCheckCondition");
  int commented_out= myx_grt_dict_item_get_as_int(view, "commentedOut");
  char *sql;

  sql= g_strdup_printf("CREATE OR REPLACE VIEW %s%s%s%s AS" _br
    "%s" _br,
    quoted_schema_name,
    quote_char, view_name, quote_char,
    queryExpression);      

  if (with_check_condition == 1)
    sql= str_g_append(sql, "  WITH CHECK OPTION" _br);

  // remove last \n
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  sql[strlen(sql)-2]= 0;
#else
  sql[strlen(sql)-1]= 0;
#endif

  sql= str_g_append(sql, ";");


  if (commented_out == 1)
  {
    char *sql_commented_out= g_strdup_printf("/*%s*/", sql);
    g_free(sql);
    sql= sql_commented_out;
  }
    
  g_free(quote_char);
  g_free(quoted_schema_name);

  return sql;
}

static char * get_sql_create_procedure(MYX_GRT *grt, MYX_GRT_VALUE *procedure)
{
  const char *procedure_code= myx_grt_dict_item_get_as_string(procedure, "procedureCode");

  return g_strdup(procedure_code);
}

static char * get_sql_drop_schema(MYX_GRT *grt, MYX_GRT_VALUE *schema)
{
  char *quote_char= g_strdup("`");
  const char *schema_name= myx_grt_dict_item_get_as_string(schema, "name");

  char *sql= g_strdup_printf("DROP DATABASE IF EXISTS %s%s%s;",
      quote_char, schema_name, quote_char);

  g_free(quote_char);

  return sql;
}

static char * get_sql_drop_table(MYX_GRT *grt, MYX_GRT_VALUE *table)
{
  char *quote_char= g_strdup("`");
  const char *table_name= myx_grt_dict_item_get_as_string(table, "name");
  char *quoted_schema_name= get_owner_name(grt, table, quote_char);

  // build drop table sql
  char *sql= g_strdup_printf(
    "DROP TABLE IF EXISTS %s%s%s%s;", 
    quoted_schema_name,
    quote_char, table_name, quote_char);

  g_free(quote_char);
  g_free(quoted_schema_name);

  return sql;
}

static char * get_sql_drop_view(MYX_GRT *grt, MYX_GRT_VALUE *view)
{
  char *quote_char= g_strdup("`");
  const char *view_name= myx_grt_dict_item_get_as_string(view, "name");
  char *quoted_schema_name= get_owner_name(grt, view, quote_char);

  // build drop view sql
  char *sql= g_strdup_printf(
    "DROP VIEW IF EXISTS %s%s%s%s;", 
    quoted_schema_name,
    quote_char, view_name, quote_char);

  g_free(quote_char);
  g_free(quoted_schema_name);

  return sql;
}

static char * get_sql_drop_procedure(MYX_GRT *grt, MYX_GRT_VALUE *procedure)
{
  char *quote_char= g_strdup("`");
  const char *procedure_name= myx_grt_dict_item_get_as_string(procedure, "name");
  char *quoted_schema_name= get_owner_name(grt, procedure, quote_char);
  const char *procedure_type= myx_grt_dict_item_get_as_string(procedure, "procedureType");

  char *sql= g_strdup_printf("DROP %s IF EXISTS %s%s%s%s;",
    procedure_type,
    quoted_schema_name,
    quote_char, procedure_name, quote_char);
    
  g_free(quote_char);
  g_free(quoted_schema_name);
    
  return sql;
}
