unit GrtDbConn;

interface

uses
  GnuGetText, Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, Buttons, TntButtons, StdCtrls, TntStdCtrls, VirtualTrees,
  ExtCtrls, TntExtCtrls, ComCtrls, PNGImage, PNGTools,
  AuxFuncs, TntForms, TntClasses, Grt, TntSysUtils,
  myx_grt_public_interface, Menus, TntMenus;

{$include Consts.ini}

const ParamLineHeight = 32;

type
  TDriverFilter = function (PDriver: Pointer): Boolean of object;

  TGrtDbConnForm = class(TTntForm)
    HeaderPnl: TTntPanel;
    HeaderImg: TTntImage;
    ParamsMainPnl: TTntPanel;
    AdvParamsMainPnl: TTntPanel;
    ConnectGBox: TTntGroupBox;
    BottomPnl: TTntPanel;
    OKBtn: TTntButton;
    CancelBtn: TTntButton;
    ClearFieldsBtn: TTntButton;
    AdvancedBtn: TTntBitBtn;
    ConnTypeMainPnl: TTntPanel;
    ConnTypeGBox: TTntGroupBox;
    ConnTypePnl: TTntPanel;
    DriverLbl: TTntLabel;
    DriverComboBox: TTntComboBox;
    RdbmsComboBox: TTntComboBox;
    RdbmsLbl: TTntLabel;
    StoredConnPnl: TTntPanel;
    ConnectionLbl: TTntLabel;
    StoredConnComboBox: TTntComboBox;
    StoredConnAddBtn: TTntSpeedButton;
    StoredConnDelBtn: TTntSpeedButton;
    ParamsPnl: TTntPanel;
    ConnectToInstanceAni: TAnimate;
    DriverNotInstalledPnl: TTntPanel;
    TntLabel2: TTntLabel;
    TntLabel9: TTntLabel;
    LocateDriverBtn: TTntButton;
    DownloadDriverPnl: TTntPanel;
    TntLabel12: TTntLabel;
    DownloadDriverBtn: TTntButton;
    LookupMenu: TTntPopupMenu;
    TntGroupBox1: TTntGroupBox;
    AdvParamPnl: TTntPanel;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure DriverComboBoxCloseUp(Sender: TObject);
    procedure RdbmsComboBoxCloseUp(Sender: TObject);
    procedure StoredConnComboBoxCloseUp(Sender: TObject);
    procedure AdvancedBtnClick(Sender: TObject);
    procedure OKBtnClick(Sender: TObject);
    procedure CancelBtnClick(Sender: TObject);
    procedure StoredConnAddBtnClick(Sender: TObject);
    procedure ClearFieldsBtnClick(Sender: TObject);
    procedure StoredConnDelBtnClick(Sender: TObject);
    procedure LocateDriverBtnClick(Sender: TObject);
    procedure DownloadDriverBtnClick(Sender: TObject);
  private
    { Private declarations }
    FGrt: TGrt;
    FParamMappingList: TList;
    FSettingConnectionValues,
      FDisplayRdbmsSelection,
      FDisplayDescriptions,
      FDisplaySchemaSelection,
      FDisplayOnlyJdbcDrivers: Boolean;

    FConnInfoPath,
      FConnTargetPath: WideString;

    FOkBtn: TTntButton;
    FFirstParamControl: TWinControl;

    FDriverFilter: TDriverFilter;

    FRdbmsDescLbl,
      FDriverDescLbl: TTntLabel;

    procedure FreeChildControls(Control: TWinControl);
    procedure BuildDriverControls(
      Target: TWinControl; PDriver: Pointer;
      DoAdvancedParams: Boolean = False);

    procedure DriverParamValueChanged(Sender: TObject);
    procedure ParamLookupBtnClick(Sender: TObject);
    procedure ParamLookupMIClick(Sender: TObject);
    procedure BrowseDirBtnClick(Sender: TObject);
    procedure BrowseFileBtnClick(Sender: TObject);

    function BuildDriverComboBoxItems(PRdbms: Pointer): Integer;

    procedure FillDropdownWithStoredConnections(
      StoredConnComboBox: TTntComboBox; PDriver: Pointer);

    procedure SetSelectedRdbms(SelectedRdbms: WideString);
    function GetSelectedRdbms: WideString;

    procedure SetDisplayRdbmsSelection(DisplayRdbmsSelection: Boolean);

    function GetParamValue(const Name: WideString): WideString;
    procedure SetParamValue(const Name: WideString; Value: WideString);

    procedure SetDisplayDescriptions(DisplayDescriptions: Boolean);
  public
    { Public declarations }
    property ConnInfoPath: WideString read FConnInfoPath write FConnInfoPath;
    property ConnTargetPath: WideString read FConnTargetPath write FConnTargetPath;

    property DisplayRdbmsSelection: Boolean read FDisplayRdbmsSelection write SetDisplayRdbmsSelection;
    property DisplayDescriptions: Boolean read FDisplayDescriptions write SetDisplayDescriptions;
    property DisplaySchemaSelection: Boolean read FDisplaySchemaSelection write FDisplaySchemaSelection;
    property DisplayOnlyJdbcDrivers: Boolean read FDisplayOnlyJdbcDrivers write FDisplayOnlyJdbcDrivers;

    property DriverFilter: TDriverFilter read FDriverFilter write FDriverFilter;
    property OkButton: TTntButton read FOkBtn write FOkBtn;
    property Grt: TGrt read FGrt write FGrt;
    property SelectedRdbms: WideString read GetSelectedRdbms write SetSelectedRdbms;

    property ParamValue[const Name: WideString]: WideString read GetParamValue write SetParamValue;

    procedure RefreshConnInfo;
    procedure WriteConnectionToTarget;

    procedure SetConnection(PConnection: Pointer);   
  end;

  PParamMapping = ^TParamMapping;
  TParamMapping = record
    PParam: Pointer;
    ParamControl: TControl;
    Lbl: TTntLabel;
    BrowseBtn: TTntButton;
  end;

implementation

{$R *.dfm}

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.FormCreate(Sender: TObject);

begin
  InitForm(self);

  //Load resources
  LoadPNGImageFromResource('connection_dialog_header', HeaderImg, True);
  ConnectToInstanceAni.ResName := 'progress_indicator';

  Caption := Application.Title + ' '+
    product_version + ' ' +
    product_build_level;

  FParamMappingList := TList.Create;
  FSettingConnectionValues := False;
  FDisplayRdbmsSelection := False;
  FDisplayDescriptions := False;
  FDisplaySchemaSelection := True;
  FDisplayOnlyJdbcDrivers := False;

  FDriverFilter := nil;

  FOkBtn := OKBtn;
  FFirstParamControl := nil;

  FRdbmsDescLbl := nil;
  FDriverDescLbl := nil;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.FormDestroy(Sender: TObject);

var
  I: Integer;

begin
  for I := 0 to FParamMappingList.Count - 1 do
    dispose(FParamMappingList[I]);
  FParamMappingList.Free;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.RefreshConnInfo;

var
  I: integer;
  PRdbmsList,
    PRdbms: Pointer;

begin
  PRdbmsList := FGrt.Global[FConnInfoPath + '/driverInfos'];

  // Clear controls
  RDBMSComboBox.Clear;
  DriverComboBox.Items.Clear;
  DriverComboBoxCloseUp(self);

  //Set RDBMSCBox Items
  for I := 0 to FGrt.ListCount(PRdbmsList) - 1 do
  begin
    PRdbms := FGrt.ListItem[PRdbmsList, I];

    if (BuildDriverComboBoxItems(PRdbms) > 0) then
      RDBMSComboBox.AddItem(
        FGrt.DictString[PRdbms, 'caption'],
        PRdbms);
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.FreeChildControls(Control: TWinControl);

begin
  while (Control.ControlCount > 0) do
    Control.Controls[0].Free;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.BuildDriverControls(
  Target: TWinControl; PDriver: Pointer; DoAdvancedParams: Boolean);

var
  I: Integer;
  PParams,
    PParam: Pointer;
  Caption,
    Desc,
    ParamType: WideString;
  OffsetLeft,
    CurrentRow,
    MaxRow,
    CurrentTop: Integer;
  CurrentLeft: Array [1 .. 30] of Integer;
  RowDescription: TTntStringList;
  ParamMapping: PParamMapping;
  DescLbl: TTntLabel;

begin
  RowDescription := TTntStringList.Create;
  PParams := FGrt.DictItem[PDriver, 'parameters'];

  OffsetLeft := 116;
  MaxRow := 0;

  for I := 1 to 30 do
  begin
    CurrentLeft[I] := 0;
    RowDescription.Add('');
  end;

  for I := 0 to FGrt.ListCount(PParams) - 1 do
  begin
    PParam := FGrt.ListItem[PParams, I];

    if (FGrt.DictInt[PParam, 'layoutAdvanced'] <>
      Ord(DoAdvancedParams)) then
      continue;

    // if the "schema" should not be displayed
    if (Not(FDisplaySchemaSelection)) and
      (WideSameText(FGrt.DictString[PParam, 'name'], 'schema')) then
      continue;

    CurrentRow := FGrt.DictInt[PParam, 'layoutRow'];
    if (CurrentRow = -1) then
      CurrentRow := MaxRow + 1;
    MaxRow := CurrentRow;

    CurrentTop := 1 + (CurrentRow - 1) * ParamLineHeight;

    Caption := FGrt.DictString[PParam, 'caption'];
    Desc := FGrt.DictString[PParam, 'description'];
    ParamType := FGrt.DictString[PParam, 'paramType'];

    // Create mapping
    New(ParamMapping);
    FParamMappingList.Add(ParamMapping);
    ParamMapping.PParam := PParam;


    if (Not(WideSameText(ParamType, 'boolean'))) and
      (Not(WideSameText(ParamType, 'tristate'))) then
    begin
      // Create Edit with Caption
      if (Caption <> '') then
      begin
        ParamMapping.Lbl := TTntLabel.Create(self);

        //CaptionLbl.Name := 'ParamLbl' + IntToStr(I);
        ParamMapping.Lbl.Caption := _(Caption);

        // if this is the first param on that row,
        // move the CaptionLbl to the left
        // so the Param edits are aligned left
        if (CurrentLeft[CurrentRow] = 0) then
          ParamMapping.Lbl.Left := OffsetLeft + CurrentLeft[CurrentRow] -
            ParamMapping.Lbl.Width - 10
        else
        begin
          ParamMapping.Lbl.Left := OffsetLeft + CurrentLeft[CurrentRow];
          CurrentLeft[CurrentRow] := CurrentLeft[CurrentRow] +
            ParamMapping.Lbl.Width + 10;
        end;

        ParamMapping.Lbl.Top := CurrentTop + 4;

        ParamMapping.Lbl.Parent := Target;
      end;


      // Create Param Edit
      ParamMapping.ParamControl := TTntEdit.Create(self);

      ParamMapping.ParamControl.Left :=
        OffsetLeft + CurrentLeft[CurrentRow];

      TTntEdit(ParamMapping.ParamControl).OnChange :=
        DriverParamValueChanged;

      TTntEdit(ParamMapping.ParamControl).Text :=
        FGrt.DictString[PParam, 'defaultValue'];

      // Set password char for password fields
      if (WideSameText(ParamType, 'password')) then
        TTntEdit(ParamMapping.ParamControl).PasswordChar := '*';
    end
    else
    begin
      // Create Checkbox
      ParamMapping.ParamControl := TTntCheckbox.Create(self);

      ParamMapping.ParamControl.Left :=
        OffsetLeft + CurrentLeft[CurrentRow] - 100;

      TTntCheckbox(ParamMapping.ParamControl).OnClick :=
        DriverParamValueChanged;

      TTntCheckbox(ParamMapping.ParamControl).Checked :=
        (FGrt.DictString[PParam, 'defaultValue'] = '1');

      TTntCheckbox(ParamMapping.ParamControl).Caption := _(Caption);
    end;

    if (FParamMappingList.Count = 1) then
      FFirstParamControl := TWinControl(ParamMapping.ParamControl);

    // Set common options
    //ParamControl.Name := 'Param' + IntToStr(I);
    ParamMapping.ParamControl.Top := CurrentTop;

    ParamMapping.ParamControl.Width :=
      FGrt.DictInt[PParam, 'layoutWidth'];

    if (Desc <> '') then
    begin
      ParamMapping.ParamControl.Hint := Desc;
      ParamMapping.ParamControl.ShowHint := True;
    end;

    // move CurrentLeft
    CurrentLeft[CurrentRow] := CurrentLeft[CurrentRow] +
      ParamMapping.ParamControl.Width + 20;

    ParamMapping.ParamControl.Parent := Target;

    // Add lookup button
    if (FGrt.DictString[PParam, 'lookupValueMethod'] <> '') or
      WideSameText(ParamType, 'file') or
      WideSameText(ParamType, 'dir') then
    begin
      ParamMapping.BrowseBtn := TTntButton.Create(self);
      //BrowseBtn.Name := 'LookupBtn' + IntToStr(I);
      ParamMapping.BrowseBtn.Tag := FParamMappingList.Count - 1;

      ParamMapping.BrowseBtn.Caption := _('...');
      ParamMapping.BrowseBtn.Width := 27;
      ParamMapping.BrowseBtn.Height := 23;
      ParamMapping.BrowseBtn.Left :=
        ParamMapping.ParamControl.Left +
        ParamMapping.ParamControl.Width -
        ParamMapping.BrowseBtn.Width;
      ParamMapping.BrowseBtn.Top := CurrentTop;

      ParamMapping.ParamControl.Width :=
        ParamMapping.ParamControl.Width -
        ParamMapping.BrowseBtn.Width - 6;

      // Add lookup action
      if (FGrt.DictString[PParam, 'lookupValueMethod'] <> '') then
        ParamMapping.BrowseBtn.OnClick := ParamLookupBtnClick
      else
        if (WideSameText(ParamType, 'file')) then
          ParamMapping.BrowseBtn.OnClick := BrowseFileBtnClick
        else
          if (WideSameText(ParamType, 'dir')) then
            ParamMapping.BrowseBtn.OnClick := BrowseDirBtnClick;

      ParamMapping.BrowseBtn.Parent := Target;
    end;

    // build parameter description
    if (Desc <> '') and (CurrentRow - 1 >= 0) and
      (CurrentRow - 1 < RowDescription.Count) then
    begin
      if (RowDescription[CurrentRow - 1] = '') then
        RowDescription[CurrentRow - 1] := _(Desc)
      else
        RowDescription[CurrentRow - 1] :=
          RowDescription[CurrentRow - 1] + ' / ' + _(Desc);
    end;
  end;

  // Add descriptions for all lines
  if (FDisplayDescriptions) then
    for I := 0 to 9 do
    begin
      CurrentTop := 1 + I * ParamLineHeight;

      if (RowDescription[I] <> '') then
      begin
        DescLbl := TTntLabel.Create(self);
        //CaptionLbl.Name := 'ParamDescLbl' + IntToStr(I + 1);
        DescLbl.Caption := RowDescription[I];

        DescLbl.Left := OffsetLeft + CurrentLeft[I + 1];

        DescLbl.Top := CurrentTop + 4;

        DescLbl.Parent := Target;
      end;
    end;

  RowDescription.Free;

  Target.Height := MaxRow * ParamLineHeight;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.DriverParamValueChanged(Sender: TObject);

var
  I: Integer;
  RequiredParamsSet: Boolean;
  ParamMapping: PParamMapping;

begin
  if (Not(FSettingConnectionValues)) then
  begin
    // Check if all required fields are set
    RequiredParamsSet := True;
    for I := 0 to FParamMappingList.Count - 1 do
    begin
      ParamMapping := FParamMappingList[I];

      if (FGrt.DictInt[ParamMapping.PParam, 'required'] = 1) then
      begin
        if (ParamMapping.ParamControl is TTntEdit) then
          if (TTntEdit(ParamMapping.ParamControl).Text = '') then
          begin
            RequiredParamsSet := False;
            break;
          end;
      end;
    end;

    FOkBtn.Enabled := RequiredParamsSet;

    StoredConnAddBtn.Enabled := True;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.ParamLookupBtnClick(Sender: TObject);

var
  I: Integer;
  Res: Pointer;
  MenuItem: TTntMenuItem;
  ScreenPoint: TPoint;
  ParamMapping: PParamMapping;

begin
  if (Sender is TControl) and
    (TControl(Sender).Tag < FParamMappingList.Count) then
  begin
    ParamMapping := FParamMappingList[TControl(Sender).Tag];

    LookupMenu.Items.Clear;

    // Store current settings
    WriteConnectionToTarget;

    try
      // Call LookupValueMethod
      Res := FGrt.ExecuteModalFunction(
        FGrt.DictString[ParamMapping.PParam, 'lookupValueModule'],
        FGrt.DictString[ParamMapping.PParam, 'lookupValueMethod'],
        [FGrt.GetGlobalAsParam(FConnTargetPath)]);

      for I := 0 to FGrt.ListCount(Res) - 1 do
      begin
        MenuItem := TTntMenuItem.Create(LookupMenu);
        MenuItem.Caption := FGrt.ListString[Res, I];

        MenuItem.OnClick := ParamLookupMIClick;
        MenuItem.Tag := TControl(Sender).Tag;

        LookupMenu.Items.Add(MenuItem);
      end;
    except
      MenuItem := TTntMenuItem.Create(LookupMenu);
        MenuItem.Caption := _('Fetching of list failed.');

      MenuItem.Tag := -1;

      LookupMenu.Items.Add(MenuItem);
    end;

    ScreenPoint := TControl(Sender).ClientToScreen(Point(0, 0));

    LookupMenu.Popup(ScreenPoint.X, ScreenPoint.Y + TControl(Sender).Height);
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.ParamLookupMIClick(Sender: TObject);

var
  ParamMapping: PParamMapping;

begin
  if (Sender is TTntMenuItem) and
    (TTntMenuItem(Sender).Tag > -1) then
  begin
    ParamMapping := FParamMappingList[TTntMenuItem(Sender).Tag];

    if (ParamMapping.ParamControl is TTntEdit) then
      TTntEdit(ParamMapping.ParamControl).Text :=
        TTntMenuItem(Sender).Caption;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.BrowseFileBtnClick(
  Sender: TObject);

var
  OpenDlg: TOpenDialog;
  ParamMapping: PParamMapping;
  PParamTypeDetails: Pointer;
  FileType,
    FileExtension: WideString;

begin
  if (Sender is TControl) and
    (TControl(Sender).Tag < FParamMappingList.Count) then
  begin
    ParamMapping := FParamMappingList[TControl(Sender).Tag];

    PParamTypeDetails :=
      FGrt.DictItem[ParamMapping.PParam, 'paramTypeDetails'];

    OpenDlg := TOpenDialog.Create(self);
    try
      if (PParamTypeDetails <> nil) then
      begin
        OpenDlg.Title :=
          FGrt.DictString[PParamTypeDetails, 'fileOpenDialogCaption'];

        FileType :=
          FGrt.DictString[PParamTypeDetails, 'fileType'];

        FileExtension :=
          FGrt.DictString[PParamTypeDetails, 'fileExtension'];
      end;

      if (OpenDlg.Title = '') then
        OpenDlg.Title := _('Open File ...');

      if (FileType <> '') and (FileExtension <> '') then
        OpenDlg.Filter := FileType +
          ' (*.' + FileExtension + ')|*.' + FileExtension + '|' +
          _('All files') + ' (*.*)|*.*'
      else
        OpenDlg.Filter := _('All files') + ' (*.*)|*.*';

      if (OpenDlg.Execute) then
      begin
        if (ParamMapping.ParamControl is TTntEdit) then
          TTntEdit(ParamMapping.ParamControl).Text := OpenDlg.FileName;
      end;
    finally
      OpenDlg.Free;
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.BrowseDirBtnClick(Sender: TObject);

begin
  //
end;

// -----------------------------------------------------------------------------

function TGrtDbConnForm.BuildDriverComboBoxItems(PRdbms: Pointer): Integer;

var
  I: integer;
  PDrivers,
    PDriver: Pointer;

begin
  PDrivers := FGrt.DictItem[PRdbms, 'drivers'];

  DriverComboBox.Clear;

  for I := 0 to FGrt.ListCount(PDrivers) - 1 do
  begin
    PDriver := FGrt.ListItem[PDrivers, I];

    if (FDisplayOnlyJdbcDrivers) and
      (FGrt.DictStructName[PDriver] <> 'db.conn.JdbcDriver') then
      continue;

    if (Assigned(FDriverFilter)) then
      if (DriverFilter(PDriver)) then
        continue;

    // place default driver on top of dropdown
    if (FGrt.DictRef[PRdbms, 'defaultDriver'] = PDriver) then
      DriverComboBox.Items.InsertObject(
        0,
        FGrt.DictString[PDriver, 'caption'],
        PDriver)
    else
      DriverComboBox.AddItem(
        FGrt.DictString[PDriver, 'caption'],
        PDriver);
  end;

  DriverComboBox.Enabled := (DriverComboBox.Items.Count > 1);
  DriverLbl.Enabled := (DriverComboBox.Items.Count > 1);
  if (FDriverDescLbl <> nil) then
    FDriverDescLbl.Enabled := DriverComboBox.Enabled;

  Result := DriverComboBox.Items.Count;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.DriverComboBoxCloseUp(Sender: TObject);

var
  PRdbms: Pointer;
  PDriver: Pointer;
  I: Integer;

begin
  // Clear mapping
  for I := 0 to FParamMappingList.Count - 1 do
    dispose(FParamMappingList[I]);
  FParamMappingList.Clear;
  FFirstParamControl := nil;

  FreeChildControls(ParamsPnl);
  FreeChildControls(AdvParamPnl);


  if (DriverComboBox.ItemIndex > -1) then
  begin
    PRdbms := RdbmsComboBox.Items.Objects[RdbmsComboBox.ItemIndex];
    PDriver :=
      DriverComboBox.Items.Objects[DriverComboBox.ItemIndex];

    if (FGrt.DictInt[PDriver, 'isInstalled'] = 1) then
    begin
      DriverNotInstalledPnl.Visible := False;

      StoredConnComboBox.ItemIndex := -1;
      StoredConnDelBtn.Enabled := False;

      // Build params
      BuildDriverControls(ParamsPnl, PDriver);
      ParamsMainPnl.Height := StoredConnPnl.Height +
        ParamsPnl.Height + 38;

      // Build advanced params
      BuildDriverControls(AdvParamPnl, PDriver, True);
      AdvParamsMainPnl.Height := AdvParamPnl.Height + 38;

      // Get stored connections
      FillDropdownWithStoredConnections(StoredConnComboBox, PDriver);

      {JdbcDriverNameLbl.Caption := _(myx_grt_dict_item_get_as_string(
        PRdbms, 'caption'));
      JdbcDriverDescLbl.Caption := _(myx_grt_dict_item_get_as_string(
        PJdbcDriver, 'description'));}

      StoredConnAddBtn.Enabled := False;

      StoredConnPnl.Visible := True;
    end
    else
    begin
      DownloadDriverPnl.Visible :=
        (FGrt.DictString[PDriver, 'downloadUrl'] <> '');

      DriverNotInstalledPnl.BringToFront;
      DriverNotInstalledPnl.Visible := True;
      ParamsMainPnl.Height := 201;

      AdvParamsMainPnl.Height := 38;

      StoredConnPnl.Visible := False;
    end;
  end
  else
  begin
    {JdbcDriverNameLbl.Caption := _('No RDBMS selected');
    JdbcDriverDescLbl.Caption := _('Please choose a Database System from the list above.');}

    StoredConnPnl.Visible := False;

    StoredConnAddBtn.Enabled := False;

    FreeChildControls(ParamsPnl);
  end;

  try
    if (FFirstParamControl <> nil) then
      GetParentForm(FFirstParamControl).ActiveControl :=
        FFirstParamControl;
  except
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.FillDropdownWithStoredConnections(
  StoredConnComboBox: TTntComboBox; PDriver: Pointer);

var
  I: Integer;
  PConnections,
    PConnection: Pointer;

begin
  PConnections := FGrt.Global[FConnInfoPath + '/storedConns'];

  StoredConnComboBox.Items.Clear;
  StoredConnComboBox.ItemIndex := -1;

  for I := 0 to FGrt.ListCount(PConnections) - 1 do
  begin
    PConnection := FGrt.ListItem[PConnections, I];

    if (FGrt.DictString[PDriver, '_id'] =
        FGrt.DictString[PConnection, 'driver']) then
    begin
      StoredConnComboBox.Items.AddObject(
        FGrt.DictString[PConnection, 'name'],
        PConnection);
    end;
  end;

  StoredConnComboBox.Items.Add(_('<New Connection>'));
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.SetConnection(PConnection: Pointer);

var
  PParameterValues,
    PRdbmsList,
    PRdbms,
    PDrivers,
    PDriver: Pointer;
  I,
    J,
    Index: Integer;
  ParamMapping: PParamMapping;
  DriverId: WideString;

begin
  StoredConnDelBtn.Enabled := (PConnection <> nil);
  StoredConnAddBtn.Enabled := False;

  // if the last entry is selected <New Connection>, clear edits
  if (PConnection = nil) then
  begin
    for I := 0 to FParamMappingList.Count - 1 do
    begin
      ParamMapping := FParamMappingList[I];

      if (ParamMapping^.ParamControl is TTntEdit) then
        TTntEdit(ParamMapping^.ParamControl).Text :=
          FGrt.DictString[ParamMapping^.PParam, 'defaultValue']
      else
        if (ParamMapping^.ParamControl is TTntCheckbox) then
          TTntCheckbox(ParamMapping^.ParamControl).Checked :=
            StrToIntDef(FGrt.DictString[ParamMapping^.PParam, 'defaultValue'], 0) = 1;
    end;

    StoredConnComboBox.ItemIndex := -1;

    OKButton.Enabled := False;
  end
  else
  begin
    PRdbmsList := FGrt.Global[FConnInfoPath + '/driverInfos'];
    PDriver := FGrt.DictRef[PConnection, 'driver'];
    DriverId := FGrt.DictString[PDriver, '_id'];

    PDriver := nil;
    for I := 0 to FGrt.ListCount(PRdbmsList) - 1 do
    begin
      PRdbms := FGrt.ListItem[PRdbmsList, I];
      PDrivers := FGrt.DictItem[PRdbms, 'drivers'];

      for J := 0 to FGrt.ListCount(PDrivers) - 1 do
      begin
        PDriver := FGrt.ListItem[PDrivers, J];

        if (FGrt.DictString[
          FGrt.ListItem[PDrivers, J], '_id'] = DriverId) then
          break;

        PDriver := nil;
      end;

      if (PDriver <> nil) then
        break;

      PRdbms := nil;
    end;

    if (PDriver = nil) then
      raise Exception.Create(_('The driver used by the given ' +
        'connection is not available.'));

    // Select correct RDBMS
    Index := RDBMSComboBox.Items.IndexOfObject(PRdbms);
    if (Index > -1) then
    begin
      if (RDBMSComboBox.ItemIndex <> Index) then
      begin
        RDBMSComboBox.ItemIndex := Index;
        RDBMSComboBoxCloseUp(self);
      end;

      // Select correct Driver
      Index := DriverComboBox.Items.IndexOfObject(PDriver);
      if (Index > -1) then
      begin
        if (DriverComboBox.ItemIndex <> Index) then
        begin
          DriverComboBox.ItemIndex := Index;
          DriverComboBoxCloseUp(self);
        end;
      end
      else
        raise Exception.CreateFmt(_('The Driver %s is not '+
          'available for selection'),
          [FGrt.DictString[PDriver, 'caption']]);
    end
    else
      raise Exception.CreateFmt(_('The RDBMS %s is not '+
        'available for selection'),
        [FGrt.DictString[PRdbms, 'caption']]);

    PParameterValues := FGrt.DictItem[PConnection, 'parameterValues'];

    for I := 0 to FParamMappingList.Count - 1 do
    begin
      ParamMapping := FParamMappingList[I];
      if (ParamMapping.ParamControl is TTntEdit) then
        TTntEdit(ParamMapping.ParamControl).Text :=
          FGrt.DictString[PParameterValues,
            FGrt.DictString[ParamMapping.PParam, 'name']]
      else
        if (ParamMapping.ParamControl is TTntCheckBox) then
          TTntCheckBox(ParamMapping.ParamControl).Checked :=
            (FGrt.DictString[PParameterValues,
              FGrt.DictString[ParamMapping.PParam, 'name']] = '1');
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.RdbmsComboBoxCloseUp(Sender: TObject);

begin
  if (RDBMSComboBox.ItemIndex > -1) then
  begin
    BuildDriverComboBoxItems(
      RDBMSComboBox.Items.Objects[RDBMSComboBox.ItemIndex]);

    DriverComboBox.ItemIndex := 0;
    DriverComboBoxCloseUp(self);
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.SetDisplayRdbmsSelection(DisplayRdbmsSelection: Boolean);

begin
  ConnTypeMainPnl.Visible := DisplayRdbmsSelection;
  FDisplayRdbmsSelection := DisplayRdbmsSelection;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.SetSelectedRdbms(SelectedRdbms: WideString);

var
  I: Integer;

begin
  for I := 0 to RdbmsComboBox.Items.Count - 1 do
  begin
    if (WideSameText(
        FGrt.DictString[RdbmsComboBox.Items.Objects[I], 'name'],
        SelectedRdbms)) then
    begin
      RdbmsComboBox.ItemIndex := I;
      RdbmsComboBoxCloseUp(self);
      break;
    end;
  end;
end;

// -----------------------------------------------------------------------------

function TGrtDbConnForm.GetSelectedRdbms: WideString;

begin
  Result := '';

  if (RdbmsComboBox.ItemIndex > -1) then
  begin
    Result := FGrt.DictString[
      RdbmsComboBox.Items.Objects[RdbmsComboBox.ItemIndex],
      'name'];
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.StoredConnComboBoxCloseUp(Sender: TObject);

begin
  if (StoredConnComboBox.ItemIndex =
    StoredConnComboBox.Items.Count - 1) or
    (StoredConnComboBox.ItemIndex = -1) then
    SetConnection(nil)
  else
    SetConnection(StoredConnComboBox.Items.Objects[
      StoredConnComboBox.ItemIndex]);

  if (FFirstParamControl <> nil) then
    GetParentForm(FFirstParamControl).ActiveControl :=
      FFirstParamControl;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.AdvancedBtnClick(Sender: TObject);

begin
  AdvParamsMainPnl.Visible := Not(AdvParamsMainPnl.Visible);

  if (Not(FDisplayRdbmsSelection)) then
    ConnTypeMainPnl.Visible := AdvParamsMainPnl.Visible;

  BottomPnl.Top := 1000;

  if (AdvParamsMainPnl.Visible) then
    AdvancedBtn.Caption := _('<< Advanced')
  else
    AdvancedBtn.Caption := _('Advanced >>');
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.WriteConnectionToTarget;

var
  I: Integer;
  PConnection,
    PRdbms,
    PDriver,
    PParamValues: Pointer;
  ParamMapping: PParamMapping;

begin
  if ((RdbmsComboBox.ItemIndex = -1) or
      (DriverComboBox.ItemIndex = -1)) then
    Exit;

  PRdbms := RdbmsComboBox.Items.Objects[RdbmsComboBox.ItemIndex];
  PDriver := DriverComboBox.Items.Objects[DriverComboBox.ItemIndex];

  // create new connection
  PConnection := FGrt.DictNew('db.conn.Connection');

  // add a reference to the driver
  FGrt.DictRef[PConnection, 'driver'] := PDriver;

  // create parameter dict
  PParamValues := FGrt.DictNewTyped(MYX_STRING_VALUE, '');
  FGrt.DictItem[PConnection, 'parameterValues'] := PParamValues;

  // set values from controls
  for I := 0 to FParamMappingList.Count - 1 do
  begin
    ParamMapping := FParamMappingList[I];

    // set the new value
    if (ParamMapping.ParamControl is TTntEdit) then
      FGrt.DictString[PParamValues,
        FGrt.DictString[ParamMapping.PParam, 'name']] :=
          TTntEdit(ParamMapping.ParamControl).Text
    else
      if (ParamMapping.ParamControl is TTntCheckbox) then
        FGrt.DictString[PParamValues,
          FGrt.DictString[ParamMapping.PParam, 'name']] :=
            IntToStr(Ord(TTntCheckbox(ParamMapping.ParamControl).Checked));
  end;

  // create modules list
  FGrt.DictItem[PConnection, 'modules'] :=
    FGrt.ValueDuplicate(FGrt.DictItem[PDriver, 'defaultModules']);

  FGrt.Global[FConnTargetPath] := PConnection;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.OKBtnClick(Sender: TObject);

begin
  WriteConnectionToTarget;

  ModalResult := mrOK;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.CancelBtnClick(Sender: TObject);

begin
  ModalResult := mrCancel;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.StoredConnAddBtnClick(Sender: TObject);

var
  ConnectionName: WideString;
  PNewConnection,
    PConnections,
    PConnection: Pointer;
  I: Integer;

begin
  // if not the last item (<New Connection>) is selected
  if (StoredConnComboBox.ItemIndex <
    StoredConnComboBox.Items.Count - 1) then
    ConnectionName := StoredConnComboBox.Text;

  if (ShowModalEditDialog(_('Store Connection.'),
      _('Please enter a name for the connection.'),
      myx_mtInformation, _('Ok') + #13#10 + _('Cancel'),
      True, _('Name:'),
      ConnectionName) = 1) then
  begin
    WriteConnectionToTarget;

    PNewConnection := Grt.Global[FConnTargetPath];

    // set the connection's name
    Grt.DictString[PNewConnection, 'name'] := ConnectionName;

    // check if there already is a connection with this name stored
    PConnections := Grt.Global[FConnInfoPath + '/storedConns'];

    for I := 0 to Grt.ListCount(PConnections) - 1 do
    begin
      PConnection := Grt.ListItem[PConnections, I];

      // if so, delete it
      if (WideSameText(Grt.DictString[PConnection, 'driver'],
        Grt.DictString[PNewConnection, 'driver'])) and
        (WideSameText(Grt.DictString[PConnection, 'name'],
        ConnectionName)) then
      begin
        Grt.ListDel(PConnections, I);
        break;
      end;
    end;

    // Store connection
    PNewConnection := Grt.ValueDuplicate(PNewConnection);

    Grt.ListAdd(PConnections, PNewConnection);

    FillDropdownWithStoredConnections(StoredConnComboBox,
      DriverComboBox.Items.Objects[DriverComboBox.ItemIndex]);
    StoredConnComboBox.ItemIndex :=
      StoredConnComboBox.Items.Count - 2;
    StoredConnComboBoxCloseUp(self);
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.ClearFieldsBtnClick(Sender: TObject);

begin
  SetConnection(nil);

  if (FFirstParamControl <> nil) then
    FFirstParamControl.SetFocus;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.StoredConnDelBtnClick(Sender: TObject);

var
  PConnections,
    PConnection,
    PDelConnection: Pointer;
  I: Integer;

begin
  if (StoredConnComboBox.ItemIndex <> -1) then
  begin
    if (ShowModalDialog(_('Delete stored connection.'),
      _('Are you sure you want to delete the stored connection?'),
      myx_mtInformation, _('Yes') + #13#10 + _('No')) = 1) then
    begin
      PDelConnection := StoredConnComboBox.Items.Objects[
        StoredConnComboBox.ItemIndex];

      PConnections := Grt.Global[FConnInfoPath + '/storedConns'];

      for I := 0 to Grt.ListCount(PConnections) - 1 do
      begin
        PConnection := Grt.ListItem[PConnections, I];

        // if so, delete it
        if (WideSameText(Grt.DictString[PConnection, 'driver'],
          Grt.DictString[PDelConnection, 'driver'])) and
          (WideSameText(Grt.DictString[PConnection, 'name'],
          Grt.DictString[PDelConnection, 'name'])) then
        begin
          Grt.ListDel(PConnections, I);

          DriverComboBoxCloseUp(self);

          break;
        end;
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.LocateDriverBtnClick(Sender: TObject);

var
  OpenDlg: TOpenDialog;
  SourceDir,
    TargetDir: WideString;
  FilenameList: TTntStringList;
  I: Integer;
  PDriver,
    PFiles: Pointer;

begin
  if (DriverComboBox.ItemIndex = -1) then
    Exit;

  PDriver := DriverComboBox.Items.Objects[
    DriverComboBox.ItemIndex];

  PFiles := FGrt.DictItem[PDriver, 'files'];

  TargetDir := GetApplDir + WideStringReplace(
    FGrt.DictString[PDriver, 'filesTarget'], '/', '\',
    [rfReplaceAll]);
  WideForceDirectories(TargetDir);

  FilenameList := TTntStringList.Create;
  try
    for I := 0 to Grt.ListCount(PFiles) - 1 do
      FilenameList.Add(Grt.ListString[PFiles, I]);

    OpenDlg := TOpenDialog.Create(self);
    try
      OpenDlg.Title := 'Attach Driver';
      while (FilenameList.Count > 0) do
      begin
        OpenDlg.Filter := _('Driver file')+
          ' (' + FilenameList[0] + ')|' + FilenameList[0] + '|'+
          _('All files')+' (*.*)|*.*';

        if (OpenDlg.Execute) then
        begin
          SourceDir := WideIncludeTrailingBackslash(
            WideExtractFilePath(OpenDlg.FileName));

          I := FilenameList.Count - 1;
          while (I >= 0) do
          begin
            if (FileExists(SourceDir + FilenameList[I])) then
            begin
              CopyDiskFile(SourceDir + FilenameList[I],
                TargetDir + FilenameList[I], False);

              FilenameList.Delete(I);
            end;

            dec(I);
          end;
        end
        else
          break;
      end;
    finally
      OpenDlg.Free;
    end;

    if (FilenameList.Count = 0) then
    begin
      if (ShowModalDialog(_('The application needs to be restarted.'),
        _('The application needs to be restarted. ' +
          'Please click on Ok to close the application.'),
          myx_mtConfirmation, _('Close') + #13#10 + _('Cancel')) = 1) then
        Application.Terminate;
    end;
  finally
    FilenameList.Free;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.DownloadDriverBtnClick(Sender: TObject);

var
  PDriver: Pointer;

begin
  if (DriverComboBox.ItemIndex = -1) then
    Exit;

  PDriver := DriverComboBox.Items.Objects[
    DriverComboBox.ItemIndex];

  BrowseWebPage(Grt.DictString[PDriver, 'downloadUrl']);
end;

// -----------------------------------------------------------------------------

function TGrtDbConnForm.GetParamValue(const Name: WideString): WideString;

var
  I: Integer;
  ParamMapping: PParamMapping;

begin
  for I := 0 to FParamMappingList.Count - 1 do
  begin
    ParamMapping := FParamMappingList[I];

    if (WideSameText(FGrt.DictString[ParamMapping^.PParam, 'name'],
      name)) then
    begin
      if (ParamMapping^.ParamControl is TTntEdit) then
        Result := TTntEdit(ParamMapping^.ParamControl).Text
      else
        if (ParamMapping^.ParamControl is TTntCheckbox) then
          Result := IntToStr(Ord(TTntCheckbox(ParamMapping^.ParamControl).Checked));

      break;
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.SetParamValue(const Name: WideString; Value: WideString);

var
  I: Integer;
  ParamMapping: PParamMapping;

begin
  for I := 0 to FParamMappingList.Count - 1 do
  begin
    ParamMapping := FParamMappingList[I];

    if (WideSameText(FGrt.DictString[ParamMapping^.PParam, 'name'],
      name)) then
    begin
      if (ParamMapping^.ParamControl is TTntEdit) then
      begin
        TTntEdit(ParamMapping^.ParamControl).Text := Value;
        TTntEdit(ParamMapping^.ParamControl).SelStart :=
          Length(TTntEdit(ParamMapping^.ParamControl).Text);
      end
      else
        if (ParamMapping^.ParamControl is TTntCheckbox) then
          TTntCheckbox(ParamMapping^.ParamControl).Checked := (Value = '1');

      break;
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.SetDisplayDescriptions(DisplayDescriptions: Boolean);

begin
  FDisplayDescriptions := DisplayDescriptions;

  if (FDisplayDescriptions) and (FRdbmsDescLbl = nil) then
  begin
    FRdbmsDescLbl := TTntLabel.Create(self);
    FRdbmsDescLbl.Caption := _('Select a RDBMS from the list of supported systems');
    FRdbmsDescLbl.Top := RdbmsLbl.Top;
    FRdbmsDescLbl.Left := RdbmsComboBox.Left +
      RdbmsComboBox.Width + 20;
    FRdbmsDescLbl.Parent := ConnTypePnl;

    FDriverDescLbl := TTntLabel.Create(self);
    FDriverDescLbl.Caption := _('Choose from the list of available ' +
      'drivers for this RDBMS');
    FDriverDescLbl.Top := DriverLbl.Top;
    FDriverDescLbl.Left := RdbmsComboBox.Left +
      RdbmsComboBox.Width + 20;
    FDriverDescLbl.Parent := ConnTypePnl;
  end
  else
    if (FDisplayDescriptions) and (FRdbmsDescLbl <> nil) then
    begin
      FRdbmsDescLbl.Free;
      FRdbmsDescLbl := nil;
      FDriverDescLbl.Free;
      FDriverDescLbl := nil;
    end;
end;

// -----------------------------------------------------------------------------

end.
