//
//  MQueryBrowser.m
//  MySQL QueryBrowser
//
//  Created by Alfredo Kojima on Sat Mar 5 2005.
//  Copyright (c) 2005 MySQL AB. All rights reserved.
//

#import "MQueryBrowserController.h"
#import "MQueryBrowser.h"
#import "MQHistory.h"
#import "MQAddBookmarkPanel.h"
#import "MQBookmark.h"
#import "MQParameters.h"
#import "MQSyntaxHelp.h"

#import <MySQLToolsCommon/MQResultSetView.h>
#import <MySQLToolsCommon/MAccessoryScrollView.h>
#import "MQScriptController.h"
#import "MQQueryController.h"
#import "MQResultSetTab.h"
#import "MQScriptTab.h"
#import "MQHelpTab.h"

#import <MySQLToolsCommon/MSchemaEditHelper.h>
#include <MySQLToolsCommon/mxUtils.h>

#include "myx_public_interface.h"

//#include "myx_qb_public_interface.h"

@interface MQueryBrowser(Private)
- (void)fetchSchemasFor:(MSchemaDataSource*)ds;
- (void)tablesArrived:(NSArray*)args result:(MYX_SCHEMA_TABLES*)tables;
- (void)requestTablesForSchema:(MSchemaDataSource*)sender item:(MSchemaItem*)item;
- (void)selectSchema:(NSString*)schema;
- (void)updateHistory:(NSNotification*)notif;
- (NSString*)defaultSchema;
@end

@implementation MQueryBrowser(Private)

- (void)fetchSchemasFor:(MSchemaDataSource*)ds
{
  MYX_CATALOGS *cats;
  
  cats= [_dispatcher performCallback:(void*(*)(MYSQL*))myx_get_catalogs
                       waitForWindow:[self window]
                             message:@"Retrieving catalog list..."];
  if (!cats)
  {
    NSLog(@"ERROR: Could not retrieve catalog list");
    return;
  }

  [ds updateWithCatalogs:cats];
  [catalogList reloadData];
  
  if (_catalogs)
    myx_free_catalogs(_catalogs);
  _catalogs= cats;
}


- (void)requestTablesForSchema:(MSchemaDataSource*)sender item:(MSchemaItem*)item
{
  MYX_SCHEMA *schema= [item schema];
 
  [_columnDS resetSchemaChildren:item];
  
  [_dispatcher performCallback:(void*(*)(MYSQL*,void*,void*))myx_get_schema_tables
                      argument:schema->catalog_name
                      argument:schema->schema_name
              finishedSelector:@selector(tablesArrived:result:)
                      argument:[[NSArray arrayWithObjects:sender,item,nil] retain]
                        target:self];
  
  if (mysql_version_is_later_or_equal_than(_mysql,5,0))
  {
    [_dispatcher performCallback:(void*(*)(MYSQL*,void*,void*))myx_get_schema_sps
                        argument:schema->catalog_name
                        argument:schema->schema_name
                finishedSelector:@selector(spsArrived:result:)
                        argument:[[NSArray arrayWithObjects:sender,item,nil] retain]
                          target:self];
  }
}


- (void)tablesArrived:(NSArray*)args result:(MYX_SCHEMA_TABLES*)tables
{
  MSchemaDataSource *ds= [args objectAtIndex:0];
  MSchemaItem *sitem= [args objectAtIndex:1];
  MYX_SCHEMA *schema= [sitem schema];

  [args autorelease];

  if (tables)
  {
    if (schema->schema_tables)
      myx_free_schema_tables(schema->schema_tables);
    schema->schema_tables= tables;
  }
  if (!tables)
  {
    NSLog(@"could not retrieve tables list");
    return;
  }
  
  [ds updateSchema:sitem 
		withTables:tables];
  
  [catalogList reloadData];
}

- (void)spsArrived:(NSArray*)args result:(MYX_SCHEMA_STORED_PROCEDURES*)sps
{
  MSchemaDataSource *ds= [args objectAtIndex:0];
  MSchemaItem *sitem= [args objectAtIndex:1];
  MYX_SCHEMA *schema= [sitem schema];
  
  [args autorelease];
  
  if (sps)
  {
    if (schema->schema_sps)
      myx_free_schema_sps(schema->schema_sps);
    schema->schema_sps= sps;
  }
  if (!sps)
  {
    NSLog(@"could not retrieve sp list");
    return;
  }
  
  [ds updateSchema:sitem 
           withSPs:sps];
  
  [catalogList reloadData];
}


- (void)selectSchema:(NSString*)schema
{
  unsigned int i;
  
  g_free(_info->schema);
  _info->schema= schema == nil ? NULL : g_strdup([schema UTF8String]);
  
  if (schema)
	myx_use_schema(_mysql, [schema UTF8String]);  
  
  for (i= 0; i < [tabView numberOfPages]; i++)
  {
    if ([[tabView pageAtIndex:i] respondsToSelector:@selector(setDefaultSchema:)])
      [[tabView pageAtIndex:i] performSelector:@selector(setDefaultSchema:)
                                    withObject:schema];
  }
  [queryCtl setDefaultSchema:schema];
  
  [catalogList reloadData];
  
  if (schema)
    MXExpandOutlineItems(catalogList,
                         [NSDictionary dictionaryWithObject:[NSDictionary dictionaryWithObject:[NSNull null]
                                                                                        forKey:schema]
                                                     forKey:@"def"],
                         nil);
}

- (void)updateHistory:(NSNotification*)notif
{
  [historyList reloadData];
}  


- (NSString*)defaultSchema
{
  if (_mysql->db)
    return [NSString stringWithUTF8String:_mysql->db];
  return nil;
}
@end

//------------------------------------------------------------------------------

@implementation MQueryBrowser

- (id)init
{ 
  self = [super initWithWindowNibName:@"QueryBrowser" owner:self];
  if (!self)
    return nil;
  
  [self window]; // force load
  if (![self isWindowLoaded])
    NSLog(@"ERROR: Could not load document window");

  [tabView setMaxTabSize:100];
  
  _mysqlLock= [[NSLock alloc] init];
  
  _historyGroupIcon= [[NSImage imageNamed:@"history_interval_14x14.png"] retain];
  _historyItemIcon= [[NSImage imageNamed:@"query_14x14.png"] retain];

  _bookmarkGroupIcon= [[NSImage imageNamed:@"folder_closed_14x14.png"] retain];
  _bookmarkItemIcon= [[NSImage imageNamed:@"bookmark_14x14.png"] retain];

  _boldFont= [[NSFont boldSystemFontOfSize:[NSFont smallSystemFontSize]] retain];
  _normalFont= [[NSFont systemFontOfSize:[NSFont smallSystemFontSize]] retain];

  [bookmarkList registerForDraggedTypes:[NSArray arrayWithObjects:MQBookmarkPboardType, NSStringPboardType, nil]];

  [[NSNotificationCenter defaultCenter] addObserver:self
										   selector:@selector(updateHistory:)
											   name:MQHistoryDidChangeNotification
											 object:nil];
  
  _parameters= [[MQParameters alloc] init];
  [parameterList setDataSource:_parameters];
  [parameterList reloadData];
  MXExpandOutline(parameterList, YES);
  {
    MAccessoryScrollView *sv= [parameterList enclosingScrollView];
    NSButton *button;
    button= [[[NSButton alloc] initWithFrame:NSMakeRect(0,0,12,12)] autorelease];
    [button setButtonType: NSMomentaryChangeButton];
    [button setBordered:NO];
    [button setEnabled:NO];
    [button setTag:1];
    [[button cell] setBezelStyle:NSShadowlessSquareBezelStyle];
    [[button cell] setImagePosition:NSImageOnly];
    [button setImage: MXGetImageFromBundle([NSBundle bundleForClass:[MAccessoryScrollView class]], @"mini_add_12.png")];
    [button setAlternateImage: MXGetImageFromBundle([NSBundle bundleForClass:[MAccessoryScrollView class]], @"mini_add_pressed_12.png")];    
    [button setTarget:self];
    [button setAction:@selector(addParameter:)];
    [sv addSubview:button];
    [sv addHorizontalAccessoryView:button];
    
    button= [[[NSButton alloc] initWithFrame:NSMakeRect(0,0,12,12)] autorelease];
    [button setButtonType: NSMomentaryChangeButton];
    [button setBordered:NO];
    [button setEnabled:NO];
    [button setTag:2];
    [[button cell] setBezelStyle:NSShadowlessSquareBezelStyle];
    [[button cell] setImagePosition:NSImageOnly];
    [button setImage: MXGetImageFromBundle([NSBundle bundleForClass:[MAccessoryScrollView class]], @"mini_del_12.png")];
    [button setAlternateImage: MXGetImageFromBundle([NSBundle bundleForClass:[MAccessoryScrollView class]], @"mini_del_pressed_12.png")];
    [button setTarget:self];
    [button setAction:@selector(deleteParameter:)];
    [sv addSubview:button];
    [sv addHorizontalAccessoryView:button];
  }
  [tabView setDelegate:self];

  NSImage *folderIcon= [NSImage imageNamed:@"folder_closed_14x14.png"];
  [[syntaxList dataSource] loadFile:@"mysqlqb_statements"];
  [[syntaxList dataSource] setGroupIcon:folderIcon itemIcon:[NSImage imageNamed:@"syntax_14x14.png"]];
  [syntaxList setTarget:self];
  [syntaxList setDoubleAction:@selector(showKeywordHelp:)];
  [syntaxList reloadData];
  [[functionList dataSource] loadFile:@"mysqlqb_functions"];
  [[functionList dataSource] setGroupIcon:folderIcon itemIcon:[NSImage imageNamed:@"function_14x14.png"]];
  [functionList setDoubleAction:@selector(showKeywordHelp:)];
  [functionList reloadData];
  
  _groupIcon= [folderIcon retain];
  _paramIcon= [[NSImage imageNamed:@"param_14x14.png"] retain];
  
  _toolbarHeight= 58;
  
  return self;
}


- (id)initWithConnection: (MYSQL*)mysql
                    info: (MYX_USER_CONNECTION*)info
{
  self = [self init];
  
  if (self)
  {
    _mysql=mysql;
    _info= info;
    
    _dispatcher= [[MMySQLDispatcher alloc] initWithMySQL:mysql instance:info];
	_columnDS= [[MSchemaDataSource alloc] initWithRoot:MSchemaItemType 
												  leaf:MColumnItemType
												 icons:[NSDictionary dictionaryWithObjectsAndKeys:
												   [NSImage imageNamed:@"16x16_Catalog.png"], @"catalog",
												   [NSImage imageNamed:@"myx_schema_16x16.png"], @"schema",
												   [NSImage imageNamed:@"myx_schema_table_16x16.png"], @"table",
                                                   [NSImage imageNamed:@"myx_schema_view_16x16.png"], @"view",
                                                   [NSImage imageNamed:@"myx_schema_sp_16x16.png"], @"sp",
												   [NSImage imageNamed:@"16x16_Field.png"], @"column",
												   [NSImage imageNamed:@"16x16_KeyColumn.png"], @"key",
												   nil]];
    [_columnDS setTableFetcher:self selector:@selector(requestTablesForSchema:item:)];
	[self refreshSchemata:nil];
    [schemaHelper setConnection:mysql];

	[catalogList setDataSource:_columnDS];
	[catalogList reloadData];
	MXExpandOutline(catalogList, NO);
    [catalogList setDoubleAction:@selector(schemataDoubleClicked:)];

	[historyList setDataSource:[[MQueryBrowserController sharedDocumentController] history]];
	MXExpandOutline(historyList, NO);
    [historyList reloadData];
    [historyList setDoubleAction:@selector(executeHistory:)];

    [bookmarkList setDataSource:[[MQueryBrowserController sharedDocumentController] bookmarks]];
	MXExpandOutline(bookmarkList, NO);
    [bookmarkList reloadData];

    if (mysql->unix_socket)
      _connectedInstance= [[NSString alloc] initWithFormat:@"%s via socket", info->hostname];
    else
      _connectedInstance= [[NSString alloc] initWithFormat:@"%s:%i", info->hostname, info->port];

    [self selectSchema: info->schema?[NSString stringWithUTF8String:info->schema]:nil];
	[self addQueryTab: self];

    [drawer open];
  }
  return self;
}


- (void)dealloc
{
  [_dispatcher release];
  [_connectedInstance release];
  [_mysqlLock release];
  [_columnDS release];
  [_parameters release];
  [_groupIcon release];
  [_paramIcon release];

  if (_catalogs)
    myx_free_catalogs(_catalogs);
  
  if (_mysql)
    myx_mysql_close(_mysql);
  if (_info)
    myx_free_user_connection_content(_info);
  
  [[NSNotificationCenter defaultCenter] removeObserver:self];
  
  [super dealloc];
}


- (void)selectQueryArea:(id)sender
{
  if ([[self currentTab] isKindOfClass:[MQResultSetTab class]])
    [queryCtl selectQueryArea:sender];
}


- (BOOL)setupQueryWithSchemaItem:(MSchemaItem*)item
{
  switch (item->type)
  {
    case MSchemaItemType:
      [self setDefaultSchema:nil];
      break;
    case MViewItemType:
    case MTableItemType:
      if ([[tabView selectedPage] isKindOfClass:[MQResultSetTab class]])
        return [queryCtl setupQueryWithItem:item];
      break;
    case MSPItemType:
      if ([[tabView selectedPage] isKindOfClass:[MQResultSetTab class]])
        return [queryCtl setupQueryWithItem:item];
      break;
  }
  return NO;
}


- (IBAction)showHelp:(id)sender
{
  unsigned int i, c= [tabView numberOfPages];
  MQHelpTab *page= nil;
  
  for (i= 0; i < c; i++)
  {
    if ([[tabView pageAtIndex:i] isKindOfClass:[MQHelpTab class]])
    {
      page= (MQHelpTab*)[tabView pageAtIndex:i];
      [tabView selectPage:i];
      break;
    }
  }
  if (!page)
  {
    page= [[[MQHelpTab alloc] initWithIdentifier:@"Help"] autorelease];
    [tabView addTabViewItem:page];
    [tabView selectPage:[tabView numberOfPages]-1];
  }

  switch ([sender tag])
  {
    case 2: // QuickStart help
      [page showQuickStartHelp];
      break;
  }
}


- (IBAction)showKeywordHelp:(id)sender
{
  unsigned int i, c= [tabView numberOfPages];
  MQHelpTab *page= nil;
  const char *topicId;

  topicId= [[sender dataSource] topicIdForItem:[sender itemAtRow:[sender selectedRow]]];
  if (!topicId)
    return;
  
  for (i= 0; i < c; i++)
  {
    if ([[tabView pageAtIndex:i] isKindOfClass:[MQHelpTab class]])
    {
      page= (MQHelpTab*)[tabView pageAtIndex:i];
      [tabView selectPage:i];
      break;
    }
  }
  if (!page)
  {
    page= [[[MQHelpTab alloc] initWithIdentifier:@"Help"] autorelease];
    [tabView addTabViewItem:page];
    [tabView selectPage:[tabView numberOfPages]-1];
  }
  

  if (sender == syntaxList)
  {
    [page loadFile:@"mysqlqb_statements" topic:[NSString stringWithUTF8String:topicId]];
  }
  else
  {
    [page loadFile:@"mysqlqb_functions" topic:[NSString stringWithUTF8String:topicId]];
  }
}


- (IBAction)addQueryTab:(id)sender
{
  MQResultSetTab *page= [queryCtl createPage];
  if (page)
  {
    [page setDefaultSchema: [self defaultSchema]];
    [tabView addTabViewItem:page];
    [tabView selectPage:[tabView numberOfPages]-1];
  }
}


- (IBAction)addScriptTab:(id)sender
{
  MQScriptTab *page= [scriptCtl createPage];
  if (page)
  {
    [page setDefaultSchema: [self defaultSchema]];
    [tabView addTabViewItem:page];
    [tabView selectPage:[tabView numberOfPages]-1];
  }
}


- (void)explainQuery:(id)sender
{
  [queryCtl explainQuery:sender];
}


- (void)compareResultsets:(id)sender
{
  [queryCtl compareQuery:sender];
}


- (void)performCloseTab:(id)sender
{
  [self tabView:tabView closeTab:[self currentTab]];
}


- (void)closeTab:(MTabViewItem*)tabItem
{
  // remove tabsheet
  [tabView removeTabViewItem:tabItem];
}


- (IBAction)toggleDrawer:(id)sender
{
  if ([drawer state] == NSDrawerOpenState)
    [sender setTitle:@"Show Side Drawer"];
  else
    [sender setTitle:@"Hide Side Drawer"];
  [drawer toggle:sender];
}

- (IBAction)schemataDoubleClicked:(id)sender
{
  int row= [catalogList selectedRow];
  MSchemaItem *item= [catalogList itemAtRow:row];
  
  if (item->type == MSchemaItemType)
  {
    [self setDefaultSchema:nil];
  }
  else if ([self setupQueryWithSchemaItem:item])
  {
    [queryCtl selectQueryArea:nil];
    
    if ([[tabView selectedPage] isKindOfClass:[MQResultSetTab class]])
      [queryCtl executeQuery:nil];
  }
}


- (IBAction)refreshSchemata:(id)sender
{
  NSDictionary *expandedItems= MXGetExpandedOutlineItems(catalogList, 2, nil);
  [_columnDS resetTree];
  [catalogList reloadData];
  [self fetchSchemasFor:_columnDS];
  MXExpandOutlineItems(catalogList, expandedItems, nil);
}

- (IBAction)setDefaultSchema:(id)sender
{  
  NSString *schema;
  
  schema= [NSString stringWithUTF8String:[[catalogList itemAtRow:[catalogList selectedRow]] schema]->schema_name];
  [self selectSchema:schema];
}

- (IBAction)copyHistory:(id)sender
{
  int row= [historyList selectedRow];
  id item= [historyList itemAtRow:row];
  
  if ([item isKindOfClass:[MQHistoryItem class]])
  {
    NSPasteboard *pasteboard= [NSPasteboard generalPasteboard];
    [pasteboard declareTypes:[NSArray arrayWithObject:NSStringPboardType]
                       owner:nil];
    [pasteboard setString:[item query] forType:NSStringPboardType];
  }  
}


- (IBAction)executeHistory:(id)sender
{
  int row= [historyList selectedRow];
  id item= [historyList itemAtRow:row];
  
  [queryCtl performQuery:[item query]];
}


- (IBAction)deleteHistory:(id)sender
{
  int row= [historyList selectedRow];
  id item= [historyList itemAtRow:row];

  [[[MQueryBrowserController sharedDocumentController] history] removeItem:item];
  [historyList reloadData];
}



- (IBAction)copyBookmark:(id)sender
{
  int row= [bookmarkList selectedRow];
  id item= [bookmarkList itemAtRow:row];
  
  if (![item isKindOfClass:[MQBookmarkGroup class]])
  {
    NSPasteboard *pasteboard= [NSPasteboard generalPasteboard];
    [pasteboard declareTypes:[NSArray arrayWithObject:NSStringPboardType]
                       owner:nil];
    [pasteboard setString:[item query:YES] forType:NSStringPboardType];
  }
}

- (IBAction)executeBookmark:(id)sender
{
  int row= [bookmarkList selectedRow];
  id item= [bookmarkList itemAtRow:row];

  [queryCtl performQuery:[item query:YES]];
}

- (IBAction)newBookmarkGroup:(id)sender
{
  int row= [bookmarkList selectedRow];
  id item= [bookmarkList itemAtRow:row];
  int nrow;
  id nitem;

  nitem= [[[MQueryBrowserController sharedDocumentController] bookmarks]
    addGroupInItem:item];
  [bookmarkList reloadData];
  
  nrow= [bookmarkList rowForItem:nitem];
  
  [bookmarkList selectRow:nrow byExtendingSelection:NO];
  [bookmarkList editColumn:0 row:nrow withEvent:nil select:YES];
}

- (IBAction)deleteBookmarkItem:(id)sender
{
  int row= [bookmarkList selectedRow];
  id item= [bookmarkList itemAtRow:row];

  [[bookmarkList dataSource] removeBookmark:item];
  [bookmarkList reloadData];
}

- (IBAction)renameBookmarkItem:(id)sender
{
  int row= [bookmarkList selectedRow];
  [bookmarkList editColumn:0 row:row withEvent:nil select:YES];
}


- (IBAction)bookmarkQuery:(id)sender
{
  NSString *query;
  MQAddBookmarkPanel *panel;
  
  query= [queryCtl currentQuery];
  if (!query)
  {
    NSBeep();
    return;
  }
  
  panel= [[[MQAddBookmarkPanel alloc] init] autorelease];

  [panel showSheetForWindow:[self window]
                  bookmarks:[[MQueryBrowserController sharedDocumentController] bookmarks]
                    catalog:@"def"
                     schema:_info->schema ? [NSString stringWithUTF8String:_info->schema] : nil
                      query:[NSString stringWithString:query]];
  [bookmarkList reloadData];
}


- (IBAction)addParameter:(id)sender
{
  int row= [parameterList selectedRow];
  if (row >= 0)
  {
    [_parameters addParameter:[parameterList itemAtRow:row]];
    [parameterList reloadData];
  }
}


- (IBAction)deleteParameter:(id)sender
{
  int row= [parameterList selectedRow];
  if (row >= 0)
  {
    [_parameters deleteParameter:[parameterList itemAtRow:row]];
    [parameterList reloadData];
  }  
}


- (BOOL)keyDown:(NSEvent*)event
{
  if (([event modifierFlags] & (NSCommandKeyMask|NSShiftKeyMask))==(NSCommandKeyMask|NSShiftKeyMask))
  {
    switch ([[event characters] characterAtIndex:0])
    {
      case NSLeftArrowFunctionKey:
        [tabView selectPreviousPage];
        break;
      case NSRightArrowFunctionKey:
        [tabView selectNextPage];
        break;
    }
  }
  return NO;
}


- (void)killConnection:(unsigned long)pid
{
  mysql_kill(_mysql, pid);
}

- (void)schemaHelperChangedSchemata:(MSchemaEditHelper*)helper
{
  [self refreshSchemata:nil];
}


- (void)schemaHelper:(MSchemaEditHelper*)helper editScript:(NSString*)data
{
  MQScriptTab *page= [scriptCtl createPage];
  if (page)
  {
    [page setDefaultSchema: [self defaultSchema]];
    [page setScript:data];
    [tabView addTabViewItem:page];
    [tabView selectPage:[tabView numberOfPages]-1];
  }  
}


- (MMySQLDispatcher*)dispatcher
{
  return _dispatcher;
}

- (void)setTitleForDocumentName: (NSString*)name
{
  if (name)
    [[self window] setTitle: [NSString stringWithFormat:@"%@ @ %@", 
      name, _connectedInstance]];
  else
    [[self window] setTitle: _connectedInstance];
}


- (void)windowWillClose:(NSNotification *)aNotification
{
  [_parameters saveGlobals];
  
  [[MQueryBrowserController sharedDocumentController] removeInstance:self];
}


- (void)unlockMySQL
{
  [_mysqlLock unlock];
}

- (MYSQL*)mysql
{
  return _mysql;
}

- (MYSQL*)mysqlLock
{
  [_mysqlLock lock];
  return _mysql;
}

- (MYX_USER_CONNECTION*)serverInfo
{
  return _info;
}


- (MSchemaEditHelper*)schemaHelper
{
  return schemaHelper;
}

- (MQParameters*)parameters
{
  return _parameters;
}

- (NSOutlineView*)parameterList
{
  return parameterList;
}

//==============================================================================

- (int)numberOfItemsInMenu:(NSMenu *)menu
{
  return [menu numberOfItems];
}


- (BOOL)menu:(NSMenu *)menu
  updateItem:(NSMenuItem *)item
	 atIndex:(int)index
shouldCancel:(BOOL)shouldCancel
{
  if (menu == catalogMenu)
  {
    int selRow= [catalogList selectedRow];
    int level= selRow < 0 ? -1 : [catalogList levelForRow:selRow];
    BOOL flag= NO;
    
    switch (index)
    {
      case 0: // Set Def. Schema
        if (level > 0)
          flag= YES;
        else
          flag= NO;
        break;
      case 2: // Create Schema
        if (level >= 0)
          flag= YES;
        else
          flag= NO;
        break;
      case 3: // Create Table
        if (level >= 1)
          flag= YES;
        else
          flag= NO;
        break;
      case 4: // Create SP
        if (level >= 1 && mysql_version_is_later_or_equal_than(_mysql,5,0))
          flag= YES;
        else
          flag= NO;
        break;
      case 5: // Create View
        if (level >= 1 && mysql_version_is_later_or_equal_than(_mysql,5,0))
          flag= YES;
        else
          flag= NO;
        break;
      case 6: // Edit
      {
        MSchemaItem *sitem= [catalogList itemAtRow:selRow];
        switch (level)
        {
          case 0:
            [item setTitle:@"Edit Object"];
            flag= NO;
            break;		  
          case 1:
            [item setTitle:@"Edit Schema"];
            flag= NO;//XXX NOT SUPPORTED YET
            break;		  
          case 2:
            if ([sitem type] == MTableItemType)
            {
              [item setTitle:@"Edit Table"];
              flag= YES;
            }
            else if ([sitem type] == MViewItemType)
            {
              [item setTitle:@"Edit View"];
              flag= mysql_version_is_later_or_equal_than(_mysql,5,0)?YES:0;
            }
            else
            {
              [item setTitle:@"Edit Stored Procedure"];
              flag= mysql_version_is_later_or_equal_than(_mysql,5,0)?YES:0;
            }
            break;
          default:
            [item setTitle:@"Edit Object"];
            flag= NO;
            break;
        } 
        break;
      }
      case 7: // Drop
        switch (level)
        {
          case 0:
            [item setTitle:@"Drop Object"];
            flag= NO;
            break;
          case 1:
            [item setTitle:@"Drop Schema"];
            flag= YES;
            break;
          case 2:
          {
            MSchemaItem *sitem= [catalogList itemAtRow:selRow];
            if ([sitem type] == MTableItemType)
            {
              [item setTitle:@"Drop Table"];
              flag= YES;
            }
            else if ([sitem type] == MViewItemType)
            {
              [item setTitle:@"Drop View"];
              flag= mysql_version_is_later_or_equal_than(_mysql,5,0)?YES:0;
            }
            else
            {
              [item setTitle:@"Drop Stored Procedure"];
              flag= mysql_version_is_later_or_equal_than(_mysql,5,0)?YES:0;
            }
            break;
          }
          default:
            [item setTitle:@"Drop Object"];
            flag= NO;
            break;
        }
        break;
      case 9: // Copy to Clipboard
      {
        MSchemaItem *sitem= [catalogList itemAtRow:selRow];
        switch ([sitem type])
        {
          case MTableItemType:
          case MViewItemType:
          case MSchemaItemType:
          case MSPItemType:
            flag= YES;
            break;
          default:
            flag= NO;
            break;
        }
        break;
      }
      case 11: // Refresh
        flag= YES;
        break;
    }
    
    [item setEnabled:flag];
  }
  else if (menu == bookmarkMenu)
  {
    int sel= [bookmarkList selectedRow];
    id bitem= sel < 0 ? nil : [bookmarkList itemAtRow:sel];
    BOOL isGroup= [bitem isKindOfClass:[MQBookmarkGroup class]];
    BOOL flag= NO;

    switch (index)
    {
      case 0:
        flag= !isGroup;
        break;
      case 1:
        flag= !isGroup;
        break;
      case 3:
        flag= !isGroup;
        break;
      case 4:
        flag= !isGroup;
        break;
      case 6:
        flag= YES;
        break;
      case 7:
        flag= isGroup;
        break;
      case 8:
        flag= isGroup;
        break;
    }
    [item setEnabled:flag];
  }
  else if (menu == historyMenu)
  {
    int sel= [historyList selectedRow];
    id hitem= sel < 0 ? nil : [historyList itemAtRow:sel];
    [item setEnabled:[hitem isKindOfClass:[MQHistoryItem class]]];
  }
  return YES;
}


- (void)tabViewChangedPage:(MTabView*)sender
{
  id page= [sender selectedPage];
  
  if ([page isKindOfClass:[MQResultSetTab class]])
  {
    [queryCtl willShowPage:page];
    [scriptCtl didHidePage];
  }
  else if ([page isKindOfClass:[MQScriptTab class]])
  {
    [scriptCtl willShowPage:page];
    [queryCtl didHidePage];
  }
}


- (IBAction)switchToolbarType:(id)sender
{
  BOOL small;

  if ([sender tag] != 100)
  {
    [sender setTag:100];
    [sender setTitle:@"Normal Query Area"];
    small= YES;
  }
  else
  {
    [sender setTag:101];
    [sender setTitle:@"Maximize Query Area"];
    small= NO;
  }
  [self setSmallToolbar:small];
}


- (void)setToolbarHeight:(float)height
{  
  if (!_smallToolbar)
  {
    NSRect rect= [tabView frame];
    
    rect.size.height= (rect.size.height + height) - _toolbarHeight;    
    [tabView setFrame:rect];
    
    [queryCtl resizeToolbar:height];
    [scriptCtl resizeToolbar:height];
  }
  _toolbarHeight= height;
}


- (void)setSmallToolbar:(BOOL)flag
{
  if (_smallToolbar != flag)
  {
    NSRect rect= [tabView frame];

    _smallToolbar= flag;
    
    if (flag)
      rect.size.height= (rect.size.height + _toolbarHeight) - 38;
    else
      rect.size.height= (rect.size.height + 38) - _toolbarHeight;
 
    [tabView setFrame:rect];

    [queryCtl switchToolbar:flag];
    [scriptCtl switchToolbar:flag];
    
    [[[self window] contentView] setNeedsDisplay:YES];
  }
}


- (BOOL)smallToolbar
{
  return _smallToolbar;
}


- (id)currentTab
{
  return [tabView selectedPage];
}


- (void)closeTabAlertDidEnd:(NSAlert *)alert returnCode:(int)returnCode contextInfo:(void *)contextInfo
{
  if (returnCode == NSAlertDefaultReturn)
    [self close];
}


- (void)tabView:(MTabView*)aTabView closeTab:(MTabViewItem*)tab
{
  if ([aTabView numberOfPages] == 1)
  {
    [[NSAlert alertWithMessageText:@"Close Tabsheet"
                    defaultButton:@"Close"
                  alternateButton:@"Cancel"
                      otherButton:nil
        informativeTextWithFormat:@"Closing the last tabsheet in the window will close the window as well."]
      beginSheetModalForWindow:[self window]
                 modalDelegate:self
                didEndSelector:@selector(closeTabAlertDidEnd:returnCode:contextInfo:) 
                   contextInfo:tab];
  }
  else
    [self closeTab:tab];
}


- (void)outlineView:(NSOutlineView *)outlineView 
    willDisplayCell:(id)cell 
     forTableColumn:(NSTableColumn *)tableColumn 
               item:(id)item
{
  if (outlineView == catalogList)
  {
	NSString *schema= _info->schema ? [NSString stringWithUTF8String:_info->schema] : nil;
	
	if ([item respondsToSelector:@selector(icon)])
	  [cell setImage:[item icon]];
	else
	  [cell setImage:nil];
    
	if ([catalogList levelForItem:item] == 1)
	{
	  if ([[item repr] isEqualToString:schema])
		[cell setFont:_boldFont];
	  else
		[cell setFont:_normalFont];
	}
	else
	  [cell setFont:_normalFont];
  }
  else if (outlineView == historyList)
  {
	if ([historyList levelForItem:item] == 0)
	  [cell setImage:_historyGroupIcon];
	else
	  [cell setImage:_historyItemIcon];
  }
  else if (outlineView == bookmarkList)
  {
    if ([item isKindOfClass:[MQBookmarkGroup class]])
      [cell setImage:_bookmarkGroupIcon];
    else
      [cell setImage:_bookmarkItemIcon];
  }
  else if (outlineView == parameterList)
  {
    if ([item isKindOfClass:[NSString class]])
      [cell setImage:_groupIcon];
    else
      [cell setImage:_paramIcon];
  }
}


- (void)outlineViewSelectionDidChange:(NSNotification *)notification
{
  if ([notification object] == parameterList)
  {
    int row= [parameterList selectedRow];
    BOOL add= NO;
    BOOL del= NO;
    if (row >= 0)
    {
      id item= [parameterList itemAtRow:row];
      if ([_parameters itemCanBeAdded:item])
        add= YES;
      if ([_parameters itemCanBeDeleted:item])
        del= YES;
    }
    [[[parameterList enclosingScrollView] viewWithTag:1] setEnabled:add];
    [[[parameterList enclosingScrollView] viewWithTag:2] setEnabled:del];
  }
}


@end
