/* Copyright (C) 2005 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#ifndef _MGRT_H_
#define _MGRT_H_

#include "myx_grt_public_interface.h"
#include <glibmm.h>
#include <sigc++/sigc++.h>
#include <list>
#include "MGRTValue.h"
#include <myg_utils.h>


class MGRTShellView;

class MGRT {
  public:
    class Request {
      protected:
        MGRTValue _result;
        MYX_GRT_ERROR _error;
      public:
        virtual ~Request() {};
        virtual void execute(MGRT *grt)= 0;
        virtual void wait()= 0;
        virtual MGRTValue result() { return _result; };
        virtual MYX_GRT_ERROR error() { return _error; };
    };

  protected:    
    MYX_GRT *_grt;
    
    std::list<Request*> _requestQueue;
    Glib::Mutex _requestQueueLock;
    Glib::Cond _requestReady;
    
    MGRTShellView *_console;
    
    sigc::slot<void,Glib::ustring> _outputHandler;
    sigc::slot<void,Glib::ustring,Glib::ustring> _messageHandler;

    MYX_GRT_ERROR _lastError;
    Glib::ustring _lastErrorMessage;
    
    Glib::ustring _appDictPath;
    
    bool _thread_ready;

    static void process_grt_output(const char *text, void *data);
    static void process_grt_message(MYX_GRT_MSGS *msgs, void *data);
    
    void init_thread(const std::string &path);
    void request_thread(std::string path);
    
    void queue_request(Request *req);
  public:
    MGRT();
    virtual ~MGRT();
    
    void initialize_grt_thread(const Glib::ustring &resourcePath);
    
    inline MYX_GRT *grt() const { return _grt; };

    void set_console(MGRTShellView *shell);
    void out_text(const Glib::ustring &text);
    void out_message(const Glib::ustring &text,
                    const Glib::ustring &type);
    void perform_shell_command(const Glib::ustring &command);
    Glib::ustring shell_prompt();

    bool save_app_dict();
    bool load_app_dict_from_path(const std::string &path);
    MGRTValue global_app_dict();

    bool scan_structs_in_path(const std::string &path);
    bool scan_modules_in_path(const std::string &path);


    bool call_procedure(const std::string &module,
                       const std::string &procedure,
                       const MGRTValue &arguments);
    MGRTValue call_function(const std::string &module,
                            const std::string &function,
                            const MGRTValue &arguments);
    
    int call_int_function(const std::string &module,
                        const std::string &function,
                        const MGRTValue &arguments);

    Glib::ustring call_string_function(const std::string &module,
                                     const std::string &function,
                                     const MGRTValue &arguments);

    void set_output_handler(sigc::slot<void,Glib::ustring> handler);
    void reset_output_handler();

    void set_message_handler(sigc::slot<void,Glib::ustring,Glib::ustring> handler);
    void reset_message_handler();
    
    void report_error(MYX_GRT_ERROR error);
    
    MYX_GRT_ERROR last_error();
    Glib::ustring last_error_description();

    bool result_is_error(const MGRTValue &value);
    void report_error_result(const MGRTValue &value);

    Glib::ustring error_text(MYX_GRT_ERROR error);

    MGRTValue global_value(const char *path);
    MGRTValue global_ref_value(const char *path);
    void set_global_value(const char *path, const MGRTValue &value);
    
    MGRTValue global_value(const Glib::ustring &path) { return global_value(path.c_str()); };
    MGRTValue global_ref_value(const Glib::ustring &path) { return global_ref_value(path.c_str()); };
    void set_global_value(const Glib::ustring &path, const MGRTValue &value) { set_global_value(path.c_str(), value); };
};

#endif /* _MGRT_H_ */

