# -*- coding: utf-8 -*-
#
# «core» - Abstract lirc creator
#
# Copyright (C) 2007, Mario Limonciello, for Mythbuntu
# Copyright (C) 2007, Nick Fox, for Mythbuntu
#
#
# Mythbuntu is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 3 of the License, or at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this application; if not, write to the Free Software Foundation, Inc., 51
# Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##################################################################################

import os
import re
import string

class ProgramHandler():
    button_dictionary= { \
        "power":"^power|^on_off|^esc|^halt", \
        "up":"^up|^arrow_up|^up_arrow|^arrowup", \
        "down":"^down|^down_arrow|^arrow_down|^arrowdown", \
        "left":"^left|^leftarrow|^seek\-|^right_arrow_term|^arrow_left|^arrowleft|^skipback", \
        "right":"^right|^rightarrow|^seek\+|^left_arrow_term|^arrow_right|^arrowright|^skipfwd", \
        "enter":"^enter|^ent|^start", \
        "rewind":"^rewind|^fastrwd|^rev|^backward|^rrev|^rwnd|^rrew|^grew", \
        "forward":"^forward|^fastfwd|^fwd|^fforward|^ffwd|^ff|^rff|^gff|^ffw|^fwdwind", \
        "vol-":"^vol\-|^voldown|^vol-down|^volume\-|^vol_down|^volumedown|^RVolumeDown|^Volume_Minus|^Vol_Minus", \
        "vol+":"^vol\+|^volup|^vol-up|^volume\+|^vol_up|^volumeup|^RVolumeUp|^Volume_Plus|^Vol_Plus", \
        "mute":"^mute|^rmute|^gmute", \
        "chan-":"^chan\-|^chandown|^chan-down|^channel\-|^ch\-|^chdown|^ch_down|^channeldown|^chnldn|^channel_down|^tune\-|^RChannelDown|^Channel_Minus|^Chan_Minus|^PG_DN|^pg\-", \
        "chan+":"^chan\+|^chanup|^chan-up|^channel\+|^ch\+|^chup|^ch_up|^channelup|^chnlup|^channel_up|^tune\+|^RChannelUp|^Channel_Plus|^Chan_Plus|^PG_UP|^pg\+", \
        "ok":"^ok|^start|^ent", \
        "home":"^home", \
        "menu":"^menu|^rmenu|^gmenu", \
        "info":"^info|^more|^osd|^display|^preview", \
        "pause":"^pause|^freeze|^playpause|^play-pause|^play_pause|^still", \
        "play":"^play|playpause|^play-pause|^play_pause", \
        "stop":"^stop|^esc|^cancel|^return", \
        "exit":"^exit|^return|^close|^halt|^back/exit|^backexit", \
        "record":"^record|^rec|^capture", \
        "one":"^1|^one|^r1|^G1", \
        "two":"^2|^two|^r2|^G2", \
        "three":"^3|^three|^r3|^G3", \
        "four":"^4|^four|^r4|^G4", \
        "five":"^5|^five|^r5|^G5", \
        "six":"^6|^six|^r6|^G6", \
        "seven":"^7|^seven|^r7|^G7", \
        "eight":"^8|^eight|^r8|^G8", \
        "nine":"^9|^nine|^r9|^G9", \
        "zero":"^0|^zero|^r0|^G0", \
        "skip":"^skip", \
        "replay":"^replay", \
        "guide":"^guide", \
        "dvd":"^dvd|^dvd_vcd|^powerdvd|^power_dvd", \
        "red":"^red", \
        "green":"^green", \
        "yellow":"^yellow", \
        "blue":"^blue" \
    }

    def __init__(self,application_name= None, application_dictionary=None,lircd="/etc/lirc/lircd.conf", lircrc=".lircrc", repeat="0", delay="0"):
        if application_name == None or application_dictionary == None:
            self.valid=False
        else:
            self.valid=True
        self.application_name = application_name
        self.application_dictionary = application_dictionary
        self.lircd = lircd
        self.lircrc = lircrc
        self.repeat = repeat
        self.delay = delay

    def is_valid(self):
        return self.valid

    def generate(self):
        """Generates a config for this program type"""
        def parse_lircd():
            """Reads in a lircd.conf and returns all buttons found"""
            in_f=open(self.lircd).readlines()
            filterpattern = re.compile("^#")
            remotes = {}
            codes=False
            remotename=""
            for line in in_f:
                if filterpattern.search(line) is None:
                    if not codes:
                        if re.compile("name").search(line) is not None:
                            list = string.split(line)
                            if len(list) >1 :
                                remotename=list[1]
                        if re.compile("begin codes").search(line) is not None:
                            codes=True
                            remotes[remotename]=[]
                    else:
                        if re.compile("end codes").search(line) is not None:
                            codes=False
                        else:
                            list = string.split(line)
                            if len(list) > 1:
                                #buttons[list[0]] = remotename
                                remotes[remotename].append(list[0])
            return remotes
        self.remotes = parse_lircd()
        out_f = open(self.lircrc,'a')
        self.match_dictionary(out_f)
        out_f.close()

    def get_app_functionality(self):
        """Returns all required functionality keys"""
        return self.application_dictionary

    def get_app_buttons(self):
        """Returns all mapping created"""
        return self.button_dictionary

    def get_remotes(self):
        """Returns all parsed remotes"""
        return self.remotes

    def debug(self):
        """Outputs a plethora of useful debug info"""
        print "---" + self.application_name + "---"
        print "Parsed Remotes:"
        print self.get_remotes()
        print "App Functionality:"
        print self.get_app_functionality()
        print "Mapped Buttons:"
        print self.get_app_buttons()

    def match_dictionary(self,stream):
        #Go through all possible remotes
        #Remotes are represented as a dictionary item that matches a list
        for remote in self.remotes:
            #Go through all possible buttons for this application
            for item in self.application_dictionary:
                pattern = re.compile(self.button_dictionary[item],re.IGNORECASE)
                for button in self.remotes[remote]:
                    if pattern.search(button) is not None:
                        self.write_key(stream,self.application_dictionary[item],button,remote)
                        break #we only want the first instance (ignore _DOWN or _UP)


    def write_key(self,stream,function,button,remote):
        stream.write("begin\n")
        stream.write("    remote = " + remote +                "\n")
        stream.write("    prog = "   + self.application_name + "\n")
        stream.write("    button = " + button +                "\n")
        stream.write("    config = " + function +              "\n")
        stream.write("    repeat = " + self.repeat +           "\n") 
        stream.write("    delay = " + self.delay +             "\n")
        stream.write("end\n")
        stream.write("\n")
