/*
 * Mausezahn - A fast versatile traffic generator
 * Copyright (C) 2008 Herbert Haas
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 as published by the 
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more 
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with 
 * this program; if not, see http://www.gnu.org/licenses/gpl-2.0.html
 * 
*/



////////////////////////////////////////////////////////////////////////////////////////////
//
//  Contains various tools to ease development of new modules:
//  
//  getarg  ............. scans string for arguments and returns assigned value
//  str2int ............. Converts a string into long int in a safe way
//  get_ip_range_dst .... Parses string for an IP range and sets start/stop addresses
//  get_ip_range_src .... Same for source addresses
//  check_eth_mac_txt ... Scans tx.eth_dst|src_txt and sets tx.eth_dst|src appropriately
//  get_port_range ...... Parses string for a dst|src-port range and sets start/stop values
//  get_tcp_flags ....... Parses string for TCP arguments and sets tx.tcp_control
//  get_mpls_params ..... Parses string for MPLS parameters (label, exp, BOS, TTL)
//  exists .............. Parses a string for a single character and returns "1" if found
//  
////////////////////////////////////////////////////////////////////////////////////////////

#include "mz.h"



// Scan 'str' for an argument 'arg_name' and returns its value in arg_value
// Return value: number of occurences of arg_name
// Note that if arg_name occurs multiple times, the last found value is returned.
// If last argument (arg_value) is set to NULL it will be ignored.
// Example: 
//   int i;
//   char ip[64];
//   i = getarg ("request, da=10.1.1.2, SYN", "da", ip);
// ...will assign "10.1.1.2" to ip and the occurence i is set to 1.
int getarg(char *str, char *arg_name, char *arg_value)
{
   char tmp[MAX_PAYLOAD_SIZE];
   char *str1, *str2, *token, *subtoken;
   char *saveptr1, *saveptr2;
   int j, occurence=0;
   
   strncpy(tmp,str,MAX_PAYLOAD_SIZE); // only operate on local strings
   
   for (j = 1, str1 = tmp; ; j++, str1 = NULL) 
     {
	
	token = strtok_r(str1, ",", &saveptr1);
	if (token == NULL)
	  break;

	str2 = token; 
	if ( (subtoken = strtok_r(str2, " =", &saveptr2))!=NULL)
	  {
	     if (strcmp(subtoken,arg_name)==0)
	       {
		  occurence+=1;
		  //printf("found %s\n",arg_name);
		  if ( (subtoken = strtok_r(NULL, " =", &saveptr2))!=NULL)
		    {
		       // argument has a value!
		       //printf("%s has value: [%s]\n",arg_name, subtoken);
		       if (arg_value!=NULL)
			 {
			    strcpy(arg_value,subtoken);
			 }
		    }
	       }
	  }
	else
	  break;
     }
   return occurence;
}


// Convert str to (long) int
// Return value: the unsigned long int
unsigned long int str2int(char *str)
{  
   unsigned long int i;
   
   errno=0;
   
   i = strtoul(str, (char **)NULL, 10);

   if ((errno == ERANGE && (i == LONG_MAX || i == LONG_MIN))
       || (errno != 0 && i == 0))
     {
	
	perror("strtol");
     }
   
   return i;
}



// Parses string 'arg' for an IP range and finds start and stop IP addresses.
// Return value: 0 upon success, 1 upon failure.
// 
// NOTE: The results are written in the following variables:
// 
//   (u_int32_t) tx.ip_dst_start    ... contains start value 
//   (u_int32_t) tx.ip_dst_stop     ... contains stop value
//   (u_int32_t) tx.ip_dst          ... initialized with start value 
//   int         tx.ip_dst_isrange  ... set to 1 if above values valid
//      
// Possible range specifications:
// 
//   1) 192.168.0.0-192.168.0.12
//   2) 10.2.11.0-10.55.13.2
//   3) 172.18.96.0/19
// 
// That is: 
// 
//   FIRST detect a range by scanning for the "-" OR "/" chars
//   THEN determine start and stop value and store them as normal unsigned integers
// 
int get_ip_range_dst (char *arg)
{

   int 
     i, len, 
     found_slash=0, found_dash=0;

   unsigned int q, b;
   u_int32_t mask, invmask;
   
   char *start_str, *stop_str;
   
   len = strlen(arg);
   
   if ( (len>31) || (len<9) ) // 255.255.255.200-255.255.255.255 (31 chars) OR 1.0.0.0/8 (9 chars)
     return 1; // ERROR: no range
   
   // Find "-" or "/"
   for (i=0; i<len; i++)
     {
	if  (arg[i]=='/')  found_slash=1;
	if  (arg[i]=='-')  found_dash=1;
     }

   if ((found_slash) && (found_dash)) 
     exit (1); // ERROR: Wrong range string syntax (cannot use both "/" and "-" !!!
   
   if (found_dash)
     {
	start_str = strtok (arg, "-");
	stop_str = strtok (NULL, "-");

	// These are the start and stop IP addresses of the range:
	tx.ip_dst_start = str2ip32 (start_str);  
	tx.ip_dst_stop = str2ip32 (stop_str);
	tx.ip_dst_h = tx.ip_dst_start; 
	tx.ip_dst = str2ip32_rev (start_str);
	
	if (tx.ip_dst_start < tx.ip_dst_stop)
	  {
	     // Set range flag:
	     tx.ip_dst_isrange = 1;
	     return 0;
	  }
	else
	  {
	     tx.ip_dst_isrange = 0;
	     return 1; // ERROR: stop value must be greater than start value !!!
	  }
     }
   else if (found_slash)
     {
	start_str = strtok (arg, "/");
	stop_str = strtok (NULL, "/"); // Actually contains the prefix length, e. g. "24" 
	
	q = (unsigned int) str2int (stop_str);
	
	// IDEA: if the prefix q is e. g. 27 then take 32-27=5 and create
	// the sum 1+2+4+8+16 = 31 and finally the required mask is (2^32-1)-31.
	
	if (q<32)
	  {
	     q = 32-q;
	  }
	else
	  {
	     return 1; // ERROR: Specified prefix greater than 31 !!!
	  }
	
	invmask=1;
	b=2;
	
	for (i=1; i<q; i++)
	  {
	     invmask = invmask + b;
	     b=2*b;
	  }
	
	mask = 0xffffffff;
	mask = mask - invmask;  // invert
	  
	tx.ip_dst_start = (str2ip32 (start_str) +1) & mask; // the '+1' is to ensure that we do not start with the net-id
	tx.ip_dst_stop = tx.ip_dst_start | invmask; 
	tx.ip_dst_h = tx.ip_dst_start; 
	tx.ip_dst = str2ip32_rev (start_str) | 0x01000000; // the '0x01000000' is to ensure that we do not start with the net-id
	tx.ip_dst_isrange = 1;
	return 0;
     }
   
    
   return 1; // ERROR: The specified argument string is not a range!
   
}




// Parses string 'arg' for an IP range and finds start and stop IP addresses.
// Return value: 0 upon success, 1 upon failure.
// 
// NOTE: The results are written in the following variables:
// 
//   (u_int32_t) tx.ip_src_start    ... contains start value 
//   (u_int32_t) tx.ip_src_stop     ... contains stop value
//   (u_int32_t) tx.ip_src          ... initialized with start value
//   int         tx.ip_src_isrange  ... set to 1 if above values valid
//   
// Possible range specifications:
// 
//   1) 192.168.0.0-192.168.0.12
//   2) 10.2.11.0-10.55.13.2
//   3) 172.18.96.0/19
// 
// That is: 
// 
//   FIRST detect a range by scanning for the "-" OR "/" chars
//   THEN determine start and stop value and store them as normal unsigned integers
// 
int get_ip_range_src (char *arg)
{

   int 
     i, len, 
     found_slash=0, found_dash=0;

   unsigned int q, b;
   u_int32_t mask, invmask;
   
   char *start_str, *stop_str;
   
   len = strlen(arg);
   
   if ( (len>31) || (len<9) ) // 255.255.255.200-255.255.255.255 (31 chars) OR 1.0.0.0/8 (9 chars)
     return 1; // ERROR: no range
   
   // Find "-" or "/"
   for (i=0; i<len; i++)
     {
	if  (arg[i]=='/')  found_slash=1;
	if  (arg[i]=='-')  found_dash=1;
     }

   if ((found_slash) && (found_dash)) 
     exit (1); // ERROR: Wrong range string syntax (cannot use both "/" and "-" !!!
   
   if (found_dash)
     {
	start_str = strtok (arg, "-");
	stop_str = strtok (NULL, "-");

	// These are the start and stop IP addresses of the range:
	tx.ip_src_start = str2ip32 (start_str);  
	tx.ip_src_stop = str2ip32 (stop_str);
	tx.ip_src_h = tx.ip_src_start;
	tx.ip_src = str2ip32_rev (start_str);
	
	if (tx.ip_src_start < tx.ip_src_stop)
	  {
	     // Set range flag:
	     tx.ip_src_isrange = 1;
	     return 0;
	  }
	else
	  {
	     tx.ip_src_isrange = 0;
	     return 1; // ERROR: stop value must be greater than start value !!!
	  }
     }
   else if (found_slash)
     {
	start_str = strtok (arg, "/");
	stop_str = strtok (NULL, "/"); // Actually contains the prefix length, e. g. "24" 
	
	q = (unsigned int) str2int (stop_str);
	
	// IDEA: if the prefix q is e. g. 27 then take 32-27=5 and create
	// the sum 1+2+4+8+16 = 31 and finally the required mask is (2^32-1)-31.
	
	if (q<32)
	  {
	     q = 32-q;
	  }
	else
	  {
	     return 1; // ERROR: Specified prefix greater than 31 !!!
	  }
	
	invmask=1;
	b=2;
	
	for (i=1; i<q; i++)
	  {
	     invmask = invmask + b;
	     b=2*b;
	  }
	
	mask = 0xffffffff;
	mask = mask - invmask;  // invert
	  
	tx.ip_src_start = (str2ip32 (start_str)+1) & mask; // the '+1' is to ensure that we do not start with the net-id
	tx.ip_src_stop = tx.ip_src_start | invmask; 
	tx.ip_src_h = tx.ip_src_start; 
	tx.ip_src = str2ip32_rev (start_str) | 0x01000000; // the '0x01000000' is to ensure that we do not start with the net-id
	tx.ip_src_isrange = 1;
	return 0;
     }
   
   return 1; // ERROR: The specified argument string is not a range!
   
}


// Scans tx.eth_dst_txt or tx.eth_src_txt and sets the corresponding
// MAC addresses (tx.eth_dst or tx.eth_src) accordingly.
// Argument: What string should be checked, ETH_SRC or ETH_DST.
// Return value: 
//       0 when a MAC address has been set or
//       1 when not set (or wrongly set)
// Currently eth_src|dst_txt can be:
//   'rand', 'own', 'bc'|'bcast', 'stp', 'pvst', 'cisco', 
//  or a real mac address.
// 
// TODO: implement other important MAC addresses
int check_eth_mac_txt(int src_or_dst)
{
   char *eth_mac_txt;
   u_int8_t  *eth_mac;
   int *eth_rand;
   int i;
   
   // Check argument
   if (src_or_dst == ETH_SRC)
     {
	eth_mac_txt = tx.eth_src_txt;
	eth_mac = tx.eth_src;
	eth_rand = &tx.eth_src_rand;
     }
   else if (src_or_dst == ETH_DST)
     {
	eth_mac_txt = tx.eth_dst_txt;
	eth_mac = tx.eth_dst;	
	eth_rand = &tx.eth_dst_rand;
     }
   else
     {
	return 1; // wrong argument
     }
   
   
   // Did the user really specify a dst-address?
   if (strlen(eth_mac_txt)==0)  
     {
	return 1; // No.
     }
   
   
   // Okay, lets check the commandline argument:
   // 
   // Do you want a random MAC?
   // TODO: Consider enforcement of unicast addresses
   if (strncmp(eth_mac_txt, "rand", 4)==0)
     {
	eth_mac[0] = (u_int8_t) ( ((float) rand()/RAND_MAX)*256);
	eth_mac[1] = (u_int8_t) ( ((float) rand()/RAND_MAX)*256);
	eth_mac[2] = (u_int8_t) ( ((float) rand()/RAND_MAX)*256);
	eth_mac[3] = (u_int8_t) ( ((float) rand()/RAND_MAX)*256);
	eth_mac[4] = (u_int8_t) ( ((float) rand()/RAND_MAX)*256);
	eth_mac[5] = (u_int8_t) ( ((float) rand()/RAND_MAX)*256);
	*eth_rand = 1;
     }
   // Do you want your own interface MAC?
   else if (strncmp(eth_mac_txt, "own", 3)==0)
     {
	for (i=0; i<6; i++)
	  {
	     eth_mac[i] = tx.eth_mac_own[i];
	  }
     }
   // Do you want a broadcast MAC?
   else if (strncmp(eth_mac_txt, "bc", 2)==0) // NOTE that this also fetches "bcast"
     {
	str2hex_mac("FF:FF:FF:FF:FF:FF", eth_mac);
     }
   // Do you want the IEEE address 'all bridges' used for STP?
   else if (strncmp(eth_mac_txt, "stp", 3)==0) // 
     {
	str2hex_mac("01:80:C2:00:00:00", eth_mac); // IEEE for all bridges	
     }
   // Do you want the Cisco address e. g. for CDP, VTP?
   else if (strncmp(eth_mac_txt, "cisco", 5)==0) 
     {
	str2hex_mac("01:00:0C:CC:CC:CC", eth_mac); 
     }
   // Do you want the Cisco address e. g. for CDP, VTP?
   else if (strncmp(eth_mac_txt, "pvst", 5)==0)
     {
	str2hex_mac("01:00:0C:CC:CC:CD", eth_mac); 
     }
   // The string MUST contain a mac address
   // TODO: CHECK whether the string has correct format for a mac address!
   else 
     {
	str2hex_mac(eth_mac_txt, eth_mac);
     }

   return 0;
}


// Scans argument for a port number or range and sets 
// the corresponding values in the tx struct:
// 
// a)  tx.sp_start, tx.sp_stop, tx.sp = tx.sp_start
//   
//         ** OR **
//            
// b)  tx.dp_start, tx.dp_stop, tx.dp = tx.dp_start
//   
// Arguments:
// 
//    - 'sp_or_dp' is either SRC_PORT or DST_PORT
//    - 'arg' contains the port range as string such as 1-1024
//    
// Return value: 0 on success, 1 upon failure
// 
int get_port_range (int sp_or_dp, char *arg)
{

   int i, len, found_dash=0;

   u_int32_t tmp1, tmp2;
   
   u_int16_t
     *port,
     *start,
     *stop;
   int 
     *isrange;

   char *start_str, *stop_str;
   
   
   // Check which port to manage
   if (sp_or_dp == DST_PORT)
     {
	port    =  &tx.dp;
	start   =  &tx.dp_start;
	stop    =  &tx.dp_stop;
	isrange =  &tx.dp_isrange;
     }
   else if (sp_or_dp == SRC_PORT)
     {
	port    =  &tx.sp;
	start   =  &tx.sp_start;
	stop    =  &tx.sp_stop;
	isrange =  &tx.sp_isrange;
     }
   else
     {
	return 1;  // error
     }

   
   len = strlen(arg);
   if (len==0) return 1; // error
   
   // Find "-" 
   for (i=0; i<len; i++)
     {
	if  (arg[i]=='-')  found_dash=1;
     }   
   
   if (found_dash)  // range specified
     {
	start_str = strtok (arg, "-");
	stop_str = strtok (NULL, "-");
	
	tmp1 = str2int (start_str);
	if ( (tmp1<0)||(tmp1>65535)) 
	  { 
	     fprintf(stderr," mz/get_port_range: Invalid port range!\n");
	     exit (-1);
	  }
	*start = tmp1;

	tmp2 = str2int (stop_str);
	if ( (tmp2<0)||(tmp2>65535)) 
	  { 
	     fprintf(stderr," mz/get_port_range: Invalid port range!\n");
	     exit (-1);
	  }
	*stop  = tmp2;
	
	if (tmp1>tmp2) // swap start/stop values!
	  {
	     *start = tmp2;
	     *stop  = tmp1;
	  }

	*port = *start;
	*isrange = 1;

	return 0;
     }
   else // single port number
     {
	tmp1 = str2int (arg);
	if ( (tmp1<0)||(tmp1>65535)) tmp1=0;
	*port    = tmp1; 
 	*isrange = 0;
	return 0;
     }
   
   return 1; // error
}




// Scans argument for TCP flags and sets 
// tx.tcp_control accordingly.
// 
// Valid keywords are: fin, syn, rst, psh, ack, urg, ecn, cwr
// Valid delimiters are: | or + or -
// Return value: 0 on success, 1 upon failure
// 
int get_tcp_flags (char*  flags)
{
   char *f;
   
   // From LSB to MSB: fin, syn, reset, push, ack, urg, ecn, cwr
   // ecn...ECN-Echo, cwr...Congestion Window Reduced
   
   if (strlen(flags)==0) return 1; // error
   

   f = strtok (flags, "|+-");
   do 
     {
	if (strncmp(f,"fin",3)==0) 
	  {
	     tx.tcp_control = tx.tcp_control | 1; 
	  }
	else if (strncmp(f,"syn",3)==0)
	  {
	     tx.tcp_control = tx.tcp_control | 2;
	  }
	else if (strncmp(f,"rst",3)==0)
	  {
	     tx.tcp_control = tx.tcp_control | 4;
	  }
	else if (strncmp(f,"psh",3)==0)
	  {
	     tx.tcp_control = tx.tcp_control | 8;
	  }
	else if (strncmp(f,"ack",3)==0)
	  {
	     tx.tcp_control = tx.tcp_control | 16;
	  }
	else if (strncmp(f,"urg",3)==0)
	  {
	     tx.tcp_control = tx.tcp_control | 32;
	  }
	else if (strncmp(f,"ecn",3)==0)
	  {
	     tx.tcp_control = tx.tcp_control | 64;
	  }
	else if (strncmp(f,"cwr",3)==0)
	  {
	     tx.tcp_control = tx.tcp_control | 128;
	  }
	
     } while ( (f=strtok(NULL, "|+-")) != NULL);
   
   return 0;
}



// Scans string 'params' for MPLS parameters 
// and sets tx.mpls_* accordingly.
// 
// CLI Syntax Examples:
// 
// -M help       .... shows syntax
// 
// -M 800        .... label=800
// -M 800:S      .... label=800 and BOS flag set
// -M 800:S:64   .... label=800, BOS, TTL=64
// -M 800:64:S   .... same
// -M 64:77      .... label=64, TTL=77
// -M 64:800     .... INVALID
// -M 800:64     .... label=800, TTL=64
// -M 800:3:S:64 .... additionall the experimental bits are set (all fields required!)
// 
// Note: S = BOS(1), s = NOT-BOS(0)
// 
// Valid delimiters: :-.,+
// Return value: 0 on success, 1 upon failure
int get_mpls_params(char *p)
{

   char *f1, *f2, *f3, *f4;
   char params[256];

   tx.mpls_exp = 0;
   tx.mpls_ttl = 255;
   
   strncpy(params, p, 256);
   
   if (strncmp(params,"help",4)==0)
     {
	fprintf(stderr,"\n"
		MAUSEZAHN_VERSION
		"\n"
		"| MPLS header Syntax: -M header[,header[,header[,...]]]\n"
		"| where each header may consist of the following parameters:\n"
		"|\n"
		"|   label ... the MPLS label (mandatory, 0..1048575)\n"
		"|   exp ..... experimental/CoS (default: 0, allowed values: 0..7)\n"
		"|   TTL ..... Time To Live (default: 255)\n"
		"|   BOS ..... marks bottom-of-stack; per default the last (most inner) header\n"
		"|             will have BOS=1. If desired you can set this flag for any header\n"  
		"|             inbetween but this will lead to an invalid packet. Simply use\n"
	        "|             'S' to set BOS=1, or 's' to set BOS=0. Note that this flag MUST be\n"
                "|             the LAST argument.\n"
		"|\n"
		"| Examples:\n"
		"|\n"
		"|  -M 800        .... label=800\n"
		"|  -M 800:6      .... label=800 and CoS=6\n"
		"|  -M 800:6:55   .... label=800, CoS=6, TTL=55\n"
		"|  -M 800:S      .... label=800 and BOS=1\n"
		"|  -M 800:6:s    .... label=800, CoS=6, and BOS=0\n"
		"|\n"
		"|  multiple headers:\n"
		"|\n"
		"|  -m 30,20:7,800:5:128 ... outer label=800 with CoS=5 and TTL=128,\n"
		"|                           middle label=20 with CoS=7,\n"
		"|                           inner label=30 (this one is closest to L3).\n"
		"|\n"
		"|   Valid delimiters inside a header: : - . +\n"
		"|\n"
		"\n");
	exit (0);
     }
   else
     {

	if ( (f1 = strtok (params, ":-.+")) == NULL )
	  {
	     return 1; // error!
	  }
	
	tx.mpls_label = (u_int32_t) str2int (f1);
	if (tx.mpls_label>1048575) 
	  { 
	     tx.mpls_label = 1048575; // 2^20
	     fprintf(stderr," Warning: MPLS label too big! Reduced to maximum allowed value.\n");
	  }
     }

      
   if ( (f2 = strtok (NULL, ":-.+")) != NULL ) // 2nd param set
     {
	if  (strncmp(f2,"S",1)==0) 
	  {
	     tx.mpls_bos = 1;
	     return 0;
	  }
	else if (strncmp(f2,"s",1)==0) 
	  {
	     tx.mpls_bos = 0;
	     return 0;
	  }
	else
	  {
	     tx.mpls_exp = (u_int8_t) str2int (f2);
	     if (tx.mpls_exp > 7)
	       {
		  tx.mpls_exp = 7;
		  fprintf(stderr," Warning: MPLS CoS too big! Reduced to maximum allowed value.\n");
	       }
	  }
	

	if ( (f3 = strtok (NULL, ":-.+")) != NULL ) // 3rd param set
	  {
	     if  (strncmp(f3,"S",1)==0) 
	       {
		  tx.mpls_bos = 1;
		  return 0;
	       }
	     else if (strncmp(f3,"s",1)==0) 
	       {
		  tx.mpls_bos = 0;
		  return 0;
	       }
	     else 
	       {
		  if ((u_int16_t) str2int (f3)>255)
		    {
		       fprintf(stderr," Warning: MPLS TTL too big! Reduced to maximum allowed value.\n");
		       tx.mpls_ttl = 255;
		    }
		  else
		    {
		       tx.mpls_ttl = (u_int8_t) str2int (f3);
		    }
	       }
	     
	     if ( (f4 = strtok (NULL, ":-.+")) != NULL ) // 4th param set
	       {

		  if  (strncmp(f3,"S",1)==0) 
		    {
		       tx.mpls_bos = 1;
		    }
		  else if (strncmp(f3,"s",1)==0) 
		    {
		       tx.mpls_bos = 0;
		    }

	       }
	     
	  }
     }
   
   
   return 0;
}


// Parses str for occurence of character or sequence ch.
// Returns number of occurences 
int exists(char* str, char* ch)
{
   int i,match;
   
   size_t len_str, len_ch;
   
   len_str = strlen(str);
   len_ch = strlen(ch);
   match=0;
   
   for (i=0; i<len_str; i++)
     {
	if (strcmp(str++,ch)==0) match++;
     }

   return match;
}
