#!/usr/bin/perl -w

# The script merge to NagiosGrapher RRD locations
# Copyright (C) 2005  Marius Hein, NETWAYS GmbH, www.netways.de
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
# $Id: rrd_commix.pl 967 2005-11-10 12:19:42Z mhein $

use strict;
use Getopt::Long;
use Pod::Usage;
use File::Basename;
use Data::Dumper;
use XML::Simple;
use File::Copy;

use subs qw/print_help read_xml add_slash read_dumper/;
use vars qw(
    $exec
    $progname
    $which
    $rrdtool
    $opt_help
    $opt_old
    $opt_new
    $opt_out
    
    $opt_cfg_format
    @cfg_formats
    
    $opt_xmlfile
    $xml_old
    $xml_new
    
    $rrd_old
    $rrd_new
    
    $rrd_merge_script
    
    $merge_count
    
    $start_time
    $time
    
    $xs
);

# defaults
$which = '/usr/bin/which';
$rrdtool = `$which rrdtool`;
$rrdtool =~ s/\n//;
$opt_xmlfile = 'index.xml';
$opt_cfg_format = 'dumper:dumper';

$rrd_merge_script = './merged-rrd.py';
$progname = basename($0);

GetOptions ("help"      =>  \$opt_help,
            "h"         =>  \$opt_help,
            "old=s"     =>  \$opt_old,
            "new=s"     =>  \$opt_new,     
            "out=s"     =>  \$opt_out,
            "script=s"  =>  \$rrd_merge_script,
            "format=s"  =>  \$opt_cfg_format)
            || print_help(1, 'Please check your arguments !');

print_help if ($opt_help);

unless ($opt_old && $opt_new && $opt_out && $opt_cfg_format) {
    print_help(1, 'Too few arguments !');
}
else {
    if ($opt_cfg_format =~ m/:/) {
        @cfg_formats = split(/:/, $opt_cfg_format);
    }
    else {
        push @cfg_formats, $opt_cfg_format;
        push @cfg_formats, $opt_cfg_format;
    }
    
    foreach (@cfg_formats) {
        unless ($_ =~ m/dumper|xml/) {
            print_help(1, 'CFG file formats are only \'dumper\' or \'xml\'!', 1);
        }
    }
    
    print_help(1, 'Old file does not exists: '. $!, 1) if (! -e $opt_old);
    print_help(1, 'New file does not exists: '. $!, 1) if (! -e $opt_new);
    print_help(1, 'No merge script found: '. $!, 1) if (! -e $rrd_merge_script);
    
    $start_time = time();
    
    # Loading the 'NEW' XML HASH
    print "Loading the 'new' xml file ... ";
    
    $opt_new = add_slash($opt_new);
    if ($cfg_formats[1] eq 'xml') {
        $xml_new = read_xml($opt_new. 'index.xml');
    }
    elsif ($cfg_formats[1] eq 'dumper') {
        $xml_new = read_dumper($opt_new. 'index.ngraph');
    }
    
    print "done\n";
    
    # Loading the 'OLD' XML HASH
    print "Loading the 'old' xml file ... ";
    
    $opt_old = add_slash($opt_old);
    if ($cfg_formats[0] eq 'xml') {
        $xml_old = read_xml($opt_old. 'index.xml');
    }
    elsif ($cfg_formats[0] eq 'dumper') {
        $xml_old = read_dumper($opt_old. 'index.ngraph');
    }
    
    print "done\n";
    
    # Creating the Out directory
    print "Creating output dir '$opt_out' ... ";
    $opt_out = add_slash($opt_out);
    mkdir($opt_out) unless (-e $opt_out);
    print "done\n";
    
    print "Starting merging the RRD's:\n";
    $merge_count = 0;
    # running through the new hash ...
    while (my ($host_name, $service) = each(%{ $xml_new->{host} })) {
        while (my ($service_name, $service_data) = each(%{ $service->{service} })) {
            if (exists($xml_old->{host}->{$host_name}->{service}->{$service_name})) {
                $rrd_old = $opt_old. $host_name. '/'.
                $xml_old->{host}->{$host_name}->{service}->{$service_name}->{rrd_file};
                
                $rrd_new = $opt_new. $host_name. '/'.
                $service_data->{rrd_file};
                
                print "Creating '$opt_out.$host_name' ... ";
                mkdir ($opt_out.$host_name) unless (-e ($opt_out.$host_name));
                print "done\n";
                
                $exec = $rrd_merge_script. ' '. $rrd_old. ' '. $rrd_new. ' '
                . add_slash($opt_out.$host_name). $service_data->{rrd_file};
                
                print "[" . ($merge_count+1). "] Merge RRD for ($host_name / $service_name) ... ";
                `$exec`;
                print "done\n\n";
                $merge_count += 1;
            }
        }
    }
    print "$merge_count RRD's successfully merged !\n";
    print "Copy the 'new' index file to the out directory ... ";
    if ($cfg_formats[1] eq 'xml') {
        copy ($opt_new. 'index.xml', $opt_out. 'index.xml');
    }
    elsif ($cfg_formats[1] eq 'dumper') {
        copy ($opt_new. 'index.ngraph', $opt_out. 'index.ngraph');
    }
    print "done\n";
    
    $time = time() - $start_time;
    $time = sprintf("The operation took %.2f minutes.\n", $time / 60);
    print $time;
    
    print "\n";
}

exit (0);

sub print_help {
    my ($level, $message, $exitval) = @_;
    
    $level = 2 if (!defined $level);
    $exitval = 1 if (!defined $exitval);
    
    pod2usage( {-message => $message,
                -exitval => $exitval,
                -verbose => $level} );
}

sub add_slash {
    my ($dir) = @_;
    unless ($dir =~ m/\/$/) {
        $dir .= '/';
    }
    return $dir;
}

sub read_dumper {
    my ($file) = @_;
    if (-e $file) {
        my $dump = 'my ';
        open (CFG, "<$file");
        while (<CFG>) {
            $dump .= $_;
        }
        close (CFG);
        
        my $ref_data = eval($dump);
        return $ref_data;
    }
    else {
        return;
    }
    
    return;
}

sub read_xml {
    my ($file) = @_;
    my $xs = XML::Simple->new(XMLDecl       =>  1,
                              RootName      =>  'main',
                              KeyAttr       =>  'name',
                              ForceArray    =>  1);
    if (-e $file) {
        my $ref_xml = $xs->XMLin($file);
        return $ref_xml;
    }
    else {
        return;
    }
    
    return
}

# SCRIPT END
__END__

=head1 NAME

rrd_commix.pl - merges rrdfiles single and recursive

It only works with structures created by the NagiosGrapher. The Script will need
the index.xml file for the rrd file links.

=head1 SYNOPSIS

rrd_comix.pl --old=<DIR> --new=<DIR> --out=<DIR> [options]

=head1 OPTIONS

=over 8

=item B<--help>

Display's a helpscreen.

=item B<--old>

The NagiosGrapher directory with the old rrd databases.

=item B<--new>

The Nagiosgrapher directory with the new rrd databases.

=item B<--out>

The output directory where the merged files are located after the operation.

=item B<--script>

The location of the merge script if its located on a different place.

=item B<--format>

The format for the index file. Possible values are 'xml' or 'dumper'.
You can mix this values with --format=[old]:[new] or --format=[both]

=back

=head1 AUTHOR

rrd_commix.pl is written by Marius Hein <mhein@netways.de>

Copyright 2005, NETWAYS GmbH <http://www.netways.de>

Published under the GPL v2 license

=head1 VERSION

This script is in alpha state, testin only, v1.0.

SVN ID: $Id: rrd_commix.pl 967 2005-11-10 12:19:42Z mhein $

=cut