--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with System.Storage_Elements;
with Ada.Characters.Handling;
with Ada.Strings.Fixed;
with Ada.Streams;
with GNAT.OS_Lib;
with GNAT.Sockets;
with GNAT.Directory_Operations;
with Interfaces;
with Interfaces.C.Strings;

with Options;

with McKae.XML.EZ_Out.String_Stream;
with Events_Receivers;

with Narval.Communication.Handling;
with Low_Level_Network.Utils;
with Narval.Configurator.Abstract_Actors_Coordination;
with Narval.Local_Configuration;
with Narval.Narval_Naming_Registry;

with Socket_Send;

package body Narval.Actors.Actives is

   --  local parameters handling
   use Interfaces.C.Strings;
   --  definition used for exporting code to C that need access to actor stuff
   Actor_Handle : Active_Actor_Access;

   procedure Subscribe_Parameter (Active_Actor : access Active_Actor_Type);
   procedure Ada_Add_New_Parameter (C_Item : Interfaces.C.Strings.chars_ptr;
                                    Error_Code : in out Integer);
   pragma Export (C, Ada_Add_New_Parameter, "ada_add_new_parameter");
   procedure Ada_Set_Parameter
     (Parameter_Name : Interfaces.C.Strings.chars_ptr;
      Parameter_Value : Interfaces.C.Strings.chars_ptr;
      Error_Code : in out Integer);
   pragma Export (C, Ada_Set_Parameter, "ada_set_parameter");
   procedure Ada_Get_Parameter
     (Parameter_Name : Interfaces.C.Strings.chars_ptr;
      Buffer : System.Address;
      Buffer_Length : Integer;
      Error_Code : in out Integer);
   pragma Export (C, Ada_Get_Parameter, "ada_get_parameter");

   procedure Ada_Log_Message (Id : Integer;
                              Message : Interfaces.C.Strings.chars_ptr;
                              Level : Integer);
   pragma Export (C, Ada_Log_Message, "ada_log_message");

   use Log4ada.Loggers;
   use Parameters.Parameter_Vector_Package;
   use Ada.Strings.Unbounded;

   procedure Trig_Event (Active_Actor : access Active_Actor_Type;
                         Event : String) is
      Event_Receiver : Events_Receivers.Events_Receivers_Class_Access;
   begin
      if Event = "log_reload" then
         begin
            Event_Receiver := Narval_Naming_Registry.Get_Event_Receiver;
         exception
            when Narval_Naming_Registry.No_Event_Receiver =>
               Log4ada.Appenders.Annex_E.Disable
                 (Active_Actor.Remote_Appender'Access);
               Log4ada.Loggers.Warn_Out (Active_Actor.Logger'Access,
                                         "log_reload asked but " &
                                         "no remote logging");
               return;
         end;
         Log4ada.Appenders.Annex_E.Set_Receiver
           (Active_Actor.Remote_Appender, Event_Receiver);
         Log4ada.Appenders.Annex_E.Enable
           (Active_Actor.Remote_Appender'Access);
      end if;
   end Trig_Event;

   procedure Parameter_Changed (Active_Actor : access Active_Actor_Type;
                                Parameter : Parameters.Parameter_Access) is
      Parameter_Low_Case : constant String :=
        Ada.Characters.Handling.To_Lower (To_String (Parameter.Name));
   begin
      if Parameter_Low_Case = P_LOG_LEVEL then
         Set_Level (Active_Actor.Logger'Access,
                    Parameter.Level_Value);
      elsif Parameter_Low_Case = P_WATCHER_RUNNING then
         if Parameter.Boolean_Value then
            select
               Active_Actor.Buffer_Dispatcher.Start;
            or
               delay 0.1;
               Log4ada.Loggers.Warn_Out
                 (Active_Actor.Logger'Access,
                  "attempt to start already running task");
            end select;
         else
            select
               Active_Actor.Buffer_Dispatcher.Stop;
            or
               delay 0.1;
               Log4ada.Loggers.Warn_Out
                 (Active_Actor.Logger'Access,
                  "attempt to stop already stopped task");
            end select;
         end if;
      elsif Parameter_Low_Case = P_WATCHER_PORT then
         Active_Actor.Duplication_Port :=
           Positive (Parameter.Unsigned_16_Value);
      elsif Parameter_Low_Case = P_WATCHER_OUTPUT_NUMBER then
         Active_Actor.Duplication_Output :=
           Positive (Parameter.Unsigned_16_Value);
      elsif Parameter_Low_Case = P_WATCHER_DELAY then
         Active_Actor.Watcher_Delay := Duration (Parameter.Float_Value);
      end if;
   end Parameter_Changed;

   procedure Set (Active_Actor : access Active_Actor_Type;
                  Parameter : String;
                  Value : String) is
      Parameter_Low_Case : constant String :=
        Ada.Characters.Handling.To_Lower (Parameter);
      Parameter_Type : Parameters.Parameter_Access;
   begin
      if Parameter_Low_Case = P_NAME then
         Ada.Exceptions.Raise_Exception
           (Parameters.Read_Only_Parameter'Identity,
            "parameter name is read only");
      elsif Parameter_Low_Case = P_CURRENT_DIRECTORY then
         GNAT.Directory_Operations.Change_Dir (Value);
         return;
      elsif Parameter_Low_Case = P_FILES then
         Ada.Exceptions.Raise_Exception
           (Parameters.Read_Only_Parameter'Identity,
            "parameter " & P_FILES & " is read only");
      elsif Parameter_Low_Case = P_BYTES_IN then
         Ada.Exceptions.Raise_Exception
           (Parameters.Read_Only_Parameter'Identity,
            "parameter " & P_BYTES_IN & " is read only");
      elsif Parameter_Low_Case = P_BYTES_OUT then
         Ada.Exceptions.Raise_Exception
           (Parameters.Read_Only_Parameter'Identity,
            "parameter " & P_BYTES_OUT & " is read only");
      elsif Parameter_Low_Case = P_CLIENTS then
         if Active_Actor.Outputs = null then
            null;
         else
            Ada.Exceptions.Raise_Exception
              (Parameters.Read_Only_Parameter'Identity,
               "parameter " & P_CLIENTS & " is read only");
         end if;
      elsif Parameter_Low_Case = P_SERVERS then
         if Active_Actor.Inputs = null then
            null;
         else
            Ada.Exceptions.Raise_Exception
              (Parameters.Read_Only_Parameter'Identity,
               "parameter " & P_SERVERS & " is read only");
         end if;
      elsif Parameter_Low_Case = P_PRELOAD_LIBRARY then
         declare
            Handle : Shared_Library.Handle_Type;
            pragma Warnings (Off, Handle);
         begin
            Handle := Shared_Library.Open_Library (Value,
                                                   Shared_Library.RTLD_NOW +
                                                   Shared_Library.RTLD_GLOBAL);
            Library_Vector_Package.Append
              (Active_Actor.Libraries,
               (Name => To_Unbounded_String (Value),
                Handle => Handle));
         exception
            when E : Shared_Library.Library_Loading_Failed =>
               Error_Out (Active_Actor.Logger'Access,
                          P_PRELOAD_LIBRARY & ":" &
                          Shared_Library.Library_Error,
                          E);
               raise;
         end;
         return;
      elsif Parameter_Low_Case = P_REMOVE_LIBRARIES then
         declare
            Number_Of_Libraries_To_Remove : constant Natural :=
              Natural'Value (Value);
            Index : Natural := Number_Of_Libraries_To_Remove;
            Last : Library_Full_Description_Type;
            Close_Return : Integer;
         begin
            loop
               exit when Index = 0;
               exit when Library_Vector_Package.Is_Empty
                 (Active_Actor.Libraries);
               Last := Library_Vector_Package.Last_Element
                 (Active_Actor.Libraries);
               Library_Vector_Package.Delete_Last (Active_Actor.Libraries);
               Close_Return := Shared_Library.Close_Library (Last.Handle);
               if Close_Return /= 0 then
                  Warn_Out (Active_Actor.Logger'Access,
                            "closing library " & To_String (Last.Name) &
                            " failed");
               end if;
               Index := Index - 1;
            end loop;
         end;
         return;
      elsif Parameter_Low_Case = P_LIBRARIES then
         Ada.Exceptions.Raise_Exception
           (Parameters.Read_Only_Parameter'Identity,
            "parameter " & P_LIBRARIES & " is read only");
      elsif Parameter_Low_Case = P_RUN_NUMBER then
         Ada.Exceptions.Raise_Exception
           (Parameters.Read_Only_Parameter'Identity,
            "parameter " & P_SERVERS & " is read only");
      else
         Parameter_Type := Parameters.Find_Parameter
           (Active_Actor.Parameters_List, Parameter_Low_Case);
         Parameters.Set (Parameter_Type.all, Value);
         Parameter_Changed (Active_Actor_Class_Access (Active_Actor),
                            Parameter_Type);
         return;
      end if;
      Error_Out (Active_Actor.Logger'Access,
                 Parameter_Low_Case & " is unknown");
      raise Unknown_Parameter;
   exception
      when Parameters.Parameter_Not_Found =>
         Error_Out (Active_Actor.Logger'Access,
                    Parameter_Low_Case & " is unknown");
         raise Unknown_Parameter;
   end Set;

   function Get_Image (Active_Actor : access Active_Actor_Type;
                       Parameter_Name : String) return String is
   begin
      declare
         Parameter_Low_Case : constant String :=
           Ada.Characters.Handling.To_Lower (Parameter_Name);
         use Narval.Communication;
         Parameter : Parameters.Parameter_Access;
      begin
         if Parameter_Low_Case = P_NAME then
            return To_String (Active_Actor.Name);
         elsif Parameter_Low_Case = P_CURRENT_DIRECTORY then
            return GNAT.Directory_Operations.Get_Current_Dir;
         elsif Parameter_Low_Case = P_FILES then
            raise Deprecated_Code;
         elsif Parameter_Low_Case = P_BYTES_IN then
            declare
               Res : Communication.Bytes_Count_Type := 0;
            begin
               if Active_Actor.Inputs /= null then
                  for I in Active_Actor.Inputs'Range loop
                     Res := Res +
                       Active_Actor.Inputs (I).Link.all.Bytes_In_Count;
                  end loop;
               else
                  raise Unknown_Parameter;
               end if;
               return Res'Img;
            end;
         elsif Parameter_Low_Case = P_BYTES_OUT then
            declare
               Res : Communication.Bytes_Count_Type := 0;
            begin
               if Active_Actor.Outputs /= null then
                  for I in Active_Actor.Outputs'Range loop
                     for J in Active_Actor.Outputs (I).Links'Range loop
                        Res := Res +
                          Active_Actor.Outputs
                          (I).Links (J).Bytes_Out_Count;
                     end loop;
                  end loop;
               else
                  raise Unknown_Parameter;
               end if;
               return Res'Img;
            end;
         elsif Parameter_Low_Case = P_CLIENTS then
            if Active_Actor.Outputs = null then
               raise Unknown_Parameter;
            end if;
            declare
               Function_Return : Unbounded_String := Null_Unbounded_String;
               Number_Of_Client : Natural := 0;
               First_Output : Positive;
               First_Link : Positive;
            begin
               for I in Active_Actor.Outputs'Range loop
                  for J in Active_Actor.Outputs (I).Links'Range loop
                     Number_Of_Client := Number_Of_Client + 1;
                  end loop;
               end loop;
               First_Output := Active_Actor.Outputs'First;
               First_Link := Active_Actor.Outputs
                 (First_Output).Links'First;
               if Number_Of_Client = 1 then
                  return Communication.Asker_Link
                    (Active_Actor.Outputs
                     (First_Output).Links (First_Link));
               else
                  for I in Active_Actor.Outputs'Range loop
                     for J in Active_Actor.Outputs (I).Links'Range loop
                        if I = First_Output and J = First_Link then
                           null;
                        else
                           Function_Return := Function_Return & ",";
                        end if;
                        Function_Return := Function_Return &
                          Communication.Asker_Link
                          (Active_Actor.Outputs (I).Links (J));
                     end loop;
                  end loop;
                  return To_String (Function_Return);
               end if;
            end;
         elsif Parameter_Low_Case = P_SERVERS then
            if Active_Actor.Inputs = null then
               raise Unknown_Parameter;
            end if;
            declare
               Function_Return : Unbounded_String := Null_Unbounded_String;
            begin
               if Active_Actor.Inputs'Length = 1 then
                  return Communication.Provider_Link
                    (Active_Actor.Inputs
                     (Active_Actor.Inputs'First).Link);
               end if;
               for I in Active_Actor.Inputs'Range loop
                  Function_Return := Function_Return &
                    Communication.Provider_Link
                    (Active_Actor.Inputs (I).Link);
                  if I /= Active_Actor.Inputs'Last then
                     Function_Return := Function_Return & ",";
                  end if;
               end loop;
               return To_String (Function_Return);
            end;
         elsif Parameter_Low_Case = P_LIBRARIES then
            declare
               Cursor : Library_Vector_Package.Cursor;
               use type Library_Vector_Package.Cursor;
               Function_Return : Unbounded_String := Null_Unbounded_String;
            begin
               Cursor := Library_Vector_Package.First (Active_Actor.Libraries);
               while Cursor /= Library_Vector_Package.No_Element loop
                  if Function_Return /= Null_Unbounded_String then
                     Function_Return := Function_Return & ",";
                  end if;
                  Function_Return := Function_Return &
                    Library_Vector_Package.Element (Cursor).Name;
                  Cursor := Library_Vector_Package.Next (Cursor);
               end loop;
               return To_String (Function_Return);
            end;
         elsif Parameter_Low_Case = P_RUN_NUMBER then
            return Configurator.Abstract_Actors_Coordination.Get_Image
              (Local_Configuration.Config_Local, "run_number");
         end if;
         Parameter := Parameters.Find_Parameter (Active_Actor.Parameters_List,
                                                 Parameter_Low_Case);
         return Parameters.Image (Parameter.all);
      end;
   exception
      when Parameters.Parameter_Not_Found =>
         raise Unknown_Parameter;
      when E : others =>
         Error_Out (Active_Actor.Logger'Access,
                    "get image",
                    E);
         raise;
   end Get_Image;

   procedure Add_Value
     (Active_Actor : access Active_Actor_Type;
      Parameter : String;
      Xml_Buffer : in out McKae.XML.EZ_Out.String_Stream.String_Buffer;
      Write_Only_Parameter : Boolean := False);

   procedure Add_Value
     (Active_Actor : access Active_Actor_Type;
      Parameter : String;
      Xml_Buffer : in out McKae.XML.EZ_Out.String_Stream.String_Buffer;
      Write_Only_Parameter : Boolean := False) is
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
      use McKae.XML.EZ_Out.String_Stream.XML_String_Buffer;
      use Narval.Communication;
      Res : Communication.Bytes_Count_Type := 0;
      Parameter_Type : Parameters.Parameter_Access;
      use Parameters;
   begin
      if Parameter = P_NAME then
         Start_Element (Xml_Buffer, "data",
                        ("mode" = Read_Only'Img,
                         "monitor" = Never'Img,
                         "type" = "string",
                         "name" = P_NAME));
         Output_Element (Xml_Buffer, "value", To_String (Active_Actor.Name));
      elsif Parameter = P_CURRENT_DIRECTORY then
         Start_Element (Xml_Buffer, "data",
                        ("mode" = Read_Write'Img,
                         "monitor" = Never'Img,
                         "type" = "string",
                         "name" = P_CURRENT_DIRECTORY));
         Output_Element (Xml_Buffer, "value",
                         GNAT.Directory_Operations.Get_Current_Dir);
      elsif Parameter = P_FILES then
         declare
            Directory : GNAT.Directory_Operations.Dir_Type;
            use GNAT.Directory_Operations;
            Output_String : String (1 .. 1024);
            Last : Natural;
            Is_Directory : Boolean;
            Is_Regular_File : Boolean;
         begin
            Start_Element (Xml_Buffer, "data",
                           ("mode" = Read_Only'Img,
                            "monitor" = Never'Img,
                            "type" = "string",
                            "name" = P_FILES));
            Open (Directory, Get_Current_Dir);
            loop
               Read (Directory, Output_String, Last);
               exit when Last = 0;
               Is_Directory := GNAT.OS_Lib.Is_Directory
                 (Output_String (1 .. Last));
               Is_Regular_File := GNAT.OS_Lib.Is_Regular_File
                 (Output_String (1 .. Last));
               Output_Element (Xml_Buffer, "value",
                               Output_String (1 .. Last),
                               ("directory" = Is_Directory'Img,
                                "regular_file" = Is_Regular_File'Img));
            end loop;
            Close (Directory);
         end;
      elsif Parameter = P_BYTES_IN then
         if Active_Actor.Inputs /= null then
            Start_Element (Xml_Buffer, "data",
                           ("mode" = Read_Only'Img,
                            "monitor" = Request'Img,
                            "type" = "constant unsigned_integer",
                            "size" = "64",
                            "name" = P_BYTES_IN));
            for I in Active_Actor.Inputs'Range loop
               Res := Res +
                 Active_Actor.Inputs (I).Link.all.Bytes_In_Count;
            end loop;
         else
            raise Unknown_Parameter;
         end if;
         Output_Element (Xml_Buffer, "value", Res'Img);
      elsif Parameter = P_BYTES_OUT then
         if Active_Actor.Outputs /= null then
            Start_Element (Xml_Buffer, "data",
                           ("mode" = Read_Only'Img,
                            "monitor" = Request'Img,
                            "type" = "constant unsigned_integer",
                            "size" = "64",
                            "name" = P_BYTES_OUT));
            for I in Active_Actor.Outputs'Range loop
               for J in Active_Actor.Outputs (I).Links'Range loop
                  Res := Res +
                    Active_Actor.Outputs
                    (I).Links (J).Bytes_Out_Count;
               end loop;
            end loop;
         else
            raise Unknown_Parameter;
         end if;
         Output_Element (Xml_Buffer, "value", Res'Img);
      elsif Parameter = P_CLIENTS then
         if Active_Actor.Outputs = null then
            raise Unknown_Parameter;
         end if;
         Start_Element (Xml_Buffer, "data",
                        ("mode" = Read_Only'Img,
                         "monitor" = Never'Img,
                         "type" = "constant string",
                         "name" = P_CLIENTS));
         for I in Active_Actor.Outputs'Range loop
            for J in Active_Actor.Outputs (I).Links'Range loop
               Output_Element (Xml_Buffer, "value",
                               Communication.Asker_Link
                               (Active_Actor.Outputs (I).Links (J)));
            end loop;
         end loop;
      elsif Parameter = P_SERVERS then
         if Active_Actor.Inputs = null then
            raise Unknown_Parameter;
         end if;
         Start_Element (Xml_Buffer, "data",
                        ("mode" = Read_Only'Img,
                         "monitor" = Never'Img,
                         "type" = "constant string",
                         "name" = P_SERVERS));
         for I in Active_Actor.Inputs'Range loop
            Output_Element (Xml_Buffer, "value",
                            Communication.Provider_Link
                            (Active_Actor.Inputs (I).Link));
         end loop;
      elsif Parameter = P_LIBRARIES then
         declare
            Cursor : Library_Vector_Package.Cursor;
            use type Library_Vector_Package.Cursor;
            Length : constant Ada.Containers.Count_Type :=
              Library_Vector_Package.Length (Active_Actor.Libraries);
            use type Ada.Containers.Count_Type;
         begin
            Start_Element (Xml_Buffer, "data",
                        ("mode" = Read_Only'Img,
                         "monitor" = Never'Img,
                         "type" = "string",
                         "libraries_number" =
                         Ada.Containers.Count_Type'Image (Length),
                         "name" = P_LIBRARIES));
            if Length = 0 then
               Output_Element (Xml_Buffer, "value", "");
            else
               Cursor := Library_Vector_Package.First (Active_Actor.Libraries);
               while Cursor /= Library_Vector_Package.No_Element loop
                  Output_Element
                    (Xml_Buffer, "value",
                     To_String (Library_Vector_Package.Element (Cursor).Name));
                  Cursor := Library_Vector_Package.Next (Cursor);
               end loop;
            end if;
         end;
      elsif Parameter = P_RUN_NUMBER then
         Start_Element (Xml_Buffer, "data",
                        ("mode" = Read_Only'Img,
                         "monitor" = Never'Img,
                         "type" = "integer",
                         "size" = "32",
                         "name" = P_RUN_NUMBER));
         Output_Element (Xml_Buffer, "value",
                         Configurator.Abstract_Actors_Coordination.Get_Image
                         (Local_Configuration.Config_Local, "run_number"));
      else
         Parameter_Type := Parameters.Find_Parameter
           (Active_Actor.Parameters_List, Parameter);
         Start_Element (Xml_Buffer, "data",
                        Get_Attributes (Parameter_Type.all) +
                        ("name" = Parameter_Type.Name));
         if Write_Only_Parameter then
            Output_Element (Xml_Buffer, "value", "");
         else
            Parameters.To_Xml (Parameter_Type.all, Xml_Buffer);
         end if;
      end if;
      End_Element (Xml_Buffer, "data");
   exception
      when Parameters.Parameter_Not_Found =>
         raise Unknown_Parameter;
   end Add_Value;

   function Get_Xml (Active_Actor : access Active_Actor_Type;
                     Parameter : String) return String is
      Parameter_Low_Case : constant String :=
        Ada.Characters.Handling.To_Lower (Parameter);
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
      use McKae.XML.EZ_Out.String_Stream.XML_String_Buffer;
      Xml_Buffer : String_Buffer;
      Configuration_Name : constant String :=
        Options.Get_Option ("config_name");
   begin
      Current_Format := McKae.XML.EZ_Out.Continuous_Stream;
      Start_Element (Xml_Buffer, "result",
                     ("cmd" = "get",
                      "status" = "OK",
                      "sub_system_name" = Configuration_Name,
                      "actor_name" = To_String (Active_Actor.Name)
                     ));
      Add_Value (Active_Actor, Parameter_Low_Case, Xml_Buffer);
      End_Element (Xml_Buffer, "result");
      declare
         String_To_Return : constant String := Get_String (Xml_Buffer);
      begin
         Full_Clear (Xml_Buffer);
         return String_To_Return;
      end;
   end Get_Xml;

   function Arguments (Active_Actor : access Active_Actor_Type;
                       Xml : Boolean := True)
                      return String is
      Parameter_Type : Parameters.Parameter_Access;
      Index : Natural;
      List_Length : Natural;
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
      use McKae.XML.EZ_Out.String_Stream.XML_String_Buffer;
      Xml_Buffer : String_Buffer;
      Configuration_Name : constant String :=
        Options.Get_Option ("config_name");
   begin
      if not Xml then
         raise Deprecated_Code;
      end if;
      Start_Element (Xml_Buffer, "result",
                     ("cmd" = "get", "status" = "OK",
                      "sub_system_name" = Configuration_Name,
                      "actor_name" = To_String (Active_Actor.Name)));
      Add_Value (Active_Actor, P_NAME, Xml_Buffer);
      Add_Value (Active_Actor, P_CURRENT_DIRECTORY, Xml_Buffer);
      Add_Value (Active_Actor, P_FILES, Xml_Buffer);
      if Active_Actor.Outputs /= null then
         Add_Value (Active_Actor, P_CLIENTS, Xml_Buffer);
      end if;
      if Active_Actor.Inputs /= null then
         Add_Value (Active_Actor, P_SERVERS, Xml_Buffer);
      end if;
      Add_Value (Active_Actor, P_LIBRARIES, Xml_Buffer);
      Index := First_Index (Active_Actor.Parameters_List);
      List_Length := Natural (Length (Active_Actor.Parameters_List));
      for I in Index .. Index + List_Length - 1 loop
         Parameter_Type := Element (Active_Actor.Parameters_List, I);
         Add_Value (Active_Actor,
                    To_String (Parameter_Type.Name),
                    Xml_Buffer,
                    Write_Only_Parameter =>
                      Parameter_Type.Mode = Write_Only);
      end loop;
      if Active_Actor.Inputs /= null then
         Add_Value (Active_Actor, P_BYTES_IN, Xml_Buffer);
      end if;
      if Active_Actor.Outputs /= null then
         Add_Value (Active_Actor, P_BYTES_OUT, Xml_Buffer);
      end if;
      Add_Value (Active_Actor, P_RUN_NUMBER, Xml_Buffer);
      End_Element (Xml_Buffer, "result");
      declare
         String_To_Return : constant String := Get_String (Xml_Buffer);
      begin
         Full_Clear (Xml_Buffer);
         return String_To_Return;
      end;
   end Arguments;

   ----------------
   -- Initialise --
   ----------------

   procedure Initialise (Active_Actor : access Active_Actor_Type;
                         Actor_Name : String) is
      Logger_Name : constant String := "log from " &
        Ada.Strings.Unbounded.To_String
        (Local_Configuration.Configuration_Name) &
        "." & Actor_Name;
      Event_Receiver : Events_Receivers.Events_Receivers_Class_Access;
      Parameter : Parameters.Parameter_Access;
      use Parameters;
      Informations : constant String :=
        Configurator.Abstract_Actors_Coordination.Get_Informations
        (Local_Configuration.Config_Local, Actor_Name);
      use Configurator.Abstract_Actors_Coordination;
      Description : Configurator.Actors_Description.Actor_Description_Type
        renames Active_Actor.Description;
   begin
      Active_Actor.Name := To_Unbounded_String (Actor_Name);
      Configurator.Actors_Description.Fill (Active_Actor.Description,
                                            Informations);
      Configurator.Actors_Description.Display (Active_Actor.Description);
      Subscribe_Parameter (Active_Actor);
      Parameter := new Parameter_Type'(Container_Kind => String_Type,
                                       Mode => Read_Only,
                                       Monitor => Never,
                                       Name => To_Unbounded_String (P_KIND),
                                       Run_Parameter => False,
                                       Editor => None,
                                       String_Value => To_Unbounded_String
                                         (Active_Actor.Kind'Img));
      Parameter_Vector_Package.Append (Active_Actor.Parameters_List,
                                       Parameter);
      Parameter := new Parameter_Type'(Container_Kind => Boolean_Type,
                                       Mode => Read_Write,
                                       Monitor => Never,
                                       Name => To_Unbounded_String (P_ACTIVE),
                                       Run_Parameter => False,
                                       Editor => None,
                                       Boolean_Value => True);
      Parameter_Vector_Package.Append (Active_Actor.Parameters_List,
                                       Parameter);
      Parameter := new Parameter_Type'(Container_Kind => Unsigned_32_Type,
                                       Mode => Read_Write,
                                       Monitor => Never,
                                       Name => To_Unbounded_String
                                         (P_PORT),
                                       Run_Parameter => False,
                                       Editor => None,
                                       Unsigned_32_Value => 7080);
      Parameter_Vector_Package.Append (Active_Actor.Parameters_List,
                                       Parameter);
      Parameter := new Parameter_Type'(Container_Kind => Log_Level_Type,
                                       Mode => Read_Write,
                                       Monitor => Never,
                                       Name => To_Unbounded_String
                                         (P_LOG_LEVEL),
                                       Run_Parameter => False,
                                       Editor => None,
                                       Level_Value => Description.Log_Level);
      Parameter_Vector_Package.Append (Active_Actor.Parameters_List,
                                       Parameter);
      Parameter := new Parameter_Type'(Container_Kind => String_Type,
                                       Mode => Read_Only,
                                       Monitor => Never,
                                       Name => To_Unbounded_String
                                         (P_HOST_NAME),
                                       Run_Parameter => False,
                                       Editor => None,
                                       String_Value => To_Unbounded_String
                                         (GNAT.Sockets.Host_Name));
      Parameter_Vector_Package.Append (Active_Actor.Parameters_List,
                                       Parameter);
      Parameter := new Parameter_Type'(Container_Kind => Natural_Type,
                                       Mode => Read_Only,
                                       Monitor => Never,
                                       Name => To_Unbounded_String (P_RANK),
                                       Run_Parameter => False,
                                       Editor => None,
                                       Natural_Value =>
                                         Natural (Description.Rank_Number));
      Parameter_Vector_Package.Append (Active_Actor.Parameters_List,
                                       Parameter);
      Parameter := new Parameter_Type'(Container_Kind => String_Type,
                                       Name => To_Unbounded_String
                                         (P_PRELOAD_LIBRARY),
                                       Run_Parameter => False,
                                       Editor => None,
                                       Mode => Write_Only,
                                       Monitor => Never,
                                       String_Value =>
                                         Null_Unbounded_String);
      Parameter_Vector_Package.Append (Active_Actor.Parameters_List,
                                       Parameter);
      Parameter := new Parameter_Type'(Container_Kind => Unsigned_16_Type,
                                       Name => To_Unbounded_String
                                         (P_REMOVE_LIBRARIES),
                                       Run_Parameter => False,
                                       Editor => None,
                                       Mode => Write_Only,
                                       Monitor => Never,
                                       Unsigned_16_Value => 0);
      Parameter_Vector_Package.Append (Active_Actor.Parameters_List,
                                       Parameter);
      --  parameter for watcher system
      Parameter := new Parameter_Type'(Container_Kind => Unsigned_16_Type,
                                       Mode => Read_Write,
                                       Monitor => Never,
                                       Name =>
                                         To_Unbounded_String (P_WATCHER_PORT),
                                       Run_Parameter => False,
                                       Editor => None,
                                       Unsigned_16_Value =>
                                         Interfaces.Unsigned_16
                                         (Active_Actor.Duplication_Port));
      Parameter_Vector_Package.Append (Active_Actor.Parameters_List,
                                       Parameter);
      Parameter := new Parameter_Type'(Container_Kind => Boolean_Type,
                                       Mode => Read_Write,
                                       Monitor => Never,
                                       Name => To_Unbounded_String
                                         (P_WATCHER_RUNNING),
                                       Run_Parameter => False,
                                       Editor => None,
                                       Boolean_Value => False);
      Parameter_Vector_Package.Append (Active_Actor.Parameters_List,
                                       Parameter);
      Parameter := new Parameter_Type'(Container_Kind => Float_Type,
                                       Mode => Read_Write,
                                       Monitor => Never,
                                       Name => To_Unbounded_String
                                         (P_WATCHER_DELAY),
                                       Run_Parameter => False,
                                       Editor => None,
                                       Float_Value => 0.1);
      Parameter_Vector_Package.Append (Active_Actor.Parameters_List,
                                       Parameter);
      Parameter := new Parameter_Type'(Container_Kind => Unsigned_16_Type,
                                       Mode => Read_Write,
                                       Monitor => Never,
                                       Name => To_Unbounded_String
                                         (P_WATCHER_OUTPUT_NUMBER),
                                       Run_Parameter => False,
                                       Editor => None,
                                       Unsigned_16_Value => 1);
      Parameter_Vector_Package.Append (Active_Actor.Parameters_List,
                                       Parameter);
      --  end of watcher parameters
      Set_Name (Active_Actor.Logger'Access, Logger_Name);
      Set_Level (Active_Actor.Logger'Access, Description.Log_Level);
      Add_Appender (Active_Actor.Logger'Access,
                    Active_Actor.Console'Access);
      Add_Appender (Active_Actor.Logger'Access,
                    Active_Actor.Remote_Appender'Access);
      begin
         Event_Receiver := Narval_Naming_Registry.Get_Event_Receiver;
         Log4ada.Appenders.Annex_E.Set_Receiver
           (Active_Actor.Remote_Appender, Event_Receiver);
         Log4ada.Loggers.Info_Out (Active_Actor.Logger'Access,
                                   "remote logging on");
      exception
         when Narval_Naming_Registry.No_Event_Receiver =>
            Log4ada.Appenders.Annex_E.Disable
              (Active_Actor.Remote_Appender'Access);
            Log4ada.Loggers.Warn_Out (Active_Actor.Logger'Access,
                                      "no remote logging");
      end;
      declare
         use Configurator.Actors_Description;
         Cursor : String_Vector.Cursor;
         use type String_Vector.Cursor;
      begin
         Cursor := Description.Arguments.First;
         loop
            exit when Cursor = String_Vector.No_Element;
            declare
               Argument : constant String :=
                 To_String (String_Vector.Element (Cursor));
            begin
               if Argument'Length > 13 then
                  if Argument (1 .. 13) = "new_parameter" then
                     begin
                        Parameter :=
                          New_Parameter (Argument (15 .. Argument'Last));
                        Parameter_Vector_Package.Append
                          (Active_Actor.Parameters_List, Parameter);
                     exception
                        when Parameter_Bad_Format =>
                           Log4ada.Loggers.Warn_Out
                             (Active_Actor.Logger'Access,
                              "try to add some parameter from .xml conf file" &
                                " but lacking information");
                        when Constraint_Error =>
                           Log4ada.Loggers.Error_Out
                             (Active_Actor.Logger'Access,
                              "add new parameter failed " &
                                "some of the informations " &
                                "are not well formated " &
                                Argument (15 .. Argument'Last));
                     end;
                  end if;
               else
                  Log4ada.Loggers.Info_Out (Active_Actor.Logger'Access,
                                            "argument : " & Argument);
               end if;
               Cursor := String_Vector.Next (Cursor);
            end;
         end loop;
      end;
      Active_Actor.Start_Stop := new Start_Stop_Handling.Synchro_Start_Type;
   end Initialise;

   ------------------
   -- Changer_Etat --
   ------------------

   procedure Change_State
     (Active_Actor : access Active_Actor_Type;
      Order : Action) is
   begin
      case Order is
         when Partial_Reset | Full_Reset =>
            Log4ada.Loggers.Warn_Out
              (Active_Actor.Logger'Access,
               "reset actions shouldn't reach this procedure");
         when Unload | Unload_Unconfigure =>
            On_Unload (Active_Actor_Class_Access (Active_Actor));
         when Start =>
            On_Start (Active_Actor_Class_Access (Active_Actor));
         when Stop =>
            On_Stop (Active_Actor_Class_Access (Active_Actor));
         when Pause =>
            On_Suspend (Active_Actor_Class_Access (Active_Actor));
         when Resume =>
            On_Resume (Active_Actor_Class_Access (Active_Actor));
         when Initialise =>
            On_Initialise (Active_Actor_Class_Access (Active_Actor));
         when Reset_Com =>
            On_Reset_Com (Active_Actor_Class_Access (Active_Actor));
         when Unconfigure | Configure | Load | Configure_Load =>
            raise Invalid_Order;
      end case;
   exception
      when E : others =>
         Log4ada.Loggers.Fatal_Out (Active_Actor.Logger'Access,
                                    "Changer_State", E);
         raise;
   end Change_State;

   --------------------------
   -- Distribution_Travail --
   --------------------------

   procedure Distribute_Work
     (Active_Actor : access Active_Actor_Type) is
   begin
      Buffer_Handling (Active_Actor_Class_Access (Active_Actor));
   exception
      when E : others =>
         Log4ada.Loggers.Fatal_Out (Active_Actor.Logger'Access,
                                    "Distribute_Work", E);
         raise;
   end Distribute_Work;

   -----------------
   -- Sur_Arreter --
   -----------------

   procedure On_Stop (Active_Actor : access Active_Actor_Type) is
   begin
      Active_Actor.Start_Stop.Stop;
      if Active_Actor.Inputs /= null then
         for I in Active_Actor.Inputs'Range loop
            Active_Actor.Inputs (I).Input_Task.Stop;
         end loop;
      end if;
      Active_Actor.Working.Stop;
      if Active_Actor.Outputs /= null then
         for I in Active_Actor.Outputs'Range loop
            Active_Actor.Outputs (I).Output_Task.Stop;
         end loop;
      end if;
   exception
      when E : others =>
         Log4ada.Loggers.Fatal_Out (Active_Actor.Logger'Access,
                                    "On_Stop", E);
         raise;
   end On_Stop;

   -------------------
   -- Sur_Decharger --
   -------------------

   procedure On_Unload (Active_Actor : access Active_Actor_Type) is
      pragma Unreferenced (Active_Actor);
   begin
      null;
   end On_Unload;

   ------------------
   -- Sur_Demarrer --
   ------------------

   procedure On_Start (Active_Actor : access Active_Actor_Type) is
   begin
      Active_Actor.Start_Stop.Start;
      if Active_Actor.Outputs /= null then
         for I in Active_Actor.Outputs'Range loop
            Active_Actor.Outputs (I).Output_Task.Start;
         end loop;
      end if;
      Active_Actor.Working.Start;
      if Active_Actor.Inputs /= null then
         for I in Active_Actor.Inputs'Range loop
            Active_Actor.Inputs (I).Input_Task.Start;
         end loop;
      end if;
   exception
      when E : others =>
         Log4ada.Loggers.Fatal_Out (Active_Actor.Logger'Access,
                                    "On_Start", E);
         raise;
   end On_Start;

   -------------------
   -- Sur_Reprendre --
   -------------------

   procedure On_Resume  (Active_Actor : access Active_Actor_Type) is
      pragma Unreferenced (Active_Actor);
   begin
      null;
   end On_Resume;

   -------------------
   -- Sur_Suspendre --
   -------------------

   procedure On_Suspend  (Active_Actor : access Active_Actor_Type) is
      pragma Unreferenced (Active_Actor);
   begin
      null;
   end On_Suspend;

   procedure On_Initialise (Active_Actor : access Active_Actor_Type) is
      Actor_Name : constant String := To_String (Active_Actor.Name);
      use type System.Bit_Order;
      Endian_To_Send : System.Bit_Order := System.Default_Bit_Order;
      Description : Configurator.Actors_Description.Actor_Description_Type
        renames Active_Actor.Description;
      use type Ada.Containers.Count_Type;
   begin
      Debug_Out (Active_Actor.Logger'Access,
                 "*** enter on_initialise ***");
      if Description.Inputs.Length /= 0 then
         Active_Actor.Inputs := new Input_Interface_Array
           (1 .. Natural (Description.Inputs.Length));
         for I in Active_Actor.Inputs'Range loop
            Initialise (Active_Actor.Inputs (I),
                        Actor_Name,
                        Description.Inputs.Element (I).all,
                        Active_Actor.Start_Stop,
                        Active_Actor.Logger'Access);
         end loop;
      end if;
      Debug_Out (Active_Actor.Logger'Access,
                 "*** on_initialise step 1 ***");
      if Description.Outputs.Length /= 0 then
         Active_Actor.Outputs := new Output_Interface_Array
           (1 .. Natural (Description.Outputs.Length));
         for I in Active_Actor.Outputs'Range loop
            Initialise (Active_Actor.Outputs (I),
                        Description.Outputs.Element (I).all,
                        Active_Actor.Start_Stop,
                        Active_Actor.Logger'Access);
         end loop;
      end if;
      Debug_Out (Active_Actor.Logger'Access,
                 "*** on_initialise step 2 ***");
      if Active_Actor.Inputs /= null then
         for I in Active_Actor.Inputs'Range loop
            Debug_Out (Active_Actor.Logger'Access,
                       "*** on_initialise : input link binding" &
                       I'Img & " ***");
            Connect (Active_Actor.Inputs (I));
            Debug_Out (Active_Actor.Logger'Access,
                       "*** on_initialise step 2,1,1 ***");
            if Active_Actor.Inputs (I).Swapped_Link /=
              Active_Actor.Inputs (Active_Actor.Inputs'First).Swapped_Link
            then
               raise Incoherent_Links;
            end if;
            Debug_Out (Active_Actor.Logger'Access,
                       "*** on_initialise step 2,1,2 ***");
         end loop;
         if (Active_Actor.Reverse_Endian and
             Active_Actor.Inputs
             (Active_Actor.Inputs'First).Swapped_Link = False) or
           (not Active_Actor.Reverse_Endian and
            Active_Actor.Inputs
            (Active_Actor.Inputs'First).Swapped_Link) then
            pragma Warnings (Off);
            if System.Default_Bit_Order = System.High_Order_First then
               Endian_To_Send := System.Low_Order_First;
            else
               Endian_To_Send := System.High_Order_First;
            end if;
            pragma Warnings (On);
         end if;
      else
         if Active_Actor.Reverse_Endian then
            pragma Warnings (Off);
            if System.Default_Bit_Order = System.High_Order_First then
               Endian_To_Send := System.Low_Order_First;
            else
               Endian_To_Send := System.High_Order_First;
            end if;
         end if;
      end if;
      Debug_Out (Active_Actor.Logger'Access,
                 "*** on_initialise step 3 ***");
      if Active_Actor.Outputs /= null then
         for I in Active_Actor.Outputs'Range loop
            Debug_Out (Active_Actor.Logger'Access,
                       "*** on_initialise : output link binding" &
                       I'Img & " ***");
            Connect (Active_Actor.Outputs (I), Endian_To_Send);
         end loop;
      end if;
      Debug_Out (Active_Actor.Logger'Access,
                 "*** on_initialise ***");
   exception
      when E : others =>
         Log4ada.Loggers.Fatal_Out (Active_Actor.Logger'Access,
                                    "on_initialise", E);
         raise;
   end On_Initialise;

   procedure On_Reset_Com (Active_Actor : access Active_Actor_Type) is
   begin
      if Active_Actor.Inputs /= null then
         for I in Active_Actor.Inputs'Range loop
            Communication.Close (Active_Actor.Inputs (I).Link);
            Free (Active_Actor.Inputs (I));
         end loop;
         Free (Active_Actor.Inputs);
         Active_Actor.Inputs := null;
      end if;
      if Active_Actor.Outputs /= null then
         for I in Active_Actor.Outputs'Range loop
            for J in Active_Actor.Outputs (I).Links'Range loop
               Communication.Close (Active_Actor.Outputs (I).Links (J));
            end loop;
            Free (Active_Actor.Outputs (I));
         end loop;
         Free (Active_Actor.Outputs);
         Active_Actor.Outputs := null;
      end if;
   exception
      when E : others =>
         Log4ada.Loggers.Fatal_Out (Active_Actor.Logger'Access,
                                    "On_Reset_Com", E);
         raise;
   end On_Reset_Com;

   procedure Free (Interface_Arg : in out Type_Interface) is
   begin
      Interface_Arg.Memory.Free;
      Protected_Memory.Free (Interface_Arg.Memory);
   end Free;

   procedure Connect (Interface_Arg : in out Input_Interface_Type) is
      Endian : System.Bit_Order;
      use type System.Bit_Order;
   begin
      Communication.Connect (Interface_Arg.Link, Endian);
      if Endian /= System.Default_Bit_Order then
         Interface_Arg.Swapped_Link := True;
      else
         Interface_Arg.Swapped_Link := False;
      end if;
   end Connect;

   procedure Initialise
     (Interface_Arg : in out Input_Interface_Type;
      Asker : String;
      Input : Configurator.Actors_Description.Receiver_Descriptor_Type;
      Start_Stop : Start_Stop_Handling.Synchro_Start_Access;
      Logger : Log4ada.Loggers.Logger_Class_Access) is
   begin
      Interface_Arg.Memory := new Protected_Memory.Buffered_Memory_Type;
      Interface_Arg.Name := Input.Name;
      Interface_Arg.Memory.Initialise
        (System.Storage_Elements.Storage_Count (Input.Size));
      Interface_Arg.Link := Communication.Handling.New_Client_Link
        (Input, Asker, Logger);
      Interface_Arg.Input_Task := new Communication.Input_Task_Type
        (Interface_Arg.Memory, Interface_Arg.Link, Logger);
   end Initialise;

   procedure Free (Interface_Arg : in out Input_Interface_Type) is
   begin
      Communication.Handling.Free (Interface_Arg.Link);
      Communication.Free (Interface_Arg.Input_Task);
      Free (Type_Interface (Interface_Arg));
   end Free;

   procedure Connect (Interface_Arg : in out Output_Interface_Type;
                      Bit_Order_To_Send : System.Bit_Order) is
      Local_Bit_Order_To_Send : System.Bit_Order := Bit_Order_To_Send;
   begin
      for I in Interface_Arg.Links'Range loop
         Communication.Connect (Interface_Arg.Links (I),
                                Local_Bit_Order_To_Send);
      end loop;
   end Connect;

   procedure Initialise
     (Interface_Arg : in out Output_Interface_Type;
      Output : Configurator.Actors_Description.Sender_Descriptor_Type;
      Start_Stop : Start_Stop_Handling.Synchro_Start_Access;
      Logger : Log4ada.Loggers.Logger_Class_Access) is
      Total_Consumers_Number : Actor_Numbering_Type := 0;
      Index_Link : Positive := 1;
   begin
      Interface_Arg.Memory := new Protected_Memory.Buffered_Memory_Type;
      Interface_Arg.Name := Output.Name;
      Interface_Arg.Memory.Initialise
        (System.Storage_Elements.Storage_Count (Output.Size));
      Interface_Arg.Links :=
        Communication.Handling.New_Server_Link (Output, Logger);
      Interface_Arg.Output_Task := new Communication.Output_Task_Type
        (Interface_Arg.Memory, Interface_Arg.Links, Start_Stop, Logger);
   end Initialise;

   procedure Free (Interface_Arg : in out Output_Interface_Type) is
   begin
      for I in Interface_Arg.Links'Range loop
         Communication.Handling.Free (Interface_Arg.Links (I));
      end loop;
      Communication.Free (Interface_Arg.Output_Task);
      Free (Type_Interface (Interface_Arg));
   end Free;

   ---------------------------------------------
   -- Recuperer_Lien_Communication_Disponible --
   ---------------------------------------------

   function Get_Available_Link
     (Active_Actor : access Active_Actor_Type;
      Asker : String;
      Port : String;
      Service_Name : String)
     return Link_Descriptor_Type is
      Port_Inactive : exception;
      Inactive_Service : exception;
      Connection_Number : Natural := 0;
      Service_Number : Natural;
   begin
      if Active_Actor.Outputs = null then
         raise Actor_Not_Ready;
      end if;
      for I in Active_Actor.Outputs'Range loop
         if Active_Actor.Outputs (I).Name = Service_Name then
            Service_Number := I;
            exit;
         end if;
      end loop;
      Connection_Number := Active_Actor.Outputs (Service_Number).Links'Length;
      if Connection_Number = 0 then
         raise Port_Inactive;
      end if;
      if Ada.Characters.Handling.To_Lower (Port) = "fifo" then
         declare
            Function_Return : Link_Descriptor_Type :=
              (Port_For_Link => Fifo,
               Number_Of_Connection => Connection_Number,
               Queue =>  (others => 0));
            Index : Positive := 1;
         begin
            for I in Active_Actor.Outputs
              (Service_Number).Links'Range loop
               if Index = Active_Actor.Outputs
                 (Service_Number).Position_Fifo then
                  Function_Return.Queue (Index) :=
                    Communication.Get_Init_Info
                    (Active_Actor.Outputs (Service_Number).Links (I));
                  Active_Actor.Outputs
                    (Service_Number).Position_Fifo :=
                    Active_Actor.Outputs
                    (Service_Number).Position_Fifo + 1;
                  Communication.Set_Asker
                    (Active_Actor.Outputs (Service_Number).Links (I),
                     Asker);
                  exit;
               end if;
               Index := Index + 1;
            end loop;
            return Function_Return;
         end;
      else
         for I in Active_Actor.Outputs
           (Service_Number).Links'Range loop
            declare
               Temp_Address : String (1 .. 15) :=  (others => ' ');
               Ethernet_Address_To_Return : String :=
                 Low_Level_Network.Utils.Get_Address (Port);
               Function_Return : Link_Descriptor_Type :=
                 --  valeur port lien bidon pb avec compilo
                 (Port_For_Link => Network,
                  Number_Of_Connection => Connection_Number,
                  Link =>  (Address => Temp_Address,
                            Port_Number =>  1,
                            Data_Available => False,
                            Size => Kilo_Bytes));
            begin
               Function_Return.Link.Port_Number :=
                 Communication.Get_Init_Info
                 (Active_Actor.Outputs (Service_Number).Links (I));
               Function_Return.Link.Address
                 (1 .. Ethernet_Address_To_Return'Length) :=
                 Ethernet_Address_To_Return;
               Function_Return.Link.Data_Available := False;
               Function_Return.Link.Size := Bytes_Buffer_Size_Type
                 (Active_Actor.Outputs
                    (Service_Number).Memory.Buffer_Size);
               Communication.Set_Asker
                 (Active_Actor.Outputs (Service_Number).Links (I),
                  Asker);
               return Function_Return;
            end;
         end loop;
      end if;
      raise Port_Inactive;
   end Get_Available_Link;

   -------------------
   -- tache Travail --
   -------------------

   task body Work_Task_Type is
      Empty_Buffer : Boolean;
   begin
      loop
         select
            accept Start;
         or
            terminate;
         end select;
         loop
            Distribute_Work (Active_Actor_Class_Access (Wrapper));
            if Wrapper.Start_Stop.Is_Stopped and
              Wrapper.Can_Be_Stopped then
               Empty_Buffer := True;
               if Wrapper.Inputs /= null then
                  for I in Wrapper.Inputs'Range loop
                     Empty_Buffer := Empty_Buffer and
                       Wrapper.Inputs (I).Memory.Is_Empty;
                  end loop;
               end if;
               if Empty_Buffer then
                  accept Stop;
                  exit;
               end if;
            end if;
         end loop;
      end loop;
   exception
      when E : others =>
         Log4ada.Loggers.Fatal_Out (Wrapper.Logger'Access,
                                    "Work_Task_Type", E);
   end Work_Task_Type;

   procedure Log_Message (Active_Actor : access Active_Actor_Type;
                          Message : String;
                          Level : Log4ada.Level_Type) is
   begin
      Log4ada.Loggers.Logger_Output (Active_Actor.Logger'Access,
                                     Message,
                                     Level);
   end Log_Message;

   procedure Log_Message
     (Active_Actor : access Active_Actor_Type;
      Message : String;
      Level : Log4ada.Level_Type;
      Exception_To_Send : Ada.Exceptions.Exception_Occurrence) is
   begin
      Log4ada.Loggers.Logger_Output (Active_Actor.Logger'Access,
                                     Message,
                                     Level,
                                     Exception_To_Send);
   end Log_Message;

   type Arguments_Array is array (Positive range <>) of
     Ada.Strings.Unbounded.Unbounded_String;
   function Arguments_Comma (Shell_Line : String) return Arguments_Array;
   function Arguments_Comma (Shell_Line : String) return Arguments_Array is
      First_Comma_Position : constant Natural :=
        Ada.Strings.Fixed.Index (Shell_Line, ",");
      use Ada.Strings.Unbounded;
      Manipulation_String : Unbounded_String :=
        To_Unbounded_String (Shell_Line);
      Number_Of_Argument : Natural := 0;
      Comma_Position : Natural;
      No_Argument : exception;
      use Ada.Exceptions;
   begin
      if First_Comma_Position = 0 then
         Raise_Exception (No_Argument'Identity, "Shell_Line = " & Shell_Line &
                          " seem invalid");
      end if;
      loop
         Comma_Position := Index (Manipulation_String, ",");
         exit when Comma_Position = 0;
         Number_Of_Argument := Number_Of_Argument + 1;
         Manipulation_String := To_Unbounded_String
           (Slice (Manipulation_String,
                   Comma_Position + 1,
                   Length (Manipulation_String)));
      end loop;
      declare
         Arguments_To_Return : Arguments_Array (1 .. Number_Of_Argument);
      begin
         Manipulation_String := To_Unbounded_String (Shell_Line);
         Comma_Position := Index (Manipulation_String, ",");
         Manipulation_String := To_Unbounded_String
           (Slice (Manipulation_String,
                   Comma_Position + 1,
                   Length (Manipulation_String)));
         for I in Arguments_To_Return'Range loop
            Comma_Position := Index (Manipulation_String, ",");
            if Comma_Position = 0 then
               Arguments_To_Return (I) := Manipulation_String;
            else
               Arguments_To_Return (I) :=
                 To_Unbounded_String
                 (Slice (Manipulation_String,
                         1,
                         Comma_Position - 1));
               Manipulation_String := To_Unbounded_String
                 (Slice (Manipulation_String,
                         Comma_Position + 1,
                         Length (Manipulation_String)));
            end if;
         end loop;
         return Arguments_To_Return;
      end;
   end Arguments_Comma;

   procedure Subscribe_Parameter (Active_Actor : access Active_Actor_Type) is
   begin
      Actor_Handle := Active_Actor_Access (Active_Actor);
   end Subscribe_Parameter;

   procedure Ada_Add_New_Parameter (C_Item : chars_ptr;
                                    Error_Code : in out Integer) is
      Argument : constant String := Value (C_Item);
      Parameter : Parameters.Parameter_Access;
   begin
      Error_Code := 0;
      Parameter := Parameters.New_Parameter (Argument);
      Parameters.Parameter_Vector_Package.Append
        (Actor_Handle.Parameters_List,
         Parameter);
   exception
      when E : others =>
         Error_Code := 1;
         Error_Out (Actor_Handle.Logger'Access,
                    "Ada_Add_New_Parameter", E);
   end Ada_Add_New_Parameter;

   procedure Ada_Set_Parameter (Parameter_Name : chars_ptr;
                                Parameter_Value : chars_ptr;
                                Error_Code : in out Integer) is
   begin
      declare
         Name : constant String := Value (Parameter_Name);
         Par_Value : constant String := Value (Parameter_Value);
         Parameter : Parameters.Parameter_Access;
      begin
         Error_Code := 0;
         Parameter := Parameters.Find_Parameter (Actor_Handle.Parameters_List,
                                                 Name,
                                                 True);
         Parameters.Set (Parameter.all, Par_Value);
      end;
   exception
      when E : others =>
         Error_Code := 1;
         Error_Out (Actor_Handle.Logger'Access,
                    "Ada_Set_Parameter", E);
   end Ada_Set_Parameter;

   procedure Ada_Get_Parameter (Parameter_Name : chars_ptr;
                                Buffer : System.Address;
                                Buffer_Length : Integer;
                                Error_Code : in out Integer) is
      Name : constant String := Value (Parameter_Name);
   begin
      Error_Code := 0;
      declare
         Param_Value : constant String := Get_Image (Actor_Handle, Name);
      begin
         if Param_Value'Length > (Buffer_Length - 1) then
            Error_Code := 1;
            return;
         else
            declare
               Fake_Buffer : String (1 .. Buffer_Length);
               for Fake_Buffer'Address use Buffer;
               pragma Convention (C, Fake_Buffer);
            begin
               Fake_Buffer (1 .. Param_Value'Length) := Param_Value;
               Fake_Buffer (Param_Value'Length + 1) := ASCII.NUL;
            end;
         end if;
      end;
   exception
      when E : others =>
         Error_Code := 2;
         Error_Out (Actor_Handle.Logger'Access,
                    "Ada_Get_Parameter", E);
   end Ada_Get_Parameter;

   procedure Switch_Output_Block
     (Used_Length : Interfaces.Unsigned_32;
      Newblockaddr : System.Address;
      New_Available_Length : out Interfaces.Unsigned_32);
   pragma Export (C, Switch_Output_Block, "switch_output_block");
   procedure Switch_Output_Block
     (Used_Length : Interfaces.Unsigned_32;
      Newblockaddr : System.Address;
      New_Available_Length : out Interfaces.Unsigned_32) is
      Available_Memory_Size : System.Storage_Elements.Storage_Count;
      Output_Address : System.Address;
   begin
      Actor_Handle.Outputs (1).Memory.Release_Memory
        (System.Storage_Elements.Storage_Count (Used_Length));
      loop
         select
            Actor_Handle.Outputs (1).Memory.Get_Memory
              (Output_Address, Available_Memory_Size);
            declare
               C_Output_Address : System.Address;
               for C_Output_Address'Address use Newblockaddr;
            begin
               C_Output_Address := Output_Address;
               New_Available_Length := Interfaces.Unsigned_32
                 (Available_Memory_Size);
            end;
            exit;
         or
            delay 1.0;
            Info_Out
              (Actor_Handle.Logger'Access,
               "Switch_Output_Block producer : waiting for free memory");
            Warn_Out (Actor_Handle.Logger'Access,
                      "Switch_Output_Block potential blocking call");
         end select;
      end loop;
   exception
      when E : others =>
         Error_Out (Actor_Handle.Logger'Access,
                    "Switch_Output_Block error occured",
                    E);
         declare
            C_Output_Address : System.Address;
            for C_Output_Address'Address use Newblockaddr;
         begin
            C_Output_Address := System.Null_Address;
            New_Available_Length := 0;
         end;
   end Switch_Output_Block;

   task body Task_Buffer_Dispatcher_Type is
      use GNAT.Sockets;
      Network_Address  : Sock_Addr_Type;
      Server   : Socket_Type;
      Socket   : Socket_Type;
      Request : Request_Type := (Non_Blocking_IO, True);
      Size : System.Storage_Elements.Storage_Count;
      Data_Address : System.Address;
   begin
      loop
         select
            accept Start;
         or
            terminate;
         end select;
         if Wrapper.Outputs = null then
            Log4ada.Loggers.Warn_Out (Wrapper.Logger'Access,
                                      "trying to start buffer dispatcher on" &
                                      " actor without output");
            select
               accept Stop;
            or
               terminate;
            end select;
         else
            Wrapper.Duplicate := True;
            Network_Address.Addr := Addresses (Get_Host_By_Name (Host_Name),
                                               1);
            Network_Address.Port := GNAT.Sockets.Port_Type
              (Wrapper.Duplication_Port);
            Create_Socket (Server);
            Set_Socket_Option
              (Server,
               Socket_Level,
               (Reuse_Address, True));
            Bind_Socket (Server, Network_Address);
            Listen_Socket (Server);
            Control_Socket (Server, Request);
            loop
               begin
                  Accept_Socket (Server, Socket, Network_Address);
                  Size := 0;
                  select
                     Wrapper.Outputs (Wrapper.Duplication_Output)
                       .Memory.Get_Duplicated_Data (Size, Data_Address);
                     declare
                        Size_Data : Ada.Streams.Stream_Element_Array (1 .. 4);
                        for Size_Data'Address use Size'Address;
                        Data : Ada.Streams.Stream_Element_Array
                          (1 .. Ada.Streams.Stream_Element_Offset (Size));
                        for Data'Address use Data_Address;
                     begin
                        Socket_Send (Socket, Size_Data);
                        Socket_Send (Socket, Data);
                        Wrapper.Outputs (Wrapper.Duplication_Output)
                          .Memory.Release_Duplicated_Data;
                     end;
                  or
                     delay 0.1;
                     Log4ada.Loggers.Info_Out (Wrapper.Logger'Access,
                                               "no data available");
                     declare
                        Size_Data : Ada.Streams.Stream_Element_Array (1 .. 4);
                        for Size_Data'Address use Size'Address;
                     begin
                        Socket_Send (Socket, Size_Data);
                     end;
                  end select;
                  Close_Socket (Socket);
               exception
                  when GNAT.Sockets.Socket_Error =>
                     delay Wrapper.Watcher_Delay;
                  when E : others =>
                     Log4ada.Loggers.Warn_Out (Wrapper.Logger'Access,
                                               "exception in Task_Test_Type",
                                               E);
               end;
               select
                  accept Stop;
                  Wrapper.Duplicate := False;
                  exit;
               or
                  delay 0.0;
               end select;
            end loop;
            Close_Socket (Server);
         end if;
      end loop;
   end Task_Buffer_Dispatcher_Type;

   procedure Put_Sub_System_In_Error (Active_Actor : access Active_Actor_Type;
                                      Message : String) is
   begin
      Fatal_Out (Active_Actor.Logger'Access,
                 To_String (Active_Actor.Name &
                            " put sub system " &
                            Local_Configuration.Configuration_Name &
                            " in error with message :" & Message));
      Configurator.Abstract_Actors_Coordination.Special_Domi
        (Local_Configuration.Config_Local);
   end Put_Sub_System_In_Error;

   procedure Ada_Log_Message (Id : Integer;
                              Message : Interfaces.C.Strings.chars_ptr;
                              Level : Integer) is
      Ada_Message : constant String := "id=" & Id'Img & " " & Value (Message);
   begin
      case Level is
         when 0 =>
            Debug_Out (Actor_Handle.Logger'Access, Ada_Message);
         when 1 =>
            Info_Out (Actor_Handle.Logger'Access, Ada_Message);
         when 2 =>
            Warn_Out (Actor_Handle.Logger'Access, Ada_Message);
         when 3 =>
            Error_Out (Actor_Handle.Logger'Access, Ada_Message);
         when others =>
            Fatal_Out (Actor_Handle.Logger'Access, Ada_Message);
      end case;
   end Ada_Log_Message;

end Narval.Actors.Actives;
