/*
 * NASPRO - The NASPRO Architecture for Sound Processing
 * Core library
 *
 * Copyright (C) 2007-2012 NASPRO core development team
 *
 * See the COPYING file for license conditions.
 */

/*
   Title: File system

   File system interaction.

   This API may be thread-safe or not depending on the platform.

   If it is not thread-safe on the target platform, calls to these functions in
   multi-threaded applications should be also synchronized w.r.t. other
   functions that operate on the involed filesystem(s).
 */

#ifndef _NASPRO_CORE_FS_H
#define _NASPRO_CORE_FS_H

#ifndef _NASPRO_CORE_LIB_H
# error Only <NASPRO/core/lib.h> can be included directly.
#endif

NACORE_BEGIN_C_DECLS

/*
   Type: nacore_fs_dir

   Directory stream.
 */
typedef struct _nacore_fs_dir * nacore_fs_dir;

/*
   Type: nacore_fs_dir_entry

   Directory entry.
 */
typedef struct _nacore_fs_dir_entry * nacore_fs_dir_entry;

/*
   Variable: nacore_fs_dir_sep

   Platform-specific directory separator.
 */
_NACORE_DEF extern const char * const nacore_fs_dir_sep;

/*
   Function: nacore_fs_dir_open()

   Opens a directory stream.

   On Windows platforms, it doesn't disable WOW64 file system redirection, but,
   if you want, you can do it yourself disabling it before calling this function
   and restoring it later.

   Parameters:

     name		- Absolute path name (without trailing separator).

   Returns:

     New directory stream or NULL if some error occurred, in which case errno is
     set to EACCES if permission denied, NACORE_ELOOP if there is a loop in
     symbolic links or there were too many links to be solved, ENAMETOOLONG if
     the length of a component of the path name is too long or the intermediate
     result of a symbolic link was too long, ENOENT if the specified directory
     does not exist, ENOTDIR if the specified file is not a directory, EMFILE if
     there are too many open files for the process, ENFILE if there are too many
     open files in the system, ENOMEM if there was not enough memory or
     <NACORE_EUNKNOWN> if another kind of error happened.
 */
_NACORE_DEF nacore_fs_dir
nacore_fs_dir_open(const char *name);

/*
   Function: nacore_fs_dir_get_next_entry()

   Gets the next entry in the given directory stream.

   Pointers to the current and the parent directories (e.g., "." and ".." on
   UNIX-like platforms) are automatically skipped.

   The returned entry must be passed to <nacore_fs_dir_entry_free()> when it is
   no longer needed.

   On Windows platforms, it doesn't disable WOW64 file system redirection, but,
   if you want, you can do it yourself disabling it before calling this function
   and restoring it later.

   Parameters:

     dir	- Directory stream.

   Returns:

     Directory entry or NULL if either the end of the directory stream is
     reached or some error occurred, in which case errno is set to ENOMEM if
     there was not enough memory or <NACORE_EUNKNOWN> if another kind of error
     happened.
 */
_NACORE_DEF nacore_fs_dir_entry
nacore_fs_dir_get_next_entry(nacore_fs_dir dir);

/*
   Function: nacore_fs_dir_close()

   Closes a directory stream.

   Parameters:

     dir	- Directory stream.

   Returns:

     0 on success, EINTR if the function was interrupted by a signal or
     <NACORE_EUNKNOWN> if another kind of error happened.
 */
_NACORE_DEF int
nacore_fs_dir_close(nacore_fs_dir dir);

/*
   Function: nacore_fs_dir_entry_get_name()

   Gets the filename of the directory entry.

   The returned string must not be modified or free()d.

   Parameters:

     entry	- Directory entry.

   Returns:

     String holding the filename.
 */
_NACORE_DEF const char *
nacore_fs_dir_entry_get_name(nacore_fs_dir_entry entry);

/*
   Function: nacore_fs_dir_entry_free()

   Dismisses a directory entry returned by <nacore_fs_dir_get_next_entry()>.

   Once this function is called, referring to entry will cause undefined
   behavior.

   Parameters:

     entry	- Directory entry.
 */
_NACORE_DEF void
nacore_fs_dir_entry_free(nacore_fs_dir_entry entry);

NACORE_END_C_DECLS

#endif
