/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Utility;

namespace Nemerle.Internal
{
  public class NemerleAttribute : System.Attribute
  { }
  
  /**
   * Marks a volatile field
   */
  public class VolatileModifier : NemerleAttribute
  {  }

  
  /**
   * Marks an immutable field
   */
  public class ImmutableAttribute : NemerleAttribute
  {  }

  
  /**
   * Used to store the Nemerle types in genereated assembly metadata
   */    
  public class TypeAttribute : NemerleAttribute
  {
    public this (encoded_type : string) 
    {    
      _encoded_type = encoded_type
    }
    
    public GetEncodedType () : string 
    { 
      _encoded_type
    }
    
    public mutable _encoded_type : string;
  }

  /**
   * Used to store the custom operator priorities loaded as syntax extensions
   */
  [Record]
  public class OperatorAttribute : NemerleAttribute
  {
    public mutable env : string;
    public mutable name : string;
    public mutable IsUnary : bool;
    public mutable left : int;
    public mutable right : int;
  }
  
  /**
   * Marks a Nemerle variant type
   */
  public class VariantAttribute : NemerleAttribute
  {
    public this () { }
    public this (variant_options : string) 
    {    
      _variant_options = variant_options
    }
    
    public GetVariantOptions () : string 
    {
      _variant_options
    }
    
    public mutable _variant_options : string;
  }

  /**
   * Marks a Nemerle variant option
   */    
  public class VariantOptionAttribute : NemerleAttribute
  {  }
  
  /**
   * Marks a constant Nemerle variant option
   */    
  public class ConstantVariantOptionAttribute : NemerleAttribute
  {  }
  
  /**
   * Container for type aliases.
   */    
  public class TypeAliasAttribute : NemerleAttribute
  {
    public this (variant_options : string) 
    {    
      _variant_options = variant_options
    }
    
    public GetAliasedType () : string 
    {
      _variant_options
    }
    
    public _variant_options : string;
  }
  
  /**
   * Used on assembly to list contained macros.
   */    
  [System.AttributeUsage (System.AttributeTargets.Assembly,
                          AllowMultiple = true)]
  public class ContainsMacroAttribute : NemerleAttribute
  {
    public this (name : string) 
    {    
      _name = name
    }
    
    public GetName () : string 
    {
      _name
    }

    public _name : string;
  }

  [System.AttributeUsage (System.AttributeTargets.All)]
  public class MacroAttribute : NemerleAttribute
  {
    public name : string;
    public parameters : string;    
    public global_ctx : int;
   
    public this (name : string, ctx : int, parameters : string)
    {
      this.name = name;
      this.parameters = parameters;
      global_ctx = ctx;
    }
  }
  
  [System.AttributeUsage (System.AttributeTargets.Class, AllowMultiple = true)]
  [Record]
  public sealed class ExtensionPatternEncodingAttribute : NemerleAttribute
  {
    [Accessor]
    name : string;
    
    [Accessor]
    identifiers : string;
    
    [Accessor]
    pattern : string;
  }
}

namespace Nemerle.Utility
{
  public class Identity ['a, 'b] : Builtins.Function ['a, 'b] where 'a : 'b 
  {
    // do not work :/
    //public static Instance : Identity ['a, 'b] = Identity.['a, 'b] ();  
  
    public override apply (x : 'a) : 'b {
      x
    }
  }
}
