//
// CodeCompletionEngine.n
//
// Author:
//  Alejandro Serrano (trupill at yahoo dot es)
//

/*
 * Copyright (c) 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 
using Nemerle.Collections;
using Nemerle.Compiler;

namespace Nemerle.Completion
{
    # region For Code Completion
    
    public abstract class CompletionInfo
    {
        public abstract CompletionKind : CompletionKind { get; }
    }
    
    public enum CompletionKind
    {
    | Member
    | Type
    }
    
    public class CompletionMember : CompletionInfo
    {
        public override CompletionKind : CompletionKind
        { get { CompletionKind.Member } }
        
        public MemberInfo : NemerleMemberInfo;
        
        public this (member : NemerleMemberInfo)
        {
            MemberInfo = member;
        }
    }
    
    public class CompletionType : CompletionInfo
    {
        public override CompletionKind : CompletionKind
        { get { CompletionKind.Type } }
        
        public TypeInfo : NemerleTypeInfo;
        
        public this (t : NemerleTypeInfo)
        {
            TypeInfo = t;
        }
    }
    
    # endregion
    
    public class TypeTree
    {
        public mutable Types : array[DeclaredTypeInfo];
    }
    
    public class CodeLocation
    {
        public mutable Filename : string;
        public mutable Line : int;
        public mutable Column : int;
        public mutable EndLine : int;
        public mutable EndColumn : int;
        
        public this (filename : string, line : int, column : int, endline : int, endcolumn : int)
        {
            Filename = filename;
            Line = line;
            Column = column;
            EndLine = endline;
            EndColumn = endcolumn;
        }
    }
    
    public abstract class NemerleTypeInfo
    {
        public abstract TypeKind : NemerleTypeKind { get; }
    }
    
    public enum NemerleTypeKind
    {
    | DeclaredType
    | ExternalType
    }
    
    public class DeclaredTypeInfo : NemerleTypeInfo
    {
        public override TypeKind : NemerleTypeKind
        {
            get { NemerleTypeKind.DeclaredType }
        }
        
        public mutable Namespace : string;
        public mutable Name : string;
        public mutable Location : CodeLocation;        
        
        public mutable IsSealed : bool;
        public mutable IsAbstract : bool;
        
        public mutable IsInterface : bool;
        public mutable IsDelegate : bool;
        public mutable IsEnum : bool;
        public mutable IsValueType : bool;
        public mutable IsModule : bool;
        public mutable IsAlias : bool;
        public mutable IsStruct : bool;
        public mutable IsVariant : bool;
        public mutable IsNested : bool;
        
        public mutable IsPublic : bool;
        public mutable IsInternal : bool;
        public mutable IsProtected : bool;
        public mutable IsPrivate : bool;
        
        public mutable DefaultIndexerName : string;
        public mutable DeclaringType : DeclaredTypeInfo;
        
        public mutable Fields : array[FieldInfo];
        public mutable Methods : array[MethodInfo];
        public mutable Properties : array[PropertyInfo];
        public mutable Events : array[EventInfo];
        public mutable NestedTypes : array[DeclaredTypeInfo];
        public mutable VariantOptions : array[DeclaredTypeInfo];
        
        public mutable BaseType : NemerleTypeInfo;
        public mutable Interfaces : array[NemerleTypeInfo];
        
        public mutable TypeParameters : array[TypeParameterInfo];
    }
    
    public class TypeParameterInfo
    {
        public mutable Name : string;
        public mutable TypeConstraints : array[ConstructedTypeInfo];
        public mutable SpecialConstraints : Constraint;
    }
    
    [System.Flags]
    public enum Constraint
    {
        | Class = 0x001;
        | Struct = 0x002;
        | New = 0x004;
        | None = 0;
    }
    
    public class ReferencedTypeInfo : NemerleTypeInfo
    {
        public override TypeKind : NemerleTypeKind
        {
            get { NemerleTypeKind.ExternalType }
        }

        public mutable Type : System.Type;
        
        public this (t : System.Type)
        {
            Type = t;
        }
    }
    
    public class ConstructedTypeInfo
    {
        public class Class : ConstructedTypeInfo
        {
            public mutable Type : NemerleTypeInfo;
            public mutable SubstitutedArguments : array[ConstructedTypeInfo];
            
            public this()
            {}
            
            public this (NemerleType : NemerleTypeInfo, substitutedArgs : array[ConstructedTypeInfo])
            {
                Type = NemerleType;
                SubstitutedArguments = substitutedArgs;
            }
        }
        
        public class GenericSpecifier : ConstructedTypeInfo
        {
            public mutable Name : string;
            public mutable TypeConstraints : array[ConstructedTypeInfo];
            public mutable SpecialConstraints : Constraint;
        }
        
        public class Function : ConstructedTypeInfo
        {
            public mutable From : ConstructedTypeInfo;
            public mutable To : ConstructedTypeInfo;
        }
        
        public class Tuple : ConstructedTypeInfo
        {
            public mutable Types : array[ConstructedTypeInfo];
        }
        
        public class Array : ConstructedTypeInfo
        {
            public mutable Type : ConstructedTypeInfo;
            public mutable Rank : int;
        }
        
        public class Void : ConstructedTypeInfo 
        { }
    }
    
    public class NemerleMemberInfo
    {
        public mutable Name : string;
        public mutable Location : CodeLocation;

        // tmp hack
        public mutable _member : IMember;
        
        public mutable IsStatic : bool;
        public mutable IsPublic : bool;
        public mutable IsPrivate : bool;
        public mutable IsProtected : bool;
        public mutable IsInternal : bool;
        
        public mutable Type : ConstructedTypeInfo;
    }
    
    public class FieldInfo : NemerleMemberInfo
    {
        public mutable IsMutable : bool;
        public mutable IsVolatile : bool;
        public mutable IsLiteral : bool;
        
        public mutable Value : object;
    }
    
    public class MethodInfo : NemerleMemberInfo
    {
        public mutable IsAbstract : bool;        
        public mutable IsFinal :  bool;
        public mutable IsOverride : bool;
        public mutable IsVirtual : bool;
        public mutable IsNew : bool;
        public mutable IsExtern : bool;

        public mutable IsVarArgs : bool;
        
        public mutable IsConstructor : bool;
        public mutable IsStaticConstructor : bool;
        
        public mutable ReturnType : ConstructedTypeInfo;
        public mutable TypeParameters : array[TypeParameterInfo];
        public mutable Parameters : array[ParameterInfo];
    }
    
    public class ParameterInfo
    {
        public mutable Name : string;
        public mutable Type : ConstructedTypeInfo;
        
        public mutable HasDefaultValue : bool;
        public mutable DefaultValue : object;
        
        public mutable IsRefParameter : bool;
        public mutable IsOutParameter : bool;
    }
    
    public class PropertyInfo : NemerleMemberInfo
    {
        public mutable IsMutable : bool;
        public mutable IsIndexer : bool;
        public mutable IndexerParameters : array[ConstructedTypeInfo];

        public mutable IsAbstract : bool;        
        public mutable IsFinal :  bool;
        public mutable IsOverride : bool;
        public mutable IsVirtual : bool;
        public mutable IsNew : bool;
        public mutable IsExtern : bool;
        
        public mutable Getter : MethodInfo;
        public mutable Setter : MethodInfo;
    }
    
    public class EventInfo : NemerleMemberInfo
    {
        public mutable Adder : MethodInfo;
        public mutable Remover : MethodInfo;
    }
}

