/*
 * Copyright (c) 2003-2005 University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Compiler.Typedtree;
using Nemerle.Collections;
using Nemerle.Utility;

using System.Reflection;
using System.IO;
using System.Diagnostics.SymbolStore;
using System.Diagnostics;

using NC = Nemerle.Compiler;
using SRE = System.Reflection.Emit;

namespace Nemerle.Compiler
{
  /**
   * This part of TypesManages is responsible for generation of System.Reflection.Emit structures 
   * composing program hierarchy. It executes emission of all classes and their members.
   */
  public partial class TypesManager
  {
    /* -- PUBLIC CONSTRUCTORS ---------------------------------------------- */

    private resolve_hack (_ : object, _ : System.ResolveEventArgs) : Assembly
    {
      //Message.Debug ($ "resolve_hack: $(args.Name)");
      this._assembly_builder
    }

    public this () 
    {
      _OutputFileName = Options.OutputFileName;

      unless (Options.TargetIsLibrary) {
        _need_entry_point = true;
        _entry_point = None ();
      };
    }

    public CreateAssembly () : void
    {
      // we need to process global assembly attributes before creating assembly name
      _assembly_name = AttributeCompiler.CreateAssemblyName (); 

      _assembly_name.Name = Path.GetFileNameWithoutExtension (_OutputFileName);

      def assembly_requirements =
        if (Options.CompileToMemory)
          Emit.AssemblyBuilderAccess.Run
        else
          Emit.AssemblyBuilderAccess.Save;
 
      // workaround MS.NET bugs with some specific value / generic types hierarchy
      def delegate_ = resolve_hack : System.ResolveEventHandler;
      System.AppDomain.CurrentDomain.TypeResolve += delegate_;
      // we need to later remove it to avoid garbage
      Passes.CleanupOnce.Push (fun () { 
        System.AppDomain.CurrentDomain.TypeResolve -= delegate_;
      });

      def dir = Path.GetDirectoryName(Path.GetFullPath (_OutputFileName));
      unless (Directory.Exists (dir))
        Message.FatalError ($"specified output directory `$dir' does not exist");
      
      /* define a dynamic assembly */
      this._assembly_builder =
        System.AppDomain.CurrentDomain.DefineDynamicAssembly
          (this._assembly_name,
           assembly_requirements,
           dir);

      when (_assembly_name.Name == "") Message.FatalError ("name of output assembly cannot be empty");

      /* create a dynamic module */
      this._module_builder =
        if (Options.CompileToMemory)
          // we cannot give output filename if we are compiling only to Run
          this._assembly_builder.DefineDynamicModule (_assembly_name.Name, Options.EmitDebug); 
        else
          this._assembly_builder.DefineDynamicModule (_assembly_name.Name,
                                                      Path.GetFileName (_OutputFileName),
                                                      Options.EmitDebug); 

      when (Options.EmitDebug) _debug_emit = _module_builder.GetSymWriter ();
    }

    public IsEmitting : bool {
      get { _assembly_builder != null }    
    }

    add_resources_to_assembly () : void 
    {
      def escape_resource (x : string) {
        def cp = x.IndexOf (',');
        if (cp != -1)
          (x.Substring (0, cp), x.Substring (cp + 1))
        else
          // change name from /bar/bar/file to bar.bar.file namespace  
          (x, x.Replace ('/','.').Replace ('\\', '.'));
      }

      def escape_linked_resource (x : string) {
        def cp = x.IndexOf (',');
        if (cp != -1)
          // change name from /bar/bar/file to bar.bar.file namespace          
          (x.Substring (0, cp), x.Substring (cp + 1))
        else
          (x, x);
      }

      /* we can embed resources only on mono or .NET 2.0 */
      foreach (element in Options.EmbeddedResources) {
        def (file, name) = escape_resource (element);
        try {
          def resource_builder = _module_builder.DefineResource (file, name);        
          if (File.Exists (file)) {
            using (fs = File.OpenRead (file)) {
              def res = array (fs.Length :> int);
              def loop (pos) {
                if (res.Length == pos) {}
                else
                  loop (pos + fs.Read(res, pos, res.Length - pos));
              }
              loop (0);
              resource_builder.AddResource (name, res);
            }
          }
          else
            Message.Error ("Could not find resource " + file);
        }
        catch {
          | e => Message.Error ("Could not embed resource: " + e.Message);
        }
      }

      foreach (element in Options.LinkedResources) {
        def (file, name) = escape_linked_resource (element);
        try {
          _assembly_builder.AddResourceFile (name, file);
        }
        catch {
          | _ is FileNotFoundException =>
            Message.Error ("Could not find resource " + file);
          | e is System.ArgumentException =>
            Message.Error ("Could not link resource: " + e.Message);
        }
      }
    }

    /**
     *
     */
    public EmitAuxDecls () : void
    {
      compile_all_tyinfos (true)
    }


    /**
     *
     */
    public EmitDecls () : void
    {
      Passes.Solver.Enqueue (fun () {
        compile_all_tyinfos (false);
        foreach (x in AttributeCompiler.GetCompiledAssemblyAttributes ())
          _assembly_builder.SetCustomAttribute (x);
          
        // emit debug attributes
        when (Options.EmitDebug) {
          def attr = AttributeCompiler.MakeEmittedAttribute (SystemType.DebuggableAttribute, array [SystemType.DebuggableAttribute_DebuggingModes], 
            DebuggableAttribute.DebuggingModes.DisableOptimizations | DebuggableAttribute.DebuggingModes.Default);
          _assembly_builder.SetCustomAttribute (attr);
        }

        // do not require string literals interning
        def attr = AttributeCompiler.MakeEmittedAttribute (SystemType.CompilationRelaxationsAttribute, 8);
        _assembly_builder.SetCustomAttribute (attr);
        
        // wrap non exception throws
        /* uncomment it after we get dependency upon mono 1.1.10, since mono 1.1.9.x do not support this attribute
        def (_, attr) = AttributeCompiler.CompileAttribute (GlobalEnv.Core, null, <[ System.Runtime.CompilerServices.RuntimeCompatibilityAttribute (WrapNonExceptionThrows=true) ]>);
        _assembly_builder.SetCustomAttribute (attr);
        */
      })
    }

    /**
     * Returns generated assembly for runtime instantations of its types
     */
    public GeneratedAssembly : Assembly
    {
      get { _assembly_builder }
    }
    
    
    /**
     * Saves the constructed assembly to a file
     */
    public SaveAssembly () : void
    {
      add_resources_to_assembly ();

      // if there are some nemerle specific metadata encoded in attributes
      when (contains_nemerle_specifics) {
        def attr = AttributeCompiler.MakeEmittedAttribute (SystemType.Reflection_AssemblyConfigurationAttribute, "ContainsNemerleTypes");
        this._assembly_builder.SetCustomAttribute (attr);
      }
          
      // set the entry point
      match ((_need_entry_point, _entry_point)) {
        | (true, Some (entry_point_method_info)) =>
          _assembly_builder.SetEntryPoint (entry_point_method_info,
                                           if (Options.TargetIsWinexe)
                                             Emit.PEFileKinds.WindowApplication
                                           else
                                             Emit.PEFileKinds.ConsoleApplication)
        | (true, None) =>
          Message.Error ("no suitable entry point (Main function) found")
        | _ => ()
      };

      // save the assembly
      try {
        _assembly_builder.Save (Path.GetFileName (_OutputFileName));
        //when (_debug_emit != null) _debug_emit.Close ();
      }
      catch {
        | e is System.UnauthorizedAccessException =>
          Message.Error ($"could not write to output file `$(this._OutputFileName)'"
                         " -- `$(e.Message)'")  
          
        | e is IOException =>
          Message.Error ($"could not write to output file `$(this._OutputFileName)'"
                         " -- `$(e.Message)'")  
      }
    }

    /* -- PRIVATE METHODS -------------------------------------------------- */


    /**
     * - create S.R.E.TypeBuilders for entire hierarchy of program 
     * - add members to those TypeBuilders (only stubs for methods)
     * - emit bodies of methods
     * - finalize value types
     * - finalize all types
     */  
    protected virtual compile_all_tyinfos (aux_phase : bool) : void
    {
      def allow_it (ti : TypeBuilder) {
        !ti.IsFinalized &&
        is_aux_decl (ti) == aux_phase
      };
      def create_type_emit_builder (ti : TypeBuilder) {
        when (allow_it (ti)) {
          //Message.Debug ("make type builder for " + ti.FullName);
          ti.CreateEmitBuilder ();
          when (ti.Attributes %&& NemerleAttributes.Macro) {
            def attr = AttributeCompiler.MakeEmittedAttribute (SystemType.ContainsMacroAttribute, ti.GetTypeBuilder ().FullName);
            _assembly_builder.SetCustomAttribute (attr);
          }
        }
      };
      // create members' declarations in SRE.TypeBuilders
      def emit_decls (ti : TypeBuilder) {
        when (allow_it (ti)) {
          // Message.Debug ("generating declarations " + ti.FullName);
          ti.CreateEmitDeclarations ()
        }
      };
      def emit_impl (ti : TypeBuilder) {
        when (allow_it (ti)) {
          //Message.Debug ("generating code for " + ti.FullName);
          Passes.MarkTypeBuilderCompiled ();
          ti.EmitImplementation ()
        }
      };

      when (!aux_phase) _cgil_phase = 1;
      Iter (create_type_emit_builder);
      Iter (fun (tb) {
        when (allow_it (tb)) tb.UpdateEmittedInheritance ()
      });

      when (!aux_phase) _cgil_phase = 2;
      // first emit fields of enum types as it is required to compute their sizes,
      // when they are used as fields
      IterConditionally (emit_decls, fun (x : TypeBuilder) { x.IsEnum });
      IterConditionally (emit_decls, fun (x : TypeBuilder) { ! x.IsEnum });

      when (!aux_phase) _cgil_phase = 3;

      // we first finalize value types, because MS.NET runtime requires so
      IterConditionally (emit_impl, fun (x : TypeBuilder) { 
        x.IsValueType && x.DeclaringType == null 
      });

      // now we can finalize everything else
      Iter (emit_impl);

      // MaybeBailout inteferes with the Code Completion Engine
      unless (Nemerle.Completion.Engine.IsInCompletionMode)
        Message.MaybeBailout ();

      when (!aux_phase) _cgil_phase = 4;
    }

    internal MemberAdded (ti : TypeBuilder, _mem : IMember) : void
    {
      // Message.Debug ($"ma: $ti -> $mem ");
      when (_cgil_phase >= 1) {
        ti.CreateEmitBuilder ();
        ti.UpdateEmittedInheritance ();
      }

      // when (_cgil_phase >= 2)
      //  add_declaration (ti, mem);
    }


    internal MaybeCompile (ti : TypeBuilder, mem : MemberBuilder) : void
    {
      mem.CreateEmitBuilder (ti.GetTypeBuilder ());
      
      when (_cgil_phase >= 3)
        match (mem) { 
          | meth is MethodBuilder =>
            ti.DoBeforeFinalization (fun () {
              meth.RunBodyTyper ();
              meth.CompileAfterTyping ()
            })
          | _ => {}
        }
    }


    /**
     * Check if declaration is auxiliary, used internally etc.
     */
    private static is_aux_decl (ti : TypeBuilder) : bool
    {
      ti.FullName.StartsWith ("Nemerle.Internal.")
    }
    
    /* -- PRIVATE FIELDS --------------------------------------------------- */

    private mutable _assembly_name : System.Reflection.AssemblyName;
    private mutable _assembly_builder : Emit.AssemblyBuilder;
    internal mutable _module_builder : Emit.ModuleBuilder;
    public static mutable _debug_emit : ISymbolWriter;
    internal mutable contains_nemerle_specifics : bool = false;

    internal _need_entry_point : bool;
    /** updated when method with static Main signature is met */
    internal mutable _entry_point : option [MethodInfo];

    private _OutputFileName : string;
    private mutable _cgil_phase : int;
  }


  public partial class TypeBuilder : TypeInfo
  {
    /**
     * This method makes a skeleton of a class -- the class partial type and
     * partial types for the nested classes are created. The parent class and
     * the interfaces being implemented are assumed to have been processed
     * before constructing a dependant class. All the declarations and method
     * implementations will be created successively in subsequent passes.
     */
    internal CreateEmitBuilder () : void
    {
      when (system_type == null) {
        /* create optional custom attributes for this type */
        def custom_attribute =
          match (tydecl) {
            | TypeDeclaration.Variant (decls) =>
              make_nemerle_variant_attribute (decls)
              
            | TypeDeclaration.VariantOption =>
              def has_co_ctor = GetConstantObject () != null;
              make_nemerle_variant_option_attribute (has_co_ctor)

            | TypeDeclaration.Alias (t) =>
              make_nemerle_type_alias_attribute (t)
              
            | _ => null
          };

        /* decide the new type attributes */
        def is_nested = DeclaringType != null;

        def type_kind_attrs =
          match (tydecl) {
            | TypeDeclaration.Variant 
            | TypeDeclaration.VariantOption 
            | TypeDeclaration.Class          => TypeAttributes.Class
            | TypeDeclaration.Alias // we pretend type alias is an interface
            | TypeDeclaration.Interface      => TypeAttributes.Interface %| TypeAttributes.Abstract
            | TypeDeclaration.Enum           => TypeAttributes.Sealed %| TypeAttributes.Class
          };

        def attrs = make_type_attributes (Attributes, is_nested) %| type_kind_attrs;
        mutable typarms_len = TyparmsCount;

        when (is_nested)
          typarms_len -= DeclaringType.TyparmsCount;

        def generic_mark_suffix =
          match (typarms_len) {
            | 0 => ""
            | l => "`" + l.ToString ()
          }

        /* create the type builder for a top-level or nested class declaration */
        type_builder =
          if (!is_nested)
            Manager._module_builder.DefineType (FullName + generic_mark_suffix, attrs)
          else {
            def containing_builder = (DeclaringType :> TypeBuilder).GetTypeBuilder ();
            containing_builder.DefineNestedType (Name + generic_mark_suffix, attrs)
          };

        // creates and store generic parameters in our StaticTyVars

        unless (typarms.IsEmpty) {
          def names = typarms.MapToArray (fun (x) { x.Name });
          def generic_parms = type_builder.DefineGenericParameters (names);
          typarms.IterI (0, fun (idx, x) { 
            x.SetGenericBuilder (generic_parms [idx]); 
          });
        }

        when (custom_attribute != null) {
          Manager.contains_nemerle_specifics = true;
          type_builder.SetCustomAttribute (custom_attribute);
        }

        when (extension_patterns.Count > 0)
          Manager.contains_nemerle_specifics = true;

        // Structs with no fields need to have at least one byte.
        // The right thing would be to set the PackingSize in a DefineType
        // but there are no functions that allow interfaces *and* the size to
        // be specified.
        // maybe in 2.0 there is a better API
        when (IsStruct && GetFields (BindingFlags.Instance %|
                                     BindingFlags.Public %| 
                                     BindingFlags.NonPublic).IsEmpty)
          _ = type_builder.DefineField ("$PLACE_HOLDER$", SystemType.Byte,
                                        FieldAttributes.Private %| FieldAttributes.SpecialName);
        
        system_type = type_builder;
      }
    }

    /**
     * Builds a Nemerle variant attribute
     */
    private static make_nemerle_variant_attribute (decls : list [TypeInfo]) : Emit.CustomAttributeBuilder
    {
      def names = decls.Map (fun (decl) { decl.FullName });
      AttributeCompiler.MakeEmittedAttribute (SystemType.VariantAttribute, NString.Concat (",", names))
    }


    /**
     * Builds a Nemerle type alias
     */
    private static make_nemerle_type_alias_attribute (t : MType) : Emit.CustomAttributeBuilder
    {
      AttributeCompiler.MakeEmittedAttribute (SystemType.TypeAliasAttribute, TyCodec.EncodeType (t))
    }


    /**
     * Builds a Nemerle variant option attribute
     */
    private static make_nemerle_variant_option_attribute (is_const : bool) : Emit.CustomAttributeBuilder
    {
      AttributeCompiler.MakeEmittedAttribute (if (is_const)
                             SystemType.ConstantVariantOptionAttribute
                           else
                             SystemType.VariantOptionAttribute)
    }


    /**
     * Converts Nemerle modifiers to the Framework type attributes.
     */
    private static make_type_attributes (attrs : NemerleAttributes, is_nested : bool) : TypeAttributes
    {
      mutable result = TypeAttributes.AutoLayout;
      when (attrs %&& NemerleAttributes.Public)
        if (is_nested) result |= TypeAttributes.NestedPublic
        else result |= TypeAttributes.Public;
      when (attrs %&& NemerleAttributes.Private)
        if (is_nested) result |= TypeAttributes.NestedPrivate
        else Message.Error ("Private is not allowed for top level types");
      when (attrs %&& NemerleAttributes.Protected)
        if (is_nested) result |= TypeAttributes.NestedFamily
        else Message.Error ("Protected is not allowed for top level types");
      when (attrs %&& NemerleAttributes.Internal && is_nested)
        result |= TypeAttributes.NestedFamORAssem;
      when (attrs %&& NemerleAttributes.Abstract) result |= TypeAttributes.Abstract;
      when (attrs %&& NemerleAttributes.Sealed) result |= TypeAttributes.Sealed;
      when (attrs %&& NemerleAttributes.SpecialName) result |= TypeAttributes.SpecialName;
      // we cannot do Sealed at the same time on MS.NET 1.1
      when (attrs %&& NemerleAttributes.Static) result |= TypeAttributes.Abstract;
        
      result
    }

    
    /**
     *  Set up custom attributes on members of this TypeBuilder. Compile method's bodies.
     */
    internal EmitImplementation () : void
    {
      IsFinalized = true;
      
      unless (GetModifiers ().IsEmpty) {
        GetModifiers ().SaveCustomAttributes (this, fun (target, a) {
          def valid = target %&& System.AttributeTargets.Class ||
            type_builder.IsEnum && target %&& System.AttributeTargets.Enum;
          if (valid) {
            type_builder.SetCustomAttribute (a);
            null
          }
          else "type " + ToString ()
        })
      }

      foreach (m is MemberBuilder in member_list) m.Compile ();
      
      //Message.Debug ("finalizing " + FullName);
      unless (Message.SeenError)
        this.FinalizeType ();
      //Message.Debug ("finalized " + FullName);
    }

    /**
     * This methods walks the class and adds field and method declarations.
     * The method implementation will be filled in a separate pass.
     */
    internal CreateEmitDeclarations () : void
    {
      // nested types are skipped here     
      foreach (m is MemberBuilder in member_list) {
        // Message.Debug ($ "emit $m from $type_builder");
        m.CreateEmitBuilder (type_builder);
      }
    }

    /**
     * Reads the inheritance list of a class and retrieves the base
     * type and builds the array of interfaces implemented by the class.
     */
    determine_inheritance () : (System.Type * array [System.Type])
    {
      match (tydecl) {
        | TypeDeclaration.Enum => (SystemType.Enum, array [])
        | _ =>
          def interfaces = InterfacesToImplement ().MapToArray (fun (i) {
              GetMemType ().GetInstantiatedSuperType (i).SystemType
            });

          if (parent_type != null)
            (parent_type.SystemType, interfaces)
          else
            (null, interfaces)
      }
    }
    
    internal UpdateEmittedInheritance () : void
    {
      when (! reflection_inheritance_emitted) {
        reflection_inheritance_emitted = true;
        
        /* determine the inheritance relation for this type
           generic builders are now properly stored */
        def (extends_class, implements_interfaces) = determine_inheritance ();

        // save inheritance information in our type builder
        when (extends_class != null)
          type_builder.SetParent (extends_class);

        foreach (inter in implements_interfaces) 
          type_builder.AddInterfaceImplementation (inter);

        foreach (gp in typarms) gp.UpdateConstraints ();
      }
    }
  }
  
  public partial class MethodBuilder : MemberBuilder 
  {
    internal override CreateEmitBuilder (tb : Emit.TypeBuilder) : void
    {
      if (fun_kind is FunKind.Constructor)
        CreateConstructorBuilder (tb)
      else
        CreateMethodBuilder (tb)
    }

    /**
     * Emits a method's body
     */
    internal CompileAfterTyping () : void
    {
      def setup_method (emitter) {
        // make sure type is not finalized
        def type_builder = declaring_type.GetTypeBuilder ();      
        assert (type_builder != null);

        def mb = method_builder;
        Util.cassert (mb != null, $"method builder is null for $this");

        /* update the entry point settings, if necessary */
        when (emitter._need_entry_point && Name == "Main" &&
              (Options.MainClass == null ||
               declaring_type.FullName == Options.MainClass)) {
          def has_correct_signature () {
            mb.IsStatic &&
            (ReturnType.Equals (InternalType.Int32) || 
             ReturnType.Equals (InternalType.Void)) &&
            match (fun_header.parms) {
              // workaround bug in MS PEVerify
              | [fp] =>
                fp.ty.Fix ().Equals (MType.Array (InternalType.String, 1))
              | [] => true
              | _ => false
            }
          }

          if (has_correct_signature ()) {
            match (emitter._entry_point) {
              | Some =>
                Message.Error ($ "more then one entry point (Main function) "
                                 "found in $(mb.DeclaringType)")
              | None =>
                when (TypesManager._debug_emit != null)
                  TypesManager._debug_emit.SetUserEntryPoint (SymbolToken (1));
                emitter._entry_point = Some (mb : MethodInfo)
            }
          }
          else
            Message.Warning (28, $"$this has the wrong signature to be an entry point");
        };

        /* update the overriding relationships */
        match (fun_kind) {
          | FunKind.BoundMethod (impl) =>
            foreach (overridden : IMethod in impl) {

              // Message.Debug ("override " + method_info.Name + " " + overridden_info.Name + " in " +
              //                type_builder.FullName);
              when (mb.IsPrivate || Name != overridden.Name) {
                def overridden_info =
                  ILEmitter.GetMethodInfo (declaring_type.GetMemType (), overridden);
                type_builder.DefineMethodOverride (mb, overridden_info)
              }
            }
          | _ => {}
        }
      }
      
  //    Message.Debug (this.ToString ());
      match (fun_header.body) {
        | FunBody.ILed => {}
        | _ =>
          // maybe create additional method for implementing interface method
          // (in case of co/contravariant methods)
          CreateImplementsWrapper ();

          // Message.Debug (Location, $ "compile: $this");

          Util.locate (Location, {
            match (fun_header.body) {
              | FunBody.ILed | FunBody.Parsed => Util.ice (fun_header.body.GetType().ToString ())

              | FunBody.Typed when Message.SeenError =>
                // just skip it
                fun_header.body = FunBody.ILed ()

              | FunBody.Typed =>
                unless (declaring_type.IsDelegate) {
                  assign_parameter_indexes (GetMethodBase ().IsStatic);
                  emit_parameter_attributes ();

                  unless (fun_kind is FunKind.Constructor)
                    setup_method (declaring_type.Manager);
                  
                  declaring_type.DoBeforeFinalization2 (fun () {
                    // Message.Debug ($"making il generator for $meth");
                    def gen =
                      match (GetMethodBase ()) {
                        | mb is Emit.MethodBuilder =>
                          ILEmitter (mb, declaring_type, fun_header);

                        | cb is Emit.ConstructorBuilder =>
                          ILEmitter (cb, declaring_type, fun_header);

                        | _ => Util.ice ()
                      }

                    when (Options.EmitDebug) gen.SetDocument (Location);
                    gen.Run ();
                  });
                }

              | FunBody.Abstract => 
                  emit_parameter_attributes ();
            }
          });

          when (Options.EarlyExit)
            Message.MaybeBailout ();
      }
    }

    static parameter_attributes (fp : Fun_parm) : ParameterAttributes
    {
      (match (fp.kind) {
        | ParmKind.Out => ParameterAttributes.Out
        | ParmKind.Ref
        | ParmKind.Normal => ParameterAttributes.None
      }) 
      | 
      (if (fp.default_value.IsSome)
         ParameterAttributes.HasDefault | ParameterAttributes.Optional
       else
         ParameterAttributes.None)
    }


    emit_parameter_attributes () : void
    {
      foreach (parm in fun_header.parms) {
        parm.modifiers.SaveCustomAttributes (declaring_type, fun (target, attr) {
          if (target %&& System.AttributeTargets.Parameter) {
            parm.builder.SetCustomAttribute (attr);
            null
          }
          else "parameter " + parm.name
        })
      }
    }


    /**
     * Assigns an ordinal index to each of a method's parameter
     */
    assign_parameter_indexes (is_static : bool) : void
    {
      mutable index = if (is_static) 0 else 1;
      foreach (parm in fun_header.parms) {
        parm.decl.SetParmIndex (index,
                                is_by_ref = parm.kind != ParmKind.Normal);
        index++;
      }
    }
  
    
    /* build the parameter types array */
    param_types () : array [System.Type] 
    { 
      GetParameters ().MapToArray (fun (parm) { parm.SystemType });
    }


    /**
     * Converts Nemerle modifiers to the CLI method attributes.
     */
    static make_method_attributes (attrs : NemerleAttributes) : MethodAttributes
    {
      mutable result = MethodAttributes.HideBySig;

      when (attrs %&& NemerleAttributes.Public) result |= MethodAttributes.Public;
      when (attrs %&& NemerleAttributes.Private) result |= MethodAttributes.Private;
      if (attrs %&& NemerleAttributes.Protected)
        if (attrs %&& NemerleAttributes.Internal)
          result |= MethodAttributes.FamORAssem;
        else
          result |= MethodAttributes.Family;
      else
        when (attrs %&& NemerleAttributes.Internal)
          result |= MethodAttributes.Assembly;
      when (attrs %&& NemerleAttributes.Static) result |= MethodAttributes.Static;
      when (attrs %&& NemerleAttributes.Override)
        result |= MethodAttributes.Virtual %| MethodAttributes.ReuseSlot;
      // when method is static, then new is only for compile-time information
      when (!(attrs %&& NemerleAttributes.Static) && attrs %&& NemerleAttributes.New)
        result |= MethodAttributes.NewSlot | MethodAttributes.Virtual;
      when (attrs %&& NemerleAttributes.Abstract)
        result |= MethodAttributes.Virtual %| MethodAttributes.Abstract;
      when (attrs %&& NemerleAttributes.Virtual) result |= MethodAttributes.Virtual;
      when (attrs %&& NemerleAttributes.Sealed)
        result |= MethodAttributes.Final %| MethodAttributes.Virtual;
      when (attrs %&& NemerleAttributes.SpecialName) result |= MethodAttributes.SpecialName;

      result
    }
    
    /**
     * Adds a method builder to a type builder
     */
    CreateMethodBuilder (tb : Emit.TypeBuilder) : void
    {
      // Message.Debug ($"-- AddMethodBuilder: $meth $(Attributes) ");

      /* update the modifiers list */
      match (fun_kind) {
        | FunKind.BoundMethod (_ :: _) =>
          Attributes |= NemerleAttributes.Virtual;
        | _ => {}
      }
      
      def attrs = make_method_attributes (Attributes);
      def attrs =
        if (Name == ".cctor")
          attrs | MethodAttributes.RTSpecialName | MethodAttributes.SpecialName
        else
          attrs;
    
      /* add the method to the type builder */
      mutable pinvoke = false;

      if (fun_header.typarms.IsEmpty) {
        def parm_types_array = param_types ();

        method_builder = AttributeCompiler.CheckPInvoking (this, tb, attrs, parm_types_array);
      
        if (method_builder == null)
          method_builder = tb.DefineMethod (Name, attrs, ReturnType.SystemType,
                                            parm_types_array);
        else
          pinvoke = true;
      }
      else {
        method_builder = tb.DefineMethod (Name, attrs);
        
        def names = fun_header.typarms.MapToArray (fun (x) { x.Name });

        def generic_parms = method_builder.DefineGenericParameters (names);
        fun_header.typarms.IterI (0, fun (idx, x) { 
          x.SetGenericBuilder (generic_parms [idx]); 
        });
        foreach (gp in fun_header.typarms) gp.UpdateConstraints ();

        method_builder.SetReturnType (ReturnType.SystemType);
    
        method_builder.SetParameters (param_types ());
      }

      /* add the runtime modifiers for delegate methods */
      when (DeclaringType.IsDelegate) {
        assert (!pinvoke);
        method_builder.SetImplementationFlags (
          MethodImplAttributes.Runtime | MethodImplAttributes.Managed
        )
      }

      when (!pinvoke && (Attributes & NemerleAttributes.Extern != 0))
        Message.Error (Location, "method is marked `extern' but do not have DllImport attribute");

      // CompileTypedMethod.compile_parms (hd);
      
      def name_parms (pos, parms) {
        match (parms) {
          | [] => ()
          | (p : Fun_parm) :: ps =>
            p.builder = method_builder.DefineParameter (pos, parameter_attributes (p), p.name);

            match (p.default_value) {
              | Some (lit) => p.builder.SetConstant (lit.AsObject ());
              | None => {}
            }
            name_parms (pos + 1, ps)
        }
      };
      name_parms (1, GetParameters ());
    }

    /**
     * Adds a constructor builder to a type builder
     */
    CreateConstructorBuilder (tb : Emit.TypeBuilder) : void
    {
      // Message.Debug ($"-- AddConstructorBuilder: $this");

      /* create the constructor builder */
      ctor_builder =
        tb.DefineConstructor (make_method_attributes (Attributes) |
                              MethodAttributes.RTSpecialName |
                              MethodAttributes.SpecialName,
                              CallingConventions.Standard, param_types ());

      /* add the runtime modifiers for delegate constructors */
      when (DeclaringType.IsDelegate) {
        ctor_builder.SetImplementationFlags (
          MethodImplAttributes.Runtime | MethodImplAttributes.Managed
        )
      }

      def name_parms (pos, parms) {
        match (parms) {
          | [] => ()
          | (p : Fun_parm) :: ps =>
            p.builder = ctor_builder.DefineParameter (pos, parameter_attributes (p), p.name);

            match (p.default_value) {
              | Some (lit) => p.builder.SetConstant (lit.AsObject ());
              | None => {}
            }
            name_parms (pos + 1, ps)
        }
      };
      name_parms (1, GetParameters ());
    }

    internal override Compile () : void
    {
      unless (modifiers.IsEmpty) {
        def adder =
          if (fun_kind is FunKind.Constructor) 
            fun (target, attribute) {
              if (target %&& System.AttributeTargets.Method) {
                ctor_builder.SetCustomAttribute (attribute);
                null
              }
              else "constructor " + ToString ()
            }
          else
            fun (target, a) {
              if (target %&& System.AttributeTargets.Method) {
                method_builder.SetCustomAttribute (a);
                null
              }
              else "method " + ToString ()
            }
    
        modifiers.SaveCustomAttributes (declaring_type, adder);
      }
      
      RunBodyTyper ();
      CompileAfterTyping ()
    }
  }

  public partial class FieldBuilder : MemberBuilder 
  {
    /**
     * Adds a field builder
     */
    internal override CreateEmitBuilder (tb : Emit.TypeBuilder) : void
    {
      // Converts Nemerle modifiers to the CLI field attributes.
      def make_field_attributes (attrs)
      {
        mutable result = FieldAttributes.PrivateScope;

        when (attrs %&& NemerleAttributes.Public) result |= FieldAttributes.Public;
        when (attrs %&& NemerleAttributes.Private) result |= FieldAttributes.Private;
        
        if (attrs %&& NemerleAttributes.Internal)
          if (attrs %&& NemerleAttributes.Protected)
            result |= FieldAttributes.FamORAssem;
          else
            result |= FieldAttributes.Assembly
        else
          when (attrs %&& NemerleAttributes.Protected)
            result |= FieldAttributes.Family;
          
        when (attrs %&& NemerleAttributes.Static) result |= FieldAttributes.Static;
        when (attrs %&& NemerleAttributes.SpecialName) result |= FieldAttributes.SpecialName %|
            FieldAttributes.RTSpecialName;

        result
      }
      
      mutable attrs = make_field_attributes (Attributes);
      
      when (IsLiteral)
        attrs |= FieldAttributes.Literal;

      // prevent verification failure
      when (Name != "value__")
        attrs &= ~FieldAttributes.RTSpecialName;

      field_builder = tb.DefineField (Name, GetMemType ().SystemType, attrs);

      when (IsLiteral)
        field_builder.SetConstant (const_value.AsObject ());
    }

    internal override Compile () : void
    {
      unless (modifiers.IsEmpty) {
        modifiers.SaveCustomAttributes (declaring_type, fun (target, a) {
          if (target %&& System.AttributeTargets.Field) {
            field_builder.SetCustomAttribute (a);
            null
          }
          else "field " + ToString ()
        });
      }
      
      when (IsVolatile) {
        def volatile_attr = AttributeCompiler.MakeEmittedAttribute (SystemType.VolatileModifier);
        field_builder.SetCustomAttribute (volatile_attr)
      }

      unless (IsMutable) {
        def imm_attr = AttributeCompiler.MakeEmittedAttribute (SystemType.ImmutableAttribute);
        field_builder.SetCustomAttribute (imm_attr)
      }
    }
  }

  public partial class PropertyBuilder : MemberBuilder, IProperty
  {
    /**
     * Adds a property builder
     */
    internal override CreateEmitBuilder (tb : Emit.TypeBuilder) : void
    {
      // Converts Nemerle modifiers to the CLI property attributes.
      def make_property_attributes (attrs)
      {
        mutable result = PropertyAttributes.None;

        when (attrs %&& NemerleAttributes.SpecialName) 
          result |= PropertyAttributes.SpecialName %| PropertyAttributes.RTSpecialName;

        result
      }

      def parms = parms.MapToArray (fun (t : MType) { t.SystemType });
      
      def attrs = make_property_attributes (Attributes);
      def ty = GetMemType ().SystemType;

      property_builder = tb.DefineProperty (Name, attrs, ty, parms);
    }
    
    internal override Compile () : void
    {
      unless (modifiers.IsEmpty) {
        modifiers.SaveCustomAttributes (declaring_type, fun (target, a) {
          if (target %&& System.AttributeTargets.Property) {
            property_builder.SetCustomAttribute (a);
            null
          }
          else "property " + ToString ()
        })
      }
      
      def declaring = DeclaringType : object;

      // emit getter only if it was defined in this type, not derived  
      when (getter != null && getter.DeclaringType == declaring)
        property_builder.SetGetMethod (getter.GetMethodInfo ());

      // emit setter only if it was defined in this type, not derived  
      when (setter != null && setter.DeclaringType == declaring)
        property_builder.SetSetMethod (setter.GetMethodInfo ());
    }
  }

  public partial class EventBuilder : MemberBuilder, IEvent
  {
    /**
     * Adds an event builder
     */
    internal override CreateEmitBuilder (tb : Emit.TypeBuilder) : void
    {
      def ty = GetMemType ().SystemType;
      event_builder = tb.DefineEvent (Name, EventAttributes.None, ty);
    }

    internal override Compile () : void
    {
      unless (modifiers.IsEmpty) {
        modifiers.SaveCustomAttributes (declaring_type, fun (target, a) {
          if (target %&& System.AttributeTargets.Event) {
            event_builder.SetCustomAttribute (a);
            null
          }
          else "event " + ToString ()
        });
      }
      
      event_builder.SetAddOnMethod (adder.GetMethodInfo ());
      event_builder.SetRemoveOnMethod (remover.GetMethodInfo ());
    }
  }
}
