/*
 * Copyright (c) 2004, 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Matching
 */
using Nemerle.Collections;

using Nemerle.Compiler.Typedtree;
using Nemerle.Compiler.MatchingCompiler;

using System.Reflection;
using System.Reflection.Emit;

namespace Nemerle.Compiler 
{
  /**
   * A match effect
   */
  class MatchEffect
  {
    public this (effect : TExpr)
    {
      m_label_id = Util.next_id ();
      m_effect = TExpr.Label (effect.loc, effect.Type, m_label_id, effect);
    }


    public GetGoto () : TExpr.Goto
    {
      TExpr.Goto (m_label_id, 1)
    }

    
    public GetEffect () : TExpr
    {
      m_effect
    }

#if EXTRAMATCHINGDEBUG    
    public Dump () : void
    {
      Message.Debug ($ "MatchEffect::Dump: $m_label_id");
    }
#endif 
            
    private mutable m_label_id : int;
    private mutable m_effect : TExpr;
  }
  
  
  /**
   * Top-level pattern with a guard and effect index
   */
  class MatchClause
  {
    public this (pattern : Pattern, guard : TExpr, 
                 assigns : list [LocalValue * TExpr], effect : MatchEffect)
    {
      m_pattern = pattern;

      if (guard_is_always_true (guard))
        m_guard = None ()
      else
        m_guard = Some (guard);

      m_effect = effect;
      m_assigns = assigns;
    }
        
    public GetPattern () : Pattern
    {
      m_pattern
    }
    
    public Effect : MatchEffect
    {
      get { m_effect }
    }      

    private static guard_is_always_true (guard : TExpr) : bool
    {
      match (guard) {
        | TExpr.Literal (Literal.Bool (true)) => 
          true
          
        | _ =>
          false
      }
    }

    public Dump () : void
    {
      def dump_pattern (pattern : Pattern) 
      {      
        match (pattern) {
          | Pattern.Wildcard => 
            "Wildcard"

          | Pattern.As (as_pattern, decl) => 
            "As (" + dump_pattern (as_pattern) + ", '" + decl.Name + "')"

          | Pattern.Tuple (args) =>
            "Tuple {" + List.FoldLeft (
              args, "", 
              fun (tuple_pattern, desc : string) { 
                (if (desc.Length > 0) desc + ", " else "") + dump_pattern (tuple_pattern)
              }
            ) + "}"

          | Pattern.Record (args) =>
            "Record {" + List.FoldLeft (
              args, "", 
              fun (field_and_tuple_pattern, desc) {
                def (field, tuple_pattern) = field_and_tuple_pattern;
                                                
                (if (desc.Length > 0) desc + ", " else "") +
                 field.Name + ":" + dump_pattern (tuple_pattern)
              }
            ) + "}"

          | Pattern.Application (name, arg) =>
            "Application (" + name.FullName + ", " + dump_pattern (arg) + ")"

          | Pattern.Enum (fld, l) =>
            $ "Enum ($(fld.Name), $l)"
            
          | Pattern.Literal (l) =>
            $ "Literal ($l)"

          | Pattern.HasType (t) =>
            "HasType (" + t.FullName + ")"

          | Pattern.Error => "Error"
        }
      };
    
      Message.Debug ("MatchClause::Dump: " + dump_pattern (m_pattern))
    }


    /**
     * Builds a pattern-matcher expression
     */
    public BuildMatchExpr (val : TExpr, general_matcher : bool) : TExpr 
    {
      def pattern_expr =
        Util.locate (m_pattern.Location, 
                     build_pattern_expr (val, m_pattern, ! general_matcher));

      def guarded_expr =
        match (m_guard) {
          | Some (guard) =>
            If (pattern_expr, guard, TExpr.FalseLiteral)
          | None => pattern_expr
        }

      def with_assigns =
        if (m_assigns.IsEmpty) guarded_expr
        else {
          If (guarded_expr, 
              m_assigns.FoldRight (TExpr.TrueLiteral, fun (assign, acc) {
                def (name, value) = assign;
                Util.locate (value.loc,
                  TExpr.Sequence (InternalType.Boolean,
                    TExpr.Assign (InternalType.Void, 
                                  TExpr.LocalRef (name.Type, name),
                                  value),
                    acc))
              }),
              TExpr.FalseLiteral)
        }
      
      with_assigns
    }


    /**
     * Builds an expression that evaluates to true if the pattern is a value
     */
    private build_pattern_expr (val : TExpr, pattern : Pattern, is_top_level : bool) : TExpr
    {
      match (pattern) {
        | Pattern.Error => assert (false)

        | Pattern.Wildcard => 
          //FIXME: check for non-nullness of non-value types
          TExpr.TrueLiteral

        | Pattern.Literal (lit) =>
          MatchingCompiler.emit_compare_with (val, lit)
        
        | Pattern.HasType (ti) =>
          HasType (val, ti)

        | Pattern.Application (cons_name, Pattern.Wildcard) =>
          assert (! (cons_name.GetTydecl () is TypeDeclaration.Alias));

          if (is_top_level)
            // no nullness check is required here -- it is guaranteed by the effects switch
            TExpr.TrueLiteral
          else {
            match (get_constant_object (cons_name, pattern.Type)) {
              | Some ((from, field)) =>
                def sref = TExpr.StaticRef (from, from, field, []);
                TExpr.Call (InternalType.Boolean,
                            TExpr.OpCode ("==.ref"),
                            [Parm (val), Parm (sref)], false)
                
              | _ =>
                HasType (val, cons_name)
             }
          }
          
        | Pattern.Application (cons_name, subpattern) =>
          assert (! (cons_name.GetTydecl () is TypeDeclaration.Alias));
          
          def ty = cons_name.GetFreshType ();
          val.Type.ForceProvide (ty);

          def cast_expr = Cast (val, ty);

          if (is_top_level) {
            // no nullness check is required here -- it is guaranteed by the effects switch
            build_pattern_expr (cast_expr, subpattern, false)
          }
          else {
            def has_type_expr = HasType (val, cons_name);
                    
            If (has_type_expr,
                build_pattern_expr (cast_expr, subpattern, false),
                TExpr.FalseLiteral)
          }

        | Pattern.As (subpattern, decl) =>
          def assign_expr = 
            TExpr.Assign (TExpr.LocalRef (decl.Type, decl), Cast (val, decl.Type));

          match (build_pattern_expr (val, subpattern, false)) {
            | TExpr.If (c, e_then, e_else) =>
              If (c, Sequence (assign_expr, e_then), e_else)
            | expr =>
              If (expr, Sequence (assign_expr, TExpr.TrueLiteral), TExpr.FalseLiteral)
          }

        | Pattern.Tuple (args) =>
          def len = List.Length (args);

          def f (arg, acc)
          {
            def (pos, ex) = acc;
            def cond = 
              build_pattern_expr (TExpr.TupleIndexer (arg.Type, val, pos, len), arg, false);
                                  
            (pos - 1, If (cond, ex, TExpr.FalseLiteral))
          };

          def ini = (len - 1, TExpr.TrueLiteral);            
          def (_, ex) = List.FoldLeft (List.Rev (args), ini, f);
          ex
            
        | Pattern.Record (args) =>
          def val = Cast (val, pattern.Type);
            
          def f (np, ex) 
          {
            def (name, pat) = np;
            def cond = 
              build_pattern_expr (build_record_field_ref (pat.Type, val, name), pat, false);
              
            If (cond, ex, TExpr.FalseLiteral)
          };
          
          List.FoldLeft (List.Rev (args), TExpr.TrueLiteral, f)

        | Pattern.Enum => assert (false)
      }
    }
    
    private mutable m_pattern : Pattern;
    private mutable m_guard : option [TExpr];
    private mutable m_assigns : list [LocalValue * TExpr];
    private mutable m_effect : MatchEffect;
  }



  /* -- MATCH COLLECTOR ---------------------------------------------------- */
  
  /**
   * Collects match effects and clauses
   */
  class MatchCollector
  {
    public this (matchType : TyVar, cases : list [Match_case])
    {
      m_matchType = matchType;
      
      collect_cases (cases, [], [])
    }
  
    public GetClauses () : list [MatchClause]
    {
      m_clauses
    }

    public IsTopLevelMatchingOverCons () : bool
    {
      mutable seen_application = false;
      mutable seen_null = false;
      mutable count = 0;
      
      foreach (clause in m_clauses) {
        count++;
        match (clause.GetPattern ()) {
          | Application =>
            seen_application = true;
          | Literal (Null) =>
            seen_null = true;
          | _ => {}
        }
      }

      seen_application &&
      ! seen_null &&
      count > 10
    }
  
    public GetTopLevelConsName () : TypeInfo
    {
      assert (IsTopLevelMatchingOverCons ());

      def walk_clauses (clauses : list [MatchClause]) 
      {
        | [] => Util.ice ("MatchCollector::GetTopLevelConsName")
        | clause :: rest =>
          match (clause.GetPattern ()) {
            | Pattern.Application (name, _) => name
            | _ => walk_clauses (rest)
          } 
      };
      
      walk_clauses (m_clauses)
    }

    public BuildEffectsSwitch (matching_expr : TExpr) : TExpr
    {
      def ty = m_effects.Head.GetEffect ().Type;
      def end_label_id = Util.next_id ();
      def end_label = 
        TExpr.Label (ty, end_label_id, TExpr.Literal (Literal.Void ()));
        
      def effects = 
        m_effects.RevMap (fun (effect) {
          TExpr.Sequence (effect.GetEffect (),
                          TExpr.Goto (end_label_id, 1))
        });
      
      Typer3.BuildRevSequence (end_label :: effects + [matching_expr])
    }


    private collect_cases (cases : list [Match_case], 
                           effects : list [MatchEffect], 
                           clauses : list [MatchClause]) : void
    {
      match (cases) {
        | [] =>
          m_effects = List.Rev (effects);
          m_clauses = List.Rev (clauses)
          
        | case :: rest =>
          def new_effect = MatchEffect (case.body);
            
          def walk_patterns (patterns, 
                             new_clauses : list [MatchClause]) {
            match (patterns) {
              | [] => new_clauses // they will be reversed when all cases are processed
              | (pattern, guard, assigns) :: rest =>
                def new_clause =
                  MatchClause (
                    pattern, 
                    guard, 
                    assigns,
                    new_effect
                  );
                          
                walk_patterns (rest, new_clause :: new_clauses) 
            }                             
          };
    
          def new_clauses = walk_patterns (case.patterns, []);
                  
          collect_cases (
            rest, 
            new_effect :: effects, 
            new_clauses + clauses
          )
      }
    }
  
    private mutable m_effects : list [MatchEffect];
    private mutable m_clauses : list [MatchClause];
    private mutable m_matchType : TyVar;
  }



  /* ----------------------------------------------------------------------- */
  /* -- MATCHING PROBLEM CATEGORIZATION ------------------------------------ */
  /* ----------------------------------------------------------------------- */

  /**
   * Recognized matching problem categories
   */
  variant MatchProblemCategory {
    | Cons_list_like
    | Cons_tree_like
    | Cons_sparse
    | Cons_sparse_with_const
    | Cons_general
    | Other
  }


  /**
   * A set of heuristics used to categorize matching problems.
   */
  class MatchProblemCategorizer
  {
    public this (variant_options : list [TypeInfo],
                 match_collector : MatchCollector,
                 matched_value_type : TyVar)
    {
      m_variant_options = variant_options;
      m_match_collector = match_collector;
      this.matched_value_type = matched_value_type;
          
      categorize_match_problem ()
    }
    
    
    /**
     * Returns the category of the matching problem
     */
    public GetCategory () : MatchProblemCategory
    {
#if EXTRAMATCHINGDEBUG
      Message.Debug ("MatchProblemCategorizer::GetCategory: " + m_category.ToString ());
#endif

      m_category
    }
    
    
    /**
     * Returns the list of constant variants
     */
    public GetConstVariants () : list [TypeInfo * MType.Class * IField]
    {
      m_const_variants
    }


    /**
     * Returns the list of used const variants
     */
    public GetConstVariantsUsed () : list [TypeInfo * MType.Class * IField]
    {
      m_const_variants_used
    }


    /**
     * Returns the list of used non-const variants
     */
    public GetNonConstVariantsUsed () : list [TypeInfo]
    {
      m_non_const_variants_used    
    }
    
    
    /*
     * Calculates some statistics on the variant options. 
     * Collect the field infos of the const variants for later use.
     */
    private calculate_variant_statistics (variant_options : list [TypeInfo]) : void
    {
      match (variant_options) {
        | variant_option :: rest =>
          match (MatchingCompiler.get_constant_object (variant_option, matched_value_type)) {
            | Some ((from, const_variant_field_info)) => 
              m_const_variant_names =
                m_const_variant_names.Add (variant_option.FullName, 0);

              m_const_variants =
                (variant_option, from, const_variant_field_info) :: m_const_variants;

              ++m_const_variants_count
              
            | _ => ()
          };

          m_variant_names =
            m_variant_names.Add (variant_option.FullName, 0);
          
          m_variants_lookup =
            m_variants_lookup.Add (variant_option.FullName, variant_option);
          
          calculate_variant_statistics (rest)
          
        | [] => ()
      }
    }


    /*
     * Calculates some statistics on clauses
     */    
    private calculate_clause_statistics (clauses : list [MatchClause]) : void
    {
      def increase_occurence_count (name : string)
      {
        def occurence_count =
          m_variant_names.Get (name);
    
        m_variant_names =
          m_variant_names.Replace (name, occurence_count + 1);
      
        ++m_cons_patterns_count;
      };

      def increase_const_occurence_count (name : string)
      {
        def occurence_count =
          m_const_variant_names.Get (name);
    
        m_const_variant_names =
          m_const_variant_names.Replace (name, occurence_count + 1);
      
        ++m_const_cons_patterns_count;
      };
      
      def classify_pattern (pattern : Pattern) : void
      {
        match (pattern) {
          | Pattern.As (subpattern, _) => 
            classify_pattern (subpattern)
            
          | Pattern.Application (name, _) =>
            def name = 
              name.FullName;
                    
            increase_occurence_count (name);
            
            when (m_const_variant_names.Member (name))
              increase_const_occurence_count (name);
            
          | _ => ()
        }
      };
      
      match (clauses) {
        | clause :: rest =>
          ++m_patterns_count;
          classify_pattern (clause.GetPattern ());
                  
          calculate_clause_statistics (rest)
          
        | [] => ()      
      }
    }


    /*
     * Calculates how many variants have actually been used in clauses
     */
    private count_unique_variants_in_clauses () : void
    {
      def variant_iterator (key : string, value : int) : void
      {
        when (value != 0) {
          def variant_option =
            m_variants_lookup.Get (key);

          ++m_unique_cons_patterns_count;
          
          match (m_const_variant_names.Find (key)) {
            | Some (value) when value != 0 =>                
              def (variant_type, const_variant_field_info) =
                Option.UnSome (MatchingCompiler.get_constant_object (variant_option, matched_value_type));
                
              m_const_variants_used =
                (variant_option, variant_type, const_variant_field_info) :: m_const_variants_used;
            
              ++m_unique_const_cons_patterns_count
              
            | _ =>
              m_non_const_variants_used =
                variant_option :: m_non_const_variants_used
          }
        }
      };
      
      m_variant_names.Iter (variant_iterator)
    }


    /*
     * Applies some heuristics to categorize the matching problem
     */    
    private categorize_match_problem () : void
    {
      // step one: count the variants
      m_variants_count = List.Length (m_variant_options);
      
      calculate_variant_statistics (m_variant_options);

      // step two: count the patterns      
      calculate_clause_statistics (m_match_collector.GetClauses ());
      
      // step three: check which variants have actually been used
      count_unique_variants_in_clauses ();
      
      // step four: given above statistics, decide the category of the problem
      m_category =
        if (m_variants_count == 0 || m_cons_patterns_count == 0)
          MatchProblemCategory.Other ()
        else {
          if (m_variants_count <= 2 && m_const_variants_count >= 1)
            MatchProblemCategory.Cons_list_like ()
          else if (m_variants_count == 3 && m_const_variants_count >= 1)
            MatchProblemCategory.Cons_tree_like ()
          else if (m_unique_cons_patterns_count <= 3 && 
                   m_unique_const_cons_patterns_count >= 1)
            MatchProblemCategory.Cons_sparse_with_const ()
          else if (m_unique_cons_patterns_count <= 2)
            MatchProblemCategory.Cons_sparse ()
          else
            MatchProblemCategory.Cons_general ()
        }      
    }


    // input data        
    private mutable m_variant_options : list [TypeInfo];
    private mutable m_match_collector : MatchCollector;
    private matched_value_type : TyVar;

    // pattern statistics
    private mutable m_patterns_count : int;
    private mutable m_cons_patterns_count : int;
    private mutable m_const_cons_patterns_count : int;
    
    // unique variants in patterns statistics
    private mutable m_unique_cons_patterns_count : int;
    private mutable m_unique_const_cons_patterns_count : int;
    
    // variants statistics
    private mutable m_variants_count : int;
    private mutable m_const_variants_count : int;

    // names of the variants and their usage counts
    private mutable m_variant_names : SystemMap [string, int] = SystemMap ();
    private mutable m_const_variant_names : SystemMap [string, int] = SystemMap ();
    
    // the list of constant variant options
    private mutable m_const_variants : list [TypeInfo * MType.Class * IField] = [];
    
    // variants lookup table
    private mutable m_variants_lookup : SystemMap [string, TypeInfo] = SystemMap ();

    // the lists of constant/non-constant variants that actually have been used
    private mutable m_non_const_variants_used : list [TypeInfo] = [];
    private mutable m_const_variants_used : list [TypeInfo * MType.Class * IField] = [];
    
    // the calculated category of the matching problem
    private mutable m_category : MatchProblemCategory;
  }
  

  
  /* ----------------------------------------------------------------------- */
  /* -- MATCHING OVER VARIANT CONSTRUCTORS CODE GENERATOR ------------------ */
  /* ----------------------------------------------------------------------- */

  /**
   * Matching over variant constructors code generator
   */
  class MatchingOverVariantCons
  {
    public this (val : TExpr, match_collector : MatchCollector)
    {
      m_val = val;
      m_match_collector = match_collector;
    
      def (variant_options, variant_code_method_info) =
        fetch_variant_options ();

      m_variant_options = variant_options;
  
      m_variant_code_call_expr =
        TExpr.Call (InternalType.Int32, 
                    TExpr.MethodRef (variant_code_method_info.GetMemType (),
                                     val, variant_code_method_info, [], false), 
                    [], false);

      m_match_categorizer =
        MatchProblemCategorizer (variant_options, match_collector, val.Type);
        
      match (m_match_categorizer.GetCategory ()) {
        | MatchProblemCategory.Cons_list_like => 
          create_list_like_matcher ()

        | MatchProblemCategory.Cons_sparse_with_const
        | MatchProblemCategory.Cons_sparse =>
          create_sparse_matcher ()

        | _ => create_general_matcher ()
      }
    }


    /**
     * Returns the resulting matcher
     */
    public GetMatcher () : TExpr
    {
      m_matcher
    }


    /**
     * Returns the method info for the _N_GetVariantCode method defined in a given tycon
     */
    private static get_variant_code_method (variant_tycon : TypeInfo) : IMethod
    {
      def members = variant_tycon.LookupMember ("_N_GetVariantCode");
      
      match (members) {
        | [variant_code_method] =>
          variant_code_method :> IMethod

        | _ =>
          Util.ice ("MatchingOverVariantCons::get_variant_code_method: expected "
                    "exactly one _N_GetVariantCode method")
      }      
    }


    /**
     * Returns the list of variants we're matching over at the top level
     * and the info on the _N_GetVariantCode method.
     */
    private fetch_variant_options () : list [TypeInfo] * IMethod
    {
      def name = 
        m_match_collector.GetTopLevelConsName ();
        
      def variant_tycon = name.BaseType;

      def variant_options =
        match (variant_tycon.GetTydecl ()) {
          | TypeDeclaration.Variant (members) => members
          | _ => Util.ice ("wrong tydecl in variant")
        };              

      (variant_options, get_variant_code_method (variant_tycon))
    }


    /**
     * Returns a list of clauses that have to do with a variant option
     */
    private collect_variant_related_clauses (variant_option : TypeInfo) : list [MatchClause]
    {
      def variant_full_name = variant_option.FullName;
        
      def walk_clauses (clauses : list [MatchClause], 
                        acc : list [MatchClause]) : list [MatchClause]
      {
        match (clauses) {
          | [] => List.Rev (acc)
            
          | clause :: rest =>
            match (clause.GetPattern ()) {
              | Pattern.As (Pattern.Application (name, _), _)
              | Pattern.Application (name, _) =>
                if (name.FullName == variant_full_name)
                  walk_clauses (rest, clause :: acc)
                else
                  walk_clauses (rest, acc);

              | Pattern.As (Pattern.Wildcard, _) | Pattern.Wildcard  =>
                walk_clauses (rest, clause :: acc)

              | Pattern.As (Pattern.HasType (ty), _) | Pattern.HasType (ty) =>
                if (ty.Equals (variant_option))
                  walk_clauses (rest, clause :: acc)
                else
                  walk_clauses (rest, acc);

              | Pattern.Literal (Literal.Null) =>
                Message.FatalError ("sorry, null patterns are not yet supported in mix with variant options");
                //walk_clauses (rest, clause :: acc)
                 
              | _ =>
                clause.Dump ();
                Util.ice ("MatchingOverVariantCons::collect_variant_related_clauses: "
                          "wrong pattern (see the dump above)")
            }
        }
      };
      
      walk_clauses (m_match_collector.GetClauses (), [])
    }


    /**
     * Returns a list of clauses that do not have anything in common with a variant option
     */
    private collect_variant_unrelated_clauses () : list [MatchClause]
    {
      def walk_clauses (clauses : list [MatchClause], acc)
      {
        match (clauses) {
          | [] => List.Rev (acc)
          | clause :: rest =>
            match (clause.GetPattern ()) {
              | Pattern.As (Pattern.Application, _) | Pattern.Application =>
                walk_clauses (rest, acc);

              | Pattern.As (Pattern.Wildcard, _) | Pattern.Wildcard =>
                walk_clauses (rest, clause :: acc)

              | Pattern.As (Pattern.HasType, _) | Pattern.HasType =>
                walk_clauses (rest, clause :: acc);
                
              | _ =>
                clause.Dump ();
                Util.ice ("MatchingOverVariantCons::collect_variant_unrelated_clauses: "
                          "wrong pattern (see the dump above)")
            }
        }
      };
      walk_clauses (m_match_collector.GetClauses (), [])
    }


    /**
     * The last step of matcher generation: wraping the matcher
     * expression with TExpr.DefValIn bindings of Pattern.As-induced variables
     * is done in AddDefines at the global level.
     */
    private finalize_matcher (matcher : TExpr) : void
    {
      m_matcher = matcher
    }
    

    /**
     * Binds clauses, creating an expression that evaluates to
     * an integer denoting the effect of each of the clauses.
     * We can perform a TExpr.Switch over that later.
     */
    private bind_clauses_for_switch (clauses : list [MatchClause]) : TExpr
    {
      match (clauses) {          
        | clause :: rest =>
          def clause_expr = clause.BuildMatchExpr (m_val, false);
            
          If (clause_expr,
              clause.Effect.GetGoto (),
              bind_clauses_for_switch (rest))

        | [] => ThrowMatchFailure ()
      }          
    }

    
    /**
     * Creates a sparse matcher for top-level variants.
     */
    private create_sparse_matcher () : void
    {
      // collects and bind clauses, builds the corresponding TExpr.HasType expression
      def process_variant (some_variant : TypeInfo) : TExpr * TExpr
      {
        def some_clauses =
          collect_variant_related_clauses (some_variant);

        def some_clauses_effect_expr =
          bind_clauses_for_switch (some_clauses);

        assert (! (some_variant.GetTydecl () is TypeDeclaration.Alias));
        
        def some_has_type_expr =
          HasType (m_val, some_variant);
          
        (some_has_type_expr, some_clauses_effect_expr)
      };

      // get the matchers for the variant unrelated clauses
      def unrelated_clauses = collect_variant_unrelated_clauses ();

      def unrelated_clauses_effect_expr =
        match (unrelated_clauses) {
          | [] => ThrowMatchFailure ()
          | _ => bind_clauses_for_switch (unrelated_clauses)
        };
            
      // get the matchers for the variant options used
      def fold_non_const_variants (some_variant : TypeInfo, matcher_expr_acc : TExpr) : TExpr
      {
        def (some_has_type_expr, some_clauses_effect_expr) =
          process_variant (some_variant);

        If (some_has_type_expr,
          some_clauses_effect_expr,
          matcher_expr_acc
        )        
      };
      
      def non_const_matcher_expr =
        List.FoldLeft (
          m_match_categorizer.GetNonConstVariantsUsed (), 
          unrelated_clauses_effect_expr,
          fold_non_const_variants
        );

      // build the comparers for the const variants
      def fold_const_variants (const_cons : TypeInfo * MType.Class * IField, matcher_expr_acc : TExpr) : TExpr
      {
        def (some_variant, from, const_cons_field_info) = 
          const_cons;
        
        def (_, some_clauses_effect_expr) =
          process_variant (some_variant);

        def compare_to_const_expr =
          TExpr.Call (InternalType.Boolean, TExpr.OpCode ("==.ref"),
                  [Parm (m_val), Parm (TExpr.StaticRef (from, from,
                                                        const_cons_field_info, []))], false);
            
        If (compare_to_const_expr,
            some_clauses_effect_expr,
            matcher_expr_acc);
      };

      def matcher_expr =
        List.FoldLeft (
          m_match_categorizer.GetConstVariantsUsed (),
          Sequence (NotNull (m_val), non_const_matcher_expr),
          fold_const_variants
        );

      // builds the effects switch and wrap the matcher with appropriate TExpr.Defines
      def effects_switch = 
        m_match_collector.BuildEffectsSwitch (matcher_expr);

      finalize_matcher (effects_switch)      
    }


    /**
     * Creates a matcher for top-level list-like variants
     *
     * NOTE: we could have two constant constructors here
     */
    private create_list_like_matcher () : void
    {
      // get the tycons for the variant options
      def (const_cons_tycon, from, const_cons_field_info) =
        match (m_match_categorizer.GetConstVariants ()) {
          | const_cons :: _ => const_cons
          | _ => Util.ice ("MatchingOverVariantCons::create_list_like_matcher: "
                           "expected at least one constant variant constructor")
        };

      def non_const_cons_tycon =
        match (m_variant_options) {
          | [left, right] => if (left.Equals (const_cons_tycon)) right else left
          | x :: _ => x
          | _ => Util.ice ("MatchingOverVariantCons::create_list_like_matcher: "
                           "expected some variant options")
        };

      // categorize the clauses, build the effects expressions
      def const_clauses =
        collect_variant_related_clauses (const_cons_tycon);

      def const_clauses_effect_expr =
        bind_clauses_for_switch (const_clauses);
        
      def non_const_clauses =
        collect_variant_related_clauses (non_const_cons_tycon);

      def non_const_clauses_effect_expr =
        bind_clauses_for_switch (non_const_clauses);

      // build the variant checking expression
        def compare_to_const_expr =
          TExpr.Call (InternalType.Boolean, TExpr.OpCode ("==.ref"),
                  [Parm (m_val), Parm (TExpr.StaticRef (from, from,
                                                        const_cons_field_info, []))], false);

      def non_const_check_expr =
        Sequence (NotNull (m_val), non_const_clauses_effect_expr);
            
      def const_check_expr =
        If (compare_to_const_expr,
            const_clauses_effect_expr,
            non_const_check_expr);

      // builds the effects switch and wrap the matcher with appropriate TExpr.Defines
      def effects_switch = 
        m_match_collector.BuildEffectsSwitch (const_check_expr);

      finalize_matcher (effects_switch)      
    }

    /**
     * Builds a general matcher over constructors on the top level
     */
    private create_general_matcher () : void
    {
      def walk_variant_options (index : int, variant_options : list [TypeInfo]) : list [int * TExpr]
      {
        match (variant_options) {
          | variant_option :: rest =>
#if EXTRAMATCHINGDEBUG
            Message.Debug (index.ToString () + ": " + variant_option.FullName);
#endif

            def patterns =
              collect_variant_related_clauses (variant_option);

            def effect_expr =
              bind_clauses_for_switch (patterns);
            
            (index, effect_expr) :: walk_variant_options (index + 1, rest)
            
          | [] => []
        }
      };

#if EXTRAMATCHINGDEBUG
      Message.Debug ("walking variant options of type " + variant_tyinfo.FullName);
#endif

      def cons_switch_cases =
        walk_variant_options (0, m_variant_options);
        
      def cons_switch =
        TExpr.Switch (
          InternalType.Int32,
          m_variant_code_call_expr,
          Some (ThrowMatchFailure ()),
          cons_switch_cases
        );
        
      def effects_switch = 
        m_match_collector.BuildEffectsSwitch (cons_switch);

      def guarded_effects_switch =
//        if (contains_nullcheck ()) effects_switch
//        else
          Sequence (NotNull (m_val), effects_switch);

      finalize_matcher (guarded_effects_switch)
    }  


    /* -- PRIVATE FIELDS --------------------------------------------------- */

    private mutable m_val : TExpr;
    private mutable m_match_collector : MatchCollector;
    private mutable m_variant_options : list [TypeInfo];
    private mutable m_variant_code_call_expr : TExpr;
    private mutable m_match_categorizer : MatchProblemCategorizer;
    
    // the resulting matcher
    private mutable m_matcher : TExpr; 
  }




  /* ----------------------------------------------------------------------- */  
  /* --  OLD APPROACH ------------------------------------------------------ */
  /* ----------------------------------------------------------------------- */

  /**
   * The matching module
   */
  partial module MatchingCompiler 
  {
    // handle both IField and IProperty
    internal build_record_field_ref (ty : TyVar, val : TExpr, member : IMember) : TExpr
    {
      match (member) {
        | field is IField =>
          TExpr.FieldMember (ty, val, field)
          
        | prop is IProperty =>
          def meth = prop.GetGetter ();
          def methref =
            TExpr.MethodRef (meth.GetMemType (), val, meth, [], false);
          TExpr.Call (ty, methref, [], false)

        | _ => Util.ice ()
      }
    }

    
    /**
     * Emit comparison code with specific literal.
     */
    public emit_compare_with (compare_to : TExpr, literal : Literal) : TExpr
    {
      def method =
        match (literal) {
          | Literal.String => 
            def meth = InternalType.String_opEquality;
            TExpr.StaticRef (meth.GetMemType (), InternalType.String, meth, [])
          | Literal.Decimal =>
            def meth = InternalType.Decimal_opEquality;
            TExpr.StaticRef (meth.GetMemType (), InternalType.Decimal, meth, [])
          | _ => TExpr.OpCode ("==") 
        };
      assert (literal != null);
      def (compare_to, lit) =
        if (compare_to.SystemType.IsValueType)
          (compare_to,
           Cast (TExpr.Literal (literal), compare_to.Type))
        else 
          (Cast (compare_to, literal.GetInternalType ()),
           TExpr.Literal (literal));
      TExpr.Call (InternalType.Boolean, method, [Parm (compare_to), Parm (lit)], false)
    }

    cg_pattern (val : TExpr, pats : list [Pattern * TExpr * list [LocalValue * TExpr]]) : TExpr
    {
      def cgpat (val, pat, guard, assigns) {
        def clause = MatchClause (pat, guard, assigns, null);
        clause.BuildMatchExpr (val, true)
      }

      def loop (pats) {
        match (pats) {
          | (pat : Pattern, expr, assigns) :: rest =>
            def cond = Util.locate (pat.Location, cgpat (val, pat, expr, assigns));
            If (cond, TExpr.TrueLiteral, loop (rest))
            
          | [] => TExpr.FalseLiteral
        }
      }

      loop (pats)
    }

    
    /**
     * Matching over arbitrary types
     */    
    cg_match_general (val : TExpr, mcs : list [Match_case]) : TExpr 
    {
      match (mcs) {
        | [] =>
          ThrowMatchFailure ()

        | m :: ms =>
          def cond = cg_pattern (val, m.patterns);
          def then_part = m.body;
          def else_part = cg_match_general (val, ms);
          If (cond, then_part, else_part)
      }
    }
    
    
    /**
     * Returns the constant object for a parameterless variant constructor
     */
    public get_constant_object (variant_tycon : TypeInfo, 
                                matched_value_type : TyVar) : option [MType.Class * IField]
    {
      def members = 
        variant_tycon.LookupMember ("_N_constant_object");
      
      match (members) {
        | [] => None ()
        
        | [constant_object_field is IField] =>
          def ty = variant_tycon.GetFreshType ();
          ty.ForceRequire (matched_value_type);
          Some ((ty, constant_object_field))

        | _ =>
          Util.ice ("get_constant_object: expected zero or exactly one _N_constant_object fields")
      }          
    }
           

    /**
     * Matching over arbitrary types
     */
    cg_match_over_any_type (t : TyVar, val : TExpr, mcs : list [Match_case]) : TExpr 
    {
      def match_collector = MatchCollector (t, mcs);
        
      if (match_collector.IsTopLevelMatchingOverCons ()) {
        def matcher_generator =
          MatchingOverVariantCons (val, match_collector);
          
        matcher_generator.GetMatcher ()
      }
      else
        cg_match_general (val, mcs)        
    }


    /**
     * Matching over booleans.
     */
    cg_match_over_booleans (val : TExpr, mcs : list [Match_case]) : TExpr
    {
      mutable true_handled = false;
      mutable false_handled = false;
      mutable last_pattern_always_matches = false;

      def is_always_true (expr : TExpr) : bool {
        match (expr) {
          | TExpr.Literal (Literal.Bool (true)) => true
          | _ => false
        }    
      };

      def make_cond (patterns : list [Pattern * TExpr * list [LocalValue * TExpr]]) : TExpr {
        match (patterns) {        
          | (Pattern.Wildcard, when_expr, _) :: _ when is_always_true (when_expr) =>
            last_pattern_always_matches = true;          
            TExpr.TrueLiteral
        
          | (Pattern.Wildcard, when_expr, _) :: rest =>
            If (when_expr, TExpr.TrueLiteral, make_cond (rest))

          | (Pattern.Literal (Literal.Bool (literal)), when_expr, _) :: rest
            when is_always_true (when_expr)
            =>
            def literal_already_handled = if (literal) true_handled else false_handled;   
            when (literal_already_handled)
              Message.Warning ("unreachable pattern in matching");
            
            if (literal) true_handled = true else false_handled = true;
                      
            if (true_handled && false_handled) {
              last_pattern_always_matches = true;          
              TExpr.TrueLiteral
            }
            else {
              if (literal) 
                If (val, TExpr.TrueLiteral, make_cond (rest))
              else
                If (val, make_cond (rest), TExpr.TrueLiteral)
            }

          | (Pattern.Literal (Literal.Bool (literal)), when_expr, _) :: rest =>
            def literal_already_handled = if (literal) true_handled else false_handled;
            when (literal_already_handled)
              Message.Warning ("unreachable pattern in matching");

            def opposite_handled = if (literal) false_handled else true_handled;

            if (opposite_handled) {
              If (when_expr, TExpr.TrueLiteral, make_cond (rest))          
            }
            else {
              def not_handled = If (when_expr, TExpr.TrueLiteral, make_cond (rest));
              if (literal)
                If (val, TExpr.TrueLiteral, not_handled)
              else                              
                If (val, not_handled, TExpr.TrueLiteral) 
            }

          | (Pattern.As (Pattern.Wildcard, decl), when_expr, _) :: _
            when is_always_true (when_expr) =>
            last_pattern_always_matches = true;
            def assign_expr = 
              TExpr.Assign (TExpr.LocalRef (decl.Type, decl), val);
            assign_expr

          | [] => TExpr.FalseLiteral

          | (Pattern.Tuple, _, _) :: _ =>
            Util.ice ("cg_match_over_booleans::make_cond: unexpected pattern kind Pattern.Tuple")
          | (Pattern.Record, _, _) :: _ =>
            Util.ice ("cg_match_over_booleans::make_cond: unexpected pattern kind Pattern.Record")
          | (Pattern.Application, _, _) :: _ =>
            Util.ice ("cg_match_over_booleans::make_cond: unexpected pattern kind Pattern.Application")

          | _ :: _ => 
            Util.ice ("cg_match_over_booleans::make_cond: unexpected pattern kind")
        }      
      };

      def walk_match_cases (mcs : list [Match_case]) : TExpr {
        match (mcs) {
          | [] =>
            ThrowMatchFailure ()

          | mc :: rest =>
            def cond = make_cond (mc.patterns);
            def body = mc.body;
            
            if (true_handled && false_handled || last_pattern_always_matches) {
              // should be handled already anyway, and here there is no way
              // to disable this warning (from @foreach)
              // unless (rest matches []) Message.Warning ("unreachable match cases");
              match (cond) {
                | TExpr.Assign =>
                  Sequence (cond, body)
                | _ => body
              }
            }
            else
              If (cond, body, walk_match_cases (rest));
        }
      };

      // handle the special cases for the if/when/unless macros
      match (mcs) {
        | [then_case, else_case] =>
          match ((then_case.patterns, else_case.patterns)) {
            | ([(Pattern.Literal (Literal.Bool (literal)), then_case_when_expr, [])],
               [(Pattern.Wildcard, else_case_when_expr, [])]) 
              when is_always_true (then_case_when_expr) &&
                   is_always_true (else_case_when_expr) =>
              if (literal)
                If (val, then_case.body, else_case.body)
              else
                If (val, else_case.body, then_case.body);
                                              
            | _ => walk_match_cases (mcs)
          }
        | _ => walk_match_cases (mcs)
      }
    }

    internal If (cond : TExpr, e1 : TExpr, e2 : TExpr) : TExpr
    {
      TExpr.If (e1.Type, cond, e1, e2)
    }

    internal Cast (expr : TExpr, ty : TyVar) : TExpr
    {
      TExpr.TypeConversion (ty, expr, ty.Fix (), ConversionKind.IL (true))
    }

    internal Sequence (e1 : TExpr, e2 : TExpr) : TExpr
    {
      TExpr.Sequence (e1.loc, e2.Type, e1, e2)
    }

    internal NotNull (e : TExpr) : TExpr
    {
      TExpr.NotNull (InternalType.Void, e)
    }

    internal HasType (e : TExpr, ti : TypeInfo) : TExpr
    {
      match (ti.GetFreshType ()) {
        | MType.Class (_, []) as ty =>
          TExpr.HasType (e, ty)
        | ty =>
          e.Type.ForceProvide (ty);
          TExpr.HasType (e, ty)
      }
    }

    internal ThrowMatchFailure () : TExpr
    {
      def ctor = InternalType.NewMatchFailureException_ctor;
      TExpr.Throw (Solver.FreshTyVar (),
                   TExpr.Call (InternalType.MatchFailureException,
                               TExpr.StaticRef (ctor.GetMemType (), 
                                                InternalType.MatchFailureException, 
                                                ctor, []), [], false))
    }


    /**
     * Matching
     */
    public Run (t : TyVar, val : TExpr,
                mcs : list [Match_case]) : TExpr
    {
      // here we choose whether to use some optimized matching algorithm or
      // just most general one
      if (val.Type.Equals (InternalType.Boolean) &&
          mcs.ForAll (fun (mc) { mc.patterns.ForAll (fun (_, _, l) { l.IsEmpty }) }))
        cg_match_over_booleans (val, mcs)
      else
        cg_match_over_any_type (t, val, mcs)
    }
   
  } // end of the module

} // end of the namespace
