/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * This is the main compiler file. But all it does it setting options
 * for compilation (CompilerOptions.n) and transferring control to
 * Passes.Run function (from passes.n).
 *
 * Main file also catches some exceptions.
 */

using Nemerle.Collections;
using Nemerle.Utility;
using Nemerle.IO;
using Nemerle.Compiler;

[assembly: System.Reflection.AssemblyTitle("Nemerle Language Compiler")]
[assembly: System.Reflection.AssemblyDescription("Nemerle (http://nemerle.org) Language Compiler")]
[assembly: System.Reflection.AssemblyCompany("University of Wroclaw")]
[assembly: System.Reflection.AssemblyProduct("Nemerle Language Compiler")]
[assembly: System.Reflection.AssemblyCopyright("Copyright @ University of Wroclaw 2003-2005")]

[assembly: System.Reflection.AssemblyVersion("0.9.2")]
[assembly: System.Reflection.AssemblyKeyFile("../misc/keys/Nemerle.Compiler.snk")]

namespace Nemerle.CommandlineCompiler
{
  module MainClass
  {
    mutable stack_kilos : int = 0;
    
    public Main () : void
    {
      parse_command_line ();
      if (stack_kilos != 0 || needs_bigger_stack ()) {
        when (stack_kilos == 0)
          stack_kilos = 16 * 1024;
        def thread = System.Threading.Thread (main_with_catching, stack_kilos * 1024);
        thread.Start ();
        thread.Join ();
      } else
        main_with_catching ()
    }
    
    needs_bigger_stack () : bool
    {
      typeof (object).Assembly.GetType ("System.RuntimeType") != null
    }

    main_with_catching () : void
    {
      try {
        def this_dir = typeof (MainClass).Assembly.Location;
        LibraryReferenceManager.AddSearchDirectory
          (LibraryReferenceManager.DirectoryOfCodebase (this_dir));
        
        // run compilation with already created options
        Passes.Run ()
      } catch {
        | e is System.IO.FileNotFoundException =>
          Message.Error (e.Message)
        | e is Recovery =>
          bomb (e, "got Recovery exception")
        | e is BailOutException =>
          bomb (e, "got bail out exception")
        | e is System.ArgumentException =>
          bomb (e, "got ArgumentException (" + e.Message + ")")
        | e is MatchFailureException =>
          bomb (e, "got MatchFailureException exception")
        | e is ICE =>
          bomb (e, e.msg)
        | e is AssertionException =>
          bomb (e, e.Message)
        | _ is AssemblyFindException =>
          System.Environment.Exit (3);
        | e =>
          bomb (e, $ "got some unknown exception of type $(e.GetType()): $(e.Message)")
      };

      Message.MaybeBailout();
    }
    
    bomb (e : System.Exception, msg : string) : void
    {
      Passes.KillProgressBar ();
      Message.MaybeBailout (true);
      Message.Error (sprintf ("internal compiler error: %s\n%s\n", msg, e.StackTrace));
      System.Environment.Exit (2);
    }

    parse_command_line () : void
    {
      Message.InitOutput (System.Console.Out);

      mutable files = [];

      def print_version () {
        System.Console.Error.Write ("Nemerle Compiler (ncc) version 0.9.2 (release)\n"
                                    "(c) 2003-2005 University of Wroclaw,"
                                    " All rights reserved.\n");
        System.Environment.Exit (0);
      };

      mutable help_opts = [];                  
      def print_help () {
        System.Console.WriteLine (Getopt.Usage (help_opts));
        System.Environment.Exit (0);
      };

      def opts = Options.GetCommonOptions () + [
        Getopt.CliOption.Int (name = "-stack-size",
                     aliases = [],
                     help = "Set stack size of the compiler to INT megabytes",
                     handler = fun (k) { stack_kilos = k * 1024; }),
        Getopt.CliOption.Int (name = "-stack-size-k",
                     aliases = [],
                     help = "NOHELP",
                     handler = fun (k) { stack_kilos = k; }),
        Getopt.CliOption.Flag (name = "-version",
                     aliases = ["-V"],
                     help = "Output version information and exit",
                     handler = print_version),
        Getopt.CliOption.Flag (name = "-help",
                     aliases = ["-h"],
                     help = "Display this usage message and exit",
                     handler = print_help),
        Getopt.CliOption.NonOption (name = "",
                          help = "Specify file to compile",
                          handler = fun (s) { files = s :: files })
      ];
      help_opts = opts;

      Getopt.Parse (opts);

      match (files) {
        | [] =>
          Getopt.Error ("need at least one file to compile\n" + Getopt.Usage (opts))
        | _ =>
          Options.Sources = files;
      }
    }
  }
}
