/*
 * Copyright (c) 2004-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle;
using Nemerle.Collections;
using Nemerle.Logging;
using System.Reflection;
using System;

using Nemerle.Compiler.SolverMacros;

//#define VERB

#if VERB
[assembly: LogFlag (SOLVER, true)]
#else
[assembly: LogFlag (SOLVER, false)]
#endif

namespace Nemerle.Compiler 
{
  /** Represents a node in constraint graph. Used during type
      inference. */
  public class TyVar : Nemerle.IComparable [TyVar]
  {
    #region PUBLIC interface
    /** Require [this] to be at least [t].
    
        To be called when we require some lower constraint on type
        variable.  Return [true] iff it's possible.  */
    public virtual Require (t : TyVar) : bool
    {
      if (t.IsFixed)
        Require (t.FixedValue)
      else if (IsFixed)
        t.Provide (FixedValue)
      else
        AddRelation (t, this)
    }


    public TryRequire (t : TyVar) : bool
    {
      Passes.Solver.PushState ();
      def res = this.Require (t);
      Passes.Solver.PopState ();
      res
    }


    /** This is much like [assert (Require (t))], but is executed regardless
        of asserts and ignores the result if there was already a
        [LocalError].  */
    public ForceRequire (t : TyVar) : void
    {
      if (LocalError)
        _ = Require (t)
      else {
        def ok = Require (t);
        Util.cassert (ok, $ "failed to require $t from $this");
      }
    }


    /** Require [this] to be at least [t].
    
        To be called when we require some lower constraint on type
        variable.  Return [true] iff it's possible.  */
    public virtual Require (t : MType) : bool
    {
      if (IsFixed)
        Self.Require (t)
      else
        AddRelation (t, this, rev = false)
    }

    
    /** Provide type [t] as the maximal type for [this].
        
        To be called when value of type [t] is assigned to cell of type
        [this]. Dual to [Require]. */
    public virtual Provide (t : TyVar) : bool
    {
      if (t.IsFixed)
        Provide (t.FixedValue)
      else if (IsFixed)
        t.Require (FixedValue)
      else
        AddRelation (this, t)
    }

    public TryProvide (t : TyVar) : bool
    {
      Passes.Solver.PushState ();
      def res = this.Provide (t);
      Passes.Solver.PopState ();
      res
    }

    public ForceProvide (t : TyVar) : void
    {
      if (LocalError)
        _ = Provide (t)
      else {
        def ok = Provide (t);
        Util.cassert (ok, $ "failed to provide $t to $this");
      }
    }
    
    /** Provide type [t] as the maximal type for [this].
        
        To be called when value of type [t] is assigned to cell of type
        [this]. Dual to [Require]. */
    public virtual Provide (t : MType) : bool
    {
      if (IsFixed)
        Self.Provide (t)
      else
        AddRelation (t, this, rev = true)
    }


    /** Make sure [t] and [this] will from now on always represent the 
        same type. */
    [ForwardThis (Self)]
    [PossiblyLooping (Passes.Solver)]
    public Unify (t : TyVar) : bool
    {
      def t = t.Self;
      
      log (SOLVER, $ "UnifyTV: $(this) =? $t");
      
      if (IsFixed)
        t.Unify (FixedValue)
      else if (t.IsFixed)
        Unify (t.FixedValue)
      else if (IsFresh) {
        this.Alias (t)
      } else if (t.IsFresh) {
        t.Alias (this)
      } else {
        assert (IsFree);
        assert (t.IsFree);
        Provide (t) && Require (t)
      }
    }

    public TryUnify (t : TyVar) : bool
    {
      Passes.Solver.PushState ();
      def res = this.Unify (t);
      Passes.Solver.PopState ();
      res
    }
    
    /** Make sure [t] and [this] will from now on always represent the 
        same type. */
    [ForwardThis (Self)]
    public virtual Unify (t : MType) : bool
    {
      log (SOLVER, $ "Unify: $(this) =? $t");
      if (IsFixed)
        FixedValue.Unify (t)
      else if (IsFresh) {
        Alias (t)
      } else {
        assert (IsFree);
        when (Alias (t)) {
          when (lower_bound != null)
            _ = t.Require (lower_bound);

          when (upper_bound != null)
            _ = t.Provide (upper_bound);

          SetLowerBound (t);
          SetUpperBound (t);
        }

        !LocalError
      }
    }


    public ForceUnify (t : TyVar) : void
    {
      if (LocalError)
        _ = Unify (t)
      else {
        def ok = Unify (t);
        Util.cassert (ok, $ "failed to unify $t and $this");
      }
    }


    public LowerBound : option [MType]
    {
      get {
        def b = Self.lower_bound;
        if (b == null) None ()
        else Some (b)
      }
    }


    public UpperBound : option [MType]
    {
      get {
        def b = Self.upper_bound;
        if (b == null) None ()
        else Some (b)
      }
    }


    public Hint : option [MType]
    {
      get {
        def s = Self;
        def ub = s.upper_bound;
        def lb = s.lower_bound;

        if (ub == null)
          if (lb == null || lb.Equals (InternalType.Object))
            None ()
          else
            Some (lb)
        else Some (ub)
      }
    }
    

    public AnyHint : option [MType]
    {
      get {
        def s = Self;
        def ub = s.upper_bound;
        def lb = s.lower_bound;

        if (ub == null)
          if (lb == null)
            None ()
          else
            Some (lb)
        else Some (ub)
      }
    }


    public virtual FixedValue : MType
    {
      get {
        def s = Self;
        assert (s.flags %&& Flags.IsMonoType);
        assert (s.lower_bound != null);
        s.lower_bound
      }
    }


    [Nemerle.OverrideObjectEquals]
    public Equals (t : TyVar) : bool
    {
      if (t.Self : object == this.Self : object)
        true
      else if (this.IsFixed && t.IsFixed)
        this.FixedValue.Equals (t.FixedValue)
      else
        false
    }


    public virtual Fix () : MType
    {
      Fixate ();
      FixedValue
    }
    

    public virtual IsAccessibleFrom (_ : TypeInfo) : bool
    {
      throw System.NotSupportedException ("it is supposed to be called on fixed type");
    }
    
    public IsFixed : bool
    {
      get {
        Self.flags %&& Flags.IsMonoType
      }
    }


    public IsFree : bool
    {
      get {
        ! (Self.flags %&& Flags.IsMonoType)
      }
    }


    public IsFresh : bool
    {
      get {
        Self.flags %&& Flags.IsFresh
      }
    }


    public IsFromNull : bool
    {
      get {
        Self.flags %&& Flags.IsFromNull
      }
      set {
        assert (IsFree);
        assert (value);

        when (!IsFromNull) {
          WillWrite ();
          def s = Self;
          s.flags |= Flags.IsFromNull;
          when (!CanBeNull)
            SaveError (Passes.Solver.CurrentMessenger,
                       $ "the `null' literal is not a valid value of type "
                         "$(Option.UnSome (Hint))");
          when (s.lower_tyvars != null)
            foreach (tv in s.lower_tyvars)
              when (tv != null && tv.IsFree)
                tv.IsFromNull = true;
          when (s.upper_tyvars != null)
            foreach (tv in s.upper_tyvars)
              when (tv != null && tv.IsFree)
                tv.IsFromNull = true;
        }
      }
    }


    /** Fix the type to be [LowerBound], so it won't change anymore.
    
        Don't use this function unless absolutely necessary. */
    [ForwardThis (Self)]
    public Fixate () : void
    {
      unless (IsFixed) {
        #if VERB
        def before = ToString ();
        #endif
        def bound =
          match (Hint) {
            | Some (t) => t
            | None => InternalType.Object
          }
        def was_local_error = LocalError;
        def cyclic_count = Passes.Solver.CyclicTypeCount;
        def res = Unify (bound);
        unless (was_local_error) {
          unless (res)
            Passes.Solver.CurrentMessenger.DumpSavedError ();
          when (cyclic_count != Passes.Solver.CyclicTypeCount)
            Message.FatalError ($ "cyclic type found: $this");
          #if !VERB
          def before = "";
          #endif
          assert (res, $ "failed to fixate $(this) [$before] with $bound");
        }
      }
    }
    
    mutable static level : int;
    
    public override ToString () : string
    {
      if (IsFree) {
          level++;
          def res =
            if (level > 4)
              "..."
            else {
              #if VERB
              def str (tv : TyVar) {
                if (tv == null) "*"
                else {
                  def tv = tv.Self;
                  $"{$(tv.id)}:"+
                  match ((tv.lower_bound, tv.upper_bound)) {
                    | (null, null) => "?"
                    | (t, null) => $ "$t+"
                    | (null, t) => $ "$t-"
                    | (t1, t2) => $ "($t1 TILL $t2)"
                  }
                }
              }
              
              def t = Self;
              mutable add = $ "{$(t.id)}:";
              when (t.upper_tyvars != null) {
                add += "upper[";
                foreach (tv in t.upper_tyvars)
                  add += str (tv) + ", ";
                add += "]:";
              }
              when (t.lower_tyvars != null) {
                add += "lower[";
                foreach (tv in t.lower_tyvars)
                  add += str (tv) + ", ";
                add += "]:";
              }
              
              add +
              #endif
              match ((Self.lower_bound, Self.upper_bound)) {
                | (null, null) =>
                  mutable res = "";

                  when (Self.upper_tyvars != null)
                    foreach (tv in Self.upper_tyvars)
                      when (tv != null)
                        match (tv.Hint) {
                          | Some (m) =>
                            if (res == "")
                              res = $"$m++"
                            else
                              res += $", $m++"
                          | None => {}
                        }
                        
                  when (Self.lower_tyvars != null)
                    foreach (tv in Self.lower_tyvars)
                      when (tv != null)
                        match (tv.Hint) {
                          | Some (m) =>
                            if (res == "")
                              res = $"$m--"
                            else
                              res += $", $m--"
                          | None => {}
                        }

                  if (res == "") "?" else res
                        
                | (t, null) => $ "$t+"
                | (null, t) => $ "$t-"
                | (t1, t2) =>
                  if (t1.Equals (InternalType.Object))
                    $ "$t2-"
                  else
                    $ "($t1 TILL $t2)"
              }
            }
          level--;
          res
      } else {
        level++;
        def res = 
          if (level > 4) "..."
          else FixedValue.ToString ();
        level--;
        res
      }
    }


    public CompareTo (other : TyVar) : int
    {
      id - other.id
    }


    public override GetHashCode () : int
    {
      id
    }


    public CurrentSolver : Solver
    {
      get { Passes.Solver }
    }


    public virtual SystemType : System.Type
    {
      get {
        Fix ().SystemType
      }
    }


    public virtual NonVoidSystemType : System.Type
    {
      get {
        Fix ().NonVoidSystemType
      }
    }


    static DoDeepFix (x : TyVar) : TyVar
    {
      x.DeepFix ()
    }
    

    static DoDeepFixM (x : MType) : MType
    {
      x.DeepFix ()
    }
    

    public DeepFix () : MType
    {
      if (Passes.Solver.CanEnterPossiblyLooping ())
        try {
          def t = Fix ();
          match (t) {
            | Class (tc, args) =>
              MType.Class (tc, args.Map (DoDeepFix))

            | Fun (from, to) =>
              MType.Fun (from.DeepFix (), to.DeepFix ())

            | Tuple (args) => 
              MType.Tuple (args.Map (DoDeepFix))
            
            | Array (t, r) =>
              MType.Array (t.DeepFix (), r)
            
            | Ref (t) =>
              MType.Ref (t.DeepFix ())
            
            | Out (t) =>
              MType.Out (t.DeepFix ())
            
            | Intersection (t) =>
              MType.Intersection (t.Map (DoDeepFixM))
              
            | TyVarRef
            | Void => t
          }
        } finally {
          Passes.Solver.LeavePossiblyLooping ()
        }
      else {
        ReportError (Passes.Solver.CurrentMessenger,
                     "deep fixation failed");
        when (Passes.Solver.CurrentMessenger.NeedMessage)
          Message.MaybeBailout ();
        InternalType.Void
      }
    }
    #endregion


    #region PRIVATE helper functions
    #if false
    Validate () : void
    {
      when (upper_tyvars != null)
        foreach (tv in upper_tyvars)
          assert (tv == null || this.IsIn (tv.lower_tyvars),
                  $ "tv=$(tv.Self.id) this=$(id)");
      when (lower_tyvars != null)
        foreach (tv in lower_tyvars)
          assert (tv == null || this.IsIn (tv.upper_tyvars),
                  $ "tv=$(tv.Self.id) this=$(id)");
    }
    #endif
    
    static AddRelation (low : TyVar, high : TyVar) : bool
    {
      def low = low.Self;
      def high = high.Self;

      assert (low.IsFree);
      assert (high.IsFree);

      log (SOLVER, $ "AddTVRel: $low :> $high");

      if (low : object == high : object || high.IsIn (low.upper_tyvars))
        true
      else {
        low.WillWrite ();
        high.WillWrite ();
        
        if (low.IsFromNull)
          high.IsFromNull = true;
        else if (high.IsFromNull)
          low.IsFromNull = true;
        else {}

        high.flags &= ~Flags.IsFresh;
        low.flags &= ~Flags.IsFresh;

        #if VERB
        def dump = false;

        when (dump) {
          Message.Debug ($"low=$(low.id) high=$(high.id), $(low.serial)");
          PrintSet ("low.lower", low.lower_tyvars);
          PrintSet ("low.upper", low.upper_tyvars);
          PrintSet ("high.lower", high.lower_tyvars);
          PrintSet ("high.upper", high.upper_tyvars);
        }
        #endif

        AddEdge (low, high);

        #if VERB
        when (dump) {
          Message.Debug ($"after, low=$(low.id) high=$(high.id), $(low.serial)");
          PrintSet ("low.lower", low.lower_tyvars);
          PrintSet ("low.upper", low.upper_tyvars);
          PrintSet ("high.lower", high.lower_tyvars);
          PrintSet ("high.upper", high.upper_tyvars);
        }
        #endif

        // low.Validate ();
        // high.Validate ();

        if (low.IsIn (low.upper_tyvars)) {
          def cycle = IntersectSets (low.upper_tyvars, high.lower_tyvars);
          
          #if VERB
          when (dump)
            PrintSet ("cycle", cycle);
          #endif

          assert (high.IsIn (cycle));
          assert (low.IsIn (cycle));

          foreach (tv in cycle)
            when (tv != null && tv : object != low)
              _ = tv.Alias (low);

          low.FixArrays ();

          foreach (tv in cycle)
            when (tv != null) {
              when (tv.upper_bound != null)
                _ = low.Provide (tv.upper_bound);
              when (tv.lower_bound != null)
                _ = low.Require (tv.lower_bound);
            }

          foreach (tv in cycle)
            when (tv != null) {
              when (tv.upper_bound != null)
                low.SetUpperBound (tv.upper_bound);
              when (tv.lower_bound != null)
                low.SetLowerBound (tv.lower_bound);
            }

        } else {
          def low = low.Self;
          when (low.lower_bound != null)
            _ = high.Require (low.lower_bound);

          def high = high.Self;
          when (high.upper_bound != null)
            _ = low.Provide (high.upper_bound);
        }

        !LocalError
      }
    }


    Alias (tv : TyVar) : bool
    {
      WillWrite ();
      assert (Self : object == this);

      log (SOLVER, $ "Alias: $(this) ---> $tv");

      if (tv.Self : object != this) {
        if (IsFromNull && !tv.CanBeNull) {
          SaveError (Passes.Solver.CurrentMessenger,
                     $ "the `null' literal is not a valid value of type $tv");
          false
        } else {
          when (tv.IsFree && IsFromNull && !tv.IsFromNull)
            tv.IsFromNull = true;
          flags |= Flags.IsAliased;
          alias_to = tv;
          true
        }
      } else true
    }


    public virtual CanBeNull : bool
    {
      get {
        def s = Self;
        (s.lower_bound == null || s.lower_bound.CanBeNull) &&
        (s.upper_bound == null || s.upper_bound.CanBeNull)
      }
    }


    internal virtual NeedNoSubst : bool
    {
      get {
        def s = Self;
        if (s.flags %&& Flags.IsMonoType)
          s.NeedNoSubst
        else
          s.flags %&& Flags.IsFresh ||
          ((s.lower_bound == null || s.lower_bound.NeedNoSubst) &&
           (s.upper_bound == null || s.upper_bound.NeedNoSubst))
      }
    }


    SetLowerBound (lb : TyVar) : void
    {
      when (lb != null && upper_tyvars != null)
        foreach (tv in upper_tyvars)
          when (tv != null)
            _ = tv.Require (lb);
    }
    

    SetUpperBound (ub : TyVar) : void
    {
      when (ub != null && lower_tyvars != null)
        foreach (tv in lower_tyvars)
          when (tv != null)
            _ = tv.Provide (ub);
    }
    

    static FixArray (s : array [TyVar], eject : TyVar) : void
    {
      for (mutable i = 0; i < s.Length; ++i) {
        when (s [i] != null) {
          def tv = s [i].Self;

          if (tv.flags %&& Flags.IsMonoType || tv : object == eject)
            s [i] = null
          else {
            s [i] = tv;
          }
        }
      }
    }


    FixArrays () : void
    {
      FixArray (lower_tyvars, this);
      FixArray (upper_tyvars, this);
    }


    [PossiblyLooping (Passes.Solver)]
    static AddRelation (low : MType, high : TyVar, rev : bool) : bool
    {
      assert (low != null);
      assert (high != null);

      whenlogging (SOLVER)
        if (rev)
          Message.Debug ($ "AddRelation: $low <: $high");
        else
          Message.Debug ($ "AddRelation: $low :> $high");

      high.WillWrite ();
      def high = high.Self;
      high.flags &= ~Flags.IsFresh;

      if (low.IsSeparated && high.IsFree)
        match (low) {
        /*
          | MType.Fun
          | MType.Array
          | MType.TyVarRef
          | MType.Tuple
          */
          | MType.Void => high.Unify (low)

          /*
          | MType.Fun (t1, t2) =>
            def tv1 = Solver.FreshTyVar ();
            def tv2 = Solver.FreshTyVar ();
            if (rev) {
              _ = tv1.Require (t1);
              _ = tv2.Provide (t2);
            } else {
              _ = tv1.Provide (t1);
              _ = tv2.Require (t2);
            }
            high.Unify (MType.Fun (tv1, tv2))
      
          | MType.Tuple (ts) =>
            def ts' = List.Map (ts, fun (t : TyVar) {
              def tv = Solver.FreshTyVar ();
              if (rev)
                _ = tv.Provide (t)
              else
                _ = tv.Require (t);
              tv
            });
            high.Unify (MType.Tuple (ts'))
            */
            
          | _ => assert (false)
        }
      else if (rev && low.Equals (InternalType.Object))
        high.Unify (low)
      else if (!high.IsFree) {
        if (rev)
          low.Require (high.FixedValue)
        else
          high.FixedValue.Require (low)
      } else if (high.IsFree && high.IsFromNull && !low.CanBeNull) {
        SaveError (Passes.Solver.CurrentMessenger,
                   $ "the `null' literal is not a valid value of type $low");
        false
      } else {
        if (rev) {
          def new_bound = Passes.Solver.Sum (high.upper_bound, low);
          def high = high.Self;
          if (high.IsFixed)
            _ = high.Provide (new_bound);
          else {
            high.WillWrite ();
            // Message.Debug ($"new bound $new_bound");
            when (high.upper_bound == null || 
                  !new_bound.Equals (high.upper_bound)) {
              high.upper_bound = new_bound;
              high.SetUpperBound (low);
            }
          }
        } else {
          def new_bound = Passes.Solver.Intersect (high.lower_bound, low);
          def high = high.Self;
          if (high.IsFixed)
            _ = high.Require (new_bound);
          else {
            high.WillWrite ();
            whenlogging (SOLVER) {
              def tv = if (high.lower_bound == null) "(null)" else high.lower_bound.ToString ();
              //Message.Debug ($"new bound $tv * $low  = $new_bound");
            }
            when (high.lower_bound == null || 
                  !new_bound.Equals (high.lower_bound)) {
              high.lower_bound = new_bound;
              high.SetLowerBound (low);
            }
          }
        }

        unless (high.IsFixed) {
          def high = high.Self;
          when (high.upper_bound != null &&
                high.lower_bound != null)
            _ = high.upper_bound.Require (high.lower_bound);
          
          def high = high.Self;

          when (high.IsFree &&
                high.upper_bound != null &&
                high.lower_bound != null &&
                high.upper_bound.TryEnforcingEquality (high.lower_bound))
            _ = high.Unify (high.lower_bound);
        }


        // Message.Debug ($"error=$(LocalError)");

        !LocalError
      }
    }

    
    public static GetStackSize () : int
    {
      def str = System.IO.StreamReader ("/proc/self/status");
      def loop () {
        def line = str.ReadLine ();
        if (line == null) -1
        else if (line.StartsWith ("VmStk:"))
          System.Int32.Parse (line.Substring (6, line.Length - 8))
        else
          loop ()
      }
      loop ()
    }


    WillRead () : void
    {
      def loop (i) {
        if (i < 0 || Passes.Solver.serial_stack [i] < serial) {
          when (next == null) {
            def the_serial = serial;
            serial = 1; // make it print
            Message.Debug ($ "oops, our_serial=$the_serial, "
                             "stack[0]=$(Passes.Solver.serial_stack[0]), "
                             "stack[1]=$(Passes.Solver.serial_stack[1]), "
                             "top=$(Passes.Solver.serial_stack_top), "
                             "id=$id, this=$(this)");
            assert (false);
          }
          def store = next;
          CopyFrom (next);
          store.next = reuse_queue;
          reuse_queue = store;
          loop (Passes.Solver.serial_stack_top)
        } else if (Passes.Solver.serial_stack [i] == serial) {}
        else loop (i - 1)
      }

      unless (Passes.Solver.top_serial == serial)
        loop (Passes.Solver.serial_stack_top)
    }


    CopyArrays () : void
    {
      unless (upper_tyvars == null)
        upper_tyvars = upper_tyvars.Clone () :> array [TyVar];
        
      unless (lower_tyvars == null)
        lower_tyvars = lower_tyvars.Clone () :> array [TyVar];
    }


    static mutable reuse_queue : TyVar;


    static CopyTyVar (tv : TyVar) : TyVar
    {
      if (reuse_queue == null)
        TyVar (tv)
      else {
        def res = reuse_queue;
        reuse_queue = reuse_queue.next;
        res.flags = Flags.None;
        res.CopyFrom (tv);
        res
      }
    }


    WillWrite () : void
    {
      WillRead ();
      when (Passes.Solver.top_serial != serial) {
        def tv = CopyTyVar (this);
        next = tv;
        CopyArrays ();
        serial = Passes.Solver.top_serial;
      }
    }


    protected Self : TyVar
    {
      get {
        WillRead ();
        if (flags %&& Flags.IsAliased)
          alias_to.Self
        else
          this
      }
    }

    
    CopyFrom (other : TyVar) : void
    {
      assert (! (other.flags %&& Flags.IsMonoType));
      assert (! (flags %&& Flags.IsMonoType));

      alias_to = other.alias_to;
      next = other.next;
      serial = other.serial;
      id = other.id;
      lower_bound = other.lower_bound;
      upper_bound = other.upper_bound;
      flags = other.flags;
      lower_tyvars = other.lower_tyvars;
      upper_tyvars = other.upper_tyvars;
    }


    static LocalError : bool
    {
      get { Passes.Solver.CurrentMessenger.LocalError }
    }
    #endregion

    
    #region State
    mutable alias_to : TyVar;
    mutable next : TyVar;
    protected mutable serial : int;
    mutable id : int;
    protected mutable lower_bound : MType;
    protected mutable upper_bound : MType;
    mutable lower_tyvars : array [TyVar];
    mutable upper_tyvars : array [TyVar];
    protected mutable flags : Flags;

    [System.Flags]
    public enum Flags {
      | None       = 0x0000
      | IsFromNull = 0x0001
      | IsMonoType = 0x0002
      | IsAliased  = 0x0004
      | IsFresh    = 0x0008
    }
    #endregion


    #region Constructors (only internal)
    static mutable current_id : int;

    internal this ()
    {
      id = current_id;
      ++current_id;
      serial = Passes.Solver.top_serial;
      flags = Flags.IsFresh;
    }
    
    this (other : TyVar)
    {
      CopyFrom (other);
    }

    static this ()
    {
      Passes.OnInit += fun () { reuse_queue = null; }
    }
    #endregion


    #region Edge operations
    static empty_set : array [TyVar] = array [];

    NullToEmpty () : void
    {
      WillWrite ();
      when (upper_tyvars == null)
        upper_tyvars = empty_set;
      when (lower_tyvars == null)
        lower_tyvars = empty_set;
    }

    /** Requires [low] and [high] to be after [Self].  */
    static AddEdge (low : TyVar, high : TyVar) : void
    {
      low.NullToEmpty ();
      high.NullToEmpty ();

      low.upper_tyvars = SumSets (low.upper_tyvars, high.upper_tyvars);
      high.InsertSelfInto (low.upper_tyvars);

      foreach (tv in low.lower_tyvars) {
        when (tv != null) {
          def tv = tv.Self;
          unless (tv.flags %&& Flags.IsMonoType) {
            tv.WillWrite ();
            tv.upper_tyvars = SumSets (tv.upper_tyvars, low.upper_tyvars);
          }
        }
      }

      high.lower_tyvars = SumSets (low.lower_tyvars, high.lower_tyvars);
      low.InsertSelfInto (high.lower_tyvars);

      foreach (tv in high.upper_tyvars) {
        when (tv != null) {
          def tv = tv.Self;
          unless (tv.flags %&& Flags.IsMonoType) {
            tv.WillWrite ();
            tv.lower_tyvars = SumSets (tv.lower_tyvars, high.lower_tyvars);
          }
        }
      }
    }
    #endregion


    #region Set operations
    #if VERB
    static PrintSet (pref : string, s : array [TyVar]) : void
    {
      def l = if (s == null) [] else 
      List.MapFromArray (s, fun (x : TyVar) {
        if (x == null) 0 
        else if (x.IsFixed) -x.Self.id else x.Self.id });
      Message.Debug ($ "set: $pref: $(l)");
    }
    #endif
    
    IsIn (s : array [TyVar]) : bool
    {
      def loop (i) {
        if (i >= s.Length) false
        else if (s [i] != null && 
                 s [i].Self : object == this : object) true
        else loop (i + 1)
      }
      
      if (s == null)
        false
      else
        loop (0)
    }

    
    static NormalizeAndCount (s : array [TyVar]) : int
    {
      mutable estimated_size = 0;
      
      for (mutable i = 0; i < s.Length; ++i) {
        when (s [i] != null) {
          def tv = s [i].Self;

          if (tv.flags %&& Flags.IsMonoType)
            s [i] = null
          else {
            s [i] = tv;
            ++estimated_size
          }
        }
      }
      
      estimated_size
    }


    InsertSelfInto (s : array [TyVar]) : void
    {
      def loop (i) {
        if (s [i] == null)
          s [i] = this
        else
          loop (i + 1)
      }
      loop (0)
    }

    
    // assumes normalized array!
    ContainedIn (s : array [TyVar], max : int) : bool
    {
      def loop (i) {
        if (i >= max) false
        else if (s [i] : object == this : object) true
        else loop (i + 1)
      }
      loop (0)
    }

    
    static SumSets (s1 : array [TyVar], s2 : array [TyVar]) : array [TyVar]
    {
      def size = NormalizeAndCount (s1) + NormalizeAndCount (s2);
      def res = array (size + 1);
      mutable res_ptr = 0;
      
      if (size > 42) {
        def ht = System.Collections.Hashtable ();
        foreach (tv in s1) {
          when (tv != null && !ht.Contains (tv.id)) {
            ht [tv.id] = ht;
            res [res_ptr] = tv;
            ++res_ptr
          }
        }
        foreach (tv in s2) {
          when (tv != null && !ht.Contains (tv.id)) {
            ht [tv.id] = ht;
            res [res_ptr] = tv;
            ++res_ptr
          }
        }
      } else {
        foreach (tv in s1) {
          when (tv != null && !tv.ContainedIn (res, res_ptr)) {
            res [res_ptr] = tv;
            ++res_ptr
          }
        }
        foreach (tv in s2) {
          when (tv != null && !tv.ContainedIn (res, res_ptr)) {
            res [res_ptr] = tv;
            ++res_ptr
          }
        }
      }

      res
    }


    static IntersectSets (s1 : array [TyVar], s2 : array [TyVar]) : array [TyVar]
    {
      def size1 = NormalizeAndCount (s1);
      def size2 = NormalizeAndCount (s2);
      def res = array (if (size1 < size2) size1 else size2);
      mutable res_ptr = 0;
      
      if (size1 + size2 > 42) {
        def ht = System.Collections.Hashtable ();
        foreach (tv in s1) {
          when (tv != null)
            ht [tv.id] = ht;
        }
        foreach (tv in s2) {
          when (tv != null && ht.Contains (tv.id)) {
            ht.Remove (tv.id);
            res [res_ptr] = tv;
            ++res_ptr
          }
        }
      } else {
        foreach (tv in s1) {
          when (tv != null && 
                tv.ContainedIn (s2, s2.Length) && 
                !tv.ContainedIn (res, res_ptr)) {
            res [res_ptr] = tv;
            ++res_ptr
          }
        }
      }

      res
    }
    #endregion
  }
}
