/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Utility;

using Nemerle.Compiler;
using Nemerle.Compiler.Parsetree;
using Nemerle.Compiler.SolverMacros;
using System;
using System.Reflection;

namespace Nemerle.Compiler
{
  public class TyVarEnv
  {
    tyvars : Map [Name, StaticTyVar];
    solver : Solver;
    messenger : Messenger;
    
    private this (tv : Map [Name, StaticTyVar])
    {
      tyvars = tv;
      solver = Passes.Solver;
      messenger = solver.CurrentMessenger;
    }

    public this ()
    {
      tyvars = Map ();
      solver = Passes.Solver;
      messenger = solver.CurrentMessenger;
    }

    public IsEmpty : bool
    {
      get { tyvars.IsEmpty }
    }

    public Find (name : Name) : option [StaticTyVar] {
      tyvars.Find (name)
    }
    
    public MonoBind (env : GlobalEnv,
                     curtc : TypeBuilder,
                     t : PExpr,
                     check_parms : bool) : MType
    {
      def t = Bind (env, curtc, t, 
                    allow_tyvars = false, 
                    check_parms = check_parms);
      t.Fix ()
    }


    bind_simple_named_type (env : GlobalEnv, curtc : TypeBuilder,
                            name : Name, idl : list [string],
                            args : list [TyVar]) : TyVar
    {
      match (tyvars.Find (name)) {
        | Some (tv) when idl is [_] =>
          when (!args.IsEmpty)
            ReportError (messenger,
                         $ "type variable `$(name.Id)' supplied with "
                           "arguments");

          MType.TyVarRef (tv)
          
        | _ =>
          def env = name.GetEnv (env);
          assert (env != null);

          mutable wrong_parms = false;
          mutable add_info = "unbound type name";
          mutable exact_hit = null;

          def symbols = env.LookupSymbol (idl, curtc);
          def typeinfos = symbols.FoldLeft ([], fun (m, acc) {
            match (m) {
              | ti is TypeInfo =>
                if (curtc == null || ti.CanAccess (curtc))
                  if (args.Length == ti.SourceTyparmsCount) {
                    when (ti.FullName == name.Id)
                      exact_hit = ti;
                    ti :: acc
                  } else { 
                    wrong_parms = true;
                    add_info = "wrong number of type parameters to";
                    acc
                  }
                else {
                  when (! wrong_parms)
                    add_info = "inaccessible type";
                  acc
                }
              | _ => acc
            }
          });

          match (typeinfos) {
            | [ti]
            | _ when exact_hit != null with ti = exact_hit =>
              if (ti.TyparmsCount == args.Length)
                MType.Class (ti, args)
              else {
                def find_nesting (nesting : TypeInfo) {
                  if (nesting == null) null
                  else if (ti.DeclaringType.Equals (nesting))
                    curtc.NestingSubst (nesting)
                      .Apply (nesting.GetMemType ())
                  else if (nesting.LookupMemberAvailable && 
                           nesting.LookupMember (ti.Name).Contains (ti))
                    curtc.NestingSubst (nesting)
                      .Apply (nesting.SubtypingSubst (ti.DeclaringType)
                        .Apply (ti.DeclaringType.GetMemType ()).Fix ())
                  else find_nesting (nesting.DeclaringType)
                }
                match (find_nesting (curtc)) {
                  | null =>
                    ReportError (messenger,
                                 $ "cannot determine nested type parameters "
                                   "for `$(idl.ToString (\".\"))', please use the fully "
                                   "qualified name");
                    InternalType.Object
                    
                  | MType.Class (_, args') =>
                    def args = args' + args;
                    Util.cassert (args.Length == ti.TyparmsCount);
                    MType.Class (ti, args)
                    
                  | _ => Util.ice ()
                }
              }
              
            | [] =>
              ReportError (messenger, $ "$add_info `$(idl.ToString (\".\"))'");
              InternalType.Object
              
            | _ =>
              ReportError (messenger,
                           $ "type name `$(idl.ToString (\".\"))' is ambiguous, it could be:");
              when (messenger.NeedMessage)
                foreach (ti in typeinfos)
                  Message.Error (ti.Location, $"   this declaration `$ti'");
              InternalType.Object
          }
      }
    }
    

    static poors_man_lookup_member (ti : TypeInfo, name : string) : list [IMember]
    {
      def loop (acc = [name], ti = ti) {
        if (ti == null) acc
        else loop (ti.Name :: acc, ti.DeclaringType)
      }
      match (NamespaceTree.LookupExactType (loop ())) {
        | Some (t) => [t]
        | None => []
      }
    }
    

    bind_nested_type (curtc : TypeInfo, base_type : TyVar, name : string, args : list [TyVar]) : TyVar
    {
      match (base_type.Fix ()) {
        | MType.Class (ti, args') =>
          mutable seen_typeinfo = "";
          
          def symbols =
            if (ti.LookupMemberAvailable)
              ti.LookupMember (name)
            else
              poors_man_lookup_member (ti, name);

          def types =
            symbols.Filter (fun (m) {
              | m is TypeInfo when curtc == null || m.CanAccess (curtc) =>
                seen_typeinfo = "with this number of type parameters";
                m.SourceTyparmsCount == args.Length
              | _ => false
            });
            
          match (types) {
            | [m is TypeInfo] =>
              MType.Class (m, args' + args)

            | [] =>
              ReportError (messenger,
                           $ "the type `$base_type' does not contain "
                             "a nested type named `$name' $seen_typeinfo");
              InternalType.Object

            | l => Util.ice ($ "$l");
          }
        | t =>
          ReportError (messenger,
                       $ "won't lookup nested types in $t");
          InternalType.Object
      }
    }

    
    /** Perform typing of Parsetree type to Typedtree type, looking up
        type constructors in given global environment (with accessibility
        information implied by given current TypeInfo) and type variables
        in current instance.

        If [check_parms] is true, then we check if type substituted for
        a given type variable conforms to all of its constraints. It
        should be true in general, but is false when we don't know the
        subtyping relations yet (during scanning of global types).
     */
    public Bind (env : GlobalEnv,
                 curtc : TypeBuilder,
                 t : PExpr,
                 allow_tyvars : bool,
                 check_parms : bool) : TyVar
    {
      def f (t) {
        f2 (t, false)
      } and f2 (t, allow_ref) {
        match (t) {
          | <[ ref $t ]> when allow_ref =>
            MType.Ref (f (t))
          
          | <[ out $t ]> when allow_ref =>
            MType.Out (f (t))
          
          | <[ ref $_ ]>
          | <[ out $_ ]> =>
            ReportError (messenger, "nested ref/out type found");
            InternalType.Void

          | <[ array [$t] ]> =>
            MType.Array (f (t), 1)
            
          | <[ array [$(rank : int), $t] ]> =>
            MType.Array (f (t), rank : int)

          | <[ $x -> $y ]> =>
            MType.Fun (f2 (x, true), f (y))

          | <[ @* (.. $args) ]> =>
            MType.Tuple (List.Map (args, fun (t) { f2 (t, allow_ref) }))

          | <[ this ]> =>
            curtc.GetMemType ()
            
          | PExpr.Void => InternalType.Void

          | PExpr.Indexer
          | PExpr.GenericSpecifier
          | PExpr.Member
          | PExpr.Ref =>
            type_class (t)

          | PExpr.TypedType (body) =>
            body
            
          | PExpr.Wildcard =>
            if (allow_tyvars)
              Solver.FreshTyVar ()
            else {
              ReportError (messenger, "type inference not allowed here");
              InternalType.Object
            }

          | PExpr.Spliced =>
            Util.ice ("Spliced type survived to typying.")

          | PExpr.Array =>
            ReportError (messenger, 
                         $ "array type must take form `array [T]' or "
                           "`array [rank, T]', not $t");
            InternalType.Void
            
          | PExpr.Ellipsis =>
            Util.ice ("Type arguments list survived to typying.")
            
          | x =>
            ReportError (messenger, $ "$x is not a legal type expression");
            InternalType.Void
        }
      } and type_class (tyexpr) : TyVar {
        def res =
          match (tyexpr) {
            | <[ $t [] ]> =>
              ReportError (messenger, $"$t[] is not a valid type, use just $t");
              when (messenger.NeedMessage)
                Message.HintOnce ("if you had array type on mind, its syntax is `array [SomeType]'");
              f (t)

            | <[ $(name : name) ]> with args = []
            | <[ $(name : name) [ .. $args ] ]>
            | <[ $(name : name) . [ .. $args ] ]> =>
              bind_simple_named_type (env, curtc, name, [name.Id], args.Map (f))
              
            | <[ $t . [ .. $args ] ]>
            | <[ $t [ .. $args ] ]>
            | <[ $t ]> with args = [] =>
              match (Util.qidl_of_expr (t)) {
                | Some ((idl, name)) =>
                  bind_simple_named_type (env, curtc, name, idl, args.Map (f))

                | None =>
                  match (t) {
                    | <[ $ns . $(fld : dyn) ]> =>
                      bind_nested_type (curtc, f (ns), fld, args.Map (f))
                    | _ =>
                      ReportError (messenger, $ "$t is not a legal type expression");
                      InternalType.Void
                  }
              }
          }

        match (res.Fix ()) {
          | MType.Class (ti, args) =>
            ti.HasBeenUsed = true;
            match (ti.GetTydecl ()) {
              | Typedtree.TypeDeclaration.Alias (t) =>
                def subst = ti.MakeSubst (args);
                subst.Apply (t)

              | _ =>
                when (check_parms)
                  _ = ti.MakeSubst (args);
                MType.Class (ti, args)
            }
          | t => t
        }
      }

      f (t)
    }
    
    
    public AddTyparms (env : GlobalEnv, tp : Typarms,
                       curtc : TypeBuilder,
                       check_parms : bool) : TyVarEnv * list [StaticTyVar]
    {
      def name_of_tv (tv) {
        def (name, color) = tv;
        Name (name, color, null)
      }
      def loop (tv : string * int, acc) {
        def (map, the_list) = acc;
        def tv_obj = StaticTyVar (Pair.First (tv));
        ((map : NemerleMap [Name, StaticTyVar]).Replace (name_of_tv (tv), tv_obj),
         tv_obj :: the_list)
      }
      def (m, l) = List.FoldLeft (tp.tyvars, (this.tyvars, []), loop);
      def tenv = TyVarEnv (m);

      def constraints = Hashtable ();
      
      def get (id) {
        if (constraints.Contains (id))
          constraints [id]
        else (GenericParameterAttributes.None, [])
      }

      // bind constraints to what they really are
      foreach (c in tp.constraints) {
        match (m.Find (name_of_tv (c.tyvar))) {
          | Some (tv) =>
            def cons = get (tv.id);
            mutable special = Pair.First (cons);
            mutable subtype = Pair.Second (cons);
            
            match (c.ty) {
              | <[ @class ]> => special |= GenericParameterAttributes.ReferenceTypeConstraint;
              | <[ @struct ]> => special |= GenericParameterAttributes.NotNullableValueTypeConstraint;
              | <[ @new ]> => special |= GenericParameterAttributes.DefaultConstructorConstraint;
              | _ =>
                def ty = tenv.MonoBind (env, curtc, c.ty, check_parms);
                if (subtype.Contains (ty))
                  Message.Error ($"constraint `$ty' is already declared for $tv");
                else
                  subtype = ty :: subtype;
                
            }
            constraints [tv.id] = (special, subtype);
            
          | None =>
            Message.Error ("unbound type variable `" + Pair.First (c.tyvar) + "' in constraint")
        }
      }  
      
      foreach (tv : StaticTyVar in l) {
        // FIXME: check Intersection invariants and flag error
        // to the user, otherwise we'll get an ICE
        tv.SetConstraints (get (tv.id)); 
      }
      
      (tenv, List.Rev (l))
    }
  }
}
