/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */ 
 
namespace Sioux
{
  using System.Xml;
  using Nemerle.Collections;


  /**
   * A class used to access Sioux XML configuration file.
   */
  public class Config
  {
    /* -- CONSTRUCTORS ----------------------------------------------------- */
    
    public this (config_file_name : option [string])
    {
      defs = Hashtable ();
      doc = XmlDocument ();

      this.config_file_name = 
        match (config_file_name) {
          | Some (config_file_name) => config_file_name
          | None => "/etc/sioux/httpd.conf"
        };

      try {
        doc.Load (this.config_file_name);
      }
      catch {
        _ => ()
      };

      put_defaults ();
    }


    /* -- PUBLIC METHODS --------------------------------------------------- */

    public Get (key : string) : string
    {
      try {
        def sect   = doc.SelectSingleNode ("config/" + key);
        def reader = XmlNodeReader (sect);
        def _      = reader.Read ();
        def _      = reader.Read ();

        reader.Value
      }
      catch {
        _ => get_default (key)
      }
    }


    /* -- PRIVATE METHODS -------------------------------------------------- */

    /**
     * Fills in the defaults
     */
    private put_defaults () : void
    {
      defs.Add ("server/name", "Sioux HTTP Server 0.1");
      defs.Add ("server/root", "d:\\cygwin\\usr\\local\\src\\svn\\nemerle\\nemerle\\trunk\\snippets\\sioux\\");
      defs.Add ("server/port", "8000");
      defs.Add ("server/webapps_dir","webapps\\");
      defs.Add ("server/url_rewriting","false");
      defs.Add ("server/max_connections","100");
      defs.Add ("server/max_sessions","20");
      
      defs.Add ("replies/invalid", "Invalid request.");
      defs.Add ("replies/not_found", "File not found.");
      defs.Add ("replies/server_full", "503 Service Unavailable.[br]Server is full. Please try later");
    }


    /**
     * Returns a default setting
     */
    private get_default (key : string) : string
    {
      match (defs.Get (key)){
        | Some (value) => value
        | None => null
      }
    }
    

    /* -- PRIVATE FIELDS --------------------------------------------------- */
    
    private config_file_name : string;
    private doc : XmlDocument;
    private defs : Hashtable [string, string];
  }
}
