/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System.Globalization;
using Nemerle.Collections;
using Nemerle.Utility;
using System.Math;

namespace Nemerle.Compiler {

  public struct Location : Nemerle.IComparable [Location]
  {
    /** The format of token is fixed and holding following info:
        19 bits for end line number
        8 bits for end column number
        19 bits for (start) line number (0-524287 line numbers)
        8 bits for (start) column number (0-255 column numbers)
        10 bits for filename index  (1024 filenames)
        = 64 bits of data
     */       
    token : ulong;

    this (t : ulong) { token = t; }

    public this (file_idx : int, line : int, col : int) {
      token = ComputeToken (file_idx, line, col, line, col);
    }    

    public this (file_idx : int, line : int, col : int, end_line : int, end_col : int) {
      token = ComputeToken (file_idx, line, col, end_line, end_col);
    }    
    
    public static Default : Location = Location (0UL);

    public static @== (x : Location, y : Location) : bool { x.token == y.token }
    public static @!= (x : Location, y : Location) : bool { x.token != y.token }
   
    // NOTE: this is not commutative
    public static @+ (x : Location, y : Location) : Location {
      Location ((x.token %& (~(end_line_mask %| end_col_mask))) %|
                (y.token %& (end_line_mask %| end_col_mask)))
    }    
    
    /** the bit number (from least important as 0) where line info begins */
    static line_bits : int = 18;
    static col_bits : int = 10;
    static end_line_bits : int = 45;
    static end_col_bits : int = 37;
    static end_line_mask : ulong = ((1UL << 19) - 1UL) << 45;
    static end_col_mask : ulong = ((1UL << 8) - 1UL) << 37;

    /** bits reserved for index of file */
    static file_mask : ulong = 0b1111111111UL;

    /** mapping from index to file name */
    static file_names : array [string] = array (1024);
    static mutable file_names_amount : int;
    
    /** encountered file names with mapping to their indices */
    static file_indices : Hashtable [string, int] = Hashtable ();

    public static Init () : void
    {
      file_indices.Clear ();
      file_indices.Add ("", 0);
      file_names [0] = "";
      file_names_amount = 1;
    }
    
    public static ComputeToken (file_idx : int, line : int, col : int, 
                                end_line : int, end_col : int) : ulong
    {
      (file_idx :> ulong) %|
      ((end_line :> ulong) << end_line_bits) %| 
      ((if (end_col > 255) 255UL else (end_col :> ulong)) << end_col_bits) %|
      ((line :> ulong) << line_bits) %| 
      ((if (col > 255) 255UL else (col :> ulong)) << col_bits)
    }

    /** Adds new filename to locations index. If filename in already in
        store, an error message is outputted.
     */
    public static AddFile (name : string) : int
    {
      when (file_indices.Contains (name))
        Message.Error ($"file $name occured twice on the list to compile");
      GetFileIndex (name)
    }
    
    /** Removes a filename from the location index */
    public static RemoveFile (name : string) : void
    {
        if (file_indices.Contains (name))
        {
            file_indices.Remove (name);
        }
        else
            throw System.ArgumentException ($"file $name do not exist");
    }

    /** Gets index of given filename in locations store. If one doesn't exist
        it is being added and new index is returned.
     */
    public static GetFileIndex (name : string) : int
    {
      match (file_indices.Get (name)) {
        | Some (idx) => idx
        | None =>
          if (file_names_amount == file_names.Length) {
            Message.Warning ("too many filenames... Location cache overflow");
            0
          }
          else {
            assert (file_names_amount > 0);
            file_indices.Add (name, file_names_amount);
            file_names [file_names_amount] = name;
            ++file_names_amount;
            file_names_amount - 1
          }
      }
    }

    public FileIndex : int
    {
      get {
        (token & file_mask) :> int
      }
    }
    
    public File : string
    {
      [Nemerle.Assertions.Ensures (value != null)]
      get {
        file_names [(token %& file_mask) :> int];
      }
    }

    public Line : int
    {
      get { ((token >> line_bits) %& ((1UL << 19) - 1UL)) :> int }
    }

    /** we allow only columns in range 0-255 */
    public Column : int
    {
      get { ((token >> col_bits) %& 255UL) :> int }
    }

    public EndLine : int
    {
      get { ((token %& end_line_mask) >> end_line_bits) :> int }
    }

    /** we allow only columns in range 0-255 */
    public EndColumn : int
    {
      get { ((token %& end_col_mask) >> end_col_bits) :> int }
    }

    public CompareTo (x : Location) : int
    {
      if (token %& file_mask == x.token %& file_mask)
        if (Line == x.Line)
          Column.CompareTo (x.Column)
        else
          Line.CompareTo (x.Line)
      else {
        File.CompareTo (x.File)
      }
    }

    public override ToString () : string {
      if (this == Default)
        ""
      else
        if (EndLine != 0)
          $"$File:$Line:$Column:$EndLine:$EndColumn: "
        else if (Line != 0)
          $"$File:$Line:$Column: "
        else
          File + ": "
    }

    [Nemerle.OverrideObjectEquals]
    public Equals (other : Location) : bool
    {
      token == other.token
    }
  }
  
  public class Located
  {
    public mutable loc : Location;
    public this () { loc = Location_stack.top (); }
    public this (loc : Location) { this.loc = loc }

    public Location : Location { get { loc } }
  }

  [System.Flags]
  public enum NemerleAttributes
  {
    | None        = 0x00000
    | Public      = 0x00001
    | Private     = 0x00002
    | New         = 0x00004 
    | Protected   = 0x00008
    | Abstract    = 0x00010
    | Virtual     = 0x00020
    | Sealed      = 0x00040
    | Static      = 0x00080
    | Mutable     = 0x00100      
    | Internal    = 0x00200
    | Override    = 0x00400
    | Struct      = 0x01000
    | Macro       = 0x02000
    | Volatile    = 0x04000
    | SpecialName = 0x08000
    | Partial     = 0x10000
    | Extern      = 0x20000            

    | AccessModifiers = Public %| Private %| Protected %| Internal
  }                 

  
  public partial class Modifiers 
  {
    public mutable mods : NemerleAttributes;
    public mutable custom_attrs : list [Parsetree.PExpr];
    internal mutable macro_attrs : list [string * Parsetree.PExpr];

    public static Empty : Modifiers;

    public IsEmpty : bool
    {
      get {
        custom_attrs.IsEmpty && macro_attrs.IsEmpty
      }
    }

    public this (mods : NemerleAttributes, custom_attrs : list [Parsetree.PExpr])
    {
      this.mods = mods;
      this.custom_attrs = custom_attrs;
      this.macro_attrs = [];
    }

    public static this ()
    {
      Empty = Modifiers (NemerleAttributes.None, []);
    }

    public Attributes : NemerleAttributes {
      get { mods }
    }

    public GetCustomAttributes () : list [Parsetree.PExpr]
    {
      custom_attrs
    }
    
    public AddCustomAttribute (expr : Parsetree.PExpr) : void
    {
      custom_attrs = expr :: custom_attrs;
    }
  }

  public variant Literal
  {
    | Void
    | Null
    | String { val : string; }
    | Float { val : float; }
    | Double { val : double; }
    | Decimal { val : decimal; }
    | Char { val : char; }
    | Bool { val : bool; }
    | Integer {
        val : ulong; 
        is_negative : bool; 
        mutable treat_as : MType.Class;
      } 

    | Enum { val : Literal.Integer; ty : TypeInfo; }


    [OverrideObjectEquals]
    public Equals (lit : Literal) : bool
    {
      match ((this, lit)) {
        | (Void, Void)
        | (Null, Null) => true
        | (String (x1), String (x2)) => x1 == x2
        | (Float (x1), Float (x2)) => x1 == x2
        | (Double (x1), Double (x2)) => x1 == x2
        | (Decimal (x1), Decimal (x2)) => x1 == x2
        | (Char (x1), Char (x2)) => x1 == x2
        | (Bool (x1), Bool (x2)) => x1 == x2
        | (Integer (val, is_neg, _), Integer (val', is_neg', _)) =>
          val == val' && is_neg == is_neg'
        | (Enum (v1, t1), Enum (v2, t2)) => v1.Equals (v2) && t1.Equals (t2)
        | _ => false
      }
    }

      
    /**
     * Converts 'this' literal to an equivalent (lexable) string
     */
    public override ToString () : string
    {
      match (this) {
        | Literal.Void           => "()"
        | Literal.Null           => "null"

        | Literal.String   (val) => "\"" + val.Replace ("\n", "\\n") + "\""
        | Literal.Float    (val) => val.ToString (NumberFormatInfo.InvariantInfo) + "f"
        | Literal.Double   (val) => val.ToString (NumberFormatInfo.InvariantInfo) + "d"
        | Literal.Decimal  (val) => val.ToString (NumberFormatInfo.InvariantInfo) + "m"
        | Literal.Char     (val) => "'" + val.ToString () + "'"
        | Literal.Bool     (val) => if (val) "true" else "false"
        | Literal.Integer  (val, is_negative, _treat_as) =>
          def s =
            if (is_negative) "-" + val.ToString ()
            else val.ToString ();
          s
          /*
          if (treat_as == null) s
          else $ "($s : $treat_as)";
          */
          
        | Literal.Enum (val, ty) => "(" + val.ToString () + " :> " + ty.FullName + ")"
      }
    }
    
    public SystemType : System.Type
    {
      get {
        GetInternalType ().SystemType
      }
    }

    public GetInternalType () : MType
    {
      match (this) {
        | Literal.Void => InternalType.Void
        | Literal.Null => InternalType.Object
        | Literal.Char => InternalType.Char
        | Literal.String => InternalType.String
        | Literal.Float => InternalType.Single
        | Literal.Double => InternalType.Double
        | Literal.Decimal => InternalType.Decimal
        | Literal.Bool => InternalType.Boolean
        // for enums we want to stay with original type
        // because e.g. ToString is called on enum not int
        | Literal.Enum (_, ty) => ty.GetMemType ()
        | Literal.Integer (_, _, t) => t
      }
    }

    public WithType (t : MType) : option [Literal]
    {
      match (this) {
        | Literal.Void => None ()
        | Literal.Null when t.CanBeNull
        | Literal.Char when t.Equals (InternalType.Char)
        | Literal.String when t.Equals (InternalType.String)
        | Literal.Float when t.Equals (InternalType.Single)
        | Literal.Double when t.Equals (InternalType.Double)
        | Literal.Decimal when t.Equals (InternalType.Decimal)
        | Literal.Bool when t.Equals (InternalType.Boolean) 
        | Literal.Enum (_, tc) when MType.Class (tc, []).Equals (t)
          => Some (this)
          
        | Literal.Integer (val, is_neg, cur) =>
          if (cur.Equals (t)) Some (this)
          else
            if (Typer.LiteralConversionPossible (this, t))
              Some (Literal.Integer (val, is_neg, t :> MType.Class))
            else
              None ()

        | _ => None ()
      }
    }
    
    public WithProperType () : Literal
    {
      match (this) {
        | Literal.Integer (0, true, _) => Literal.Integer (0, false, InternalType.Int32)
        | Literal.Integer (val, is_neg, _) =>
          def t =
            if (is_neg)
              if (val - 1 <= int.MaxValue)
                InternalType.Int32
              else
                InternalType.Int64
            else
              if (val <= int.MaxValue)
                InternalType.Int32
              else if (val <= uint.MaxValue)
                InternalType.UInt32
              else if (val <= System.Convert.ToUInt64 (long.MaxValue))
                InternalType.Int64
              else
                InternalType.UInt64;
          Literal.Integer (val, is_neg, t)
        | _ => this
      }
    }

    public AsObject () : object
    {
      match (this) {
        | Literal.Void => assert (false)
        | Literal.Null => null
        | Literal.Char (c) => c : object
        | Literal.String (s) => s
        | Literal.Float (f) => f
        | Literal.Double (d) => d
        | Literal.Decimal (d) => d
        | Literal.Bool (b) => b
        | Literal.Enum (l, t) =>
          def t = t.SystemType;
          if (t is System.Reflection.Emit.EnumBuilder || t is System.Reflection.Emit.TypeBuilder)
            l.AsObject ()
          else
            System.Enum.ToObject (t, l.AsObject ())
        | Literal.Integer (val, is_neg, t) =>
          def t = if (t == null) InternalType.Int32 else t;
          
          if (t.Equals (InternalType.UInt64)) {
            assert (!is_neg);
            val : object
          } else if (val == 0x8000000000000000UL) {
            assert (is_neg);
            assert (t.Equals (InternalType.Int64));
            long.MinValue : object
          } else {
            def val = 
              if (is_neg) 
                -System.Convert.ToInt64 (val)
              else
                System.Convert.ToInt64 (val);
            match (t.tycon.FullName) {
              | "System.UInt32" => (val :> uint) : object
              | "System.Int32" => val :> int
              | "System.Int16" => val :> short
              | "System.UInt16" => val :> ushort
              | "System.SByte" => val :> System.SByte
              | "System.Byte" => val :> System.Byte
              | "System.Int64" => val
              | _ => assert (false, t.tycon.FullName)
            }
          }
      }
    }

    public AsInt : option [int]
    {
      get {
        match (this) {
          | Literal.Integer (0x80000000ul, true, _) => Some (int.MinValue)          
          | Literal.Integer (x, neg, _) when x & 0x7FFFFFFFul == x =>
            Some (if (neg) -(x :> int) else x :> int)
          | _ => None ()
        }
      }
    }

    public AsSByte : option [sbyte]
    {
      get {
        match (this) {
          | Literal.Integer (0x80ul, true, _) => Some (sbyte.MinValue)          
          | Literal.Integer (x, neg, _) when x & 0x7Ful == x =>
            Some (if (neg) -(x :> int) :> sbyte else x :> sbyte)
          | _ => None ()
        }
      }
    }

    public AsByte : option [byte]
    {
      get {
        match (this) {
          | Literal.Integer (x, false, _) when x <= byte.MaxValue => Some (x :> byte)
          | _ => None ()
        }
      }
    }

    public AsShort : option [short]
    {
      get {
        match (this) {
          | Literal.Integer (0x8000ul, true, _) => Some (short.MinValue)          
          | Literal.Integer (x, neg, _) when x & 0x7FFFul == x =>
            Some (if (neg) -(x :> int) :> short else x :> short)
          | _ => None ()
        }
      }
    }

    public AsUShort : option [ushort]
    {
      get {
        match (this) {
          | Literal.Integer (x, false, _) when x <= ushort.MaxValue => Some (x :> ushort)
          | _ => None ()
        }
      }
    }

    public AsUInt : option [uint]
    {
      get {
        match (this) {
          | Literal.Integer (x, false, _) when x <= uint.MaxValue => Some (x :> uint)
          | _ => None ()
        }
      }
    }

    public AsLong : option [long]
    {
      get {
        match (this) {
          | Literal.Integer (0x8000000000000000ul, true, _) => Some (long.MinValue)          
          | Literal.Integer (x, neg, _) when x & 0x8000000000000000ul != 0ul =>
            Some (if (neg) -(x :> long) else x :> long)
          | _ => None ()
        }
      }
    }

    public AsULong : option [ulong]
    {
      get {
        match (this) {
          | Literal.Integer (x, false, _) => Some (x)
          | _ => None ()
        }
      }
    }

    public static FromInt (x : int) : Literal.Integer
    {
      if (x == int.MinValue)
        Literal.Integer (0x80000000UL, true, InternalType.Int32)
      else
        Literal.Integer (Abs (x) :> ulong, x < 0, InternalType.Int32)
    }

    public static FromSByte (x : sbyte) : Literal.Integer
    {
      if (x == sbyte.MinValue)
        Literal.Integer (0x80UL, true, InternalType.SByte)
      else
        Literal.Integer (Abs (x) :> ulong, x < 0, InternalType.SByte)
    }

    public static FromByte (x : byte) : Literal.Integer
    {
      Literal.Integer (x, false, InternalType.Byte)
    }

    public static FromShort (x : short) : Literal.Integer
    {
      if (x == short.MinValue)
        Literal.Integer (0x8000UL, true, InternalType.Int16)
      else
        Literal.Integer (Abs (x) :> ulong, x < 0, InternalType.Int16)
    }

    public static FromUShort (x : ushort) : Literal.Integer
    {
      Literal.Integer (x, false, InternalType.UInt16)
    }

    public static FromUInt (x : uint) : Literal.Integer
    {
      Literal.Integer (x, false, InternalType.UInt32)
    }

    public static FromLong (x : long) : Literal.Integer
    {
      if (x == long.MinValue)
        Literal.Integer (0x8000000000000000UL, true, InternalType.Int64)
      else
        Literal.Integer (Abs (x) :> ulong, x < 0, InternalType.Int64)
    }

    public static FromULong (x : ulong) : Literal.Integer
    {
      Literal.Integer (x, false, InternalType.UInt64)
    }

    public static FromObject (o : object) : Literal
    {
      if (o == null) Null ()
      else
        match (o) {
          | o is bool => Bool (o)
          | o is string => String (o)
          | o is float => Float (o)
          | o is double => Double (o)
          | o is decimal => Decimal (o)
          | o is char => Char (o)
          
          | o is int => FromInt (o)
          | o is sbyte => FromSByte (o)
          | o is byte => FromByte (o)
          | o is short => FromShort (o)
          | o is ushort => FromUShort (o)
          | o is uint => FromUInt (o)
          | o is long => FromLong (o)
          | o is ulong => FromULong (o)

          | o is System.Enum =>
            def s = o.ToString ("d");
            def tyname = NString.Split (o.GetType ().FullName, '+', '.');
            def tc = NamespaceTree.LookupInternalType (tyname);
            def t = tc.GetMemType ();
            def int_lit =
              if (s [0] == '-')
                Integer (ulong.Parse (s.Substring (1)), true, t)
              else
                Integer (ulong.Parse (s), false, t);
            Enum (int_lit, tc)

          | _ =>
            Util.ice ($ "cannot create literal from $o")
        }
    }
  }

  public variant FunBody
  {
    | Parsed { expr : Parsetree.PExpr; }
    | Typed { expr : Typedtree.TExpr; }
    | ILed
    | Abstract  // for interface method
  }

  public variant FunKind
  {
    | Method { f_implements : list [Parsetree.PExpr]; }
    | BoundMethod { f_implements : list [IMethod]; }
    | Constructor
    | StaticConstructor
    | Function
  }
} // Nemerle.Compiler
