/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */ 

namespace Sioux
{
    using Nemerle.Collections;
    
    using System;
    
    
    /**
     * An HTTP cookie
     */
    public class Cookie
    {
        /* -- CONSTRUCTORS ------------------------------------------- */
        
        /**
         * Class representing http cookie
         */
        public this (name : string)
        {
            this.name = name;
            this.path = "/"
        }

        public this (name : string,value : string)
        {
            this.name = name;
            this.value = value;
            this.path = "/"
        }

        public this (name : string,value : string,expires : DateTime)
        {
            this.name = name;
            this.value = value;
            this.expires = expires;
            this.expires_set = true;
            this.path = "/"
        }

        /* -- PUBLIC PROPERTIES -------------------------------------- */


        /**
         *  Getting name of cookie
         */
        public Name : string
        {
            get { name }
        }

        
        /**
         *  Getting and setting value of cookie
         */
        public Value : string
        {
            get { this.value }
            set { this.value = value }
        }

        
        /**
         *  Getting and setting virtual path of cookie
         */
        public Path : string
        {
            get { path }
            set { path = value }
        }

        
        /**
         * Gets or sets domain to associate the cookie with.
         */
        public  Domain : string
        {
            get { domain; }
            set { domain = value }
        }


        /**
         * Gets or sets the expiration date and time for the cookie.
         */
        public Expires : DateTime
        {
            get
            {
                if (!expires_set)
                    DateTime.MinValue
                else    
                    expires
            }
                
            set
            {
                expires_set = true;
                expires = value
            }
        }

        
        /**
         * Gets or sets a value indicating whether to transmit the cookie using SSL, that is, over HTTPS only.
         */
        public Secure : bool
        {
            get { secure; }
            set { secure = value }
        }

        
        /**
         *  Gets a header pair representing cookie, where first elem is header name, and second is hedaer value
         */
        public Header : string * string
        {
            get
            {
                def SetCookie = System.Text.StringBuilder ();
                
                when (null != name && name.Length > 0)
                {
                    ignore(SetCookie.Append (name));
                    ignore(SetCookie.Append ("="))
                }
                
                when (null != value)
                {
                    ignore(SetCookie.Append (value))
                }
            
                when (null != domain && domain.Length > 0)
                {
                    ignore(SetCookie.Append ("; domain="));
                    ignore(SetCookie.Append (domain))
                }
            
                when (null != path && path.Length > 0)
                {
                    ignore(SetCookie.Append ("; path="));
                    ignore(SetCookie.Append (path))
                }

                when (expires_set && expires != DateTime.MinValue) {
                    ignore(SetCookie.Append ("; expires="));
                    ignore(SetCookie.Append (expires.ToUniversalTime ().ToString ("ddd, dd-MMM-yyyy HH':'mm':'ss '+0000'")))
                }

                when (secure)
                    ignore(SetCookie.Append ("; secure"));
                           
                ("Set-Cookie",SetCookie.ToString())
            }
        }
        /* -- PUBLIC METHODS ------------------------------------------ */

        public override ToString() : string
        {
            name + ": " + value
        }
        
        /* -- PRIVATE FIELDS ------------------------------------------ */

        private name : string;
        private mutable value : string;
        private mutable domain : string;
        private mutable expires : DateTime;
        private mutable expires_set : bool;
        private mutable path : string;
        private mutable secure : bool;
    }
}