/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2007 Sun Microsystems, Inc.
 */
package org.netbeans.modules.editor.settings.storage.codetemplates;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.netbeans.api.editor.mimelookup.MimePath;
import org.netbeans.api.editor.settings.CodeTemplateDescription;
import org.netbeans.lib.editor.util.CharacterConversions;
import org.netbeans.modules.editor.settings.storage.SettingsType;
import org.netbeans.modules.editor.settings.storage.Utils;
import org.netbeans.modules.editor.settings.storage.XMLStorage;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileSystem;
import org.openide.filesystems.FileUtil;
import org.openide.filesystems.Repository;
import org.openide.xml.XMLUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 *
 * @author vita
 */
public final class CodeTemplatesStorage {

    private static final Logger LOG = Logger.getLogger(CodeTemplatesStorage.class.getName());
    
    public static Map<String, CodeTemplateDescription> load(MimePath mimePath, boolean defaults) {
        assert mimePath != null : "The parameter mimePath must not be null"; //NOI18N
        
        FileObject baseFolder = Repository.getDefault().getDefaultFileSystem().findResource("Editors"); //NOI18N
        Map<String, List<Object []>> files = new HashMap<String, List<Object []>>();
        SettingsType.CODETEMPLATES.getLocator().scan(baseFolder, mimePath.getPath(), null, true, true, !defaults, files);
        
        assert files.size() <= 1 : "Too many results in the scan"; //NOI18N

        List<Object []> profileInfos = files.get(null);
        if (profileInfos == null) {
            return Collections.<String, CodeTemplateDescription>emptyMap();
        }
        
        Map<String, CodeTemplateDescription> codeTemplatesMap = new HashMap<String, CodeTemplateDescription>();
        for(Object [] info : profileInfos) {
            FileObject profileHome = (FileObject) info[0];
            FileObject settingFile = (FileObject) info[1];
            boolean modulesFile = ((Boolean) info[2]).booleanValue();

            XMLStorage.Handler reader;
            if (MIME_TYPE.equals(settingFile.getMIMEType())) {
                reader = new Reader();
            } else {
                // assume legacy file
                reader = new LegacyReader();
            }
            
            // Load keybindings from the settingFile
            @SuppressWarnings("unchecked")
            Object [] loadedData = (Object []) XMLStorage.load(settingFile, reader);
            @SuppressWarnings("unchecked")
            Map<String, CodeTemplateDescription> addedTemplates = (Map<String, CodeTemplateDescription>) loadedData[0];
            @SuppressWarnings("unchecked")
            Collection<String> removedTemplates = (Collection<String>)loadedData[1];

            if (LOG.isLoggable(Level.FINE)) {
                LOG.fine("Loading codetemplates from: '" + settingFile.getPath() + "'"); //NOI18N
            }

            if (LOG.isLoggable(Level.FINEST)) {
                LOG.finest("--- Removing codetemplates: " + removedTemplates); //NOI18N
            }
            
            // First remove all code templates marked as removed
            for(String abbreviation : removedTemplates) {
                codeTemplatesMap.remove(abbreviation);
            }

            if (LOG.isLoggable(Level.FINEST)) {
                LOG.finest("--- Adding codetemplates: " + addedTemplates); //NOI18N
            }
            
            // Then add all new bindings
            codeTemplatesMap.putAll(addedTemplates);

            if (LOG.isLoggable(Level.FINEST)) {
                LOG.finest("-------------------------------------"); //NOI18N
            }
        }
        
        return Collections.unmodifiableMap(codeTemplatesMap);
    }

    public static void save(
        MimePath mimePath, 
        boolean defaults, 
        final Collection<CodeTemplateDescription> codeTemplates,
        final Collection<String> removedCodeTemplates
    ) {
        assert mimePath != null : "The parameter mimePath must not be null"; //NOI18N
        
        final FileSystem sfs = Repository.getDefault().getDefaultFileSystem();
        final String settingFileName = SettingsType.CODETEMPLATES.getLocator().getWritableFileName(
                mimePath.getPath(), null, null, defaults);

        try {
            sfs.runAtomicAction(new FileSystem.AtomicAction() {
                public void run() throws IOException {
                    FileObject baseFolder = sfs.findResource("Editors"); //NOI18N
                    FileObject f = FileUtil.createData(baseFolder, settingFileName);
                    saveCodeTemplates(f, codeTemplates, removedCodeTemplates);
                }
            });
        } catch (IOException ioe) {
            LOG.log(Level.WARNING, "Can't save editor codetempaltes for " + mimePath.getPath(), ioe); //NOI18N
        }
    }
    
    public static void delete(MimePath mimePath, boolean defaults) {
        assert mimePath != null : "The parameter mimePath must not be null"; //NOI18N
        
        FileSystem sfs = Repository.getDefault().getDefaultFileSystem();
        FileObject baseFolder = sfs.findResource("Editors"); //NOI18N
        Map<String, List<Object []>> files = new HashMap<String, List<Object []>>();
        SettingsType.CODETEMPLATES.getLocator().scan(baseFolder, mimePath.getPath(), null, true, defaults, !defaults, files);
        
        assert files.size() <= 1 : "Too many results in the scan"; //NOI18N

        final List<Object []> profileInfos = files.get(null);
        if (profileInfos != null) {
            try {
                sfs.runAtomicAction(new FileSystem.AtomicAction() {
                    public void run() {
                        for(Object [] info : profileInfos) {
                            FileObject settingFile = (FileObject) info[1];
                            try {
                                settingFile.delete();
                            } catch (IOException ioe) {
                                LOG.log(Level.WARNING, "Can't delete editor settings file " + settingFile.getPath(), ioe); //NOI18N
                            }
                        }
                    }
                });
            } catch (IOException ioe) {
                LOG.log(Level.WARNING, "Can't delete editor codetemplates for " + mimePath.getPath(), ioe); //NOI18N
            }
        }
    }

    // ---------------------------------------------------------
    // Private implementation
    // ---------------------------------------------------------
    
    private static final String E_ROOT = "codetemplates"; //NOI18N
    private static final String E_CODETEMPLATE = "codetemplate"; //NOI18N
    private static final String E_DESCRIPTION = "description"; //NOI18N
    private static final String E_CODE = "code"; //NOI18N
    private static final String A_ABBREV = "abbreviation"; //NOI18N
    private static final String A_DESCRIPTION_ID = "descriptionId"; //NOI18N
    private static final String A_CONTEXTS = "contexts"; //NOI18N
    private static final String A_UUID = "uuid"; //NOI18N
    private static final String A_REMOVE = "remove"; //NOI18N
    private static final String A_XML_SPACE = "xml:space"; //NOI18N
    private static final String V_PRESERVE = "preserve"; //NOI18N

    private static final String PUBLIC_ID = "-//NetBeans//DTD Editor Code Templates settings 1.0//EN"; //NOI18N
    private static final String SYSTEM_ID = "http://www.netbeans.org/dtds/EditorCodeTemplates-1_0.dtd"; //NOI18N
            
    private static final String MIME_TYPE = "text/x-nbeditor-codetemplatesettings"; //NOI18N
    
    private CodeTemplatesStorage() {
    }

    private static final class Reader extends XMLStorage.Handler {

        private Map<String, CodeTemplateDescription> codeTemplatesMap = new HashMap<String, CodeTemplateDescription>();
        private Collection<String> removedTemplates = new ArrayList<String>();
        
        // The code template being processed
        private String abbreviation = null;
        private String description = null;
        private String code = null;
        private List<String> contexts = null;
        private String uuid = null;

        private StringBuilder text = null;
        private StringBuilder cdataText = null;
        private boolean insideCdata = false;
        
        @Override
        public Object getResult() {
            return new Object [] { codeTemplatesMap, removedTemplates };
        }

        @Override
        public void characters(char[] ch, int start, int length) throws SAXException {
            if (text != null) {
                text.append(ch, start, length);
                
                if (insideCdata) {
                    cdataText.append(ch, start, length);
                }
            }
        }

        @Override
        public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
            if (qName.equals(E_ROOT)) {
                // We don't read anything from the root
            } else if (qName.equals(E_CODETEMPLATE)) {
                boolean removed = Boolean.valueOf(attributes.getValue(A_REMOVE));
                
                if (removed) {
                    String abbrev = attributes.getValue(A_ABBREV);
                    removedTemplates.add(abbrev);
                    
                    abbreviation = null;
                    description = null;
                    contexts = null;
                    uuid = null;
                    text = null;
                    cdataText = null;
                } else {
                    // Read the abbreviation
                    abbreviation = attributes.getValue(A_ABBREV);
                    
                    // Read the description and localize it
                    description = attributes.getValue(A_DESCRIPTION_ID);
                    if (description != null) {
                        String localizedDescription = Utils.getLocalizedName(
                                getProcessedFile(), description, null);
                        if (localizedDescription != null) {
                            description = localizedDescription;
                        }
                    }
                    
                    // Read contexts associated with this template
                    String ctxs = attributes.getValue(A_CONTEXTS);
                    if (ctxs != null) {
                        String [] arr = ctxs.split(","); //NOI18N
                        contexts = new ArrayList<String>(arr.length);
                        
                        for(String context : arr) {
                            context = context.trim();
                            if (context.length() > 0) {
                                contexts.add(context);
                            }
                        }
                    } else {
                        contexts = null;
                    }
                    
                    // Read the unique id
                    uuid = attributes.getValue(A_UUID);
                }
            } else if (qName.equals(E_CODE)) {
                if (abbreviation != null) {
                    // Initiate the new builder for the code template parametrized text
                    text = new StringBuilder();
                    cdataText = new StringBuilder();
                    insideCdata = false;
                }
            } else if (qName.equals(E_DESCRIPTION)) {
                if (abbreviation != null) {
                    // Initiate the new builder for the code template description
                    text = new StringBuilder();
                    cdataText = new StringBuilder();
                    insideCdata = false;
                }
            }
        }

        @Override
        public void endElement(String uri, String localName, String qName) throws SAXException {
            if (qName.equals(E_ROOT)) {
                // We don't read anything from the root
            } else if (qName.equals(E_CODETEMPLATE)) {
                if (abbreviation != null) {
                    CodeTemplateDescription template = new CodeTemplateDescription(
                        abbreviation,
                        description == null ? null : CharacterConversions.lineSeparatorToLineFeed(description),
                        code == null ? "" : CharacterConversions.lineSeparatorToLineFeed(code), //NOI18N
                        contexts,
                        uuid
                    );
                    codeTemplatesMap.put(abbreviation, template);
                }
            } else if (qName.equals(E_CODE)) {
                if (text != null) {
                    code = cdataText.length() > 0 ? cdataText.toString() : text.toString();
                }
            } else if (qName.equals(E_DESCRIPTION)) {
                if (text != null) {
                    if (cdataText.length() > 0) {
                        description = cdataText.toString();
                    } else if (text.length() > 0) {
                        description = text.toString();
                    }
                }
            }
        }

        @Override
        public void startCDATA() throws SAXException {
            if (cdataText != null) {
                insideCdata = true;
            }
        }
        
        @Override
        public void endCDATA() throws SAXException {
            if (cdataText != null) {
                insideCdata = false;
            }
        }
    } // End of Reader class
    
    private static final class LegacyReader extends XMLStorage.Handler {

        private static final String EL_ROOT = "abbrevs"; //NOI18N
        private static final String EL_CODETEMPLATE = "abbrev"; //NOI18N
        private static final String AL_ABBREV = "key"; //NOI18N
        private static final String AL_REMOVE = "remove"; //NOI18N
        
        private Map<String, CodeTemplateDescription> codeTemplatesMap = new HashMap<String, CodeTemplateDescription>();
        private Collection<String> removedTemplates = new ArrayList<String>();
        
        // The code template being processed
        private String abbreviation = null;
        private StringBuilder text = null;
        
        @Override
        public Object getResult() {
            return new Object [] { codeTemplatesMap, removedTemplates };
        }

        @Override
        public void characters(char[] ch, int start, int length) throws SAXException {
            if (text != null) {
                text.append(ch, start, length);
            }
        }

        @Override
        public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
            if (qName.equals(EL_ROOT)) {
                // We don't read anything from the root
            } else if (qName.equals(EL_CODETEMPLATE)) {
                boolean removed = Boolean.valueOf(attributes.getValue(AL_REMOVE));
                
                if (removed) {
                    String abbrev = attributes.getValue(AL_ABBREV);
                    removedTemplates.add(abbrev);
                    
                    abbreviation = null;
                    text = null;
                } else {
                    abbreviation = attributes.getValue(AL_ABBREV);
                    text = new StringBuilder();
                }
            }
        }

        @Override
        public void endElement(String uri, String localName, String qName) throws SAXException {
            if (qName.equals(EL_ROOT)) {
                // We don't read anything from the root
            } else if (qName.equals(EL_CODETEMPLATE)) {
                if (abbreviation != null) {
                    String parametrizedText = text.toString().replaceFirst(
                        "([^|]+)[|]([^|]+)", "$1\\${cursor}$2"); // NOI18N
                    
                    CodeTemplateDescription template = new CodeTemplateDescription(
                        abbreviation,
                        null,
                        CharacterConversions.lineSeparatorToLineFeed(parametrizedText),
                        null,
                        null
                    );
                    codeTemplatesMap.put(abbreviation, template);
                }
            }
        }
    } // End of LegacyReader class
    
    private static void saveCodeTemplates(
        FileObject fileObject,
        Collection<CodeTemplateDescription> codeTemplates,
        Collection<String> removedCodeTemplates
    ) {
        Document doc = XMLUtil.createDocument(E_ROOT, null, PUBLIC_ID, SYSTEM_ID);
        Node root = doc.getElementsByTagName(E_ROOT).item(0);
        
        for(CodeTemplateDescription codeTemplate : codeTemplates) {
            Element element = doc.createElement(E_CODETEMPLATE);
            root.appendChild(element);
            
            // Store template's abbreviation
            element.setAttribute(A_ABBREV, codeTemplate.getAbbreviation());
            
            // Store template's contexts
            List<String> contexts = codeTemplate.getContexts();
            if (contexts != null && !contexts.isEmpty()) {
                StringBuilder sb = new StringBuilder();
                for(int i = 0; i < contexts.size(); i++) {
                    String ctx = contexts.get(i);
                    if (ctx != null) {
                        ctx = ctx.trim();
                        if (ctx.length() > 0) {
                            if (i > 0) {
                                sb.append(","); //NOI18N
                            }
                            sb.append(ctx);
                        }
                    }
                }
                
                if (sb.length() > 0) {
                    element.setAttribute(A_CONTEXTS, sb.toString());
                }
            }

            // Store template's unique id
            String uuid = codeTemplate.getUniqueId();
            if (uuid != null) {
                element.setAttribute(A_UUID, uuid);
            }
            
            // Just some XML crap to preserve whitespace
            element.setAttribute(A_XML_SPACE, V_PRESERVE);

            // Store template's code
            String code = codeTemplate.getParametrizedText();
            if (code.length() > 0) {
                Element codeElement = doc.createElement(E_CODE);
                codeElement.appendChild(doc.createCDATASection(
                    CharacterConversions.lineFeedToLineSeparator(code)));
                element.appendChild(codeElement);
            }
            
            // Store template's description
            String description = codeTemplate.getDescription();
            if (description != null && description.length() > 0) {
                Element descriptionElement = doc.createElement(E_DESCRIPTION);
                descriptionElement.appendChild(doc.createCDATASection(
                    CharacterConversions.lineFeedToLineSeparator(description)));
                element.appendChild(descriptionElement);
            }
        }
        
        for(String abbreviation : removedCodeTemplates) {
            Element element = doc.createElement(E_CODETEMPLATE);
            root.appendChild(element);
            
            element.setAttribute(A_ABBREV, abbreviation);
            element.setAttribute(A_REMOVE, Boolean.TRUE.toString());
        }
        
        XMLStorage.save(fileObject, doc);
    }
}
