/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.cnd.apt.impl.support;

import java.io.IOException;
import java.util.List;
import java.util.logging.Level;
import java.util.zip.Adler32;
import java.util.zip.Checksum;
import org.netbeans.modules.cnd.apt.support.APTIncludeHandler;
import org.netbeans.modules.cnd.apt.support.APTMacroMap;
import org.netbeans.modules.cnd.apt.support.APTPreprocHandler;
import org.netbeans.modules.cnd.apt.support.IncludeDirEntry;
import org.netbeans.modules.cnd.apt.utils.APTSerializeUtils;
import org.netbeans.modules.cnd.apt.utils.APTUtils;
import org.netbeans.modules.cnd.repository.api.Repository;
import org.netbeans.modules.cnd.repository.spi.RepositoryDataInput;
import org.netbeans.modules.cnd.repository.spi.RepositoryDataOutput;
import org.openide.filesystems.FileSystem;
import org.openide.util.CharSequences;

/**
 * composition of include handler and macro map for parsing file phase
 * @author Vladimir Voskresensky
 */
public class APTPreprocHandlerImpl implements APTPreprocHandler {
    private boolean compileContext;
    private boolean isValid = true;
    private CharSequence lang;
    private CharSequence flavor;
    private long cuCRC;
    private APTMacroMap macroMap;
    private APTIncludeHandler inclHandler;
    
    /**
     * @param compileContext determine whether state created for real parse-valid
     * context, i.e. source file has always correct state, but header itself has
     * not correct state until it was included into any source file (may be recursively)
     */
    public APTPreprocHandlerImpl(APTMacroMap macroMap, APTIncludeHandler inclHandler, boolean compileContext, CharSequence lang, CharSequence flavor) {
        this.macroMap = macroMap;
        this.inclHandler = inclHandler;
        this.compileContext = compileContext;
        assert lang != null;
        this.lang = lang;
        assert flavor != null;
        this.flavor = flavor;
        this.cuCRC = countCompilationUnitCRC(inclHandler.getStartEntry().getStartFileProject().getUnitId());
    }
    
    @Override
    public APTMacroMap getMacroMap() {
        return macroMap;
    }

    @Override
    public APTIncludeHandler getIncludeHandler() {
        return inclHandler;
    }   
    
    ////////////////////////////////////////////////////////////////////////////
    // manage state (save/restore)
    
    @Override
    public State getState() {
        return createStateImpl();
    }
    
    @Override
    public void setState(State state) {
        if (state instanceof StateImpl) {
            ((StateImpl)state).restoreTo(this);
        }
    }
    
    public void setValid(boolean isValid) {
        this.isValid = isValid;
    }

    @Override
    public boolean isValid() {
        return isValid;
    }
    
    @Override
    public boolean isCompileContext() {
        return compileContext;
    }
    
    protected StateImpl createStateImpl() {
        return new StateImpl(this);
    }
    
    private void setCompileContext(boolean state) {
        this.compileContext = state;
    }

    @Override
    public CharSequence getLanguage() {
        return lang;
    }

    @Override
    public CharSequence getLanguageFlavor() {
        return flavor;
    }

    long getCompilationUnitCRC() {
        return cuCRC;
    }
    
    private long countCompilationUnitCRC(int unitId) {
	Checksum checksum = new Adler32();
	updateCrc(checksum, lang.toString());
	updateCrc(checksum, flavor.toString());
	updateCrcByFSPaths(checksum, ((APTIncludeHandlerImpl)inclHandler).getSystemIncludePaths(), unitId);
	updateCrcByFSPaths(checksum, ((APTIncludeHandlerImpl)inclHandler).getUserIncludePaths(), unitId);
	updateCrcByFSPaths(checksum, ((APTIncludeHandlerImpl)inclHandler).getUserIncludeFilePaths(), unitId);
        long value = checksum.getValue();
        value += APTHandlersSupportImpl.getCompilationUnitCRC(macroMap);        
	return value;
        
    }

    private void updateCrc(Checksum checksum, String s) {
	checksum.update(s.getBytes(), 0, s.length());
    }
    
    private void updateCrcByFSPaths(Checksum checksum, List<IncludeDirEntry> paths, int unitId) {
	for( IncludeDirEntry path : paths ) {
            int id = Repository.getFileIdByName(unitId, path.getAsSharedCharSequence());
	    checksum.update(id);
            //updateCrc(checksum, path.getPath());
	}
    }

    public final static class StateImpl implements State {
        /*package*/ final CharSequence lang;
        /*package*/ final CharSequence flavor;
        /*package*/ final APTMacroMap.State macroState;
        /*package*/ final APTIncludeHandler.State inclState;
        private final byte attributes;
        private final long cuCRC;
        
        private final static byte COMPILE_CONTEXT_FLAG = 1 << 0;
        private final static byte CLEANED_FLAG = 1 << 1;
        private final static byte VALID_FLAG = 1 << 2;
        
        private static byte createAttributes(boolean compileContext, boolean cleaned, boolean valid) {
            byte out = 0;
            if (compileContext) {
                out |= COMPILE_CONTEXT_FLAG;
            } else {
                out &= ~COMPILE_CONTEXT_FLAG;
            }
            if (cleaned) {
                out |= CLEANED_FLAG;
            } else {
                out &= ~CLEANED_FLAG;
            }
            if (valid) {
                out |= VALID_FLAG;
            } else {
                out &= ~VALID_FLAG;
            }
            return out;
        }
        
        protected StateImpl(APTPreprocHandlerImpl handler) {
            if (handler.getMacroMap() != null) {
                this.macroState = handler.getMacroMap().getState();
            } else {
                this.macroState = null;
            }
            if (handler.getIncludeHandler() != null) {
                this.inclState = handler.getIncludeHandler().getState();
            } else {
                this.inclState = null;
            }
            this.attributes = createAttributes(handler.isCompileContext(), false, handler.isValid());
            this.lang = handler.lang;
            this.flavor = handler.flavor;
            this.cuCRC = handler.cuCRC;
        }
        
        private StateImpl(StateImpl other, boolean cleanState, boolean compileContext, boolean valid) {
            boolean cleaned;
            if (cleanState && !other.isCleaned()) {
                // first time cleaning
                // own copy of include information and macro state
                this.inclState = APTHandlersSupportImpl.copyIncludeState(other.inclState, true);
                this.macroState = APTHandlersSupportImpl.createCleanMacroState(other.macroState);
                cleaned = true;
            } else {
                // share states
                this.macroState = other.macroState;
                cleaned = other.isCleaned();
                this.inclState = other.inclState;
            }
            this.attributes = createAttributes(compileContext, cleaned, valid);
            this.lang = other.lang;
            this.flavor = other.flavor;
            this.cuCRC = other.cuCRC;
        }
        
        private void restoreTo(APTPreprocHandlerImpl handler) {
            if (handler.getMacroMap() != null) {
                handler.getMacroMap().setState(this.macroState);
            }
            if (handler.getIncludeHandler() != null) {
                handler.getIncludeHandler().setState(this.inclState);
            }
            handler.setCompileContext(this.isCompileContext());
            handler.lang = this.lang;
            handler.flavor = this.flavor;
            handler.cuCRC = this.cuCRC;
            handler.setValid(this.isValid());
            if (!isValid()) {
                APTUtils.LOG.log(Level.SEVERE, "setting invalid state {0}", new Object[] { this } ); // NOI18N
            }
        }

        @Override
        public String toString() {
            StringBuilder retValue = new StringBuilder();
            retValue.append(isCleaned() ? "\nCleaned State;" : "\nNot Cleaned State;"); // NOI18N
            retValue.append(isCompileContext() ? "Compile Context;" : "Default/Null State;"); // NOI18N
            retValue.append(isValid() ? "Valid State;" : "Invalid State;"); // NOI18N
            retValue.append("\nInclude state Info:\n"); // NOI18N
            retValue.append(inclState);
            retValue.append("\nMACROS state info:\n"); // NOI18N
            retValue.append(this.macroState);
            return retValue.toString();
        }

        boolean equalsIgnoreInvalidFlag(State obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null || (obj.getClass() != this.getClass())) {
                return false;
            }
            StateImpl other = (StateImpl) obj;
            // we do not compare macroStates because in case of
            // parsing from the same include sequence they are equal
            if (this.isCompileContext() != other.isCompileContext()) {
                return false;
            }
            if (this.inclState != other.inclState && (this.inclState == null || !this.inclState.equals(other.inclState))) {
                return false;
            }
            return true;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null || (obj.getClass() != this.getClass())) {
                return false;
            }
            StateImpl other = (StateImpl)obj;
            // we do not compare macroStates because in case of 
            // parsing from the same include sequence they are equal
            if (this.isCompileContext() != other.isCompileContext()) {
                return false;
            }
            if (this.isValid() != other.isValid()) {
                return false;
            }
            if (this.inclState != other.inclState && (this.inclState == null || !this.inclState.equals(other.inclState))) {
                return false;
            }
            return true;
        }

        @Override
        public int hashCode() {
            int hash = 5;
            hash = 83 * hash + (this.isCompileContext() ? 1 : 0);
            hash = 83 * hash + (this.isValid() ? 1 : 0);
            hash = 83 * hash + (this.inclState != null ? this.inclState.hashCode() : 0);
            return hash;
        }
                
        @Override
        public boolean isCompileContext() {
            return (this.attributes & COMPILE_CONTEXT_FLAG) == COMPILE_CONTEXT_FLAG;
        }
        
        @Override
        public boolean isCleaned() {
            return (this.attributes & CLEANED_FLAG) == CLEANED_FLAG;
        }

        @Override
        public boolean isValid() {
            return (this.attributes & VALID_FLAG) == VALID_FLAG;
        }
        
        /*package*/ APTPreprocHandler.State copy() {
            return new StateImpl(this, this.isCleaned(), this.isCompileContext(), this.isValid());
        }
        
        /*package*/ APTPreprocHandler.State copyCleaned() {
            return new StateImpl(this, true, this.isCompileContext(), this.isValid());
        }
        
        /*package*/ APTPreprocHandler.State copyInvalid() {
            return new StateImpl(this, this.isCleaned(), this.isCompileContext(), false);
        }

        ////////////////////////////////////////////////////////////////////////
        // persistence support

        public void write(RepositoryDataOutput output, int unitIndex) throws IOException {
            output.writeByte(this.attributes);
            APTSerializeUtils.writeIncludeState(this.inclState, output, unitIndex);
            APTSerializeUtils.writeMacroMapState(this.macroState, output);
            output.writeCharSequenceUTF(lang);
            output.writeCharSequenceUTF(flavor);
            output.writeLong(cuCRC);
        }

        public StateImpl(FileSystem fs, RepositoryDataInput input, int unitIndex) throws IOException {
            this.attributes = input.readByte();
            this.inclState = APTSerializeUtils.readIncludeState(fs, input, unitIndex);
            this.macroState = APTSerializeUtils.readMacroMapState(input);
            this.lang = input.readCharSequenceUTF();
            this.flavor = input.readCharSequenceUTF();
            this.cuCRC = input.readLong();
        }

        @Override
        public CharSequence getLanguage() {
            return lang;
        }

        @Override
        public CharSequence getLanguageFlavor() {
            return flavor;
        }

        public long getCRC() {
            return cuCRC;
        }
    }    
    
    ////////////////////////////////////////////////////////////////////////////
    // implementation details
    
    @Override
    public String toString() {
        StringBuilder retValue = new StringBuilder();
        retValue.append(this.isCompileContext() ? "\nCompile Context" : "\nDefault/Null State"); // NOI18N
        retValue.append("\nInclude Info:\n"); // NOI18N
        retValue.append(this.inclHandler);
        retValue.append("\nMACROS info:\n"); // NOI18N
        retValue.append(this.macroMap);
        return retValue.toString();
    }
}
