/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>

#include <netdude/nd.h>
#include <netdude/nd_debug.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_protocol_inst.h>
#include <netdude/nd_protocol.h>

void
nd_proto_inst_init(ND_ProtoInst *pi,
		   ND_Protocol *proto,
		   guint nesting)
{
  if (!pi)
    return;

  memset(pi, 0, sizeof(ND_ProtoInst));
  pi->proto = proto;
  pi->nesting = nesting;
}


const char       *
nd_proto_inst_to_string(const ND_Protocol *proto, guint nesting)
{
  static char s[MAXPATHLEN];

  if (!proto)
    return NULL;

  g_snprintf(s, MAXPATHLEN, "%s_%u", proto->name, nesting);
  return s;
}


ND_ProtoData     *
nd_proto_data_new(ND_Protocol *protocol, guint nesting, guchar *data, guchar *data_end)
{
  ND_ProtoData *pd;

  pd = g_new0(ND_ProtoData, 1);
  D_ASSERT_PTR(pd);
  if (!pd)
    return NULL;

  if (data_end < data)
    {
      D(("Assertion failed -- data end smaller than data start!\n"));
      return NULL;
    }

  nd_proto_inst_init(&pd->inst, protocol, nesting);
  pd->data = data;
  pd->data_end = data_end;

  return pd;
}


void              
nd_proto_data_free(ND_ProtoData *pd)
{
  g_free(pd);
}


ND_ProtoInfo     *
nd_proto_info_new(ND_Protocol *proto, guint nesting)
{
  ND_ProtoInfo *pinf;

  pinf = g_new0(ND_ProtoInfo, 1);
  D_ASSERT_PTR(pinf);
  if (!pinf)
    return NULL;

  D(("Creating proto info for %s/%i\n", proto->name, nesting));
  
  nd_proto_inst_init(&pinf->inst, proto, nesting);
  pinf->registry = g_hash_table_new(g_str_hash, g_str_equal);

  return pinf;
}


void              
nd_proto_info_free(ND_ProtoInfo *pinf)
{
  D(("Cleaning up ND_ProtoInfo\n"));
  g_hash_table_destroy(pinf->registry);
  g_free(pinf);
}


void              
nd_proto_info_set_data(ND_ProtoInfo *pinf, const char *key, void *data)
{
  char *key_dup;

  if (!pinf || !key)
    return;

  D(("Setting %s to %p in proto info for %s/%i\n",
     key, data, pinf->inst.proto->name, pinf->inst.nesting));

  key_dup = g_strdup(key);

  g_hash_table_insert(pinf->registry, key_dup, data);
}


void             *
nd_proto_info_get_data(const ND_ProtoInfo *pinf, const char *key)
{
  void       *result;

  if (!pinf)
    return NULL;

  result = g_hash_table_lookup(pinf->registry, key);

  if (!result)
    D(("Trace table lookup for key %s failed.\n", key));

  return result;
}


void             *
nd_proto_info_remove_data(ND_ProtoInfo *pinf, const char *key)
{
  void       *result;

  result = g_hash_table_lookup(pinf->registry, key);
  g_hash_table_remove(pinf->registry, key);

  return result;
}


void           
nd_proto_info_field_set_state(ND_ProtoInfo *pinf,
			      ND_ProtoField *field,
			      ND_ProtoFieldState state)
{
  GtkWidget *button;


  /* Note -- we cannot store any information for a single field
     in the field structure directly, since the field structures
     are only templates, but the actual buttons exist once per
     trace file. */

  if (!pinf || !field)
    return;

  button = nd_proto_info_get_data(pinf, nd_proto_field_to_string(field));
  
  D_ASSERT_PTR(button);
  if (!button)
    return;
  
  switch (state)
    {
    case ND_FIELD_STATE_ERROR:
      /* Paint the field in a gorgeous bright red: */
      nd_gui_widget_set_color(button, ND_COLOR_RED);

      break;
    case ND_FIELD_STATE_UNKN:
      nd_gui_widget_set_color(button, ND_COLOR_YELLOW);
      break;

    default:
      nd_gui_widget_set_color(button, ND_COLOR_BG);
    }
}
