/* tcl_pbuild.c
  
   The "pdu" Tcl command and the "pdu" Tcl object type are implemented here.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
#include "util-tcl.h"

static pcb_t *pcb;

/********************************************************************
 *		      The "pdu" Tcl object type                     *
 ********************************************************************/

/* Forward declarations */
static void update_string(Tcl_Obj *);
static void free_int_rep(Tcl_Obj *);
static int set_from_any(Tcl_Interp *, Tcl_Obj *);

static Tcl_ObjType tclPDUType = {
    .name = "pdu",
    .freeIntRepProc = &free_int_rep,
    .dupIntRepProc = NULL,
    .updateStringProc = &update_string,
    .setFromAnyProc = &set_from_any
};

static int
set_from_any(Tcl_Interp *interp, Tcl_Obj *objPtr _U_)
{
    if (interp)
	Tcl_SetResult(interp,
		      "PDU object internal representation can't be created "
		      "from string", TCL_VOLATILE);

    return TCL_ERROR;
}

static void
update_string(Tcl_Obj *obj)
{
    pcb_t *pcb;

#ifdef DEBUG
    printf("pdu_obj's update_string() called\n");
    printf("  PDU object at %p\n", obj);
    printf("  PCB structure at %p\n", obj->internalRep.otherValuePtr);
#endif

    pcb = obj->internalRep.otherValuePtr;

    obj->bytes = ckalloc(strlen(pcb->def) + 1);
    if (!obj->bytes)
	return;

    strlcpy(obj->bytes, pcb->def, strlen(pcb->def) + 1);
    obj->length = strlen(pcb->def);
}

static void 
free_int_rep(Tcl_Obj *obj)
{
#ifdef DEBUG
    printf("pdu_obj's free_int_rep() called\n");
    printf("  PDU object at %p\n", obj);
    printf("  PCB structure at %p\n", obj->internalRep.otherValuePtr);
#endif

    pcb_destroy(obj->internalRep.otherValuePtr);
    free(obj->internalRep.otherValuePtr);
}

/********************************************************************
 *                              pdu new                             *
 ********************************************************************/

static int
process_options(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    int i, index;
    char *pdudef = NULL;
    double delay;
    char errbuf[PDU_ERRBUF_SIZE];
    static const char *options[] = {
	"-o", "-delay", "-rate", NULL
    };
    enum options {
	OPT_SPEAKER, OPT_DELAY, OPT_RATE
    };

    /*
     * Parse command-line arguments.
     */
    for (i = 1; i < objc && *Tcl_GetString(objv[i]) == '-'; i++) {
	if (Tcl_GetIndexFromObj(interp, objv[i], options, "option", 0, &index)
	    != TCL_OK)
	    return -1;

	switch ( (enum options) index) {
	case OPT_SPEAKER:
	    if (++i >= objc) {
		Tcl_WrongNumArgs(interp, 1, objv, NULL);
		goto error;
	    }

	    pcb->speaker = lookup_speaker(Tcl_GetString(objv[i]) );
	    if (!pcb->speaker) {
		nexp_error(interp, "No speaker named \"%s\". Use "
			   "\"spawn_network -info\" to find out existing "
			   "speakers.", Tcl_GetString(objv[i]) );
		goto error;
	    }
	    break;
	case OPT_DELAY:
            if (++i >= objc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-d seconds");
		goto error;
	    }

	    if (Tcl_GetDoubleFromObj(interp, objv[i], &delay) != TCL_OK)
		goto error;

	    pcb->delay.tv_sec = delay;
	    pcb->delay.tv_usec = (delay - pcb->delay.tv_sec)*1000000UL;

	    break;
	case OPT_RATE:
	    if (++i >= objc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-r PPS");
		goto error;
	    }

	    /* Convert a packets-per-second rate to a usecs delay */
	    if (Tcl_GetDoubleFromObj(interp, objv[i], &delay) != TCL_OK)
		goto error;

	    if (delay == 0.0) {
		nexp_error(interp, "Rate can't be 0 packets per second.");
		goto error;
	    }

	    delay = 1/delay;

	    pcb->delay.tv_sec = delay;
	    pcb->delay.tv_usec = (delay - pcb->delay.tv_sec)*1000000UL;
	    break;
	}
    }

    /*
     * We treat whatever is left on the command line, i.e. anything that
     * is not an option (anything that doesn't start with '-'), as a PDU
     * definition.
     */
    pdudef = copy_objv(objc - i, &objv[i]);
    if (!pdudef) {
	nexp_error(interp, "A PDU definition is required");
	goto error;
    }

#ifdef DEBUG
    printf("PDU definition = %s\n", pdudef);
#endif

    if ( (pcb->pdu = pb_parsedef(pdudef, errbuf) ) == NULL) {
	nexp_error(interp, "%s", errbuf);
	goto error;
    }

    pcb->def = pdudef;

    return 0;

error:
    if (pdudef)
	free(pdudef);

    return -1;
}

/* see Tcl_SetListObj() in tclListObj.c and Tcl_SetByteArrayObj()
 * in tclBinary.c */
static int
tcl_pdu_new(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    int retval;
    Tcl_Obj *obj;

    pcb = xmalloc(sizeof *pcb);

    memset(pcb, 0, sizeof(*pcb) );

    retval = process_options(interp, objc, objv);
    if (retval == -1)
	goto error;

    /*
     * "pdu new" produces a PDU that will not be used "on the fly". Therefore
     * we need to set this flag so any use of this PDU (by the "send_network"
     * command, for example) doesn't destroy the PDU.
     */
    pcb->keep_after_send = 1;

    obj = Tcl_NewObj();
    obj->typePtr = &tclPDUType;
    obj->internalRep.otherValuePtr = pcb;
    Tcl_InvalidateStringRep(obj);

    Tcl_SetObjResult(interp, obj);

    return TCL_OK;

error:
    return TCL_ERROR;
}

/********************************************************************
 *                            pdu delete                            *
 ********************************************************************/

static int
tcl_pdu_delete(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *obj;
    const char *pdu_name;

    if (objc != 2) {
	nexp_error(interp, "usage: pdu delete <pdu>");
	return TCL_ERROR;
    }

    pdu_name = Tcl_GetString(objv[1]);

    obj = Tcl_GetVar2Ex(interp, pdu_name, NULL, TCL_LEAVE_ERR_MSG);
    if (!obj)
	return TCL_ERROR;

    if (obj->typePtr != &tclPDUType) {
	nexp_error(interp, "\"%s\" is not a PDU object", pdu_name);
	return TCL_ERROR;
    }

    return Tcl_UnsetVar(interp, pdu_name, TCL_LEAVE_ERR_MSG);
}

/********************************************************************
 *                              pdu info                            *
 ********************************************************************/

static int
tcl_pdu_info(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *obj;
    const char *pdu_name;
    pcb_t *pcb;

    if (objc != 2) {
	nexp_error(interp, "usage: pdu info <pdu>");
	return TCL_ERROR;
    }

    pdu_name = Tcl_GetString(objv[1]);

    obj = Tcl_GetVar2Ex(interp, pdu_name, NULL, TCL_LEAVE_ERR_MSG);
    if (!obj)
	return TCL_ERROR;

    if (obj->typePtr != &tclPDUType) {
	nexp_error(interp, "\"%s\" is not a PDU object", pdu_name);
	return TCL_ERROR;
    }

#ifdef DEBUG
    printf("PDU object at %p\n", obj);
    printf("PCB structure at %p\n", obj->internalRep.otherValuePtr);
    printf("  Reference count is %d\n", obj->refCount);
#endif

    pcb = obj->internalRep.otherValuePtr;

    pcb_dump(pcb, vflag);
    pb_dumppdu(pcb->pdu);

    return TCL_OK;
}

/********************************************************************
 *                            pdu count                             *
 ********************************************************************/

static int
tcl_pdu_count(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *obj;
    const char *pdu_name;
    pcb_t *pcb;

    if (objc != 2) {
	nexp_error(interp, "usage: pdu count <pdu>");
	return TCL_ERROR;
    }

    pdu_name = Tcl_GetString(objv[1]);

    obj = Tcl_GetVar2Ex(interp, pdu_name, NULL, TCL_LEAVE_ERR_MSG);
    if (!obj)
	return TCL_ERROR;

    if (obj->typePtr != &tclPDUType) {
	nexp_error(interp, "\"%s\" is not a PDU object", pdu_name);
	return TCL_ERROR;
    }

    pcb = (pcb_t *) obj->internalRep.otherValuePtr;

    obj = Tcl_NewIntObj(pb_permutation_count(pcb->pdu) );
    Tcl_SetObjResult(interp, obj);

    return TCL_OK;
}

/********************************************************************
 *                            pdu build                             *
 ********************************************************************/

static int
tcl_pdu_build(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *pdu_obj, *packet_obj;
    pcb_t *pcb;
    size_t len;
    uint8_t *packet;
    const char *pdu_name;
    struct pcap_pkthdr pkthdr;
    int ll_type;

    if (objc != 2) {
	nexp_error(interp, "usage: pdu build <pdu>");
	return TCL_ERROR;
    }

    pdu_name = Tcl_GetString(objv[1]);

    pdu_obj = Tcl_GetVar2Ex(interp, pdu_name, NULL, TCL_LEAVE_ERR_MSG);
    if (!pdu_obj)
	return TCL_ERROR;

    if (pdu_obj->typePtr != &tclPDUType) {
	nexp_error(interp, "\"%s\" is not a PDU object", pdu_name);
	return TCL_ERROR;
    }

    pcb = (pcb_t *) pdu_obj->internalRep.otherValuePtr;

    len = pb_len(pcb->pdu);

    /*
     * Two extra bytes just in case the packet is an Ethernet frame,
     * in which case we need to adjust the location where the packet
     * will be built so the IP header is word-aligned.
     */
    packet = xmalloc(len);

    pb_build(pcb->pdu, packet, NULL);

    pkthdr.caplen = len;
    pkthdr.len = pkthdr.caplen;
    gettimeofday(&pkthdr.ts, NULL);

    /* XXX - ugly hack */
    ll_type = !strcmp(pb_getname(pcb->pdu), "ether") ? DLT_EN10MB : DLT_RAW;

    packet_obj = Tcl_NewPacketObj(packet, &pkthdr, ll_type);

    Tcl_SetObjResult(interp, packet_obj);

    free(packet);

    return TCL_OK;
}

/********************************************************************
 *                            pdu list                              *
 ********************************************************************/

static int
tcl_pdu_list(Tcl_Interp *interp _U_, int objc _U_, Tcl_Obj * const *objv _U_)
{
    pdu_list();

    return TCL_OK;
}

/********************************************************************
 *                            pdu send                              *
 ********************************************************************/

static int
tcl_pdu_send(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *pdu_obj;
    pcb_t *pcb;
    size_t len;
    uint8_t *packet;
    const char *pdu_name;

    if (objc != 2) {
	nexp_error(interp, "usage: pdu send <pdu>");
	return TCL_ERROR;
    }

    pdu_name = Tcl_GetString(objv[1]);

    pdu_obj = Tcl_GetVar2Ex(interp, pdu_name, NULL, TCL_LEAVE_ERR_MSG);
    if (!pdu_obj)
	return TCL_ERROR;

    if (pdu_obj->typePtr != &tclPDUType) {
	nexp_error(interp, "\"%s\" is not a PDU object", pdu_name);
	return TCL_ERROR;
    }

    pcb = (pcb_t *) pdu_obj->internalRep.otherValuePtr;

    len = pb_len(pcb->pdu);

    packet = xmalloc(len);

    pb_build(pcb->pdu, packet, NULL);

    nexp_pdu_output(pcb, packet, len);

    free(packet);

    return TCL_OK;
}

static int
NExp_PDUObjCmd(ClientData clientData _U_, Tcl_Interp *interp, int objc,
	       Tcl_Obj * const *objv)
{
    int index, retval;
    static const char *subcmds[] = {
	"exists", "new", "delete", "info", "count", "build", "list", "send",
	NULL
    };
    enum subcmds {
	SUBCMD_EXISTS, SUBCMD_NEW, SUBCMD_DELETE, SUBCMD_INFO, SUBCMD_COUNT,
	SUBCMD_BUILD, SUBCMD_LIST, SUBCMD_SEND
    };

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "subcmd pduName ?arg ...?");
	return TCL_ERROR;
    }

    if (Tcl_GetIndexFromObj(interp, objv[1], subcmds, "subcmd", 0, &index)
	!= TCL_OK)
	return TCL_ERROR;

    switch ( (enum subcmds) index) {
    case SUBCMD_EXISTS: {
	Tcl_Obj *obj;

	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "pduName");
	    return TCL_ERROR;
	}

	obj = Tcl_ObjGetVar2(interp, objv[2], NULL, TCL_LEAVE_ERR_MSG);
	if (!obj || obj->typePtr != &tclPDUType) {
	    /* Variable does not exist or it's not of our type */
	    obj = Tcl_NewBooleanObj(0);
	} else {
	    obj = Tcl_NewBooleanObj(1);
	}

	Tcl_SetObjResult(interp, obj);
	return TCL_OK;
    }
    case SUBCMD_NEW:
	retval = tcl_pdu_new(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_DELETE:
	retval = tcl_pdu_delete(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_INFO:
	retval = tcl_pdu_info(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_COUNT:
	retval = tcl_pdu_count(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_BUILD:
	retval = tcl_pdu_build(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_LIST:
	retval = tcl_pdu_list(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_SEND:
	retval = tcl_pdu_send(interp, objc - 1, &objv[1]);
	break;
    }

    return retval;
}

static struct nexp_cmd_data cmd_data[] = {
    {"pdu", NExp_PDUObjCmd, NULL, 0, 0},

    {NULL, NULL, NULL, 0, 0}
};

void
nexp_init_pdu_cmd(Tcl_Interp *interp)
{
    Tcl_RegisterObjType(&tclPDUType);

    nexp_create_commands(interp, cmd_data);
}
